(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                 auto.mli                                 *)
(****************************************************************************)

open Names
open Term
open Proof_trees
open Tacmach
open Clenv



type auto_tactic = 
    Res_pf     of constr * unit clausenv (* Hint Apply *)
  | ERes_pf    of constr * unit clausenv (* Hint EApply *)
  | Give_exact of constr                  
  | Res_pf_THEN_trivial_fail of constr * unit clausenv (* Hint Immediate *)
  | Unfold_nth of constr                 (* Hint Unfold *)
  | Extern of CoqAst.t                   (* Hint Extern *)

type pri_auto_tactic = 
  { hname : identifier    (* name of the hint *)
  ; pri  : int            (* A number between 0 and 4, 4 = lower priority *)
  ; pat  : constr option  (* A pattern for the concl of the Goal *)
  ; code : auto_tactic    (* the tactic to apply when the concl matches pat *)
  }

type stored_data = pri_auto_tactic 
and search_entry =
    stored_data list * stored_data list * stored_data Btermdn.t

module Hint_db :
  sig
    type t
    val empty : t
    val find : constr -> t -> search_entry
    val map_all : constr -> t -> pri_auto_tactic list
    val map_auto : constr * constr -> t -> pri_auto_tactic list
    val add_one : constr * pri_auto_tactic -> t -> t
    val add_list : (constr * pri_auto_tactic) list -> t -> t
    val iter : (constr -> stored_data list -> unit) -> t -> unit
  end

type frozen_hint_db_table = (string, Hint_db.t) Fmavm.frozen_t
and hint_db_table = (string, Hint_db.t) Fmavm.t

val searchtable : hint_db_table

(* [make_exact_entry hint_name (c, ctyp)] *)
(* hint_name is the name of then hint *)
(* c is the term given as an exact proof to solve the goal *)
(* ctyp is the type of hc *)
val make_exact_entry :
  identifier -> constr * constr -> constr * pri_auto_tactic

(* [make_apply_entry (eapply,verbose) name (c,cty)] *)
(* eapply is true if this hint will be used only with EApply *)
(* name is the name of then hint *)
(* c is the term given as an exact proof to solve the goal *)
(* cty is the type of hc *)
val make_apply_entry :
  bool * bool -> identifier -> constr * constr -> constr * pri_auto_tactic

(* A constr which is Hint'ed will be:
   (1) used as an Exact, if it does not start with a product
   (2) used as an Apply, if its HNF starts with a product, and
       has no missing arguments.
   (3) used as an EApply, if its HNF starts with a product, and
       has missing arguments.
 *)
val make_resolves :
  identifier -> bool * bool -> constr * constr -> (constr * pri_auto_tactic) list

(* [make_resolve_hyp hname htyp] *)
(* used to add an hypothesis to the local hint database *)
(* Never raises an User_exception *)
(* If the hyp cannot be used as a Hint, the empty list is returned *)
val make_resolve_hyp : identifier -> constr -> (constr * pri_auto_tactic) list

(* [make_extern name pri pattern tactic_ast] *)
val make_extern :
  identifier -> int -> constr -> CoqAst.t -> constr * pri_auto_tactic

(* Create a Hint database from the pairs (name, constr) *)
(* Useful to take the current goal hypotheses as hints *)
val make_local_hint_db : constr signature -> Hint_db.t

val priority : (int * 'a) list -> 'a list

val default_search_depth : int ref
(* Try unification with the precompiled clause, then use registered Apply *)
val unify_resolve : (constr * unit clausenv) -> tactic


(* The Auto tactic *)

val auto : int -> string list -> tactic

(* auto with default search depth and with the hint database "core" *)
val default_auto : tactic

(* auto with all hint databases except the "v62" compatibility database *)
val full_auto : int -> tactic

(* auto with default search depth and with all hint databases 
   except the "v62" compatibility database *)
val default_full_auto : tactic

(* The hidden version of auto *)
val h_auto : tactic_arg list -> tactic

(* Trivial *)
val trivial : string list -> tactic
val full_trivial : tactic
val h_trivial : tactic_arg list -> tactic

(**************************************************************************)
(*  Tout ce qui suit n'est pas encore  jour pour l'instant -- Papageno   *)
(**************************************************************************)

(* DAuto *)
val dauto : int option * int option -> tactic
val default_search_decomp : int ref
val default_dauto : tactic

(* SuperAuto *)

type autoArguments =
    UsingTDB       
  | Destructing   

val superauto    : int -> identifier list -> autoArguments list -> tactic


(* $Id: auto.mli,v 1.17 1999/06/29 07:47:40 loiseleu Exp $ *)
