(****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                 caml.ml                                  *)
(****************************************************************************)

(**************************************************************************)
(*        production of Caml Light syntax                                 *)
(**************************************************************************)

open Pp;;
open Std;;
open More_util;;
open System;;
open Names;;
open Vernacinterp;;
open Ml_import;;
open Mlterm;;
open Genpp;;

(**************************************************************************)
(*          translation of type expressions and inductive types           *)
(**************************************************************************)

let print_typeid = function
    TYPEparam id -> [< 'sTR"'" ; 'sTR(string_of_id id) >]
  | TYPEname id  -> [< 'sTR(string_of_id id) >]
;;


(* caml_of_type : bool -> MLtype -> std_ppcmds
 * [caml_of_type b t] formats the type expression t.
 * b=true if we need parentheses around the result. *)

let rec caml_of_type paren_p = function
     TYarr(t1,t2) ->
       	[< open_par paren_p ;
	   caml_of_type true t1 ; 'sTR" -> " ;
           caml_of_type false t2 ;
	   close_par paren_p
        >]

  | TYapp cl ->
      	let n = List.length cl in
	if n=1 then 
	  caml_of_type false (List.hd cl)
        else
	  [< open_par paren_p ;
	     uncurry_list (fun c -> caml_of_type false c)
      	       	       	  (List.tl cl) ; 'sTR " " ;
             caml_of_type false (List.hd cl) ;
	     close_par paren_p
	  >]

  | TYvar tid ->
      	[< print_typeid tid >]

  | TYglob id ->
      	[< 'sTR(string_of_id id) >]
;;


(* caml_of_one_inductive : MLind -> std_ppcmds
 * [caml_of_one_inductive mi] formats the declaration of mutual
 * inductive mi. *)

let caml_of_one_inductive (pl,name,cl) =
  let fmt_constructor (id,l) =
    [< 'sTR(string_of_id id) ;
       match l with
         [] -> [< >] 
       | _  -> [< 'sTR" of " ;
      	       	  prlist_with_sep (fun () -> [< 'sPC ; 'sTR"* " >]) 
      	       	       	          (fun s -> caml_of_type true s) l
	       >]
    >] in

  [< uncurry_list (fun id -> [< 'sTR"'" ; 'sTR(string_of_id id) >]) pl ;
     if pl=[] then [< >] else [< 'sTR" " >] ;
     'sTR(string_of_id name) ; 'sTR" =" ; 'fNL ;
     v 0 [< prlist_with_sep (fun () -> [< 'fNL ; 'sTR"| ">])
                            (fun c -> hOV 0 (fmt_constructor c))
		            cl
         >]
  >]
;;

let caml_of_inductive il =
  [< 'sTR"type " ;
     prlist_with_sep 
      	  (fun () -> [< 'fNL ; 'sTR"and " >])
       	  (fun i -> caml_of_one_inductive i)
	  il ;
     'fNL
  >]
;;


(**************************************************************************)
(*                  production of caml syntax for terms                   *)
(**************************************************************************)

let abst = function
    [] -> [< >]
  | l  -> [< 'sTR"fun " ;
             prlist_with_sep (fun  ()-> [< 'sTR" " >])
      	       	       	     (fun id -> [< 'sTR(string_of_id id) >]) l ;
             'sTR" -> " >]
;;

let pr_binding = function
    [] -> [< >]
  | l  -> [< 'sTR" " ; prlist_with_sep (fun () -> [< 'sTR" " >])
      	       	       	(fun id -> [< 'sTR(string_of_id id) >]) l >]
;;


(* pp_mlast : identifier list -> bool -> std_ppcmds list
 *	      -> MLast -> std_ppcmds
 * [pp_mlast idl b args t] formats the Ml term (t a1...am)
 * in the de Bruijn environment idl, where args=[a1...am].
 * b=true if we need parentheses around the result. *)

let rec pp_mlast idl paren_p args = 

  let apply st = match args with
     [] -> st
   | _  -> hOV 2 [< open_par paren_p ; st ; 'sTR" " ;
                    prlist_with_sep (fun () -> [< 'sPC >]) (fun s -> s) args ;
                    close_par paren_p
                 >] in

  function
    MLrel n ->
      	 apply [< 'sTR(string_of_id (List.nth idl (n-1))) >]

  | MLapp (h, args') ->
      	 let stl = List.map (fun t -> pp_mlast idl true [] t) args' in
         pp_mlast idl paren_p (stl@args) h

  | MLlam _ as t ->
      	 let fl,t' = collect_lambda t in
	 let st = [< abst (List.rev fl) ; pp_mlast (fl@idl) false [] t' >] in
	 if args=[] then
           [< open_par paren_p ; st ; close_par paren_p >]
         else
           apply [< 'sTR"(" ; st ; 'sTR")" >]

  | MLglob id ->
      	 apply [< 'sTR(string_of_id id) >]
	
  | MLcons (_,id,args') ->
      	 if args'=[] then
	   [< 'sTR(string_of_id id) >]
	 else
	   [< open_par paren_p ; 'sTR(string_of_id id) ; 'sTR " " ;
	      uncurry_list2 (fun t -> pp_mlast idl true [] t) args' ;
	      close_par paren_p
           >]

  | MLcase (t, pv) ->
      	 apply
      	 [< if args<>[] then [< 'sTR"(" >]  else open_par paren_p ;
      	    v 0 [< 'sTR"match " ; pp_mlast idl false [] t ; 'sTR" with" ;
		   'fNL ; 'sTR"  " ;
		   pp_mlpat idl pv >] ;
	    if args<>[] then [< 'sTR")" >] else close_par paren_p >]

  | MLfix (x_0,x_1,x_2,x_3) ->
      	 pp_mlfix idl paren_p (x_0,x_1,x_2,x_3) args

  | MLexn id ->
      	 [< open_par paren_p ; 'sTR"failwith \"" ; print_id id ; 'sTR"\"" ;
	    close_par paren_p >]

and pp_mlfix idl paren_p (j,in_p,fid,bl) args =
  let idl' = (List.rev fid)@idl in
  [< open_par paren_p ; v 0 [< 'sTR"let rec " ;
       prlist_with_sep
      	  (fun () -> [< 'fNL ; 'sTR"and " >])
	  (fun (fi,ti) -> pp_mlfunction idl' fi ti)
	  (List.combine fid bl) ;
       'fNL ;
       if in_p then 
      	 hOV 2 [< 'sTR"in " ; 'sTR(string_of_id (List.nth fid j)) ;
                  if args<>[] then
                    [< 'sTR" "; prlist_with_sep (fun () -> [<'sTR" ">])
                                 (fun s -> s) args >]
                  else [< >]
      	       >]
       else 
         [< >] >] ;
     close_par paren_p >]

and pp_mlfunction idl f t =
  let bl,t' = collect_lambda t in
  let is_function pv =
    let ktl = map_vect_list (fun (_,l,t0) -> (List.length l,t0)) pv in
    (not((List.exists (fun (k,t0) -> occurs (k+1) t0) ktl))) in

  match t' with 
    MLcase(MLrel 1,pv) ->
       if is_function pv then
	 [< 'sTR(string_of_id f) ; pr_binding (List.rev (List.tl bl)) ;
       	    'sTR" = function" ; 'fNL ;
	    v 0 [< 'sTR"  " ; pp_mlpat (bl@idl) pv >] >]
       else
         [< 'sTR(string_of_id f) ; pr_binding (List.rev bl) ; 'sTR" = match " ;
	    'sTR(string_of_id (List.hd bl)) ; 'sTR" with" ; 'fNL ;
	    v 0 [< 'sTR"  " ; pp_mlpat (bl@idl) pv >] >]
	   
  | _ -> [< 'sTR(string_of_id f) ; pr_binding (List.rev bl) ;
	    'sTR" =" ; 'fNL ; 'sTR"  " ;
	    hOV 2 (pp_mlast (bl@idl) false [] t') >]

and pp_mlpat idl pv =
  let pp_one_pat (name,ids,t) =
      let paren_p = match t with
                      MLlam _  -> true
                    | MLcase _ -> true
                    | _        -> false in

      hOV 2 [< 'sTR(string_of_id name) ;
      	       begin match ids with 
		   [] -> [< >]
		 | _  -> 
      		     [< 'sTR " " ;
			uncurry_list2 (fun id -> [< 'sTR(string_of_id id) >])
			  (List.rev ids) >]
	       end;
	       'sTR" ->" ; 'sPC ; pp_mlast (ids@idl) paren_p [] t
            >]

  in [< prvect_with_sep (fun () -> [< 'fNL ; 'sTR"| " >])
                        (fun p -> pp_one_pat p)
	                pv >]
;;


(* caml_of_decl : MLdecl -> std_ppcmds
 * [caml_of_decl d] formats the declaration d. *)

let caml_of_decl = function
    DECLtype il -> caml_of_inductive il

  | DECLabbrev (id, idl, t) ->
          [< 'sTR"type" ; if idl<>[] then [< 'sTR" " >] else [< >] ;
	     uncurry_list (fun id -> [< 'sTR"'" ; 'sTR(string_of_id id) >]) idl ;
      	     'sTR" " ; 'sTR(string_of_id id) ; 'sTR" == " ;
	      caml_of_type false t >]

  | DECLglob (id, MLfix (n,_,idl,l)) ->
      let id' = List.nth idl n in
      	if id = id' then
	  [<  hOV 2 (pp_mlfix [] false (n,false,idl,l) []) >]
	else
	  [< 'sTR"let " ; 'sTR(string_of_id id) ; 'sTR" =" ; 'fNL ;
	     v 0 [< 'sTR"  " ; 
		    hOV 2 (pp_mlfix [] false (n,true,idl,l) []) ; 'fNL >] >]
  
  | DECLglob (id, t) ->
      	  [< 'sTR"let " ; pp_mlfunction [] id t ; 'fNL >]       
;;


(**************************************************************************)
(*             translation of an environment into caml syntax.            *)
(**************************************************************************)

let caml_of_env env =
  prlist (fun d -> [< caml_of_decl d ; 'sTR";;" ; 'fNL ; 'fNL >]) env
;;

module Caml_renaming =
  struct
    let rename_type_parameter = caml_name_of;;
    let rename_type           = caml_name_of;;
    let rename_term           = caml_name_of;;
    let rename_global_type    = caml_name_of;;
    let rename_global_constructor = caml_name_of;;
    let rename_global_term    = caml_name_of;;
  end;;

module Caml_pp : MLPP = 
  struct
    let opt = Optimise.optimise;;
    let suffixe = ".ml";;
    let cofix = false;;
    let pp_of_env = caml_of_env;;
    module Renaming = Caml_renaming;;
  end;;

module Caml_pp_file = Pp_to_file(Caml_pp);;


(**************************************************************************)
(*            Write Caml File filename [ ident1 ... identn ].             *)
(**************************************************************************)

vinterp_add("WRITECAMLLIGHTFILE",
  function VARG_STRING file :: rem ->
    let prm = parse_param rem in
      (fun () -> Caml_pp_file.write_extraction_file file prm)
  | _ -> anomaly "WRITECAMLLIGHTFILE called with bad arguments.")
;;

(* $Id: caml.ml,v 1.14 1999/06/29 07:48:04 loiseleu Exp $ *)
