 (****************************************************************************)
(*                 The Calculus of Inductive Constructions                  *)
(*                                                                          *)
(*                                Projet Coq                                *)
(*                                                                          *)
(*                     INRIA        LRI-CNRS        ENS-CNRS                *)
(*              Rocquencourt         Orsay          Lyon                    *)
(*                                                                          *)
(*                                 Coq V6.3                                 *)
(*                               July 1st 1999                              *)
(*                                                                          *)
(****************************************************************************)
(*                                 point.ml                                 *)
(****************************************************************************)


open Std;;
open Initial;;
open Names;;
open Vectops;;
open Generic;;
open Term;;
open Reduction;;
open Himsg;;
open Mach;;
open Proof_trees;;
open Trad;;
open Tacmach;;
open Tactics;;
open Pp;;

open Clenv;;
open Termenv;;
open Tactics;;
open Libobject;;
open Library;;
open Vernacinterp;;
open Pattern;;
open Tacticals;;
open Tactics;;
open Termast;;
open CoqAst;;
open Ast;;
open Pcoq;;


(* The family of tactics Point is intended to do proof by pointing.
It works exactly as DHyp, apart from two main differences it is
possible to indicate a ``continuation'' for the hints, as well as a
``pointed metavariable''. For example, it is possible to decleare the
following hint:

Hint Point Conclusion prod_left  ?1->?2 1 
     [ 1       Hypothesis 1 ] [<:tactic:<Intro>>].
      ||       \-----------/
      ||            ||
      \/            ||
  pointed meta      \/
                 continuation

The hint prod_left says that once applied, the tactic should proceed
recursively with the first hypothesis of the context (ie, the last
hypothesis introduced). This is the meaning of ``Hypothesis 1'' in the
argument [ 1 Hypothesis 1 ]. Similarly, the hint:

Hint Point Conclusion prod_right  ?1->?2 1 
  [ 2 Conclusion ] 
  [<:tactic:<Intro>>].

says that after applying the rule prod_right, the tactic Point should
try to match the conclusion with a pattern from the data base. This is
the meaning of ``Conclusion'' in the expression [ 2 Conclusion ].

The argument of the tactic Point is a term containing a mark. This
mark is internally represented as an abstraction term convertible to
the identity. In this way, the tactic :

Point #[nat]->nat->nat.

means that the marked term (ie, nat) should only match with those
patterns where the metavariable pointed in the patterns (ie, ?1 for
rule prod_left) has been matched with a term containing the mark. This
enables to reduce the number of applicable tactics. In this example,
the tactic

Point #[nat]->nat->nat.

only matches with the rule prod_left, while 

Point nat->#[nat->nat].


only matches with the rule prod_right.

WARNING: This second feature does not actually work because presently
abstractions are expanded by some of the tactics.

Eduardo.
30/6/98

*)



(******************************************)
(* Information associated to the patterns *)
(******************************************)

(* two patterns - one for the type, and one for the type of the type *)
type destructor_pattern =
    {d_typ: constr; d_sort: constr};;

type ('a,'b) location = HYP of 'a | CONCL of 'b;;

(* hypothesis patterns might need to do matching on the conclusion, too.
 * conclusion-patterns only need to do matching on the hypothesis *)
type located_destructor_pattern =
    (bool * destructor_pattern * destructor_pattern,
     destructor_pattern) location
;;

type destructor_data =
    {d_pat    : located_destructor_pattern;
     d_mark   : int;     (* number of meta-variable containing mark *)
     d_pri    : int;     (* priority *)
     d_code   : Ast.act; (* should be of phylum tactic *)
     d_result : (int,unit) location option list} (* the result of the tactic *)
;;


(********************************************)
(* Definition of the discrimination network *)
(********************************************)

type t = (identifier,destructor_data) Nbtermdn.t;;
type frozen_t = (identifier,destructor_data) Nbtermdn.frozen_t;;

let tactab = (Nbtermdn.create {Nbtermdn.name_ord = id_ord} : t);;

let lookup pat = Nbtermdn.lookup tactab pat;;

let init () = Nbtermdn.empty tactab;;

let freeze () = Nbtermdn.freeze tactab;;
let unfreeze fs = Nbtermdn.unfreeze fs tactab;;

let rollback f x =
    let fs = freeze()
    in try f x
       with e -> (unfreeze fs; raise e)
;;

let add (na,dd) =
  let pat = match dd.d_pat with
             HYP(_,p,_) -> p.d_typ
           | CONCL p    -> p.d_typ
  in
  if Nbtermdn.in_dn tactab na 
  then (mSGNL [< 'sTR "Warning [Overriding Destructor Entry " ; 
                 'sTR (string_of_id na) ; 'sTR"]" >];
         Nbtermdn.remap tactab na (pat,dd))
  else Nbtermdn.add tactab (na,(pat,dd))
;;

Summary.declare_summary "point-hyp-concl"
  {Summary.freeze_function = freeze;
   Summary.unfreeze_function = unfreeze;
   Summary.init_function = init}

;;

let cache_dd (_,(na,dd)) =
    (try add (na,dd)
    with _ -> anomalylabstrm "Point.add"
              [< 'sTR"The code which adds destructor hints broke;" ;
                 'sPC ; 'sTR"this is not supposed to happen" >])
;;

let specification_dd x = x;;

type destructor_data_object = identifier * destructor_data;;

let (( inDD:destructor_data_object->obj),
     (outDD:obj->destructor_data_object)) =
    declare_object ("POINT-HYP-CONCL-DATA",
                    {load_function = (fun _ -> ());
                     cache_function = cache_dd;
                     specification_function = specification_dd});;

let add_destructor_hint na pat (mark,rl) pri code =
    add_anonymous_object
    (inDD(na,{d_pat  = pat;
              d_mark = mark;
              d_pri  = pri;
              d_code = code;
              d_result = rl}))
;;

vinterp_add
  ("HintPoint",
  function [VARG_IDENTIFIER na; VARG_AST location; VARG_COMMAND patcom;
       VARG_NUMBER pri; VARG_NUMBER subterm;
       VARG_AST (Node(_,"result_list",results));
       VARG_AST tacexp ] ->
  let loc =
    match location with
        Node(_,"CONCL",[])          -> CONCL()
      | Node(_,"DiscardableHYP",[]) -> HYP true
      | Node(_,"PreciousHYP",[])    -> HYP false
      | _ -> assert false
  in
  let rl =
    List.map (function
                  Node(_,"CONCL",[])       -> Some(CONCL())
                | Node(_,"HYP",[Num(_,n)]) -> Some(HYP n)
                | Node(_,"NONE",[])        -> None
		| _ -> assert false) 
              results
  in
    fun () ->
      let pat = raw_sopattern_of_compattern (initial_sign()) patcom in
      let code = Ast.to_act_check_vars ["$0",ETast] ETast tacexp in

        add_destructor_hint na
          (match loc with
               HYP b ->
                 HYP(b,{d_typ=pat;d_sort=DOP0(Meta(newMETA()))},
                       {d_typ=DOP0(Meta(newMETA()));
                        d_sort=DOP0(Meta(newMETA()))})
             | CONCL () ->
                 CONCL({d_typ=pat;d_sort=DOP0(Meta(newMETA()))}))
          (subterm,rl)
          pri
          code
    | _ -> assert false)
;;

let match_dpat dp cls gls =
  let cltyp = clause_type cls gls in
    match (cls,dp) with
        (Some id,HYP(_,hypd,concld)) ->
          (somatch None hypd.d_typ cltyp)@
          (somatch None hypd.d_sort (pf_type_of gls cltyp))@
          (somatch None concld.d_typ (pf_concl gls))@
          (somatch None concld.d_sort (pf_type_of gls (pf_concl gls)))
      | (None,CONCL concld) ->
          (somatch None concld.d_typ (pf_concl gls))@
          (somatch None concld.d_sort (pf_type_of gls (pf_concl gls)))
      | _ -> error "ApplyDestructor"
;;

let mark_occurs m =
    occur_opern (Abst (path_of_string"#Point#mark.cci")) m;;

let applyDestructor cls discard dd gls =
  let mvb = match_dpat dd.d_pat cls gls in
  (*
    (try
       if (not (mark_occurs (Listmap.map mvb dd.d_mark)))
       then error "ApplyDestructor: the mark does not occur"
     with Not_found -> error "ApplyDestructor: Not found");
  *)
  let astb = (match cls with
                  Some id -> ["$0", Vast (nvar (string_of_id id))]
                | None -> ["$0",Vast (nvar "$0")])
  in
  (* TODO: find the real location *)
  let tcom = match Ast.eval_act dummy_loc astb dd.d_code with
      Vast x -> x
    | _ -> assert false
  in
  let discard_0 =
    match (cls,dd.d_pat) with
        (Some id,HYP(discardable,_,_)) ->
          if discard & discardable then thin [id] else tclIDTAC
      | (None,CONCL _) -> tclIDTAC
      | _ -> error "ApplyDestructor3"
  in
    (tclTHEN (Tacinterp.interp tcom) discard_0) gls
;;

(* [PointHyp id gls]

   will take an identifier, get its type, look it up in the
   discrimination net, get the destructors stored there, and then try
   them in order of priority.
 *)

let pointHyp discard id gls =
  let hyptyp = clause_type (Some id) gls in
  let ddl    = List.map snd (lookup hyptyp) in
  let sorted_ddl = Sort.list (fun dd1 dd2 -> dd1.d_pri > dd2.d_pri) ddl in
    tclFIRST (List.map (applyDestructor (Some id) discard) sorted_ddl) gls
;;

let cPHyp id gls = pointHyp true id gls;;
let pHyp id gls  = pointHyp false id gls;;

open Tacinterp;;

tacinterp_add("PHyp",(function [IDENTIFIER id] -> pHyp id
			| _ -> assert false));;
tacinterp_add("CPHyp",(function [IDENTIFIER id] -> cPHyp id
			 | _ -> assert false));;

(* [PConcl gls]

   will take a goal, get its concl, look it up in the
   discrimination net, get the destructors stored there, and then try
   them in order of priority.
 *)

let pConcl gls =
  let ddl = List.map snd (lookup (pf_concl gls)) in
  let sorted_ddl = Sort.list (fun dd1 dd2 -> dd1.d_pri > dd2.d_pri) ddl in
    tclFIRST (List.map (applyDestructor None false) sorted_ddl) gls
;;

tacinterp_add("PConcl",(function [] -> pConcl | _ -> assert false));;

(* [PointStart n c gls] change the nth hypothesis (0=conclusion) into c *)

let pointStart cls c gl = change_option c cls gl;;

let dyn_point_start = function
    [INTEGER n;COMMAND c] ->
      (fun gl -> pointStart (nth_clause n gl) (pf_constr_of_com gl c) gl)
  | _ -> invalid_arg "PointStart called with bad arguments";;

let h_pointStart =
  let gentac = hide_tactic "PointStart" dyn_point_start
  in (fun n c -> gentac [INTEGER n;COMMAND c])
;;

(* PointEnd gls] unfolds all the remaining identifiers called Point#mark_i *)

let pointEnd gls =
  let unfolder = unfold_option [([],
                                 Nametab.sp_of_id CCI (id_of_string "mark"))]
  in onAllClauses
    (ifOnClause (fun (cls,t) -> mark_occurs t) 
                unfolder 
               (fun _ -> tclIDTAC)) gls;;


let h_pointEnd = hide_atomic_tactic "PointEnd" pointEnd;;


let rec tclFIRSTWITH = fun tacl g ->
  match tacl with
      []       -> error "FIRSTWITH : no applicable tactic"
    |  t::rest -> (try t g with UserError _ -> tclFIRSTWITH rest g);;

let rec point discard cls gls =
  let ddl = List.map snd (lookup (clause_type cls gls)) in
  let sorted_ddl = Sort.list (fun dd1 dd2 -> dd1.d_pri > dd2.d_pri) ddl in
    match 
      tclFIRSTWITH
        ((List.map (fun dd gls ->
                      Some(applyDestructor cls discard dd gls,dd)) sorted_ddl)
         @[(fun gls -> None)])
        gls
    with
        None -> tclIDTAC gls
      | Some((cl,v),dd) ->

(* now we have a successful destructor, so we use the results field to
   continue *)

          tclTHEN_i 
            (fun _ -> (cl,v))
            (fun i g ->
               match List.nth dd.d_result (i-1) with
                   None -> tclIDTAC g
                 | Some(CONCL()) -> point true None g
                 | Some(HYP n) -> point true (nth_clause n g) g)
            1
            gls
;;

let genPoint discard cls c gls =
  (tclTHEN (tclTHEN (pointStart cls c) (point discard cls)) (pointEnd)) gls;;

let point  = genPoint false;;
let cPoint = genPoint true;;

let point_tac =
  let gentac =
    hide_tactic "Point"
    (function [IDENTIFIER id;COMMAND c] ->
         (fun gl -> point (Some id) (pf_constr_of_com gl c) gl)
       | [COMMAND c] ->
         (fun gl -> point None (pf_constr_of_com gl c) gl)
       | _ -> assert false)
    (*fun p_0 p_1 p_2 -> match p_0,p_1,p_2 with
         (sigma, goal, (_,[IDENTIFIER id;COMMAND c])) ->
           [< 'sTR"Point " ; pr_com sigma goal c ;
              'sTR" in "; 'sTR(string_of_id id) >]
       | (sigma, goal, (_,[COMMAND c])) ->
         [< 'sTR"Point " ; pr_com sigma goal c >]*)
  in (fun p_0 p_1 -> match p_0,p_1 with
          ((Some id), c) -> gentac [IDENTIFIER id;COMMAND c]
        | (None, c) -> gentac [COMMAND c])
;;

let cPoint_tac =
  let gentac =
    hide_tactic "CPoint"
      (function
	   [IDENTIFIER id;COMMAND c] ->
             (fun gl -> cPoint (Some id) (pf_constr_of_com gl c) gl)
         | [COMMAND c] ->
             (fun gl -> cPoint None (pf_constr_of_com gl c) gl)
	 | _ -> assert false)
    (*fun p_0 p_1 p_2 -> match p_0,p_1,p_2 with
         (sigma, goal, (_,[IDENTIFIER id;COMMAND c])) ->
         [< 'sTR"CPoint " ; pr_com sigma goal c ;
            'sTR" in " ; 'sTR(string_of_id id) >]
       | (sigma, goal, (_,[COMMAND c])) ->
         [< 'sTR"CPoint " ; pr_com sigma goal c >]*)
  in (fun p_0 p_1 -> match p_0,p_1 with
        ((Some id), c) -> gentac [IDENTIFIER id;COMMAND c]
      | (None, c) -> gentac [COMMAND c])
;;

(* $Id: point.ml,v 1.23 1999/10/29 23:19:38 barras Exp $ *)
