/* $Header: /cvs/gnome/gIDE/src/gI_compile.c,v 1.3 1999/12/05 19:10:48 jpr Exp $ */
/* gIDE
 * Copyright (C) 1998-2000 Steffen Kern
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gdk/gdkprivate.h>
#include <gnome.h>
#include <unistd.h>
#include <time.h>
#include <string.h>
#include <fcntl.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <errno.h>
#include <ctype.h>

#include "structs.h"
#include "gI_compile.h"
#include "gI_document.h"
#include "gI_cfp.h"
#include "gI_project.h"
#include "gI_edit.h"
#include "gI_common.h"
#include "gI_file.h"
#include "gI_window.h"
#include "gI_prefs.h"


/* globals */
static gint file;
static gchar filename[STRLEN];
static glong file_is_open;
#ifdef USE_FORK_CODE
static GList *pid_data = NULL;
static long canceled = 0;
#endif

/* externs */
extern gI_window *main_window;
extern gI_config *cfg;


/*
 ---------------------------------------------------------------------
     Function: gI_compile_window_new()
     Desc: To create a new Compile Window
 ---------------------------------------------------------------------
*/

gI_compile_window *gI_compile_window_new( gchar *title )
{
	gI_compile_window *compile_window;
	GtkWidget *vbox;
	gI_document *current;
	GtkWidget *hbox;
	GtkWidget *scrolled_win;

	current = gI_document_get_current( main_window );
	if( !current )
		return( NULL );

	compile_window = g_malloc0( sizeof( gI_compile_window ) );

	compile_window->window = gtk_window_new( GTK_WINDOW_TOPLEVEL );
	gtk_widget_set_usize( compile_window->window, 500, 250 );
	gtk_window_set_title( GTK_WINDOW( compile_window->window ), title );
	gtk_signal_connect( GTK_OBJECT( compile_window->window ), "destroy",
	                    GTK_SIGNAL_FUNC( gI_compile_window_destroy ), current );

	vbox = gtk_vbox_new( FALSE, 5 );
	gtk_container_set_border_width( GTK_CONTAINER( vbox ), 5 );
	gtk_container_add( GTK_CONTAINER( compile_window->window ), vbox );
	gtk_widget_show( vbox );

	scrolled_win = gtk_scrolled_window_new( NULL, NULL );
	gtk_box_pack_start( GTK_BOX( vbox ), scrolled_win, TRUE, TRUE, 0 );
	gtk_scrolled_window_set_policy( GTK_SCROLLED_WINDOW( scrolled_win ),
	                                GTK_POLICY_ALWAYS, GTK_POLICY_ALWAYS );
	gtk_widget_show( scrolled_win );

	compile_window->list = gtk_clist_new( 1 );
	gtk_clist_column_titles_hide( GTK_CLIST( compile_window->list ) );
	gtk_clist_column_titles_passive( GTK_CLIST( compile_window->list ) );
	gtk_container_add( GTK_CONTAINER( scrolled_win ), compile_window->list );
	gtk_signal_connect( GTK_OBJECT( compile_window->list ), "select_row",
	                    GTK_SIGNAL_FUNC( compile_list_select ), (gpointer) current );
	gtk_widget_show( compile_window->list );

	hbox = gtk_hbox_new( FALSE, 0 );
	gtk_box_pack_start( GTK_BOX( vbox ), hbox, FALSE, TRUE, 0 );
	gtk_widget_show( hbox );

	compile_window->close_button = gnome_stock_button( GNOME_STOCK_BUTTON_CLOSE );
	gtk_box_pack_start( GTK_BOX( hbox ), compile_window->close_button, TRUE, TRUE, 5 );
	gtk_signal_connect( GTK_OBJECT( compile_window->close_button ), "clicked",
	                    GTK_SIGNAL_FUNC( gI_compile_window_destroy ), current );
	gtk_widget_show( compile_window->close_button );

	GTK_WIDGET_SET_FLAGS( compile_window->close_button, GTK_CAN_DEFAULT );
	gtk_widget_grab_default( compile_window->close_button );

	return( compile_window );
}


void gI_compile_window_destroy( GtkWidget *widget, gI_document *document )
{
	if( !document
	        || !document->compile_window
	        || !document->compile_window->window )
	{
		return;
	}

	if( document->compile_window->window )
	{
		if( document->compile_window->comp_dialog )
		{
			gtk_widget_destroy( document->compile_window->comp_dialog );
			document->compile_window->comp_dialog = NULL;

			if( document->compile_window->pid > 0 )
			{
				if( kill( document->compile_window->pid, SIGKILL ) < 0 )
				{
					perror( "kill" );
				}
				else
				{
					g_print("process %d killed\n",document->compile_window->pid);
					document->compile_window->pid = -1;
				}
			}

		}
		gtk_widget_destroy( document->compile_window->window );
		g_free( document->compile_window );
		document->compile_window = NULL;
		document->working = 0;
	}
}


void window_deiconify( GdkWindow *window )
{
	GdkWindowPrivate *Private;

	Private = (GdkWindowPrivate*) window;
	XMapRaised (Private->xdisplay, Private->xwindow);
}


gboolean window_is_iconified( GdkWindow *window )
{
	XWindowAttributes xattr;
	GdkWindowPrivate *Private;

	Private = (GdkWindowPrivate*) window;

	xattr.map_state = IsUnmapped;
	XGetWindowAttributes(Private->xdisplay, Private->xwindow, &xattr);

	return (xattr.map_state == IsUnmapped);
}



void update_compile_list( udata *update_data )
{
	gchar buf[MAXLEN];
	gchar tmpbuf[STRLEN];
	glong count;
	gchar *item[1];
	gchar *p;

	/* set working flag */
	update_data->doc->working = 1;

	count = read( update_data->file, buf, sizeof(buf) );
	if( count <= 0 )
		return;
	buf[count] = '\0';

	while( !strchr( buf, '\n' ) )
	{
		count = read( update_data->file, tmpbuf, sizeof(tmpbuf) );
		if( count <= 0 )
			continue;
		tmpbuf[count] = '\0';
		strcat( buf, tmpbuf );
	}

	/*
	 * there was a gtk_main_iteration() call here, i removed it,
	 * made not much sense and lead to recursive calling of
	 * the update_compile_list() function, at least on my
	 * notebook.
	 */

	if( !update_data->doc || !update_data->doc->compile_window
	        || !update_data->doc->compile_window->window )
	{
		/* remove timeout */
		gdk_input_remove( update_data->input_id );

		/* close the file */
		close( update_data->file );

		/* unlink named pipe */
		unlink( update_data->filename );

		/* set working flag */
		/*update_data->doc->working = 0;*/

		/* free memory */
		g_free( update_data->filename );
		g_free( update_data );

		return;
	}

	/* compile done? */
	if( strstr( buf, "Done." ) )
	{
		p = strstr( buf, "Done." );
		*p = '\0';

		/* new item */
		if( !isempty( buf ) )
		{
			item[0] = buf;
			gtk_clist_append( GTK_CLIST( update_data->doc->compile_window->list ), item );
		}

#ifdef USE_FORK_CODE
		if( canceled == 1 )
		{
			strcpy( buf, "Compile Canceled!\n" );
			canceled = 0;
		}
		else
#endif
		{
			strcpy( buf, "Compile Done!\n" );
		}
		item[0] = buf;
		gtk_clist_append( GTK_CLIST( update_data->doc->compile_window->list ), item );
		buf[0] = '\0';

		/* remove timeout */
		gdk_input_remove( update_data->input_id );

		/* destroy dialog */
		if( update_data->doc->compile_window->comp_dialog )
			gtk_widget_destroy( update_data->doc->compile_window->comp_dialog );
		update_data->doc->compile_window->comp_dialog = NULL;

		/* close the file */
		close( update_data->file );

		/* unlink named pipe */
		unlink( update_data->filename );

		/* correct the scrolling */
		if( update_data->doc->compile_window->list )
		{
			gtk_clist_moveto( GTK_CLIST( update_data->doc->compile_window->list ),
			                  GTK_CLIST( update_data->doc->compile_window->list )->rows-1,
			                  0,
			                  1.0,
			                  0.0 );
		}

		/* set working flag */
		update_data->doc->working = 0;

		/* free memory */
		g_free( update_data->filename );
		g_free( update_data );
	}

	/* new item */
	if( !isempty( buf ) )
	{
		/*
		 * a line splitter is needed here,
		 * but what how long is one line???
		 */

		/*
		 * MAX_LINE_LENGTH is defined in
			 * gI_compile.h, currently its set
		 * to 100. Maybe we should set dynamically
		 * depending on the current window width.		 
		 */

		if( strlen( buf ) > MAX_LINE_LENGTH )
		{
			p = &buf[0];
			while( strlen( p ) > MAX_LINE_LENGTH )
			{
				gchar *e = &p[MAX_LINE_LENGTH];
				gchar tmp[MAX_LINE_LENGTH+1];

				while( *e != ' ' )
					e--;

				strncpy( tmp, p, e-p );
				tmp[e-p] = '\0';

				item[0] = tmp;
				gtk_clist_append( GTK_CLIST( update_data->doc->compile_window->list ), item );

				p = e + 1;
			}

			if( strlen( p ) > 0 )
			{
				item[0] = p;
				gtk_clist_append( GTK_CLIST( update_data->doc->compile_window->list ), item );
			}
		}
		else
		{
			item[0] = buf;
			gtk_clist_append( GTK_CLIST( update_data->doc->compile_window->list ), item );
		}
	}

	if( !update_data->doc->compile_window->window )
	{
		return;
	}

	/* correct the scrolling */
	if( update_data->doc->compile_window->list )
	{
		gtk_clist_moveto( GTK_CLIST( update_data->doc->compile_window->list ),
		                  GTK_CLIST( update_data->doc->compile_window->list )->rows-1,
		                  0,
		                  1.0,
		                  0.0 );
	}

	/* de-iconify window */
	if( window_is_iconified( update_data->doc->compile_window->window->window ) )
	{
		window_deiconify( update_data->doc->compile_window->window->window );
	}

	gtk_widget_show_all( update_data->doc->compile_window->window );
}


#ifdef USE_FORK_CODE
void sigchld_handler( gint signal )
{
	gint status;
	udata *update_data;
	FILE *fp;

	/* lookup update_data for pid */
	update_data = table_lookup( wait( &status ) );

	g_return_if_fail( update_data != NULL );

	/* remove association */
	/*
	 * i had some problems, when removing the association,
	 * so we keep it for now,
	    	 * but we should find a way to free the complete GList
	 * at some point.
	 */	 
	/*table_rassoc( update_data );*/

	/*
	 * the child terminated, now we write the "Done." message.
	 * so we also ensure that the handler is called again,
	 * the compile list is updated and working flag is set
	 * correctly.
	 */
	fp = fopen( update_data->filename, "w" );
	if(!fp) {
		perror("open");
	} else {
		fprintf( fp, "Done.\n" );
		fclose( fp );
  	}
  	
	/*
	 * wtf do we have memory corruption (at least on my
	 * notebook) here? thats why the input handler loops
	 * again and again and never gets the data and quits.
	 */

	/*
	 * i dont know why, but the memory corruption stopped,
	 * when i replaced the open() thru fopen().
	 */

	/*
	 * but there's still something going wrong...i linked against
	 * the electric fence library ... i have a segfault after
	 * a g_free() call in the main loop.
	 */
}


gint gI_exec( gchar **argv, gchar *socket )
{
	gint pid;
	gint ret;
	gint fh;

	/*
	 * connect signal handler
	 * for SIGCHLD to catch the termination
	 * of the child process.
	 */
	signal( SIGCHLD, sigchld_handler );

	/*
	 * fork the child process
	 */
	pid = fork();
	if( pid < 0 )
	{
		/*
		 * something went wrong,
		 * return error.
		 */
		return -1;
	}

	if( pid > 0 )
	{
		/*
		 * we're in the parent process,
		 * return the process id of the
		 * child process.
		 */
		return pid;
	}
	else
	{
		/*
		 * we're in the child process,
		 * begin work.
		 */

		/* open socket */
		fh = open( socket, O_WRONLY );

		/* redirect output and input */
		close (0); dup (fh);  /* set the stdin to the in pipe */
		close (1); dup (fh);  /* set the stdout to the out pipe */
		close (2); dup (fh);  /* set the stderr to the out pipe */

		/* we should close the display here */
		close( gdk_screen );

		/* execute program */
		ret = execvp( argv[0], argv );
		if( ret < 0 )
		{
			perror( "execvp" );
			return -1;
		}

		return 0;
	}
}


void build_comp_string_fork( gchar *str, gchar **argv )
{
	gchar *dup;
	gchar *args[100];
	glong count, args_count;
	gchar *beg, *end;
	glong fields_no;

	/*
	 * dup string and get
	 * pointers to the beginning
	 * and the end.
	 */
	dup = g_strdup( str );
	beg = strchr( dup, '(' );
	beg++;
	end = strstr( dup, "1>" );
	*end = '\0';

	/*
	 * split commandline into fields
	 * for execvp().
	 */
	SK_GetFields( beg, args, ' ' );

	fields_no = atoi( args[0] );
	for(count=0,args_count=1;args_count<fields_no;args_count++)
	{
		/*
		 * abort to avoid an overflow
		 * of the array.
		 */
		if( count >= 100 )
			break;

		while( args_count < fields_no-1
		        && isempty( args[args_count] ) )
		{
			args_count++;
		}

		/*
		 * to avoid crashes on
		 * NULL-Pointers.
		 */
		if( args[args_count] )
		{
			strcpy( argv[count++], args[args_count] );
		}
		else
		{
			/*
			 * we reached a NULL-Pointer
			 * in the array...quit!
			 */

			/*
			 * increase counter
			 * to avoid argument loss.
			 */
			count++;

			break;
		}
	}
	argv[count-1] = NULL;

	/*
	 * free memory allocated
	 * by SK_GetFields().
	 */
	for(count=0;count<fields_no;count++)
		g_free( args[count] );

	/* free dup string */
	g_free( dup );
}
#endif



static void _compile_compile( gI_document *document, glong type )
{
	gchar exec_string[MAXLEN] = "";
	gchar tmp_string[STRLEN];
	gchar *ptr;
	gchar *items[1];
	gchar exec_file[STRLEN];
	gchar *path;
	gint pid = -1;
	udata *update_data;
#ifdef USE_FORK_CODE
	gchar *argv[100];
	glong i;
#endif

	document->compile_window->pid = pid;

	/* tmp filename */
	g_snprintf( filename, STRLEN, "%s/gide_compile.%d.%d", cfg->tmpdir,
	            (gint) time(0), (gint)getpid() );

	/* if there's already a file with this 'unique' name, unlink it */
	unlink( filename );

	if( mkfifo( filename, S_IRUSR|S_IWUSR ) < 0 )
	{
		sprintf( tmp_string, "File Error: %s", strerror( errno ) );
		gI_error_dialog( tmp_string);
		return;
	}

	strcpy( exec_file, document->filename );
	ptr = match_cset_fpat( get_comp_set_from_filename( document->filename ), exec_file );
	if( !ptr )	/* default is '.c' */
		ptr = strstr( exec_file, ".c" );
	if( ptr )
		*ptr = '\0';

	/* change to working dir */
	path = get_path( document->filename );
	chdir( path );
	g_free( path );

	strcpy( exec_string, build_comp_string( document, exec_file, filename, type ) );

#ifdef USE_FORK_CODE
	for(i=0;i<100;i++)
		argv[i] = (gchar *) g_malloc0( 256 );

	build_comp_string_fork( exec_string, argv );

	pid = gI_exec( argv, filename );

	for(i=0;i<100;i++)
		g_free( argv[i] );
#else
	pid = system( exec_string );
#endif

	document->compile_window->pid = pid;

	/* show command line */
	ptr = strstr( exec_string, "1>" );
	if( ptr )
		*ptr = '\0';
	ptr = strchr( exec_string, '(' );
	ptr++;
	items[0] = ptr;
	gtk_clist_append( GTK_CLIST( document->compile_window->list ), items );

	file = open( filename, O_RDONLY );
	if( !file )
	{
		perror( "open" );
		return;
	}

	update_data = (udata *) g_malloc0( sizeof( udata ) );
	update_data->file = file;
	update_data->doc = document;
	update_data->filename = g_strdup( filename );

	/* get process id of the shell */
	/*fgets( buf, sizeof(buf), update_data->file );
	*document->compile_window->pid = atoi( buf );*/

	update_data->input_id = gdk_input_add( update_data->file, GDK_INPUT_READ, (GdkInputFunction) update_compile_list, (gpointer) update_data );

#ifdef USE_FORK_CODE
	table_assoc( pid, update_data );
#endif

}


void compile_compile( GtkWidget *widget, gpointer data )
{
	gI_document *current;
	glong type;

	type = (glong) data;

	/* get current page */
	current = gI_document_get_current( main_window );

	if( current->working )
		return;

	if( current->filename == NULL )
		return;

	current->working = 1;

	if( current->compile_window == NULL )
	{
		current->compile_window = gI_compile_window_new( _("Compile Window") );
		gtk_widget_show( current->compile_window->window );
	}

	show_compiling( current, current->compile_window );

	_compile_compile( current, type );

	/*gtk_widget_destroy( current->compile_window->comp_dialog );*/

	/*current->working = 0;*/
}


/*
 ---------------------------------------------------------------------
     Function: compile_make()
     Desc: Callback-Function /Compile/Make
 ---------------------------------------------------------------------
*/

void compile_make( GtkWidget *widget, gpointer data )
{
	gchar tmp_string[STRLEN];
	gI_document *current;
	gchar *path;
	gint pid = -1;
	udata *update_data;
	gchar *items[1];
#ifdef USE_FORK_CODE
	gchar *argv[64];
#else
	gchar exec_string[MAXLEN];
#endif

	/* get current page */
	current = gI_document_get_current( main_window );
	if(!current) 
		return;
		
	/* only one make/doc allowed */
	if( current->working )
		return;

	/*if( current->filename == NULL )
	return;*/ 

	current->working = 1;

	if( current->compile_window == NULL )
	{
		current->compile_window = gI_compile_window_new( _("Make Window") );
		gtk_widget_show( current->compile_window->window );
	}

	current->compile_window->pid = pid;

	/* if there's a file open, get the path from it and try make there */
	if( current->filename != NULL )
	{
		/* change working dir */
		path = get_path( current->filename );
		if( !isempty( path ) )
		{
			chdir( path );
			g_free( path );
		}
	}

	/* generate a *really* unique filename */
	g_snprintf( filename, STRLEN, "%s/gide_make.%d.%d", cfg->tmpdir,
	            (gint) time(0), (gint)getpid() );

	/* if there's already a file with this 'unique', unlink it */
	unlink( filename );

	if( mkfifo( filename, S_IRUSR|S_IWUSR ) < 0 )
	{
		sprintf( tmp_string, "File Error: %s", strerror( errno ) );
		gI_error_dialog( tmp_string);
		return;
	}

	/* show compiling dialog */
	show_compiling( current, current->compile_window );

	/* start compile */
#ifdef USE_FORK_CODE
	argv[0] = cfg->make;
	argv[1] = NULL;

	pid = gI_exec( argv, filename );
#else
	/* build exec string */
	g_snprintf( exec_string, STRLEN, "%s -c \"(%s 1>%s 2>&1 ; echo \"Done.\" >%s) &\"", cfg->bash, cfg->make, filename, filename );

	pid = system( exec_string );
#endif

	items[0] = cfg->make;
	gtk_clist_append( GTK_CLIST( current->compile_window->list ), items );

	current->compile_window->pid = pid;

	/* try to open socket */
	file = open( filename, O_RDONLY );
	if( !file )
	{
		perror( "open" );
		return;
	}

	update_data = (udata *) malloc( sizeof( udata ) );
	update_data->file = file;
	update_data->doc = current;
	update_data->filename = g_strdup( filename );

	/* get process id of the shell */
	/*fgets( buf, sizeof(buf), update_data->file );
	*current->compile_window->pid = atoi( buf );*/

	update_data->input_id = gdk_input_add( update_data->file, GDK_INPUT_READ, (GdkInputFunction) update_compile_list, (gpointer) update_data );

#ifdef USE_FORK_CODE
	table_assoc( pid, update_data );
#endif

}



void compile_link( GtkWidget *widget, gpointer data )
{
	gchar exec_string[MAXLEN];
	gchar tmp_string[STRLEN];
	gchar exec_file[STRLEN];
	gchar *ptr;
	gI_document *current;
	gchar *items[1];
	udata *update_data;
	gint pid = -1;
#ifdef USE_FORK_CODE
	gchar *argv[64];
	glong i;
#endif

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	if( current->working )
		return;

	if( current->filename == NULL )
		return;

	current->working = 1;

	if( current->compile_window == NULL )
	{
		current->compile_window = gI_compile_window_new( _("Link Window") );
		gtk_widget_show( current->compile_window->window );
	}

	current->compile_window->pid = pid;

	show_compiling( current, current->compile_window );

	/* check if object file exists */
	strcpy( tmp_string, current->filename );
	ptr = strrchr( tmp_string, '.' );
	if( ptr )
	{
		ptr++;
		*ptr = 'o';
		ptr++;
		*ptr = '\0';
		if( !file_exist( tmp_string ) )
			_compile_compile( current, 2 );
	}
	else
	{
		return;
	}

	strcpy( exec_file, current->filename );
	ptr = match_cset_fpat( get_comp_set_from_filename( current->filename ), exec_file );
	if( !ptr )	/* default is '.c' */
		ptr = strstr( exec_file, ".c" );
	if( ptr )
		*ptr = '\0';

	/* tmp filename */
	g_snprintf( filename, STRLEN, "%s/gide_link.%d.%d", cfg->tmpdir,
	            (gint) time(0), (gint)getpid() );

	/* if there's already a file with this 'unique', unlink it */
	unlink( filename );

	if( mkfifo( filename, S_IRUSR|S_IWUSR ) < 0 )
	{
		sprintf( tmp_string, "File Error: %s", strerror( errno ) );
		gI_error_dialog( tmp_string);
		return;
	}

	/* build exec string */
	strcpy( exec_string, build_comp_string( current, exec_file, filename, COMPILE_LINK ) );

#ifdef USE_FORK_CODE
	for(i=0;i<64;i++)
		argv[i] = (char *) g_malloc0( 100 );

	build_comp_string_fork( exec_string, argv );

	pid = gI_exec( argv, filename );

	for(i=0;i<64;i++)
		g_free( argv[i] );
#else
	pid = system( exec_string );
#endif

	current->compile_window->pid = pid;

	/* show command line */
	ptr = strstr( exec_string, "1>" );
	if( ptr )
		*ptr = '\0';
	ptr = strchr( exec_string, '(' );
	ptr++;
	items[0] = ptr;
	gtk_clist_append( GTK_CLIST( current->compile_window->list ), items );

	file = open( filename, O_RDONLY );
	if( !file )
	{
		perror( "open" );
		return;
	}

	update_data = (udata *) malloc( sizeof( udata ) );
	update_data->file = file;
	update_data->doc = current;
	update_data->filename = g_strdup( filename );

	/* get process id of the shell */
	/*fgets( buf, sizeof(buf), update_data->file );
	*current->compile_window->pid = atoi( buf );*/

	update_data->input_id = gdk_input_add( update_data->file, GDK_INPUT_READ, (GdkInputFunction) update_compile_list, (gpointer) update_data );

#ifdef USE_FORK_CODE
	table_assoc( pid, update_data );
#endif

}



void show_compiling( gI_document *current, gI_compile_window *compile_window )
{
	GtkWidget *vbox;
	GtkWidget *label;
	GtkWidget *button;

	/* new dialog */
	compile_window->comp_dialog = gtk_window_new( GTK_WINDOW_DIALOG );
	gtk_window_set_policy( GTK_WINDOW( compile_window->comp_dialog ), FALSE, FALSE, FALSE );
	gtk_window_set_position( GTK_WINDOW( compile_window->comp_dialog ), GTK_WIN_POS_CENTER );
	gtk_widget_set_usize( compile_window->comp_dialog, 300, 80 );
	gtk_window_set_title( GTK_WINDOW( compile_window->comp_dialog ), _("Compile Status") );

	/* new vbox */
	vbox = gtk_vbox_new( FALSE, 5 );
	gtk_container_add( GTK_CONTAINER( compile_window->comp_dialog ), vbox );
	gtk_container_set_border_width( GTK_CONTAINER( compile_window->comp_dialog ), 10 );
	gtk_widget_show( vbox );

	/* new label */
	label = gtk_label_new( _("Compiling...") );
	gtk_box_pack_start( GTK_BOX( vbox ), label, FALSE, FALSE, 0 );
	gtk_widget_show( label );

	/* new button */
	button = gnome_stock_button( GNOME_STOCK_BUTTON_CANCEL );
	gtk_box_pack_start( GTK_BOX( vbox ), button, FALSE, FALSE, 10 );
#ifdef USE_FORK_CODE
	gtk_signal_connect( GTK_OBJECT( button ), "clicked",
	                    GTK_SIGNAL_FUNC( cancel_compiling ), (gpointer) current );
#endif
	gtk_widget_show( button );
#ifndef USE_FORK_CODE
	gtk_widget_set_sensitive( button, FALSE );
#endif

	/* show dialog */
	gtk_widget_show( compile_window->comp_dialog );
}


#ifdef USE_FORK_CODE
void cancel_compiling( GtkWidget *widget, gI_document *current )
{
	if( !current )
	{
		return;
	}

	/* set working flag */
	current->working = 0;

	if( !current->compile_window )
	{
		return;
	}

	/* kill compile process, ensure that we dont kill init */
	if( current->compile_window->pid > 1 )
	{
		if( kill( current->compile_window->pid, SIGKILL ) < 0 )
		{
			perror( "kill" );
		}
		else
		{
			g_print("process %d killed\n",current->compile_window->pid);
			current->compile_window->pid = -1;
			canceled = 1;
			current->working = 0;
		}
	}

	if( current->compile_window->comp_dialog )
		gtk_widget_destroy( current->compile_window->comp_dialog );
	/*gtk_widget_destroy( current->compile_window->window );*/
}
#endif


static void _check_file_open( gI_document *document, gchar *filename )
{
	if( document )
	{
		if( document->filename )
		{
			if( !strcmp( document->filename, filename ) )
				file_is_open = TRUE;
		}
	}
}


gint check_file_open( gchar *filename )
{
	file_is_open = FALSE;
	g_list_foreach( main_window->documents, (GFunc) _check_file_open, (gpointer) filename );
	if( file_is_open )
		return( 1 );

	return( 0 );
}



gint get_point_from_line( gint tline )
{
	gulong line,point,i,length;
	gI_document *current;
	gchar *ch;

	line=1;
	point=1;

	current = gI_document_get_current( main_window );

	length = gI_text_get_length( current->text );

	if( length > 0 )
	{
		if( tline == 1 )
			return( 1 );
	}

	for(i=1;i<=length;i++)
	{
		ch = gtk_editable_get_chars( GTK_EDITABLE( current->text ), i-1, i );
		if( ch[0] == '\n' )
		{
			g_free( ch );
			line++;
			if( line == tline )
				break;
			point++;
		}
		else
		{
			g_free( ch );
			point++;
		}
	}

	/*point++;*/

	if( point < 0 )
		point = 0;
	if( point > gI_text_get_length( current->text ) )
		point = gI_text_get_length( current->text );

	return( point );
}


gint get_line_from_point( gint tpoint )
{
	gint line,point,i,length;
	gI_document *current;
	gchar *c = NULL;

	/* Init */
	line=1;
	point=0;

	/* get current document */
	current = gI_document_get_current( main_window );

	/* get length of current document */
	length = gI_text_get_length( current->text );

	/* read... */
	for(i=1;i<=length;i++)
	{
		c =  gtk_editable_get_chars( GTK_EDITABLE( current->text ), i-1, i );
		if( *c == '\n' )
		{
			if( point >= tpoint )
				goto hit;
			point++;
			line++;
		}
		else
		{
			if( point >= tpoint )
				goto hit;
			point++;
		}
		g_free(c);
	}

	/* we got what we want */
hit:
	if( c )
		g_free( c );
	return( line );
}


glong goto_file( gchar *filename )
{
	glong nod,i,found = 0,last;
	gI_document *current;

	/* reimplement this stuff!! */

	last = gtk_notebook_get_current_page( GTK_NOTEBOOK( main_window->notebook ) );

	nod = gI_document_get_no( main_window );

	for(i=0;i<nod;i++)
	{
		gtk_notebook_set_page( GTK_NOTEBOOK( main_window->notebook ), i );
		current = gI_document_get_current( main_window );
		if( current->filename != NULL )
		{
			if( !strcmp( current->filename, filename ) )
			{
				found = 1;
				break;
			}
		}
	}

	if( !found )
	{
		gtk_notebook_set_page( GTK_NOTEBOOK( main_window->notebook ), last );
		return( 0 );
	}

	return( 1 );
}


/*
 * be careful, this function still takes a line number
 * as argument!
 */
void goto_point( gint line )
{
	gI_document *current;
	int point;
	int i;
	glong changed;
	double tl, ln, adjval;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	tl = (double) get_total_lines( current->text );
	ln = (double) line;

	if( tl < 10.0 )
	{
		/* is on the screen */
		return;
	}

	if( ln > tl )
	{
		/* should not happen */
		return;
	}

	gI_text_set_point( current->text, get_point_from_line( line ) );

	/* borrowed from gEdit */
	adjval = (ln * GTK_ADJUSTMENT( gI_text_vadj(current->text ) )->upper ) / tl - GTK_ADJUSTMENT( gI_text_vadj( current->text ) )->page_increment;

	gtk_adjustment_set_value( GTK_ADJUSTMENT( gI_text_vadj( current->text )  ), adjval );


	if( current->changed )
		changed = TRUE;
	else
		changed = FALSE;

	point = get_point_from_line( line );

	i = point;
	gtk_editable_insert_text( GTK_EDITABLE( current->text ), " ", 1, &i );
	gtk_editable_delete_text( GTK_EDITABLE( current->text ), i-1, i );
	gI_text_set_point( current->text, i-2 );

	if( changed == FALSE )
	{
		gI_document_set_unchanged( current );
		gI_window_set_statusbar( current );
	}

	/* select whole line */
	edit_select_line( NULL, NULL );

	/* track movement and set statusbar */
	gI_document_track_movement( current );
	gI_window_set_statusbar( current );
}


/*
 * this function does not really go to the line
 * it just makes the line appear on the screen and sets the point
 * if you want really to go there .. use goto_point().
 */
void goto_line( gint line )
{
	gI_document *current;
	/*gint point;
	int i;
	glong changed;*/
	double tl, ln, adjval;

	current = gI_document_get_current( main_window );
	if( !current )
		return;

	/*** this is shit, replaced
	    if( current->changed )
	        changed = TRUE;
	    else
	        changed = FALSE;

	    point = get_point_from_line( line );

	    gI_text_set_point( current->text, point );

	    i = point;
	    gtk_editable_insert_text( GTK_EDITABLE( current->text ), " ", 1, &i );
	    gtk_editable_delete_text( GTK_EDITABLE( current->text ), i-1, i );
	    gI_text_set_point( current->text, i-2 );

	    edit_select_line( NULL, NULL );
	    
	    if( changed == FALSE )
	    {
	        gI_document_set_unchanged( current );
	   		gI_window_set_statusbar( current ); 
	    }
	***/  

	tl = (double) get_total_lines( current->text );
	ln = (double) line;

	if( tl < 10.0 )
	{
		/* is on the screen */
		return;
	}

	if( ln > tl )
	{
		/* should not happen */
		return;
	}

	gI_text_set_point( current->text, get_point_from_line( line ) );

	/* borrowed from gEdit */
	adjval = (ln * GTK_ADJUSTMENT( gI_text_vadj( current->text ) )->upper ) / tl - GTK_ADJUSTMENT( gI_text_vadj( current->text ) )->page_increment;

	gtk_adjustment_set_value( GTK_ADJUSTMENT( gI_text_vadj( current->text ) ), adjval );

	/* select whole line */
	edit_select_line( NULL, NULL );

	/* track movement and set statusbar */
	gI_document_track_movement( current );
	gI_window_set_statusbar( current );
}


void jump_to_source( gchar *listitem )
{
	gint hits;
	gint line;
	gchar filename[STRLEN];
	gchar *fi[64];
	/*gI_project *project = gI_project_get_current();*/
	gchar *x;

	hits = SK_GetFields( listitem, fi, ':' );
	if( hits >= 4 )
	{
		strcpy( filename, fi[1] );

		x = get_path( filename );
		if( x[0] == '\0' )	/* no path returned */
		{
			gchar *path;
			gchar tmpfilename[STRLEN];

			path = g_get_current_dir();

			strcpy( tmpfilename, filename );
			strcpy( filename, path );
			strcat( filename, "/" );
			strcat( filename, tmpfilename );

			g_free( path );
		}

		/*if( project && !isempty( project->prjroot ) )
	{
			gchar *x;

			x = get_path( filename );
			if( x[0] == '\0' )	/- no path returned -/
			{
				gchar tmpfilename[STRLEN];

				strcpy( tmpfilename, filename );

				strcpy( filename, project->prjroot );
				strcat( filename, "/" );
				strcat( filename, tmpfilename );
			}

			g_free( x );
	}*/

		line = atoi( fi[2] );

		/* free memory */
		g_free( fi[0] ); g_free( fi[1] ); g_free( fi[2] );

		if( line <= 0 )
			return;

		if( !check_file_open( filename ) )
		{
			if( file_exist( filename ) )
			{
				file_open_by_name( main_window, filename );
			}
			else
			{
				/* cant find file */
				/* open fileselector dialog */
				GList *filelist = gI_file_sel_new( "Open File...", FALSE, FALSE );
				if( filelist != NULL )
				{
					strcpy( filename, filelist->data );
					file_open_by_name( main_window, filename );
					goto_line( line );
				}

				return;
			}
		}
		else
		{
			goto_file( filename );
		}

		goto_line( line );
	}
}


void compile_list_select( GtkWidget *widget, gint row, gint column, GdkEventButton *bevent, gI_document *document )
{
	gchar *item_data_string;

	if( !document )
		return;

	if( !bevent )
		return;

	if( bevent->type == GDK_2BUTTON_PRESS )
	{
		if( !document->compile_window )
		{
			g_warning( "document->compile_window is NULL, but someone double-clicked a row in the clist ?!" );
			return;
		}

		gtk_clist_get_text( GTK_CLIST( document->compile_window->list ), row, 0, &item_data_string );

		jump_to_source( item_data_string );
	}
}


glong get_total_lines( GtkWidget *text )
{
	gchar *c;
	glong i;
	glong lines = 1;

	for(i=0;i<(gI_text_get_length( text ));i++)
	{
		c = gtk_editable_get_chars( GTK_EDITABLE( text ), i, i+1 );
		if( *c == '\n' )
			lines++;
		g_free( c );
	}

	return( lines );
}


gchar *build_comp_string( gI_document *document, gchar *targetfile, gchar *fifoname, glong type )
{
	static gchar cmdstr[MAXLEN];
	gI_comp_set *cset;
	gchar *libpaths[64];
	gchar *incpaths[64];
	glong i;
	gchar options[MAXLEN];
	gchar *objectfile;
	gchar *ptr;

	if( !document )
		return( NULL );

	if( !get_comp_set_from_filename( document->filename ) )
	{
		/*
		 * if no compile set is given in the document struct 
		 * we build a default c compile/link string using the default
		 * parameters from the cfg struct
		 * the default parameters in the cfg struct are filled in
		 * set_cfg_defaults().
		 */

		switch( type )
		{
		case COMPILE_EXEC:
			{
				strcpy( options, cfg->ccopt );
				if( !isempty( cfg->incpath ) )
				{
					SK_GetFields( cfg->incpath, incpaths, ':' );

					for(i=1;i<(atoi(incpaths[0]));i++)
					{
						sprintf( options, "%s -I%s", options, incpaths[i] );
						g_free( incpaths[i] );
					}
					g_free( incpaths[0] );
				}

				sprintf( options, "%s %s", options, cfg->ldopt );
				if( !isempty( cfg->libpath ) )
				{
					SK_GetFields( cfg->libpath, libpaths, ':' );

					for(i=1;i<(atoi(libpaths[0]));i++)
					{
						sprintf( options, "%s -L%s", options, libpaths[i] );
						g_free( libpaths[i] );
					}
					g_free( libpaths[0] );
				}

				g_snprintf( cmdstr, MAXLEN, "%s -c \"(%s %s %s -o %s 1>%s 2>&1 ; echo \"Done.\" >%s) &\"", cfg->bash, cfg->cc, options, document->filename, targetfile, fifoname, fifoname );

				break;
			}

		case COMPILE_SYNTAX:
		case COMPILE_OBJ:
			{
				if( !isempty )
					strcpy( options, cfg->ccopt );
				else
					strcpy( options, "" );

				if( type == COMPILE_SYNTAX )
					strcat( options, " -fsyntax-only" );
				if( !isempty( cfg->incpath ) )
				{
					SK_GetFields( cfg->incpath, incpaths, ':' );

					for(i=1;i<(atoi(incpaths[0]));i++)
					{
						sprintf( options, "%s -I%s", options, incpaths[i] );
						g_free( incpaths[i] );
					}
					g_free( incpaths[0] );
				}

				g_snprintf( cmdstr, MAXLEN, "%s -c \"(%s %s -c %s -o %s.o 1>%s 2>&1 ; echo \"Done.\" >%s) &\"", cfg->bash, cfg->cc, options, document->filename, targetfile, fifoname, fifoname );

				break;
			}

		case COMPILE_LINK:
			{
				strcpy( options, cfg->ldopt );
				if( !isempty( cfg->libpath ) )
				{
					SK_GetFields( cfg->libpath, libpaths, ':' );

					for(i=1;i<(atoi(libpaths[0]));i++)
					{
						sprintf( options, "%s -L%s", options, libpaths[i] );
						g_free( libpaths[i] );
					}
					g_free( libpaths[0] );
				}

				objectfile = g_strdup( document->filename );
				ptr = strrchr( objectfile, '.' );
				if( ptr )
				{
					ptr++;
					*ptr = 'o';
					ptr++;
					*ptr = '\0';
				}
				g_snprintf( cmdstr, MAXLEN, "%s -c \"(%s %s %s -o %s 1>%s 2>&1 ; echo \"Done.\" >%s) &\"", cfg->bash, cfg->cc, options, objectfile, targetfile, fifoname, fifoname );
				g_free( objectfile );
				break;
			}
		}

		return( cmdstr );
	}

	cset = get_comp_set_from_filename( document->filename );
	if( !cset )
	{
		/* now we have a problem! */
		return( NULL );
	}

	switch( type )
	{
	case COMPILE_EXEC:	/* compile to executable*/
		{
			if( isempty( cset->c_cmdline_exec ) )
			{
				/* use default command line */
				/* maybe we need one for each language? */
				strcpy( options, "" );
				if( !isempty( cset->ccopt ) )
					strcpy( options, cset->ccopt );
				if( !isempty( cset->incpath ) )
				{
					SK_GetFields( cset->incpath, incpaths, ':' );

					for(i=1;i<(atoi(incpaths[0]));i++)
					{
						sprintf( options, "%s -I%s", options, incpaths[i] );
						g_free( incpaths[i] );
					}
					g_free( incpaths[0] );
				}

				if( !isempty( cset->ldopt ) )
					sprintf( options, "%s %s", options, cset->ldopt );
				if( !isempty( cset->libpath ) )
				{
					SK_GetFields( cset->libpath, libpaths, ':' );

					for(i=1;i<(atoi(libpaths[0]));i++)
					{
						sprintf( options, "%s -L%s", options, libpaths[i] );
						g_free( libpaths[i] );
					}
					g_free( libpaths[0] );
				}

				g_snprintf( cmdstr, MAXLEN, "%s -c \"(%s %s %s -o %s 1>%s 2>&1 ; echo \"Done.\" >%s) &\"", cfg->bash, cset->compiler, options, document->filename, targetfile, fifoname, fifoname );
			}
			else
			{
				/* parse custom command line and insert parameters */
				g_snprintf( cmdstr, MAXLEN, "%s -c \"(%s 1>%s 2>&1 ; echo \"Done.\" >%s) &\"", cfg->bash, parse_cmdline( cset->c_cmdline_exec, cset, document, targetfile, type ), fifoname, fifoname );
			}

			break;
		}

	case COMPILE_SYNTAX:
	case COMPILE_OBJ:	/* compile to object */
		{
			if( isempty( cset->c_cmdline_obj ) )
			{
				/* use default command line */
				if( !isempty( cset->ccopt ) )
					strcpy( options, cset->ccopt );
				else
					strcpy( options, "" );
				if( !isempty( cset->incpath ) )
				{
					SK_GetFields( cset->incpath, incpaths, ':' );

					for(i=1;i<(atoi(incpaths[0]));i++)
					{
						sprintf( options, "%s -I%s", options, incpaths[i] );
						g_free( incpaths[i] );
					}
					g_free( incpaths[0] );
				}

				g_snprintf( cmdstr, MAXLEN, "%s -c \"(%s %s -c %s -o %s.o 1>%s 2>&1 ; echo \"Done.\" >%s) &\"", cfg->bash, cset->compiler, options, document->filename, targetfile, fifoname, fifoname );
			}
			else
			{
				/* parse custom command line and insert parameters */
				g_snprintf( cmdstr, MAXLEN, "%s -c \"(%s 1>%s 2>&1 ; echo \"Done.\" >%s) &\"", cfg->bash, parse_cmdline( cset->c_cmdline_obj, cset, document, targetfile, type ), fifoname, fifoname );
			}

			break;
		}

	case COMPILE_LINK:	/* link */
		{
			if( isempty( cset->l_cmdline ) )
			{
				/* use default command line */
				if( !isempty( cfg->ldopt ) )
					strcpy( options, cfg->ldopt );
				else
					strcpy( options, "" );

				if( !isempty( cfg->libpath ) )
				{
					SK_GetFields( cfg->libpath, libpaths, ':' );

					for(i=1;i<(atoi(libpaths[0]));i++)
					{
						sprintf( options, "%s -L%s", options, libpaths[i] );
						g_free( libpaths[i] );
					}
					g_free( libpaths[0] );
				}

				objectfile = g_strdup( document->filename );
				ptr = strrchr( objectfile, '.' );
				if( ptr )
				{
					ptr++;
					*ptr = 'o';
					ptr++;
					*ptr = '\0';
				}
				g_snprintf( cmdstr, MAXLEN, "%s -c \"(%s %s -o %s %s 1>%s 2>&1 ; echo \"Done.\" >%s) &\"", cfg->bash, cset->compiler, options, targetfile, objectfile, fifoname, fifoname );
				g_free( objectfile );
			}
			else
			{
				/* parse custom command line and insert parameters */
				g_snprintf( cmdstr, MAXLEN, "%s -c \"(%s 1>%s 2>&1 ; echo \"Done.\" >%s) &\"", cfg->bash, parse_cmdline( cset->l_cmdline, cset, document, targetfile, type ), fifoname, fifoname );
			}

			break;
		}
	}

	return( cmdstr );
}


gchar *match_cset_fpat( gI_comp_set *cset, gchar *filename )
{
	glong i;
	gchar *exts[64];
	gchar *p;
	glong match;

	if( !cset )
		return( NULL );

	if( !cset->fpat )
		return( NULL );

	p = strrchr( filename, '.' );
	if( !p )
		return( NULL );

	SK_GetFields( cset->fpat, exts, ':' );

	for(i=1;i<atoi(exts[0]);i++)
	{
		match = strcmp( p, exts[i] );
		if( !match )
		{
			return( p );
		}
	}

	return( NULL );
}


gchar *parse_cmdline( gchar *cmdline, gI_comp_set *cset, gI_document *document, gchar *targetfile, glong type )
{
	static gchar cmdstr[MAXLEN] = "";
	gchar *ptr;
	glong cnt = 0;
	gchar *incpaths[64];
	gchar *libpaths[64];
	glong i;
	gchar options[STRLEN];
	gchar *tmp, *sptr;
	glong hit = 0;

	ptr = cmdline;

	while( *ptr )
	{
		if( *ptr != '%' )
		{
			cmdstr[cnt++] = *ptr;
			ptr++;
		}
		else
		{
			hit = FALSE;

			if( !strncmp( ptr, "%compiler%", 10 ) )
			{
				ptr += 10;
				cmdstr[cnt++] = '\0';
				strcat( cmdstr, cset->compiler );
				cnt = strlen( cmdstr );

				if( type == COMPILE_SYNTAX )
				{
					strcat( cmdstr, " -fsyntax-only" );
					cnt = strlen( cmdstr );
				}

				hit = TRUE;
			}

			if( !strncmp( ptr, "%linker%", 8 ) )
			{
				ptr += 8;
				cmdstr[cnt++] = '\0';
				strcat( cmdstr, cset->linker );
				cnt = strlen( cmdstr );

				hit = TRUE;
			}

			if( !strncmp( ptr, "%copts%", 7 ) )
			{
				ptr += 7;
				cmdstr[cnt++] = '\0';
				strcat( cmdstr, cset->ccopt );
				cnt = strlen( cmdstr );

				hit = TRUE;
			}

			if( !strncmp( ptr, "%lopts%", 7 ) && type != COMPILE_OBJ )
			{
				ptr += 7;
				cmdstr[cnt++] = '\0';
				strcat( cmdstr, cset->ldopt );
				cnt = strlen( cmdstr );

				hit = TRUE;
			}

			if( !strncmp( ptr, "%incpaths%", 10 ) )
			{
				ptr += 10;
				if( !isempty( cset->incpath ) )
				{
					cmdstr[cnt++] = '\0';
					SK_GetFields( cset->incpath, incpaths, ':' );

					for(i=1;i<(atoi(incpaths[0]));i++)
					{
						sprintf( options, "-I%s ", incpaths[i] );
						strcat( cmdstr, options );
						g_free( incpaths[i] );
					}
					g_free( incpaths[0] );
					cnt = strlen( cmdstr );
				}

				hit = TRUE;
			}

			if( !strncmp( ptr, "%libpaths%", 10 ) && type != COMPILE_OBJ )
			{
				ptr += 10;

				if( !isempty( cset->libpath ) )
				{
					cmdstr[cnt++] = '\0';
					SK_GetFields( cset->libpath, libpaths, ':' );

					for(i=1;i<(atoi(libpaths[0]));i++)
					{
						sprintf( options, "-L%s ", libpaths[i] );
						strcat( cmdstr, options );
						g_free( libpaths[i] );
					}
					g_free( libpaths[0] );
					cnt = strlen( cmdstr );
				}

				hit = TRUE;
			}

			if( !strncmp( ptr, "%source%", 8 ) )
			{
				ptr += 8;

				if( document->filename )
				{
					cmdstr[cnt++] = '\0';
					strcat( cmdstr, document->filename );
					cnt = strlen( cmdstr );
				}

				hit = TRUE;
			}

			if( !strncmp( ptr, "%output%", 8 ) )
			{
				ptr += 8;

				if( targetfile )
				{
					cmdstr[cnt++] = '\0';
					strcat( cmdstr, targetfile );
					cnt = strlen( cmdstr );
				}

				hit = TRUE;
			}

			if( !strncmp( ptr, "%object%", 8 ) )
			{
				ptr += 8;

				/* this assums that .o extensions means object file, ie C */
				tmp = g_strdup( document->filename );
				if( tmp )
				{
					sptr = strstr( tmp, ".o" );
					if( sptr )
					{
						*sptr = '\0';
						cmdstr[cnt++] = '\0';
						strcat( cmdstr, tmp );
						cnt = strlen( cmdstr );
						g_free( tmp );
					}
				}

				hit = TRUE;
			}

			/*
			 * default handling if
			 * spaceholder matches.
			 */
			if( !hit )
			{
				cmdstr[cnt++] = *ptr;
				ptr++;
			}
		}
	}

	return( cmdstr );
}


gI_comp_set *get_comp_set_from_filename( gchar *filename )
{
	GList *tmp;
	gI_comp_set *comp_set;

	tmp = cfg->csets;

	while( tmp != NULL )
	{
		comp_set = (gI_comp_set *) tmp->data;
		if( comp_set )
		{
			if( match_cset_fpat( comp_set, filename ) != NULL )
				return( comp_set );
		}
		tmp = tmp->next;
	}

	return( NULL );
}


void compile_syntax_check( GtkWidget *widget, gpointer data )
{
	gI_document *current = gI_document_get_current( main_window );
	gI_project *project;
	gchar exec_string[MAXLEN];
	gchar filename[STRLEN];
	gchar HelpString[STRLEN];
	gchar exec_file[STRLEN];
	gchar *ptr;
	udata *update_data;
	gchar *items[1];
	gint pid = -1;
#ifdef USE_FORK_CODE
	gchar *argv[64];
	glong i;
#endif

	if( !current )
		return;

	if( current->working )
		return;

	if( !(project = gI_project_get_current()) && current->filename == NULL )
		return;

	if( current->compile_window == NULL )
	{
		current->compile_window = gI_compile_window_new( _("Syntax Check Window") );
		gtk_widget_show( current->compile_window->window );
	}

	/* tmp filename */
	g_snprintf( filename, STRLEN, "%s/gide_syntax.%d.%d", cfg->tmpdir,
	            (gint) time(0), (gint)getpid() );

	/* if there's already a file with this 'unique', unlink it */
	unlink( filename );

	if( mkfifo( filename, S_IRUSR|S_IWUSR ) < 0 )
	{
		sprintf( HelpString, "File Error: %s", strerror( errno ) );
		gI_error_dialog( HelpString);
		return;
	}

	strcpy( exec_file, current->filename );
	ptr = match_cset_fpat( get_comp_set_from_filename( current->filename ), exec_file );
	if( !ptr )	/* default is '.c' */
		ptr = strstr( exec_file, ".c" );
	if( ptr )
		*ptr = '\0';


	if( project )
	{
		/* no project handling code yet */
		return;
	}

	if( current->filename )
	{
		strcpy( exec_string, build_comp_string( current, exec_file, filename, COMPILE_SYNTAX ) );

#ifdef USE_FORK_CODE
		for(i=0;i<64;i++)
			argv[i] = (char *) g_malloc0( 100 );

		build_comp_string_fork( exec_string, argv );

		pid = gI_exec( argv, filename );

		for(i=0;i<64;i++)
			g_free( argv[i] );
#else
		system( exec_string );
#endif

		current->compile_window->pid = pid;

		/* show command line */
		ptr = strstr( exec_string, "1>" );
		if( ptr )
			*ptr = '\0';
		ptr = strchr( exec_string, '(' );
		ptr++;
		items[0] = ptr;
		gtk_clist_append( GTK_CLIST( current->compile_window->list ), items );

		file = open( filename, O_RDONLY );
		if( file < 0 )
		{
			perror( "open" );
			return;
		}

		update_data = (udata *) malloc( sizeof( udata ) );
		update_data->file = file;
		update_data->doc = current;
		update_data->filename = g_strdup( filename );

		/* get process id of the shell */
		/*fgets( buf, sizeof(buf), update_data->file );
		*document->compile_window->pid = atoi( buf );*/

		update_data->input_id = gdk_input_add( update_data->file, GDK_INPUT_READ, (GdkInputFunction) update_compile_list, (gpointer) update_data );

		return;
	}
}


#ifdef USE_FORK_CODE
void table_assoc( gint pid, udata *update_data )
{
	pdata *data;

	data = (pdata *) g_malloc0( sizeof( pdata ) );

	data->pid = pid;
	data->update_data = update_data;

	pid_data = g_list_prepend( pid_data, data );
}

udata *table_lookup( gint pid )
{
	pdata *data;
	GList *tmp;

	data = NULL;

	tmp = g_list_first( pid_data );
	while( tmp )
	{
		data = (pdata *) tmp->data;
		if( data && data->pid == pid )
			break;

		tmp = g_list_next( tmp );
	}

	if( !data )
		return NULL;

	return data->update_data;
}

void table_rassoc( udata *update_data )
{
	pdata *data;
	GList *tmp;
	gint pid;

	g_return_if_fail( update_data != NULL );
	g_return_if_fail( update_data->doc != NULL );
	g_return_if_fail( update_data->doc->compile_window != NULL );

	data = NULL;
	pid = update_data->doc->compile_window->pid;

	tmp = g_list_first( pid_data );
	while( tmp )
	{
		data = (pdata *) tmp->data;
		if( data->pid == pid )
		{
			g_list_remove( pid_data, data );
			break;
		}

		tmp = g_list_next( tmp );
	}
}
#endif

