%
% Copyright (C) 1997 Thomas Nordin and Alastair Reid
%

\begin{code}
module DIS
    ( DIS(..)
    , apply
    , ppDIS, ppDIS'
    , expandDIS, DISEnv
    , freeVarsOfDIS 
    , simplify
    ) where

import Casm( Kind(..), ppKind )
import Name( Name ) 
import ErrMonad

import Pretty
import PrettyUtils( ppTuple, commaList, textline )

import Char( isAlpha
	   , isDigit
	   , isLower
	   )


\end{code}

I'd like to get rid of this definition

\begin{code}
-- The DIS table maps a user defined DIS to its definition.
type DISTable = [(Name, ([Name], DIS))]
\end{code}

%************************************************************************
%*                                                                      *
\subsection{DIS data structure}
%*                                                                      *
%************************************************************************

\begin{code}

data DIS
  = Var Name
  | CCode String
  | Kind Kind
    {-
     DIS application.
     Invariant: the argument list is never empty.
    -}
  | Apply DIS [DIS]
    {-
     Invariant: the field list is never empty.
    -}
  | Record Name [Name]
  | Constructor Name
     {-
      Invariant : declared DIS can only be a Var or CCode
     -}
  | Declare String DIS 
  | Tuple 
  | UserDIS Bool         -- are the marshall/unmarshall expressions IO actions?
            (Maybe Kind) -- the primitive type the marshall expression maps to.
            String       -- marshall (+args)
            String       -- unmarshall (+args)

instance Show DIS where
  showsPrec p d = showsPrec p (ppDIS d)

\end{code}

Always use this constructor to maintain the invariant that the
args part of an apply is non-empty.

\begin{code}

apply :: DIS -> [DIS] -> DIS
apply f [] = f
apply f as = Apply f as

\end{code}

%************************************************************************
%*                                                                      *
\subsection{Pretty Printing of DISs}
%*                                                                      *
%************************************************************************

\begin{code}

ppDIS :: DIS -> Doc
ppDIS = ppDIS' False

-- ppDIS' Can either print the type casts or not.

ppDIS' :: Bool -> DIS -> Doc
ppDIS' showCasts dis = pp dis
 where

  pp (Apply Tuple ds)            = ppTuple (pps ds)
  pp (Apply (Record name fs) ds) = text name <+> braces (commaList fields)
   where
    fields = zipWith (\n d -> textline [n, "="] <+> d) fs (pps ds)
  pp (Apply (Declare exp nm) [d]) = text ("(declare { "   ++ 
                                                    exp ++ 
						  " } ") <> 
						  pp nm  <> 
						  text  " in "  <>
						  pp d <> 
						  char ')'
  pp (Apply d ds)                = parens (pp d <+> hsep (pps ds))
  pp (Record nm fs)    		 = text "<record>"
  pp (Constructor nm)  		 = text nm
  pp Tuple             		 = text "()" -- unit
  pp (CCode s)         		 = braces (text s)
  pp (UserDIS io k ma unma) = 
        (if io then text "<<" else char '<') <> 
	   text ma   <+>
        char '/' <> 
	   text unma <+> 
        (case k of
	  Nothing -> empty
	  Just v  -> char '/' <+> ppKind v) <>
	(if io then text ">>" else char '>')
  pp (Var nm)      		 = text nm
  pp (Kind k)          		 = ppKind k

  pps = map pp

\end{code}

%************************************************************************
%*                                                                      *
\subsection{Free Variables}
%*                                                                      *
%************************************************************************

\begin{code}

freeVarsOfDIS :: DIS -> [Name]
freeVarsOfDIS = free
 where
  free (Apply d ds)   = free d ++ concatMap free ds
  free (Var nm)       = [nm]
  free (Declare  _ d) = free d
  free _              = []

\end{code}

%************************************************************************
%*                                                                      *
\subsection{Expanding DISs}
%*                                                                      *
%************************************************************************

Expanding a DIS is rather like evaluating an expression: we walk over
the DIS with an environment replacing disnames and arguments with
values from the environment. The result is a DIS in normal form.

\begin{code}

type DISEnv = [(Name, ([Name], DIS))]
type ArgEnv = [(Name, DIS)]

expandDIS :: DISEnv -> DIS -> ErrM String DIS
expandDIS denv d = expandDIS' [] d
 where
  expandDIS' :: ArgEnv -> DIS -> ErrM String DIS
  expandDIS' aenv d = xp d
   where

    xp :: DIS -> ErrM String DIS
    xp (Apply f@(UserDIS _ _ _ _) ds) = do
        ds' <- mapM xp ds
        f'  <- xp f
        return (Apply f' ds')

    xp r@(Apply f@(Var nm) ds) =
        case (lookup nm denv) of 
          Just (args, d)  
	    | length args == length ds -> do
                   ds' <- mapM xp ds
	           expandDIS' (zip args ds') d
	    | otherwise ->
	        failure 
		 (show $
		  text "" $$ 
		  hang (text "")
		   8   (hang (text "DIS application" <+> quotes (ppDIS r) <+>
			      text "incompatible with definition:")
			 4  (text "%dis" <+> hsep (map text (nm:args)) <+> 
			     equals <+> ppDIS d)))
          Nothing -> do
             ds' <- mapM xp ds
	     return (Apply f ds')

    xp (Apply d ds) = do
        d'  <- xp d
	ds' <- mapM xp ds
        return (Apply d' ds')

    xp v@(Var nm) =
        case (lookup nm aenv) of 
          Just d   -> return d
          Nothing  -> return v

    xp (Declare ctype cv) = do
        ctype' <- subst ctype
        cv'    <- xp cv
        return (Declare ctype' cv')

    xp v@(CCode ccode) = do
        ccode' <- subst ccode
        return (CCode ccode')

    xp (UserDIS io k ma unma) = do
        ma'   <- subst ma
        unma' <- subst unma
        return (UserDIS io k ma' unma')

    -- everything else is already in normal form
    xp d = return d

    -- substitute for anything of the form %[a-z][a-zA-Z0-9]*
    subst :: String -> ErrM String String
    subst ""           = return ""
    subst ('%':'%':cs) = do  -- escape code
        cs' <- subst cs
	return ('%' : cs')
    subst ('%':c:cs) | isLower c =
        case lookup nm aenv of
          Just (CCode c) -> do
	       rest' <- subst rest
	       return (c ++ rest')
          Just (Var   v) -> do
	       rest' <- subst rest
	       return (v ++ rest')
          Just d'        -> do
	       failure ("Can't substitute " ++ 
	                show (ppDIS d') ++ " for "  ++ 
			nm ++ " in DIS "    ++ show (ppDIS d))
          Nothing -> do
	       failure ("Unknown variable " ++ 
	                nm ++ " in DIS " ++ show (ppDIS d))
     where
	-- Hack around the (needless) API change that Haskell 98
	-- makes of turning 'isAlphanum' into 'isAlphaNum'.
      (cs', rest) = span (\ x -> isAlpha x || isDigit x) cs
      nm = c:cs'

    subst (c:cs)  = do
      cs' <- subst cs
      return (c:cs')
\end{code}

%************************************************************************
%*                                                                      *
\subsection{Simplify DISs}
%*                                                                      *
%************************************************************************

Simplify a DIS by pushing casts down to the leaves

\begin{code}
simplify :: DIS -> DIS
simplify (Apply (CCode cty) [dis@(Apply (CCode cty') _)]) = simplify dis
simplify (Apply f as) = Apply (simplify f) (map simplify as)
simplify dis          = dis
\end{code}

%************************************************************************
%*                                                                      *
\subsection{Example DISs}
%*                                                                      *
%************************************************************************

COMMENTED OUT:

\begin{code}
{-
dis1 = Apply (Kind Int)   [ CCode "int",   Var "x" ]
dis2 = Apply (Kind Float) [ CCode "float", Var "y" ]
dis3 = Apply Tuple [dis1,dis2]

disenv1 =
  [ ( "int",   (["x"], dis1) )
  , ( "float", (["y"], dis2) )
  ]

dis4 = Apply (Var "int") [Var "arg1"]
dis5 = expandDIS disenv1 dis4
-}
\end{code}


