/*
 * PDB.H - header file supporting PDBLIB routines
 *       -
 * Source Version: 9.0
 * Software Release #92-0043
 *
 */

#ifndef PCK_PDB

#include "cpyright.h"

#define PCK_PDB                       /* announce that this package is here */

/* Think C compiler defines THINK_C; CodeWarrior defines __MWERKS__ */

#ifdef THINK_C
# include "score.h"
# include "pml.h"
#else
# ifdef __MWERKS__
#  include "score.h"
#  include "pml.h"
# else
#  include <score.h>
#  include <pml.h>
# endif
#endif


/* PDBLib History:
 * Version 4 contains bug fixes pertaining to cast information in PDB files
 * and how it is read in (the bug was that the DATE returned by SC_date
 * had a newline at the end which buggered _PD_rd_extras) - 3/19/91
 *
 * Version 5 contains the EXTRA defining row major or column major
 * ordering for multidimensional arrays - 3/22/91
 *
 * Version 6 contains the primitive defintions in the EXTRA table
 * done in a way which is compatible with old files (pre Version 6)
 * this allows applications to define new primitive data types - 2/25/92
 *
 * Version 7 has the new interface with the functions from the old
 * interface deprecated.  The new functions are: PD_write_alt,
 * PD_write_as_alt, PD_defent, PD_defent_alt, and PD_part_read.
 * The old functions are: PD_r_write, PD_v_write, PD_l_write, PD_r_part_read,
 * and PD_v_part_read. This new interface is aimed at general partial
 * reads and writes via a hyper slab approach. 5/29/92
 *
 * Version 8 has the deprecated functions removed and has the discontiguous
 * data block stuff added.  ITAG flags have a third possible value of 2
 * This signals a discontiguous data block.  That is a pointered variable
 * may be written in discontiguous blocks of data with each succeeding
 * block pointing to the previous one.  Syment was also modified to use
 * an array of address, number structs describing the discontiguous blocks.
 * Using PD_entry_xxx macros helps with backward compatibility. 5/17/93
 *
 * Version 9 has file directories. The new functions are: PD_ls, PD_mkdir,
 * PD_ln, PD_cd, PD_pwd. 6/4/93
 *
 * Version 10 has the uncomputable additional struct alignment handled
 * properly. 7/23/93
 *
 * Version 11 has support for families of files with links to previous files
 * and maximum size per file in the family.  PD_family will spawn the next
 * file in the family. 7/27/93
 *
 * Version 12 has conversions done on a per type rather than a per file basis
 * and it read/writes the structure chart, symbol table, and extras table
 * as single blocks of data for improved performance (especially in the
 * remote file access case). 1/24/94
 *
 * Version 13 uses a LALR(1) grammar and parser for expressions which access
 * parts of entries. This is both more robust and complete than previous
 * versions. 1/6/94
 *
 * Version 14 recognizes and converts bitstreams.  This allows PDB to open
 * and read data in files written on old machines with byte sizes other
 * than 8 bits and to support the use of bit fields in data files which
 * PDB may also have to access. 1/8/94
 *
 * Version 15 adds symbol table entry for dynamically allocated spaces
 * (pointees).  This is a step toward the next generation pointer
 * handling. 10/15/97
 *
 */

#define PDB_SYSTEM_VERSION  15

/* PDBFile Layout:
 *       
 *       Header (ASCII)
 *         Identifying token
 *         Data Formats
 *         Structure Chart Address
 *         Symbol Table Address
 *       Data
 *       Structure Chart
 *       Symbol Table
 *       Extensions
 *       EOF
 */

#define NORMAL_ORDER    1
#define REVERSE_ORDER   2  /* this must not be -1 which turns out to be EOF */

#define IEEE_32_64   1        /* IEEE standard 32 bit float : 64 bit double */
#define INTEL_X86    2                        /* Intel 80x86 class machines */
#define CRAY_64      3                               /* CRAY class machines */
#define VAX_11       4                                /* VAX class machines */
#define IEEE_32_96   6        /* IEEE standard 32 bit float : 96 bit double */
#define NSTD         6       /* Number of standards currently in the system 
                                            should be same as last standard */

#define BITS_DEFAULT 6        /* Default bits per byte */
#define PD_READ   0
#define PD_WRITE  1
#define PD_APPEND 2
#define PD_OPEN   3
#define PD_CREATE 4
#define PD_CLOSE  5
#define PD_TRACE  6
#define PD_PRINT  7

#define ROW_MAJOR_ORDER     101
#define COLUMN_MAJOR_ORDER  102

#define PF_ALPHABETIC 1
#define PF_DISK_ORDER 2

#define X_AND_Y 1
#define X_ONLY  2
#define Y_ONLY  3

#define pio_open    (*(PIO_OPEN_HOOK(_t_index)))
#define pio_setvbuf (*(PIO_SETVBUF_HOOK(_t_index)))
#define pio_tell    (*(PIO_TELL_HOOK(_t_index)))
#define pio_read    (*(PIO_READ_HOOK(_t_index)))
#define pio_write   (*(PIO_WRITE_HOOK(_t_index)))
#define pio_close   (*(PIO_CLOSE_HOOK(_t_index)))
#define pio_seek    (*(PIO_SEEK_HOOK(_t_index)))
#define pio_printf  (*(PIO_PRINTF_HOOK(_t_index)))
#define pio_puts    (*(PIO_PUTS_HOOK(_t_index)))
#define pio_getc    (*(PIO_GETC_HOOK(_t_index)))
#define pio_ungetc  (*(PIO_UNGETC_HOOK(_t_index)))
#define pio_flush   (*(PIO_FLUSH_HOOK(_t_index)))
#define pio_gets    (*(PIO_GETS_HOOK(_t_index)))

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

#define PN_sizeof(type, tab)  _PD_lookup_size(type, tab)

#define PD_get_mode(file)              ((file)->mode)
#define PD_set_mode(file, v)           (file)->mode = (v)
#define PD_get_offset(file)            ((file)->default_offset)
#define PD_set_offset(file, v)         (file)->default_offset = (v)
#define PD_get_track_pointers(file)    ((file)->track_pointers)
#define PD_set_track_pointers(file, v) (file)->track_pointers = (v)

/* #define PD_file_maximum_size(file) file->maximum_size           */
/* replaced by PD_get_max_file_size and PD_set_max_file_size below */

#define PD_get_max_file_size(file)     ((file)->maximum_size)
#define PD_set_max_file_size(file, v)  (file)->maximum_size = (v)


#define PD_entry_type(ep)          ((ep)->type)
#define PD_entry_dimensions(ep)    ((ep)->dimensions)
#define PD_entry_address(ep)       ((ep)->blocks[0].diskaddr)
#define PD_entry_number(ep)        ((ep)->number)
#define PD_entry_indirects(ep)     ((ep)->indirects)
#define PD_entry_blocks(ep)        ((ep)->blocks)
#define PD_block_address(ep, n)    ((ep)->blocks[n].diskaddr)
#define PD_block_number(ep, n)     ((ep)->blocks[n].number)
#define PD_n_blocks(ep)            (SC_arrlen((ep)->blocks)/sizeof(symblock))

#define PD_get_major_order(file)       ((file)->major_order)
#define PD_set_major_order(file, type) (file)->major_order = (type)

#define PD_get_file_name(file)       ((file)->name)

#define PD_has_directories(file)                                             \
 (file->current_prefix != NULL)

#define PD_inquire_type(file, name)                                          \
 (defstr *) SC_def_lookup(name, file->chart)

#define PD_inquire_host_type(file, name)                                     \
 (defstr *) SC_def_lookup(name, file->host_chart)

#define PD_inquire_table_type(table, name)                                   \
 (defstr *) SC_def_lookup(name, table)

#define PD_hyper_refp(name, commnd)                                          \
   {char c, s[MAXLINE];                                                      \
    strcpy(s, name);                                                         \
    c = s[strlen(s) - 1];                                                    \
    if ((c == ')') || (c == ']'))                                            \
       (commnd);}

#define PD_hyper_ref(s, name)                                                \
   {char c;                                                                  \
    strcpy(s, name);                                                         \
    c = s[strlen(s) - 1];                                                    \
    if ((c == ')') || (c == ']'))                                            \
       {s[strlen(s) - 1] = '\0';                                             \
        SC_lasttok(s, "[]()");};}

#define PD_CAST_TYPE(t, d, vg, vl, err_fnc, err_msg, err_arg)                \
    {if (d->cast_offs < 0L)                                                  \
        t = d->type;                                                         \
     else                                                                    \
        {t = DEREF(vl + d->cast_offs);                                       \
         if (t == NULL)                                                      \
            {if (DEREF(vg) != NULL)                                          \
                {err_fnc(err_msg, err_arg);}                                 \
             else                                                            \
                t = d->type;};};}

#ifdef HAVE_THREADS

/* Application interface to library state */
#define PD_BUFFER_SIZE(t)       PD_threads[t].PD_buffer_size 
#define PD_ERR(t)               PD_threads[t].PD_err

/* Internal library interface to state */
#define _PD_DIM(t)              PD_threads[t].PD_DIM
#define _PD_INT_STANDARD(t)     PD_threads[t].INT_STANDARD
#define _PD_REQ_STANDARD(t)     PD_threads[t].REQ_STANDARD
#define _PD_INT_ALIGNMENT(t)    PD_threads[t].INT_ALIGNMENT
#define _PD_REQ_ALIGNMENT(t)    PD_threads[t].REQ_ALIGNMENT
#define _PD_VIF(t)              PD_threads[t].PD_vif
#define _PD_READ_ERR(t)         PD_threads[t]._PD_read_err
#define _PD_WRITE_ERR(t)        PD_threads[t]._PD_write_err
#define _PD_PRINT_ERR(t)        PD_threads[t]._PD_print_err
#define _PD_OPEN_ERR(t)         PD_threads[t]._PD_open_err
#define _PD_TRACE_ERR(t)        PD_threads[t]._PD_trace_err
#define _PD_CLOSE_ERR(t)        PD_threads[t]._PD_close_err
#define _PD_CREATE_ERR(t)       PD_threads[t]._PD_create_err
#define _PD_WR_HOOK(t)          PD_threads[t].pdb_wr_hook
#define _PD_RD_HOOK(t)          PD_threads[t].pdb_rd_hook

#define PIO_OPEN_HOOK(t)     PD_threads[t].pio_open_hook
#define PIO_TELL_HOOK(t)     PD_threads[t].pio_tell_hook
#define PIO_READ_HOOK(t)     PD_threads[t].pio_read_hook
#define PIO_WRITE_HOOK(t)    PD_threads[t].pio_write_hook
#define PIO_SETVBUF_HOOK(t)  PD_threads[t].pio_setvbuf_hook
#define PIO_CLOSE_HOOK(t)    PD_threads[t].pio_close_hook
#define PIO_SEEK_HOOK(t)     PD_threads[t].pio_seek_hook
#define PIO_PRINTF_HOOK(t)   PD_threads[t].pio_printf_hook
#define PIO_PUTS_HOOK(t)     PD_threads[t].pio_puts_hook
#define PIO_GETC_HOOK(t)     PD_threads[t].pio_getc_hook
#define PIO_UNGETC_HOOK(t)   PD_threads[t].pio_ungetc_hook
#define PIO_FLUSH_HOOK(t)    PD_threads[t].pio_flush_hook
#define PIO_GETS_HOOK(t)     PD_threads[t].pio_gets_hook

#define PD_GET_PTR_STREAM(pp, fp)                                            \
    {PD_Pfile *_pfp_;                                                        \
     _pfp_ = ((pp) != NULL) ? (PD_Pfile *)((pp)->stream) : NULL;             \
     fp    = (_pfp_ != NULL) ? (FILE *)(_pfp_->stream) : NULL;}

#else

#define _PD_T_INDEX(x)

/* Application interface to library state */
#define PD_BUFFER_SIZE(t)       PD_buffer_size 
#define PD_ERR(t)               PD_err

/* Internal library interface to state */
#define PIO_OPEN_HOOK(t)     io_open_hook
#define PIO_TELL_HOOK(t)     io_tell_hook
#define PIO_READ_HOOK(t)     io_read_hook
#define PIO_WRITE_HOOK(t)    io_write_hook
#define PIO_SETVBUF_HOOK(t)  io_setvbuf_hook
#define PIO_CLOSE_HOOK(t)    io_close_hook
#define PIO_SEEK_HOOK(t)     io_seek_hook
#define PIO_PRINTF_HOOK(t)   io_printf_hook
#define PIO_PUTS_HOOK(t)     io_puts_hook
#define PIO_GETC_HOOK(t)     io_getc_hook
#define PIO_UNGETC_HOOK(t)   io_ungetc_hook
#define PIO_FLUSH_HOOK(t)    io_flush_hook
#define PIO_GETS_HOOK(t)     io_gets_hook

#define _PD_DIM(t)               PD_DIM
#define _PD_INT_STANDARD(t)      INT_STANDARD
#define _PD_REQ_STANDARD(t)      REQ_STANDARD
#define _PD_INT_ALIGNMENT(t)     INT_ALIGNMENT
#define _PD_REQ_ALIGNMENT(t)     REQ_ALIGNMENT
#define _PD_VIF(t)               PD_vif
#define _PD_READ_ERR(t)          _PD_read_err
#define _PD_WRITE_ERR(t)         _PD_write_err
#define _PD_PRINT_ERR(t)         _PD_print_err
#define _PD_OPEN_ERR(t)          _PD_open_err
#define _PD_TRACE_ERR(t)         _PD_trace_err
#define _PD_CLOSE_ERR(t)         _PD_close_err
#define _PD_CREATE_ERR(t)        _PD_create_err
#define _PD_WR_HOOK(t)           pdb_wr_hook
#define _PD_RD_HOOK(t)           pdb_rd_hook

#define PD_GET_PTR_STREAM(pp, fp)                                            \
    (fp = ((pp) != NULL) ? (pp)->stream : NULL)

#endif

/*--------------------------------------------------------------------------*/

/*                          TYPEDEFS AND STRUCTS                            */

/*--------------------------------------------------------------------------*/

struct s_data_alignment
   {int char_alignment;
    int ptr_alignment;
    int short_alignment;
    int int_alignment;
    int long_alignment;
    int longlong_alignment;
    int float_alignment;
    int double_alignment;
    int struct_alignment;};

typedef struct s_data_alignment data_alignment;


struct s_data_standard
   {int bits_byte;
    int ptr_bytes;
    int short_bytes;
    int short_order;
    int int_bytes;
    int int_order;
    int long_bytes;
    int long_order;
    int longlong_bytes;
    int longlong_order;
    int float_bytes;
    long *float_format;
    int *float_order;
    int double_bytes;
    long *double_format;
    int *double_order;};

typedef struct s_data_standard data_standard;

struct s_PD_Pfile
   {SC_address *f_addr;  /* current address (file) (per thread)*/
    SC_address *bf;      /* buffer (per thread) */
    FILE *stream;
    long *bf_len;        /* length of buffer (per thread) */
    int id;};            /* id  (for file manager)*/

typedef struct s_PD_Pfile PD_Pfile;

struct s_PDBfile
   {FILE *stream;
    char *name;
    char *type;                        /* used by files masqerading as PDB */
    HASHTAB *symtab;
    HASHTAB *chart;
    HASHTAB *host_chart;
    HASHTAB *attrtab;
    char *previous_file;                /* for file family bookkeeping */
    int mode;
    int default_offset;
    data_standard *std;
    data_standard *host_std;
    data_alignment *align;
    data_alignment *host_align;
    char *current_prefix;
    int flushed;
    int virtual_internal;
    int system_version;
    int major_order;
    int track_pointers;
    char *date;
    char **ptr_rd_list;
    char **ptr_wr_list;
    off_t *addr_rd_list;
    off_t *reta_rd_list;
    off_t *addr_wr_list;
    long max_rd_indx;
    long max_wr_indx;
    long rd_indx;
    long wr_indx;
    long n_dyn_spaces;
    off_t maximum_size;                  /* for file family bookkeeping */
    off_t headaddr;
    off_t symtaddr;
    off_t chrtaddr;};

typedef struct s_PDBfile PDBfile;


/* dimension descriptor - describe an array dimension range
 * NOTE: changes here must be reflected in sxhook.c
 */
struct s_dimdes
   {long index_min;
    long index_max;
    long number;
    struct s_dimdes *next;};

typedef struct s_dimdes dimdes;


/* member descriptor - describe a member efficiently
 * NOTE: changes here must be reflected in sxhook.c
 */
struct s_memdes
   {char *member;
    off_t member_offs;
    char *cast_memb;
    long cast_offs;
    char *type;
    char *base_type;
    char *name;
    dimdes *dimensions;
    long number;
    struct s_memdes *next;};

typedef struct s_memdes memdes;

FUNCTION_POINTER(memdes, *(*PFPmemdes));


/* symbol table entry block - manage discontiguous data items
 * NOTE: changes here must be reflected in sxhook.c
 */
struct s_symblock
   {long number;
    off_t diskaddr;};

typedef struct s_symblock symblock;


/* symbol table entry indirects
 * NOTE: changes here must be reflected in sxhook.c
 */
struct s_symindir
   {off_t addr;
    long n_ind_type;
    long arr_offs;};

typedef struct s_symindir symindir;


/* symbol table entry - the def member of a hashel
 * NOTE: changes here must be reflected in sxhook.c
 */
struct s_syment
   {char *type;
    dimdes *dimensions;
    long number;
    symindir indirects;
    symblock *blocks;};

typedef struct s_syment syment;


/* structure chart entry - the def member of a hashel
 * NOTE: changes here must be reflected in sxhook.c
 */
struct s_defstr
   {char *type;
    long size_bits;
    long size;
    int alignment;   /* type begins on byte boundary divisible by alignment */
    int n_indirects;                              /* number of indirections */
    int convert;
    int onescmp;      /* TRUE iff ones complement arithmetic - old machines */
    int unsgned;                  /* TRUE iff the integral type is unsigned */
    int order_flag;
    int *order;
    long *format;
    memdes *members;};

typedef struct s_defstr defstr;


/* indirect reference tag */
struct s_PD_itag
   {long nitems;                              /* number of items pointed to */
    char *type;                                  /* type of item pointed to */
    off_t addr;                          /* address of the itag owning data */
    int flag;};                          /* TRUE if this itag owns the data */

typedef struct s_PD_itag PD_itag;


/* hyper-index support structure */
struct s_dimind
   {long stride;
    long start;
    long stop;
    long step;};

typedef struct s_dimind dimind;


struct s_attribute
   {char *name;
    char *type;
    byte **data;
    long size;
    long indx;};

typedef struct s_attribute attribute;


struct s_attribute_value
   {attribute *attr;
    long indx;
    struct s_attribute_value *next;};

typedef struct s_attribute_value attribute_value;


/* PD_IMAGE is a faithful shadow of the PG_image which is not defined yet
 * any changes to PG_image must be reflected here
 */

struct s_PD_image
   {int version_id;
    char *label;
    double xmin;
    double xmax;
    double ymin;
    double ymax;
    double zmin;
    double zmax;
    char *element_type;
    unsigned char *buffer;
    int kmax;
    int lmax;
    long size;
    int bits_pixel;
    char *palette;};

typedef struct s_PD_image PD_image;

struct s_PD_thread_state
   {char PD_err[MAXLINE];
    int  PD_buffer_size;
    int  PD_DIM;
    data_standard *INT_STANDARD;
    data_standard *REQ_STANDARD;
    data_alignment *INT_ALIGNMENT;
    data_alignment *REQ_ALIGNMENT;
    PDBfile *PD_vif;
    jmp_buf _PD_read_err;
    jmp_buf _PD_write_err;
    jmp_buf _PD_print_err;
    jmp_buf _PD_open_err;
    jmp_buf _PD_trace_err;
    jmp_buf _PD_close_err;
    jmp_buf _PD_create_err;
    PFPmemdes pdb_wr_hook;
    PFPmemdes pdb_rd_hook;
    PFfopen pio_open_hook;
    PFftell pio_tell_hook;
    PFfread pio_read_hook;
    PFfwrite pio_write_hook;
    PFsetvbuf pio_setvbuf_hook;
    PFfclose pio_close_hook;
    PFfseek pio_seek_hook;
    PFfprintf pio_printf_hook;
    PFfputs pio_puts_hook;
    PFfgetc pio_getc_hook;
    PFungetc pio_ungetc_hook;
    PFfflush pio_flush_hook;
    PFfgets pio_gets_hook;};

typedef struct s_PD_thread_state PD_thread_state;
    
#ifdef __cplusplus
extern "C" {
#endif

/*--------------------------------------------------------------------------*/

/*                            PDBLIB VARIABLES                              */

/*--------------------------------------------------------------------------*/

#ifdef HAVE_THREADS

extern PD_thread_state
   *PD_threads;

extern int
 _PD_nthreads;

#endif

extern long
 PD_print_controls[10];

extern int
 FORMAT_FIELDS,
 PD_buffer_size,
 PD_DIM,
 PD_short_digits,
 PD_int_digits,
 PD_long_digits,
 PD_long_long_digits,
 PD_float_digits,
 PD_double_digits,
 PD_tolerance;

extern double
 PD_float_tol,
 PD_double_tol;

extern data_standard
 *INT_STANDARD,
 *REQ_STANDARD,
 *PD_std_standards[];

extern data_alignment
 *INT_ALIGNMENT,
 *REQ_ALIGNMENT,
 *PD_std_alignments[];

extern PDBfile
 *PD_vif;

extern char
 *PD_ALIGNMENT_S,
 *PD_DEFSTR_S,
 *PD_STANDARD_S,
 *PD_SYMENT_S,
 PD_err[],
 *PD_print_formats1[8],
 *PD_print_formats2[8],
 *PD_user_formats1[8],
 *PD_user_formats2[8];

extern jmp_buf
 _PD_read_err,
 _PD_write_err,
 _PD_print_err,
 _PD_open_err,
 _PD_trace_err,
 _PD_close_err,
 _PD_create_err;

extern PFPmemdes
 pdb_wr_hook,
 pdb_rd_hook;

extern data_standard
 DEF_STD,
 IEEEA_STD,
 IEEEB_STD,
 IEEEC_STD,
 IEEED_STD,
 IEEEE_STD,
 INTELA_STD,
 INTELB_STD,
 VAX_STD,
 CRAY_STD;

extern data_alignment
 RS6000_ALIGNMENT,
 SPARC_ALIGNMENT,
 MIPS_ALIGNMENT,
 UNICOS_ALIGNMENT,
 M68000_ALIGNMENT,
 INTELA_ALIGNMENT,
 INTELB_ALIGNMENT,
 INTELC_ALIGNMENT,
 VAX_ALIGNMENT,
 MIPS64_ALIGNMENT,
 ALPHA64_ALIGNMENT,
 DEF_ALIGNMENT;

/* IO hooks */

extern PFfopen pio_open_hook;
extern PFftell pio_tell_hook;
extern PFfread pio_read_hook;
extern PFfwrite pio_write_hook;
extern PFsetvbuf pio_setvbuf_hook;
extern PFfclose pio_close_hook;
extern PFfseek pio_seek_hook;
extern PFfprintf pio_printf_hook;
extern PFfputs pio_puts_hook;
extern PFfgetc pio_getc_hook;
extern PFungetc pio_ungetc_hook;
extern PFfflush pio_flush_hook;
extern PFfgets pio_gets_hook;

/*--------------------------------------------------------------------------*/

/*                            PDBLIB FUNCTIONS                              */

/*--------------------------------------------------------------------------*/


/* PDB.C declarations */

extern PDBfile
 *SC_DECLARE(PD_open, (char *name, char *mode)),
 *SC_DECLARE(PD_family, (PDBfile *file, int flag)),
 *SC_DECLARE(PD_create, (char *name)),
 *SC_DECLARE(PD_open_vif, (char *name));

extern syment
 *SC_DECLARE(_PD_defent,
	  (PDBfile *file, char *name, char *outtype,
	   long number, dimdes *dims)),
 *SC_DECLARE(PD_defent, (PDBfile *file, char *name, char *outtype)),
 *SC_DECLARE(PD_defent_alt,
	  (PDBfile *file, char *name, char *outtype,
	   int nd, long *ind)),
 *SC_DECLARE(_PD_write,
	  (PDBfile *file, char *name, char *intype, char *outtype,
	   byte *vr, dimdes *dims, int appnd, int *pnew));

extern defstr
 *SC_DECLARE(PD_typedef, (PDBfile *file, char *oname, char *tname)),
 *SC_DECLARE(PD_defncv,
	  (PDBfile *file, char *name, long bytespitem, int align)),
 *SC_DECLARE(PD_defix,
	  (PDBfile *file, char *name, long bytespitem, int align,
	   int flg)),
 *SC_DECLARE(PD_defixnum,
	  (PDBfile *file, char *name, long bytespitem, int align,
	   int flg, int unsgned, int onescmp)),
 *SC_DECLARE(PD_defloat,
	  (PDBfile *file, char *name, long bytespitem, int align,
	   int *ordr, long expb, long mantb, long sbs, long sbe,
	   long sbm, long hmb, long bias)),
 *SC_DECLARE(PD_defstr, (PDBfile *file, char *name, ...)),
 *SC_DECLARE(PD_defstr_alt,
	  (PDBfile *file, char *name, int nmemb, char **members));

extern int
 SC_DECLARE(PD_close, (PDBfile *file)),
 SC_DECLARE(PD_flush, (PDBfile *file)),
 SC_DECLARE(PD_append, (PDBfile *file, char *name, byte *vr)),
 SC_DECLARE(PD_append_as,
         (PDBfile *file, char *name, char *intype, byte *vr)),
 SC_DECLARE(PD_append_alt,
         (PDBfile *file, char *name, byte *vr,
          int nd, long *ind)),
 SC_DECLARE(PD_append_as_alt,
         (PDBfile *file, char *name,
          char *intype, byte *vr,
          int nd, long *ind)),
 SC_DECLARE(PD_write, 
            (PDBfile *file, char *name, char *type, byte *vr)),
 SC_DECLARE(PD_write_as,
         (PDBfile *file, char *name,
          char *intype, char *outtype, byte *vr)),
 SC_DECLARE(PD_write_alt,
         (PDBfile *file, char *name, char *type, byte *vr,
          int nd, long *ind)),
 SC_DECLARE(PD_write_as_alt,
         (PDBfile *file, char *name,
          char *intype, char *outtype, byte *vr,
          int nd, long *ind)),
 SC_DECLARE(PD_read, (PDBfile *file, char *name, byte *vr)),
 SC_DECLARE(PD_read_as, 
            (PDBfile *file, char *name, char *type, byte *vr)),
 SC_DECLARE(PD_read_as_dwim, 
	    (PDBfile *file, char *name, char *outtype,
	     long nix, byte *space)),
 SC_DECLARE(PD_read_alt, 
            (PDBfile *file, char *name, byte *vr, long *ind)),
 SC_DECLARE(PD_read_as_alt,
         (PDBfile *file, char *name, char *type, byte *vr,
          long *ind)),
 SC_DECLARE(PD_target, (data_standard *data, data_alignment *align)),
 SC_DECLARE(PD_cast, 
            (PDBfile *file, char *type, char *memb, char *contr));

extern void
 SC_DECLARE(_PD_sign_extend, (char *out, long nitems,
			   int nbo, int nbti, int *ord)),
 SC_DECLARE(_PD_pdb_pinit, (byte)),
 SC_DECLARE(_PD_ones_complement, (char *out, long nitems, int nbo, int *order)),
 SC_DECLARE(PD_error, (char *s, int n));


/* PDBIO.C declarations */

extern int
 SC_DECLARE(_PD_pio_close, (FILE *stream)),
 SC_DECLARE(_PD_pio_seek, (FILE *stream, off_t addr, int offset)),
 SC_DECLARE(_PD_pio_printf, (FILE *fp, char *fmt, ...));


/* PDBMM.C declarations */

extern PDBfile
 *SC_DECLARE(_PD_mk_pdb, (char *s));

extern data_standard
 *SC_DECLARE(_PD_mk_standard, (byte)),
 *SC_DECLARE(_PD_copy_standard, (data_standard *std));

extern data_alignment
 *SC_DECLARE(_PD_mk_alignment, (char *vals)),
 *SC_DECLARE(_PD_copy_alignment, (data_alignment *align));

extern syment
 *SC_DECLARE(_PD_mk_syment,
	  (char *type, long numb, off_t addr, symindir *indr, dimdes *dims)),
 *SC_DECLARE(PD_copy_syment, (syment *osym));

extern defstr
 *SC_DECLARE(_PD_mk_defstr,
	  (HASHTAB *chrt, char *type, memdes *lst, long sz, int align,
	   int flg, int conv, int *ordr, long *formt,
	   int unsgned, int onescmp));

extern memdes
 *SC_DECLARE(PD_copy_members, (memdes *desc)),
 *SC_DECLARE(_PD_mk_descriptor, (char *member, int defoff));

extern dimdes
 *SC_DECLARE(_PD_mk_dimensions, (long mini, long leng)),
 *SC_DECLARE(PD_copy_dims, (dimdes *odims));

extern byte
 *SC_DECLARE(PD_alloc_entry, (PDBfile *file, char *name)),
 *SC_DECLARE(_PD_alloc_entry, 
             (PDBfile *file, char *type, long nitems));

extern void
 SC_DECLARE(_PD_rl_pdb, (PDBfile *file)),
 SC_DECLARE(_PD_rl_standard, (data_standard *std)),
 SC_DECLARE(_PD_rl_alignment, (data_alignment *align)),
 SC_DECLARE(_PD_clr_table, (HASHTAB *tab, PFVoid rel)),
 SC_DECLARE(_PD_rl_syment_d, (syment *ep)),
 SC_DECLARE(_PD_rl_syment, (syment *ep)),
 SC_DECLARE(_PD_rl_defstr, (defstr *dp)),
 SC_DECLARE(_PD_rl_descriptor, (memdes *desc)),
 SC_DECLARE(_PD_rl_dimensions, (dimdes *dims));


/* PDBDIR.C declarations */

extern char
 *SC_DECLARE(*PD_ls, 
             (PDBfile *file, char *path, char *type, int *num)),
 *SC_DECLARE(PD_pwd, (PDBfile *file)),
 *SC_DECLARE(_PD_fixname, (PDBfile *file, char *inname));

extern void
 SC_DECLARE(_PD_dir_pinit, (byte));

extern int
 SC_DECLARE(PD_cd, (PDBfile *file, char *dirname)),
 SC_DECLARE(PD_mkdir, (PDBfile *file, char *dirname)),
 SC_DECLARE(PD_ln, 
            (PDBfile *file, char *oldname, char *newname)),
 SC_DECLARE(PD_def_dir, (PDBfile *file));


/* PDBNET.C declarations */

extern void
 SC_DECLARE(PN_conv_in,
         (byte *out, byte *in, char *type, long nitems,
          HASHTAB *in_chart)),
 SC_DECLARE(_PD_net_pinit, (byte)),
 SC_DECLARE(PN_conv_out,
         (byte *out, byte *in, char *type, long nitems,
          HASHTAB *out_chart));

extern int
 SC_DECLARE(PN_close, (PDBfile *file)),
 SC_DECLARE(PN_write, (PDBfile *file, char *type, long nitems, byte *vr)),
 SC_DECLARE(PN_read, (PDBfile *file, char *type, long nitems, byte *vr));

extern PDBfile
 *SC_DECLARE(PN_open, (PDBfile *fm, char *bf));

extern HASHTAB
 *SC_DECLARE(PN_target, 
             (data_standard *data, data_alignment *align));

extern defstr
 *SC_DECLARE(PN_defstr,
	  (HASHTAB *fchart, char *name,
	   data_alignment *align, int defoff, ...));


/* PDBX.C declarations */

extern byte
 *SC_DECLARE(PD_get_attribute, 
             (PDBfile *file, char *vr, char *at));

extern int
 SC_DECLARE(PD_def_attr_str, (PDBfile *file)),
 SC_DECLARE(PD_def_attribute, 
            (PDBfile *file, char *at, char *type)),
 SC_DECLARE(PD_set_attribute, 
            (PDBfile *file, char *vr, char *at, byte *vl)),
 SC_DECLARE(PD_rem_attribute, (PDBfile *file, char *at)),
 SC_DECLARE(_PD_contains_indirections, (HASHTAB *tab, char *type)),
 SC_DECLARE(PD_read_pdb_curve,
         (PDBfile *fp, char *name, REAL **pxp, REAL **pyp,
          int *pn, char *label,
          REAL *pxmn, REAL *pxmx, REAL *pymn, REAL *pymx, int flag)),
 SC_DECLARE(PD_wrt_pdb_curve,
         (PDBfile *fp, char *labl, int n,
          REAL *px, REAL *py, int icurve)),
 SC_DECLARE(PD_wrt_pdb_curve_y,
         (PDBfile *fp, char *labl, int n, int ix,
          REAL *py, int icurve)),
 SC_DECLARE(PD_put_mapping, 
            (PDBfile *file, PM_mapping *f, int mapping)),
 SC_DECLARE(PD_put_image, (PDBfile *file, byte *f, int image)),
 SC_DECLARE(PD_put_set, (PDBfile *file, PM_set *s)),
 SC_DECLARE(PD_mesh_struct, (PDBfile *file)),
 SC_DECLARE(PD_def_mapping, (PDBfile *fp)),
 SC_DECLARE(_PD_disk_addr_sort, (hashel **v, int n));

extern attribute
 *SC_DECLARE(PD_mk_attribute, (char *at, char *type)),
 *SC_DECLARE(PD_inquire_attribute,
	     (PDBfile *file, char *name, char *path));

extern attribute_value
 *SC_DECLARE(PD_inquire_attribute_value,
	     (PDBfile *file, char *name, char *path));

extern void
 SC_DECLARE(PD_rel_image, (PD_image *im)),
 SC_DECLARE(_PD_convert_attrtab, (PDBfile *file)),
 SC_DECLARE(_PD_rl_attribute, (attribute *attr));

extern syment
 *SC_DECLARE(PD_write_table, (PDBfile *file, char *name, HASHTAB *tab)),
 *SC_DECLARE(PD_replace,
	     (PDBfile *file, char *name, char *type, byte *vr, dimdes *dims));

extern char
 *SC_DECLARE(PD_process_set_name, (char *dname));

extern PD_image
 *SC_DECLARE(PD_make_image,
	     (char *name, char *type, byte *data, int kmax, int lmax,
	      int bpp, double xmin, double xmax, double ymin, double ymax,
	      double zmin, double zmax));


/* PDCONV.C declarations */

extern int
 SC_DECLARE(PD_convert,
	    (char **out, char **in, char *intype, char *outtype,
	     long nitems,
	     data_standard *istd, data_standard *ostd, data_standard *hstd,
	     long *pin_offs, long *pout_offs,
	     HASHTAB *in_chart, HASHTAB *out_chart, int boffs, int error)),
 SC_DECLARE(_PD_unp_bits, (char *out, char *in, int ityp, int nbits,
			   int padsz, int fpp, long nitems, long offs)),
 SC_DECLARE(_PD_prim_typep, (char *memb, HASHTAB *chrt, int error));

extern long
 SC_DECLARE(_PD_extract_field, (char *in, int off, int nbi,
				int nby, int *ord));


/* PDMEMB.C declarations */

extern dimdes
 *SC_DECLARE(_PD_ex_dims, (char *memb, int defoff));

extern long
 SC_DECLARE(_PD_comp_num, (dimdes *dims)),
 SC_DECLARE(_PD_str_size, (memdes *str, HASHTAB *tab)),
 SC_DECLARE(_PD_lookup_size, (char *s, HASHTAB *tab));

extern off_t
 SC_DECLARE(_PD_member_location,
	    (char *s, HASHTAB *tab, defstr *dp, memdes **pdesc));

extern int
 SC_DECLARE(_PD_wr_itag,
	    (PDBfile *file, long nitems, char *type, off_t addr, int flag)),
 SC_DECLARE(_PD_rd_itag,
	    (PDBfile *file, PD_itag *pi)),
 SC_DECLARE(_PD_align, 
            (long n, char *type, HASHTAB *tab, int *palign)),
 SC_DECLARE(_PD_adj_dimensions, 
            (PDBfile *file, char *name, syment *ep));

extern char
 *SC_DECLARE(_PD_member_type, (char *s)),
 *SC_DECLARE(_PD_member_base_type, (char *s)),
 *SC_DECLARE(_PD_member_name, (char *s)),
 *SC_DECLARE(_PD_hyper_type, (char *name, char *type));

extern defstr
 *SC_DECLARE(_PD_lookup_type, (char *s, HASHTAB *tab));

extern hashel
 *SC_DECLARE(PD_inquire_symbol,
	     (PDBfile *file, char *name, int flag, char *fullname, HASHTAB *tab));

extern syment
 *SC_DECLARE(PD_inquire_entry,
	     (PDBfile *file, char *name, int flag, char *fullname)),
 *SC_DECLARE(_PD_effective_ep,
	     (PDBfile *file, char *name, int flag, char *fullname));

extern void
 SC_DECLARE(_PD_memb_pinit, (byte));

/* PDPATH.C declarations */

extern long
 SC_DECLARE(PD_hyper_number,
	    (PDBfile *file, char *name, syment *ep)),
 SC_DECLARE(_PD_hyper_number,
	    (PDBfile *file, char *indxpr, long numb, dimdes *dims, long *poff)),
 SC_DECLARE(_PD_parse_index_expr, (char *expr, dimdes *dim,
				   long *pstart, long *pstop, long *pstep)),
 SC_DECLARE(_PD_num_indirects, (char *type, HASHTAB *tab)),
 SC_DECLARE(_PD_member_items, (char *s));


extern off_t
 SC_DECLARE(_PD_skip_over, (PDBfile *file, long skip, int noind));

extern char
 *SC_DECLARE(_PD_tr_syment_v,
	     (PDBfile *file, syment *ep, char *s, memdes **pdesc)),
 *SC_DECLARE(PD_index_to_expr,
	     (char *bf, long indx, dimdes *pd, int major_order, int def_off));


/* PDPRNT.C declarations */

extern void
 SC_DECLARE(PD_print_extras, (PDBfile *file)),
 SC_DECLARE(PD_print_syment, (syment *ep)),
 SC_DECLARE(PD_print_defstr, (defstr *dp)),
 SC_DECLARE(PD_print_entry,
            (PDBfile *file, char *name, byte *vr, syment *ep)),
 SC_DECLARE(PD_write_extras, (FILE *f0, PDBfile *file)),
 SC_DECLARE(PD_write_syment, (FILE *f0, syment *ep)),
 SC_DECLARE(PD_write_defstr, (FILE *f0, defstr *dp)),
 SC_DECLARE(PD_write_entry, 
            (FILE *f0,
             PDBfile *file, char *name, byte *vr, syment *ep)),
 SC_DECLARE(_PD_print_leaf,
            (FILE *f0,
             char *prefix, char *before, char *after,
             char *nodename, PDBfile *file, char *vr,
             long nitems, char *type, dimdes *dims,
             int mjr, int def_off, int irecursion));

extern void
 SC_DECLARE(_PD_set_user_formats, (byte)),
 SC_DECLARE(_PD_set_format_defaults, (byte)),
 SC_DECLARE(_PD_set_digits, (PDBfile *file)),
 SC_DECLARE(_PD_digits_tol, (PDBfile *file_a, PDBfile *file_b));


/* PDRDWR.C declarations */

extern char
 *SC_DECLARE(_PD_expand_hyper_name, (PDBfile *file, char *name)),
 *SC_DECLARE(PD_dereference, (char *s));

extern int
 SC_DECLARE(_PD_hyper_write,
         (PDBfile *file, char *name, syment *ep, byte *vr,
          char *intype)),
 SC_DECLARE(_PD_hyper_read,
         (PDBfile *file, char *name, char *type,
          syment *ep, byte *vr)),
 SC_DECLARE(_PD_indexed_read_as, 
            (PDBfile *file, char *fullpath, char *type,
             byte *vr, int nd, long *ind, syment *ep)),
 SC_DECLARE(_PD_rd_bits, (PDBfile *file, char *name, char *type,
		       long nitems, int sgned, int nbits, int padsz, int fpp, long offs,
		       long *pan, char **pdata)),
 SC_DECLARE(PD_read_bits, (PDBfile *file, char *name, char *type,
		       long nitems, int sgned, int nbits, int padsz, int fpp, long offs,
		       long *pan, char **pdata)),
 SC_DECLARE(_PD_indirection, (char *s)),
 SC_DECLARE(PD_reset_ptr_list, (PDBfile *file));

extern long
 SC_DECLARE(_PD_number_refd, (byte *vr, char *type, HASHTAB *tab)),
 SC_DECLARE(_PD_wr_syment,
         (PDBfile *file, char *vr, long nitems, char *intype,
          char *outtype)),
 SC_DECLARE(_PD_rd_syment,
         (PDBfile *file, syment *ep, char *outtype, byte *vr));

extern dimdes
 *SC_DECLARE(_PD_hyper_dims, 
             (PDBfile *file, char *name, dimdes *dims));

extern void
 SC_DECLARE(_PD_rdwr_pinit, (byte));

/* PDLOW.C declarations */

extern defstr
 *SC_DECLARE(_PD_defstr_inst,
	  (char *name, memdes *desc, int flg, int *ordr, long *formt,
	   HASHTAB *chrt, HASHTAB *host_chart,
	   data_alignment *align, data_alignment *host_align,
	   int flag)),
 *SC_DECLARE(_PD_container_type, (PDBfile *file, defstr *dp));

extern char
 *SC_DECLARE(_PD_get_tbuffer, (byte)),
 *SC_DECLARE(_PD_rfgets, (char *s, int n, FILE *fp)),
 *SC_DECLARE(PD_get_error, (byte));

extern int
 SC_DECLARE(_PD_rd_format, (PDBfile *file)),
 SC_DECLARE(_PD_wr_format, (PDBfile *file)),
 SC_DECLARE(_PD_compare_std,
         (data_standard *a, data_standard *b,
          data_alignment *c, data_alignment *d)),
 SC_DECLARE(_PD_rd_symt, (PDBfile *file)),
 SC_DECLARE(_PD_rd_chrt, (PDBfile *file)),
 SC_DECLARE(_PD_rd_extras, (PDBfile *file)),
 SC_DECLARE(_PD_wr_extras, (PDBfile *file)),
 SC_DECLARE(_PD_rev_chrt, (PDBfile *file)),
 SC_DECLARE(_PD_add_block, (PDBfile *file, syment *ep, dimdes *dims,
                            off_t addr)),
 SC_DECLARE(_PD_extend_file, (PDBfile *file, long nb)),
 SC_DECLARE(PD_get_buffer_size, ()),
 SC_DECLARE(PD_set_buffer_size, (int v));

extern off_t
 SC_DECLARE(_PD_wr_symt, (PDBfile *file)),
 SC_DECLARE(_PD_wr_chrt, (PDBfile *file)),
 SC_DECLARE(_PD_eod, (PDBfile *file)),
 SC_DECLARE(_PD_get_next_address,
	    (PDBfile *file, char *type, long number, byte *vr,
	     int seekf, int tellf));

extern void
 SC_DECLARE(_PD_init_chrt, (PDBfile *file)),
 SC_DECLARE(_PD_setup_chart, (HASHTAB *chart, data_standard *fstd,
			   data_standard *hstd, data_alignment *falign,
			   data_alignment *halign, int flag)),
 SC_DECLARE(_PD_e_install, (char *name, syment *entr, HASHTAB *tab, int lookup)),
 SC_DECLARE(_PD_d_install, (char *name, defstr *def, HASHTAB *tab)),
 SC_DECLARE(_PD_rd_prim_extras, 
             (PDBfile *file, int dc, int rec, char *bf)),
 SC_DECLARE(_PD_wr_prim_extras, (FILE *fp, HASHTAB *tab, int dc, int rec)),
 SC_DECLARE(_PD_low_pinit, (byte)),
 SC_DECLARE(PD_typedef_primitive_types, (PDBfile *file));


/* PDSZOF.C declarations */

extern long
 SC_DECLARE(PD_sizeof, (PDBfile *file, char *type, long n, byte *vri));

extern int
 SC_DECLARE(PN_relocate, (PDBfile *file, char *type, long n));

extern void
 SC_DECLARE(_PD_szof_pinit, (byte));

/* PDFIA.C declarations */
extern void
 SC_DECLARE(_PD_fia_pinit, (byte));

/* PDPAR.C declarations */
extern void
 SC_DECLARE(_PD_pfm_init, (byte)),
 SC_DECLARE(_PD_init_hooks, (byte)),
 SC_DECLARE(_PD_pfm_add, (PDBfile *file, off_t start_addr)),
 SC_DECLARE(_PD_pfm_setaddr, (PDBfile *file, off_t addr)),
 SC_DECLARE(_PD_pfm_remove, (PDBfile *file)),
 SC_DECLARE(_PD_smp_term, (PDBfile *file)),
 SC_DECLARE(_PD_par_pinit, (byte));

extern int
 SC_DECLARE(PD_init_threads, (int nthreads, PFVoid tid)),
 SC_DECLARE(PD_init_threads_arg, (int c, char **v, char *key, PFVoid tid)),
 SC_DECLARE(_PD_smp_init, (PDBfile *file));

extern off_t
 SC_DECLARE(_PD_pfm_getspace, (PDBfile *file, size_t nbytes));

#ifdef __cplusplus
}
#endif

#endif

