/*
 * GSATST.C - test of basic PGS drawing
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pgs.h"

#define N       100
#define N_TEST  10
#define N_DEV   5

#define PAL(i) palettes[(i+palette)%(sizeof(palettes)/sizeof(char *))]

typedef int SC_DECLARE((*gtest), (PG_device *dev));
typedef int SC_DECLARE((*dtest), (char *name, int color, int debug_mode,
				  int nt, gtest *tst));

static int
 mode = 0,
 debug_mode = 0,
 ids = 0,
 palette = 0;

static int
 sty[] = {LINE_SOLID, LINE_DOTTED, LINE_DASHED, LINE_DOTDASHED};

static char
 *palettes[] = {"rainbow", "bw", "iron", "contrast", "cym"},
 *clrs[] = {"BLACK", "WHITE", "GRAY", "DARK_GRAY",
            "BLUE", "GREEN", "CYAN", "RED", "MAGENTA", "BROWN",
            "DARK_BLUE", "DARK_GREEN", "DARK_CYAN", "DARK_RED",
            "YELLOW", "DARK_MAGENTA"};

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* COMPUTE_2D_DATA - compute data for 2D-1D rendering tests */

static void compute_2d_data(f, x, y, kmax, lmax)
   REAL *f, *x, *y;
   int kmax, lmax;
   {int k, l, i;
    REAL xmin, xmax, ymin, ymax;
    REAL dx, dy, tol, t, r, fv;

    xmin = -5.0;
    xmax =  5.0;
    ymin = -5.0;
    ymax =  5.0;

    dx = (xmax - xmin)/((REAL) kmax - 1.0);
    dy = (xmax - xmin)/((REAL) kmax - 1.0);

    tol = 1.0e-5;

    for (k = 0; k < kmax; k++)
        for (l = 0; l < lmax; l++)
            {i    = l*kmax + k;
             x[i] = k*dx + xmin;
             y[i] = l*dy + ymin;
             r    = x[i]*x[i] + y[i]*y[i];
             t    = 5.0*atan(y[i]/(x[i] + SMALL));
             r    = POW(r, 0.125);
             fv   = exp(-r)*(1.0 + 0.1*cos(t));
             f[i] = floor(fv/tol)*tol;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TEST_1 - line drawing test 
 *        - Modes: cartesian, polar, inselberg, histogram
 */

static int test_1(dev)
   PG_device *dev;
   {int i, n, ip, np, imn, imx, ok;
    REAL *x, *y;
    PG_graph *data;

    n = N;

    x = FMAKE_N(REAL, n, "TEST_1:x");
    y = FMAKE_N(REAL, n, "TEST_1:y");

/* prepare the data */
    ip = PG_get_processor_number();
    np = PG_get_number_processors();
    n  /= np;
    imn = ip*n;
    imx = imn + n;

    for (i = imn; i < imx; i++)
        {x[i] = -(i+1)/10.0;
         y[i] = 6.022e23/x[i];};

    ok = TRUE;

    data = PG_make_graph_1d('A', "Test Data #1", FALSE, n,
                            x, y, "X Values", "Y Values");

/* set up the hard copy device */
    PG_turn_data_id(dev, ids);

    data->info = (byte *) PG_set_line_info(data->info, CARTESIAN, CARTESIAN,
					   SOLID, FALSE, 0, dev->BLUE,
					   0, 0.0);
    PG_draw_graph(dev, data);

    data->info = (byte *) PG_set_line_info(data->info, POLAR, POLAR,
					   SOLID, FALSE, 0, dev->GREEN,
					   0, 0.0);
    PG_draw_graph(dev, data);

    data->info = (byte *) PG_set_line_info(data->info, INSEL, INSEL,
					   SOLID, FALSE, 0, dev->RED,
					   0, 0.0);
    PG_draw_graph(dev, data);

    data->info = (byte *) PG_set_line_info(data->info, HISTOGRAM, CARTESIAN,
					   SOLID, FALSE, 0, dev->WHITE,
					   0, 0.0);
    PG_draw_graph(dev, data);

    PG_rl_graph(data, TRUE, TRUE);

    return(ok);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TEST_2 - line drawing test 
 *        - Modes: cartesian, polar, inselberg, histogram
 *        - with grids on
 */

static int test_2(dev)
   PG_device *dev;
   {int i, n, ip, np, imn, imx, ok;
    REAL dx;
    REAL *x, *y;
    PG_graph *data;

    n = N;

    x = FMAKE_N(REAL, n, "TEST_2:x");
    y = FMAKE_N(REAL, n, "TEST_2:y");

/* prepare the data */
    ip = PG_get_processor_number();
    np = PG_get_number_processors();
    n  /= np;
    imn = ip*n;
    imx = imn + n;

    dx = 0.1*((REAL) (imx - imn))/((REAL) N - 1.0);

    for (i = imn; i < imx; i++)
        {x[i] = i*dx;
         y[i] = cos(x[i]);};

    ok = TRUE;

    data = PG_make_graph_1d('B', "Test Data #2", FALSE, n,
                            x, y, "X Values", "Y Values");

    PG_turn_grid(dev, ON);

    data->info = (byte *) PG_set_line_info(data->info, HISTOGRAM, CARTESIAN,
					   SOLID, FALSE, 0, dev->MAGENTA,
					   1, 0.0);
    PG_draw_graph(dev, data);

    data->info = (byte *) PG_set_line_info(data->info, POLAR, POLAR,
					   SOLID, FALSE, 0, dev->CYAN,
					   0, 0.0);
    PG_draw_graph(dev, data);

    data->info = (byte *) PG_set_line_info(data->info, INSEL, INSEL,
					   SOLID, FALSE, 0, dev->YELLOW,
					   0, 0.0);
    PG_draw_graph(dev, data);

    data->info = (byte *) PG_set_line_info(data->info, CARTESIAN, CARTESIAN,
					   SOLID, FALSE, 0, dev->DARK_GRAY,
					   0, 0.0);

    PG_draw_graph(dev, data);

    PG_rl_graph(data, TRUE, TRUE);

    PG_turn_grid(dev, OFF);

    return(ok);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TEST_3 - image drawing tests */

static int test_3(dev)
   PG_device *dev;
   {int k, l, X_max, Y_max, ok;
    unsigned char *bf;
    double dx, dy, xv, yv, sf;
    PG_palette *pl;
    PG_image *calc_im;

    ok = TRUE;

/* prepare the data */
    X_max  = 250;
    Y_max  = 250;
    sf     = 255;

    calc_im = PG_make_image("Test Image", SC_CHAR_S, NULL,
			    0.0, 0.0, 0.0, 0.0, 0.0, sf,
			    X_max, Y_max, 4, NULL);
    if (calc_im == NULL)
       {PRINT(STDOUT, "CAN'T ALLOCATE IMAGE");
	exit(1);};

    pl = PG_set_palette(dev, PAL(1));

/* compute the first image */
    bf = calc_im->buffer;
    for (l = 0; l < Y_max; l++)
        {for (k = 0; k < X_max; k++)
	     *bf++ = sf*((double) k)*((double) l)/
	                ((double) (Y_max - 1)*(X_max - 1));};

/* draw the first image */
    PG_draw_image(dev, calc_im, "Test Data A", NULL);

/* compute the second image */
    pl = PG_set_palette(dev, PAL(2));
    dx = 2.0*PI/((double) X_max);
    dy = 2.0*PI/((double) Y_max);
    bf = calc_im->buffer;
    for (l = 0; l < Y_max; l++)
        {for (k = 0; k < X_max; k++)
	     {xv = ((double) k)*dx;
	      yv = ((double) (Y_max - l - 1))*dy;

	      *bf++ = sf*(0.5 + 0.5*sin(xv)*cos(yv));};};

/* draw the second image */
    PG_draw_image(dev, calc_im, "Test Data B", NULL);

    PG_rl_image(calc_im);

    return(ok);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SF_DT - set the font and draw the text */

static void sf_dt(dev, x1, y1, face, style, size)
   PG_device *dev;
   double x1, y1;
   char *face, *style;
   int size;
   {REAL x2, y2, dx, dy;

    PG_set_font(dev, face, style, size);

    PG_get_text_ext(dev, "foo", &dx, &dy);

/* write some text and draw a box around it */
    PG_write_WC(dev, x1, y1, "%s", "foo");

    x2 = x1 + dx;
    y2 = y1 + dy;
    PG_draw_box(dev, x1, x2, y1, y2);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TEST_4_AUX - draw lots of text */

static void test_4_aux(dev)
   PG_device *dev;
   {PG_set_color_line(dev, dev->BLACK, TRUE);
    PG_set_color_text(dev, dev->BLACK, TRUE);

    sf_dt(dev, 0.1, 0.9, "helvetica", "medium", 12);
    sf_dt(dev, 0.1, 0.8, "helvetica", "italic", 12);
    sf_dt(dev, 0.1, 0.7, "helvetica", "bold", 12);
    sf_dt(dev, 0.1, 0.6, "helvetica", "bold-italic", 12);

    sf_dt(dev, 0.1, 0.50, "helvetica", "medium", 10);
    sf_dt(dev, 0.1, 0.45, "helvetica", "italic", 10);
    sf_dt(dev, 0.1, 0.40, "helvetica", "bold", 10);
    sf_dt(dev, 0.1, 0.35, "helvetica", "bold-italic", 10);

    sf_dt(dev, 0.1, 0.30, "helvetica", "medium", 8);
    sf_dt(dev, 0.1, 0.25, "helvetica", "italic", 8);
    sf_dt(dev, 0.1, 0.20, "helvetica", "bold", 8);
    sf_dt(dev, 0.1, 0.15, "helvetica", "bold-italic", 8);

    sf_dt(dev, 0.3, 0.9, "times", "medium", 12);
    sf_dt(dev, 0.3, 0.8, "times", "italic", 12);
    sf_dt(dev, 0.3, 0.7, "times", "bold", 12);
    sf_dt(dev, 0.3, 0.6, "times", "bold-italic", 12);

    sf_dt(dev, 0.3, 0.50, "times", "medium", 10);
    sf_dt(dev, 0.3, 0.45, "times", "italic", 10);
    sf_dt(dev, 0.3, 0.40, "times", "bold", 10);
    sf_dt(dev, 0.3, 0.35, "times", "bold-italic", 10);

    sf_dt(dev, 0.3, 0.30, "times", "medium", 8);
    sf_dt(dev, 0.3, 0.25, "times", "italic", 8);
    sf_dt(dev, 0.3, 0.20, "times", "bold", 8);
    sf_dt(dev, 0.3, 0.15, "times", "bold-italic", 8);

    sf_dt(dev, 0.5, 0.9, "courier", "medium", 12);
    sf_dt(dev, 0.5, 0.8, "courier", "italic", 12);
    sf_dt(dev, 0.5, 0.7, "courier", "bold", 12);
    sf_dt(dev, 0.5, 0.6, "courier", "bold-italic", 12);

    sf_dt(dev, 0.5, 0.50, "courier", "medium", 10);
    sf_dt(dev, 0.5, 0.45, "courier", "italic", 10);
    sf_dt(dev, 0.5, 0.40, "courier", "bold", 10);
    sf_dt(dev, 0.5, 0.35, "courier", "bold-italic", 10);

    sf_dt(dev, 0.5, 0.30, "courier", "medium", 8);
    sf_dt(dev, 0.5, 0.25, "courier", "italic", 8);
    sf_dt(dev, 0.5, 0.20, "courier", "bold", 8);
    sf_dt(dev, 0.5, 0.15, "courier", "bold-italic", 8);

    PG_update_vs(dev);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TEST_4 - text tests */

static int test_4(dev)
   PG_device *dev;
   {int i, ok;
    double x0, y0, an;

/* connect the I/O functions */
/*    PG_open_console("PGS Test", "COLOR", TRUE, 0.1, 0.7, 0.5, 0.3); */

    ok = TRUE;

/* do simple text test */

    PG_clear_screen(dev);

    PG_set_viewport(dev, 0.1, 0.9, 0.2, 0.8);
    PG_set_window(dev, 0.0, 1.0, 0.0, 1.0);
    PG_draw_box(dev, -0.02, 1.02, -0.02, 1.02);

/*    PG_expose_device(PG_console_device); */

    test_4_aux(dev);

/* reset for following tests */
    PG_set_font(dev, "helvetica", "medium", 10);

    PG_finish_plot(dev);

/* do oriented text test */
    PG_clear_screen(dev);
    PG_set_window(dev, -3.0, 3.0, -3.0, 3.0);

    for (i = 0; i < 8; i++)
        {an = i*PI/4.0;
         x0 = cos(an);
         y0 = sin(an);
         PG_set_char_path(dev, x0, y0);
         PG_write_WC(dev, x0, y0, "Sample");};

/* reset for following tests */
    PG_set_char_path(dev, 1.0, 0.0);

    PG_finish_plot(dev);

    return(ok);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TEST_5 - contour plot test */

static int test_5(dev)
   PG_device *dev;
   {int kmax, lmax, kxl;
    int id, lncolor, lnstyle, centering;
    REAL *x, *y, *f, lnwidth;
    REAL fmin, fmax;
    PG_graph *data;

/* set up data */
    kmax      = 20;
    lmax      = 20;
    kxl       = kmax*lmax;
    x         = FMAKE_N(REAL, kxl, "TEST_5:x");
    y         = FMAKE_N(REAL, kxl, "TEST_5:y");
    f         = FMAKE_N(REAL, kxl, "TEST_5:f");
    id        = 'A';
    lncolor   = dev->BLUE;
    lnwidth   = 0.0;
    lnstyle   = SOLID;
    centering = N_CENT;

    compute_2d_data(f, x, y, kmax, lmax);

    PG_iso_limit(f, kxl, &fmin, &fmax);

    data = PG_make_graph_r2_r1(id, "{x, y}->f", FALSE,
                               kmax, lmax, centering, x, y, f, "xy", "f");

    data->info = (byte *) PG_set_tds_info(data->info, PLOT_CONTOUR, CARTESIAN,
					  lnstyle, lncolor, 10, 1.0,
					  lnwidth, 0.0, 0.0, 0.0, HUGE);

    PG_turn_data_id(dev, OFF);

    PG_contour_plot(dev, data);

    PG_turn_data_id(dev, ON);

    PG_rl_graph(data, TRUE, TRUE);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TEST_6 - vector plot test */

static int test_6(dev)
   PG_device *dev;
   {int i, k, l, kmax, lmax, kxl;
    int id, lncolor, lnstyle, centering;
    REAL km, lm, lnwidth;
    REAL xmin, xmax, ymin, ymax;
    REAL *x, *y, *u, *v;
    PM_set *domain, *range;
    PG_graph *data;

/* set up data */
    kmax      = 20;
    lmax      = 20;
    xmin      = -5.0;
    xmax      = 5.0;
    ymin      = -5.0;
    ymax      = 5.0;
    id        = 'A';
    lncolor   = dev->BLUE;
    lnwidth   = 0.0;
    lnstyle   = SOLID;
    centering = N_CENT;

    kxl = kmax*lmax;
    x   = FMAKE_N(REAL, kxl, "TEST_6:x");
    y   = FMAKE_N(REAL, kxl, "TEST_6:y");
    u   = FMAKE_N(REAL, kxl, "TEST_6:u");
    v   = FMAKE_N(REAL, kxl, "TEST_6:v");

    km = 2.0*PI/((REAL) kmax - 1.0);
    lm = 2.0*PI/((REAL) lmax - 1.0);
    for (k = 0; k < kmax; k++)
        for (l = 0; l < lmax; l++)
            {i    = l*kmax + k;
             x[i] = k*km;
             y[i] = l*lm;
             u[i] = sin(k*km);
             v[i] = sin(l*lm);};

/* build the domain set */
    domain = PM_make_set("{x, y}", SC_REAL_S, FALSE,
			 2, kmax, lmax, 2, x, y);

/* build the range set */
    range = PM_make_set("{u, v}", SC_REAL_S, FALSE,
			2, kmax, lmax, 2, u, v);

    data = PG_make_graph_from_sets("{x, y}->{u, v}", domain, range,
				   centering,
				   SC_PCONS_P_S, NULL, id, NULL);

    data->info = (byte *) PG_set_tdv_info(data->info, PLOT_VECTOR, CARTESIAN,
					  lnstyle, lncolor, lnwidth);

    PG_vector_plot(dev, data);

    PG_rl_graph(data, TRUE, TRUE);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TEST_7 - fill-poly plot test */

static int test_7(dev)
   PG_device *dev;
   {int kmax, lmax, kxl;
    int id, centering;
    REAL *x, *y, *f;
    PG_graph *data;

/* set up data */
    centering = N_CENT;
    kmax      = 20;
    lmax      = 20;
    kxl       = kmax*lmax;
    x         = FMAKE_N(REAL, kxl, "TEST_7:x");
    y         = FMAKE_N(REAL, kxl, "TEST_7:y");
    f         = FMAKE_N(REAL, kxl, "TEST_7:f");
    id        = 'A';

    compute_2d_data(f, x, y, kmax, lmax);

    data = PG_make_graph_r2_r1(id, "{x, y}->f", FALSE,
			       kmax, lmax, centering, x, y, f, "xy", "f");

    PG_set_palette(dev, PAL(3));
    PG_poly_fill_plot(dev, data);

    PG_rl_graph(data, TRUE, TRUE);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TEST_8 - surface plot test */

static int test_8(dev)
   PG_device *dev;
   {int k, l, kmax, lmax, kxl;
    int maxes[2];
    char *emap;
    REAL theta, phi, chi;
    REAL *rx, *ry, *rz, *rs, *pz, *ps, *ext;
    pcons *alist;

    kmax = 10;
    lmax = 10;
    kxl  = kmax*lmax;

/* make the data */
    rx = FMAKE_N(REAL, kxl, "TEST_8:rx");
    ry = FMAKE_N(REAL, kxl, "TEST_8:ry");
    rz = pz = FMAKE_N(REAL, kxl, "TEST_8:rz");
    rs = ps = FMAKE_N(REAL, kxl, "TEST_8:rs");

/* load the data */
    compute_2d_data(rz, rx, ry, kmax, lmax);

    for (l = 0; l < lmax; l++)
        for (k = 0; k < kmax; k++)
	    *ps++ = exp(*pz++);

    maxes[0] = kmax;
    maxes[1] = lmax;

    emap = FMAKE_N(char, kxl, "TEST_8:emap");
    memset(emap, 1, kxl);
    alist = SC_add_alist(NULL, "EXISTENCE", SC_STRING_S, (byte *) emap);

    ext = FMAKE_N(REAL, 4, "TEST_8:ext");
    ext[0] = -1;
    ext[1] =  1;
    ext[2] = -1;
    ext[3] =  1;

/* draw the wire frames */
    PG_set_palette(dev, "standard");

    for (theta = 0.0; theta <= 90.0; theta += 45.0)
        for (phi = 0.0; phi <= 90.0; phi += 45.0)
	    for (chi = 30.0; chi <= 60.0; chi += 30.0)
	        PG_draw_surface(dev, rz, rz, ext, rx, ry, kxl,
				0.0, 2.0*PI, 0.0, 2.0*PI,
				DEG_RAD*theta, DEG_RAD*phi, DEG_RAD*chi,
				0.0, dev->BLUE, SOLID,
				PLOT_WIRE_MESH, "Test Data",
				PM_LR_S, maxes, alist);

/* draw the surface frames */
    PG_set_palette(dev, PAL(4));

    for (theta = 0.0; theta <= 90.0; theta += 45.0)
        for (phi = 0.0; phi <= 90.0; phi += 45.0)
	    for (chi = 30.0; chi <= 60.0; chi += 30.0)
	        PG_draw_surface(dev, rz, rs, ext, rx, ry, kxl,
				0.0, 2.0*PI, 0.0, 2.0*PI,
				DEG_RAD*theta, DEG_RAD*phi, DEG_RAD*chi,
				0.0, dev->BLUE, SOLID,
				PLOT_SURFACE, "Test Data",
				PM_LR_S, maxes, alist);

    SFREE(rx);
    SFREE(ry);
    SFREE(rz);
    SFREE(rs);
    SFREE(ext);
    SFREE(emap);
    SC_free_alist(alist, 3);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TEST_9 - mesh plot test */

static int test_9(dev)
   PG_device *dev;
   {int i, k, l, m, kmax, lmax, mmax, kxl;
    int id, lncolor, lnstyle, centering;
    REAL xmin, xmax, ymin, ymax, zmin, zmax, dx, dy;
    REAL lnwidth, theta, phi, chi;
    REAL *x, *y, *z, *f;
    PM_set *domain, *range;
    PG_graph *data;

/* set up data */
    id        = 'A';
    lncolor   = dev->BLUE;
    lnwidth   = 0.0;
    lnstyle   = SOLID;
    centering = N_CENT;

/* construct 2d mesh */
    kmax      = 5;
    lmax      = 5;
    xmin      = -5.0;
    xmax      = 5.0;
    ymin      = -5.0;
    ymax      = 5.0;
    kxl       = kmax*lmax;
    x         = FMAKE_N(REAL, kxl, "TEST_9:x");
    y         = FMAKE_N(REAL, kxl, "TEST_9:y");
    f         = FMAKE_N(REAL, kxl, "TEST_9:f");

    dx = (xmax - xmin)/((REAL) kmax - 1.0);
    dy = (xmax - xmin)/((REAL) kmax - 1.0);

    for (k = 0; k < kmax; k++)
        for (l = 0; l < lmax; l++)
            {i    = l*kmax + k;
             x[i] = k*dx + xmin;
             y[i] = l*dy + ymin;
             f[i] = 0.0;};

    domain = PM_make_set("{x, y}", SC_REAL_S, FALSE,
			 2, kmax, lmax, 2, x, y);

    range = PM_make_set("f", SC_REAL_S, FALSE,
			2, kmax, lmax, 1, f);

    data = PG_make_graph_from_sets("{x, y}->f", domain, range,
				   centering,
				   SC_PCONS_P_S, NULL, id, NULL);

    data->info = (byte *) PG_set_tds_info(data->info, PLOT_MESH, CARTESIAN,
					  lnstyle, lncolor, 10, 1.0,
					  lnwidth, 0.0, 0.0, 0.0, HUGE);

    PG_mesh_plot(dev, data);

    PG_rl_graph(data, TRUE, TRUE);

/* construct 3d mesh */
    kmax      = 5;
    lmax      = 6;
    mmax      = 7;
    xmin      = -5.0;
    xmax      = 3.0;
    ymin      = -3.0;
    ymax      = 5.0;
    zmin      = 0.0;
    zmax      = 7.0;
    kxl       = kmax*lmax*mmax;
    x         = FMAKE_N(REAL, kxl, "TEST_9:x");
    y         = FMAKE_N(REAL, kxl, "TEST_9:y");
    z         = FMAKE_N(REAL, kxl, "TEST_9:z");
    f         = FMAKE_N(REAL, kxl, "TEST_9:f");

    dx = (xmax - xmin)/((REAL) kmax - 1.0);
    dy = (xmax - xmin)/((REAL) kmax - 1.0);

    for (k = 0; k < kmax; k++)
        for (l = 0; l < lmax; l++)
	    for (m = 0; m < mmax; m++)
	        {i    = (m*lmax + l)*kmax + k;
		 x[i] = k*dx + xmin;
		 y[i] = l*dy + ymin;
		 z[i] = m;
		 f[i] = 0.0;};

    domain = PM_make_set("{x, y, z}", SC_REAL_S, FALSE,
			 3, kmax, lmax, mmax, 3, x, y, z);

    range = PM_make_set("f", SC_REAL_S, FALSE,
			3, kmax, lmax, mmax, 1, f);

    data = PG_make_graph_from_sets("{x, y, z}->f", domain, range,
				   centering,
				   SC_PCONS_P_S, NULL, id, NULL);

    theta = 30.0;
    phi   = 45.0;
    chi   = 60.0;

    data->info = (byte *) PG_set_tds_info(data->info, PLOT_MESH, CARTESIAN,
					  lnstyle, lncolor, 10, 1.0, lnwidth,
					  DEG_RAD*theta, DEG_RAD*phi, DEG_RAD*chi,
					  HUGE);

    PG_mesh_plot(dev, data);

    PG_rl_graph(data, TRUE, TRUE);


    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TEST_10 - line width/color/style test */

static int test_10(dev)
   PG_device *dev;
   {int i, j, n;
    REAL y, dy, w, xmn, xmx;

    PG_set_viewport(dev, 0.05, 0.95, 0.05, 0.95);
    PG_set_window(dev, 0.0, 1.0, 0.0, 1.0);
    PG_set_palette(dev, "standard");

/* colors */
    PG_clear_screen(dev);

    n  = 16;
    dy = 1.0/(n + 1.0);
    y  = 0.5*dy;
    for (i = 0; i < n; i++)
        {PG_set_color_line(dev, i, TRUE);
         PG_draw_line(dev, 0.0, y, 0.5, y);

         PG_set_color_text(dev, i, TRUE);
         PG_write_WC(dev, 0.6, y, "%d %s", i, clrs[i]);

         y += dy;};

    PG_finish_plot(dev);

/* widths and styles */
    PG_set_line_color(dev, dev->WHITE);
    PG_clear_screen(dev);

    n  = 10;
    dy = 1.0/(n + 1.0);
    y  = 0.5*dy;
    for (i = 0; i < n; i++)
        {w = 0.3*i;
	 PG_set_line_width(dev, w);
	 for (j = 0; j < 4; j++)
	     {xmn = 0.25*j + 0.05;
	      xmx = 0.25*(j+1) - 0.05;
	      PG_set_line_style(dev, sty[j]);
	      PG_draw_line(dev, xmn, y, xmx, y);};

         y += dy;};

    PG_finish_plot(dev);

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* RUN_TEST - run the sequence of tests for the specified device */

static int run_test(dev, nt, tst)
   PG_device *dev;
   int nt;
   gtest *tst;
   {int i, err;
    gtest t;

    err = TRUE;

    for (i = 0; i < nt; i++)
        {t = tst[i];
	 if (t != NULL)
	    err &= (*t)(dev);};

    return(err);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* TEST_CHECK - begin/end memory checks for each test */

static long test_check(name, which, debug_mode)
   char *name;
   long which;
   int debug_mode;
   {long bytaa, bytfa, rv;
    char msg[MAXLINE];
    static long bytab, bytfb;
    static int first = TRUE;

    if (debug_mode)
       rv = SC_mem_monitor(which, 2, "G", msg);

    if (which == 0)
       SC_mem_stats(&bytab, &bytfb, NULL, NULL);

    else if (which != 0)
       {SC_mem_stats(&bytaa, &bytfa, NULL, NULL);

	bytaa -= bytab;
	bytfa -= bytfb;

	if (first)
	   {first = FALSE;
	    PRINT(STDOUT, "\n\n\t\t   Allocated      Freed       Diff\n");};

	PRINT(STDOUT, "\t%s\t   %9d  %9d  %9d\n",
	      name, bytaa, bytfa, bytaa - bytfa);};

    return(rv);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* REGISTER_DEVICES - register the devices needed for the tests */

static void register_devices()
   {extern void
      SC_DECLARE(PG_setup_image_device, (PG_device *d));

#ifdef HAVE_JPEGLIB
    extern void
      SC_DECLARE(PG_setup_jpeg_device, (PG_device *d));
    
    PG_register_device("JPEG", PG_setup_jpeg_device);

#endif

    PG_register_device("IMAGE", PG_setup_image_device);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SCR_TEST - test the SCREEN device */

static int scr_test(name, color, debug_mode, nt, tst)
   char *name;
   int color, debug_mode, nt;
   gtest *tst;
   {int ok;
    long sc;
    PG_device *dev;

    ok = TRUE;

    sc = test_check(name, 0L, debug_mode);

    register_devices();

    if (color)
       dev = PG_make_device("SCREEN", "COLOR", name);
    else
       dev = PG_make_device("SCREEN", "MONOCHROME", name);

    PG_open_device(dev, 0.01, 0.01, 0.3, 0.3);

    ok &= run_test(dev, nt, tst);

    PG_close_device(dev);

    PG_rl_all();

    sc = test_check(name, sc, debug_mode);

    return(ok);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PS_TEST - test the PostScript device */

static int ps_test(name, color, debug_mode, nt, tst)
   char *name;
   int color, debug_mode, nt;
   gtest *tst;
   {int ok;
    long sc;
    PG_device *dev;

    ok = TRUE;

    sc = test_check(name, 0L, debug_mode);

    register_devices();

    if (color)
       {if (mode == PORTRAIT_MODE)
	   dev = PG_make_device("PS", "COLOR", name);
	else
	   dev = PG_make_device("PS", "COLOR LANDSCAPE", name);}
    else
       {if (mode == PORTRAIT_MODE)
	   dev = PG_make_device("PS", "MONOCHROME PORTRAIT", name);
	else
	   dev = PG_make_device("PS", "MONOCHROME LANDSCAPE", name);};

    PG_open_device(dev, 0.0, 0.0, 0.0, 0.0);

    ok &= run_test(dev, nt, tst);

    PG_close_device(dev);

    PG_rl_all();

    sc = test_check(name, sc, debug_mode);

    return(ok);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* CGM_TEST - test the CGM device */

static int cgm_test(name, color, debug_mode, nt, tst)
   char *name;
   int color, debug_mode, nt;
   gtest *tst;
   {int ok;
    long sc;
    PG_device *dev;

    ok = TRUE;

    sc = test_check(name, 0L, debug_mode);

    register_devices();

    if (color)
       dev = PG_make_device("CGM", "COLOR", name);
    else
       dev = PG_make_device("CGM", "MONOCHROME", name);

    PG_open_device(dev, 0.0, 0.0, 0.0, 0.0);

    ok &= run_test(dev, nt, tst);

    PG_close_device(dev);

    PG_rl_all();

    sc = test_check(name, sc, debug_mode);

    return(ok);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* JPEG_TEST - test the JPEG device */

static int jpeg_test(name, color, debug_mode, nt, tst)
   char *name;
   int color, debug_mode, nt;
   gtest *tst;
   {int ok;
    long sc;
    PG_device *dev;

    ok = TRUE;

#ifdef HAVE_JPEGLIB

    sc = test_check(name, 0L, debug_mode);

    register_devices();

    if (color)
       dev = PG_make_device("JPEG", "COLOR", name);
    else
       dev = PG_make_device("JPEG", "MONOCHROME", name);

    PG_open_device(dev, 0.0, 0.0, 400.0, 400.0);

    ok &= run_test(dev, nt, tst);

    PG_close_device(dev);

    PG_rl_all();

    sc = test_check(name, sc, debug_mode);

#endif

    return(ok);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* IMAGE_TEST - test the IMAGE device */

static int image_test(name, color, debug_mode, nt, tst)
   char *name;
   int color, debug_mode, nt;
   gtest *tst;
   {int ok, np;
    long sc;
    PG_device *dev;

    ok = TRUE;

    np = PG_get_number_processors();
    if (np == 1)
       {sc = test_check(name, 0L, debug_mode);

	register_devices();

	if (color)
	   dev = PG_make_device("IMAGE", "COLOR", name);
	else
	   dev = PG_make_device("IMAGE", "MONOCHROME", name);

	PG_open_device(dev, 0.0, 0.0, 400.0, 400.0);

	ok &= run_test(dev, nt, tst);

	PG_close_device(dev);

	PG_rl_all();

	sc = test_check(name, sc, debug_mode);};

    return(ok);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* HELP - print the help message */

void help()
   {

    PRINT(STDOUT, "Usage: gsatst [-cdhilABCDE123456789] [-n #] [-p #] [-r #]\n");
    PRINT(STDOUT, "   -c  do all tests in COLOR mode\n");
    PRINT(STDOUT, "   -d  debug memory usage\n");
    PRINT(STDOUT, "   -h  this help message\n");
    PRINT(STDOUT, "   -i  display data-ids in plots\n");
    PRINT(STDOUT, "   -n  number of processors to use/simulate\n");
    PRINT(STDOUT, "   -p  index of palette to be used\n");
    PRINT(STDOUT, "   -r  PS dots/inch\n");
    PRINT(STDOUT, "   -l  landscape mode for PS\n");
    PRINT(STDOUT, "   -A  don't test SCREEN device\n");
    PRINT(STDOUT, "   -B  don't test PS device\n");
    PRINT(STDOUT, "   -C  don't test CGM device\n");
    PRINT(STDOUT, "   -D  don't test JPEG device\n");
    PRINT(STDOUT, "   -E  don't test IMAGE device\n");
    PRINT(STDOUT, "   -1  don't do test #1 (first line test)\n");
    PRINT(STDOUT, "   -2  don't do test #2 (second line test)\n");
    PRINT(STDOUT, "   -3  don't do test #3 (image test)\n");
    PRINT(STDOUT, "   -4  don't do test #4 (rotated text test)\n");
    PRINT(STDOUT, "   -5  don't do test #5 (contour test)\n");
    PRINT(STDOUT, "   -6  don't do test #6 (vector test)\n");
    PRINT(STDOUT, "   -7  don't do test #7 (fill-poly test)\n");
    PRINT(STDOUT, "   -8  don't do test #8 (surface test)\n");
    PRINT(STDOUT, "   -9  don't do test #9 (mesh plot test)\n");
    PRINT(STDOUT, "   -10 don't do test #10 (line color/width/style)\n");
    PRINT(STDOUT, "\n");

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

int main(argc, argv)
   int argc;
   char *argv[];
   {int i, in, nd, nt, trm, color;
    char *tok, *n, tc;
    dtest d;
    gtest tst[N_TEST];
    dtest dev[N_DEV];
    char *name[N_DEV];

/* let's hear it for non-ANSI compilers */
    tst[0] = test_1;
    tst[1] = test_2;
    tst[2] = test_3;
    tst[3] = test_4;
    tst[4] = test_5;
    tst[5] = test_6;
    tst[6] = test_7;
    tst[7] = test_8;
    tst[8] = test_9;
    tst[9] = test_10;

    dev[0] = scr_test;
    dev[1] = ps_test;
    dev[2] = cgm_test;
    dev[3] = jpeg_test;
    dev[4] = image_test;

    name[0] = "gsscrts";
    name[1] = "gspsts";
    name[2] = "gscgmt";
    name[3] = "gsjpgt";
    name[4] = "gsimts";

    trm = PG_init_parallel(argv, NULL);

    SC_zero_space(0);

    nt = N_TEST;
    nd = N_DEV;

    color      = FALSE;
    mode       = PORTRAIT_MODE;
    ids        = FALSE;
    debug_mode = FALSE;

    PG_plot_date = FALSE;

    for (i = 1; i < argc; i++)
        if (argv[i][0] == '-')
           {tok = argv[i] + 1;
	    tc  = tok[0];
	    if (SC_intstrp(tok, 10))
	       {in      = SC_stoi(tok) - 1;
		tst[in] = NULL;}

	    else if (('A' <= tc) && (tc <= 'Z'))
	       {in      = tc - 'A';
		dev[in] = NULL;}

	    else
	       {switch (tok[0])
		   {case 'c' :
		         color = TRUE;
			 break;
		    case 'd' :
		         debug_mode = TRUE;
			 break;
		    case 'i' :
		         ids = TRUE;
			 break;
		    case 'n' :
		         SC_comm_size = SC_stoi(argv[++i]);
			 PG_parallel_simulate = TRUE;
			 PG_parallel_graphics = TRUE;
			 break;
		    case 'p' :
		         palette = SC_stoi(argv[++i]);
			 break;
		    case 'r' :
		         _PG_ps_dots_inch = SC_stof(argv[++i]);
			 break;
		    case 'l' :
		         mode = LANDSCAPE_MODE;
			 break;
		    default  :
		    case 'h' :
		         help();
			 PG_fin_parallel(trm);
			 return(1);
			 break;};};};

    for (i = 0; i < nd; i++)
        {d = dev[i];
	 n = name[i];
	 if (d != NULL)
	    (*d)(n, color, debug_mode, nt, tst);};

    PG_fin_parallel(trm);

    return(0);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
