/*
 * GSDV.C - Portable Graphics System device initialization routines
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
  
#include "pgs.h"

#define DEBUG_TEXT 0

char
 *_PG_input_bf;

int
 _PG_input_length,
 _PG_nc_input;

FILE
 *stdscr;              /* this is the effective file pointer for the screen */

PG_device
 *PG_console_device;
 
PG_event
 PG_current_event;

jmp_buf
 io_avail;

static char
 gbuffer[LRG_TXT_BUFFER];

#ifndef HAVE_WINDOW_DEVICE

PFInt
 PG_get_event_hook = NULL;

PFPPG_device
 PG_event_device_hook = NULL;

PFInt
 PG_open_console_hook = NULL;

#endif

int
 SC_DECLARE(_PG_display_page, 
            (PG_device *dev, FILE *fp, char *s));

extern void
 SC_DECLARE(_PG_get_input, (void));

/*--------------------------------------------------------------------------*/
 
/*                          HIGH LEVEL ROUTINES                             */
 
/*--------------------------------------------------------------------------*/
 
/* PG_OPEN_DEVICE - initialize a graphics device and return a pointer to it
 *                - name - a desriptive name of the device
 *                -        WINDOW - display screen window
 *                -        PS     - PostScript
 *                -        CGM    - CGM device
 *                -        RASTER - raster device
 *                - type - a qualifier for the device
 *                -        MONOCHROME - black and white display
 *                -        COLOR      - color display
 */
 
PG_device *PG_open_device(dev, xf, yf, dxf, dyf)
   PG_device *dev;
   double xf, yf, dxf, dyf;
   {

#ifdef HAVE_WINDOW_DEVICE 
    if (dev != NULL)
       {if (dev->open_screen != NULL)
           if ((*dev->open_screen)(dev, xf, yf, dxf, dyf) == NULL)
	      return(NULL);};

/* connect I/O to graphical functions */
    if (getln != (PFfgets) PG_wind_fgets)
       {if (getln == io_gets_hook)
           getln = (PFfgets) dev->ggets;};

    if (putln != (PFfprintf) PG_wind_fprintf)
       {if (putln == io_printf_hook)
           putln = (PFfprintf) PG_fprintf;};

#else
    if (dev != NULL)
       {if (dev->open_screen != NULL)
           if (strcmp(dev->name, "WINDOW") == 0)
	      return(NULL);
           else
	      if ((*dev->open_screen)(dev, xf, yf, dxf, dyf) == NULL)
		 return(NULL);};
#endif

    return(dev);}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_QUERY_DEVICE - query some physical device characteristics */

void PG_query_device(dev, pdx, pdy, pnc)
   PG_device *dev;
   int *pdx, *pdy, *pnc;
   {PG_query_screen(dev, pdx, pdy, pnc);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_QUERY_WINDOW - query the window shape */

void PG_query_window(dev, pdx, pdy)
   PG_device *dev;
   int *pdx, *pdy;
   {

    *pdx = dev->window_width;
    *pdy = dev->window_height;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_REGISTER_RANGE_EXTREMA - let the device know about the range extrema */

void PG_register_range_extrema(dev, nd, extr)
   PG_device *dev;
   int nd;
   REAL *extr;
   {int i, no;
    REAL *re;

    if ((extr == NULL) || (nd < 0))
       return;

    nd *= 2;

    re = dev->range_extrema;
    no = dev->range_n_extrema;

    if ((re == NULL) || (no != nd))
       {SFREE(re);
	re = FMAKE_N(REAL, nd, "PG_REGISTER_RANGE_EXTREMA:re");};

    for (i = 0; i < nd; i++)
        re[i] = extr[i];

    dev->range_n_extrema = nd;
    dev->range_extrema   = re;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_WRITE_NDC - print formatted output
 *              - at the specified coordinates in NDC
 */

#ifdef PCC

int PG_write_NDC(dev, x, y, fmt, va_alist)
   PG_device *dev;
   double x, y;
   char *fmt;
   va_dcl

#endif

#ifdef ANSI

int PG_write_NDC(PG_device *dev, double x, double y, char *fmt, ...)

#endif

   {if (dev->gprint_flag)
       {SC_VA_START(fmt);
        SC_VSPRINTF(gbuffer, fmt);
        SC_VA_END;

	StoW(dev, x, y);
        PG_move_tx_abs(dev, x, y);
        PG_write_text(dev, stdscr, gbuffer);

#if DEBUG_TEXT
        {double xt, yt, dx, dy;

         PG_get_text_ext(dev, gbuffer, &dx, &dy);
         xt = x + dx;
         yt = y + dy;
         PG_draw_box(dev, x, xt, y, yt);};
#endif
        };

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_WRITE_WC - print formatted output
 *             - at the specified coordinates in WC
 */

#ifdef PCC

int PG_write_WC(dev, x, y, fmt, va_alist)
   PG_device *dev;
   double x, y;
   char *fmt;
   va_dcl

#endif

#ifdef ANSI

int PG_write_WC(PG_device *dev, double x, double y, char *fmt, ...)

#endif

   {if (dev->gprint_flag)
       {SC_VA_START(fmt);
        SC_VSPRINTF(gbuffer, fmt);
        SC_VA_END;
        PG_move_tx_abs(dev, x, y);
        PG_write_text(dev, stdscr, gbuffer);

#if DEBUG_TEXT
        {double xt, yt, dx, dy;

         PG_get_text_ext(dev, gbuffer, &dx, &dy);
         xt = x + dx;
         yt = y + dy;
         PG_draw_box(dev, x, xt, y, yt);};
#endif
        };

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
 
/* _PG_DISPLAY_PAGE - write the string out to the appropriate file or
 *                  - device
 */
 
int _PG_display_page(dev, fp, s)
   PG_device *dev;
   FILE *fp;
   char *s;
   {char *p, *ps, *s1, string[LRG_TXT_BUFFER+1];

    ps = s;
    string[LRG_TXT_BUFFER] = '\0';
    while (TRUE)
       {strncpy(string, ps, LRG_TXT_BUFFER);
        ps += strlen(string);
        if (strlen(string) < 1)
           break;

        p  = string;
        s1 = string;
        while (p != NULL)
           {p = strchr(s1, '\n');
            if (p != NULL)
               {*p = '\0';
                p++;
                PG_write_text(dev, fp, s1);
                PG_next_line(dev);}
            else
               PG_write_text(dev, fp, s1);

            if (p != NULL)
               s1 = p;};};

    return(0);}
 
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_FPRINTF - variable arg version of fPRINTf for the library
 *            - return FALSE unless the flush option on More is invoked
 *            - see _PG_display_page for more details
 */

#ifdef PCC

int PG_fprintf(fp, fmt, va_alist)
   FILE *fp;
   char *fmt;
   va_dcl

#endif

#ifdef ANSI

int PG_fprintf(FILE *fp, char *fmt, ...)

#endif

/* the ifs are nested to get the right behavior wrt the print_flag
 * LEAVE THEM ALONE !!!!
 */
   {int ret;

    ret = 0;

    if (fp != NULL)
       {SC_VA_START(fmt);
	SC_VSPRINTF(gbuffer, fmt);
	SC_VA_END;
	if ((PG_console_device != NULL) && ((fp == stdout) || (fp == stdscr)))
	   {if (PG_console_device->gprint_flag)
	       ret = _PG_display_page(PG_console_device, fp, gbuffer);}
	else
	   {ret = io_puts(gbuffer, fp);};};

    FLUSH_ON(fp, gbuffer);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_WIND_FGETS - window version of fgets
 *               - event handling loop driver
 */

char *PG_wind_fgets(str, maxlen, stream)
   char *str;
   int maxlen;
   FILE *stream;
   {if (stream == stdin)
       return(PG_fgets(str, maxlen, stream));
    else
       return(io_gets(str, maxlen, stream));}

/*------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_GET_INPUT - respond to an input available interrupt */

void _PG_get_input()
   {int n, fd;
    char bf[LRG_TXT_BUFFER], eof;

    _PG_nc_input = 0;

    eof = (char) EOF;
    fd  = fileno(stdin);
    n   = read(fd, bf, LRG_TXT_BUFFER);

/* if we have no input buffer throw the input away (nobody asked for it)
 * and return to whoever called
 */
    if (_PG_input_bf == NULL)
       return;

    if (n > 0)
       {strncpy(&_PG_input_bf[_PG_nc_input], bf, n);
        _PG_nc_input += n;
        _PG_input_bf[_PG_nc_input] = '\0';
        longjmp(io_avail, ERR_FREE);}

    else if (n == 0)
       {if (bf[0] == eof)
           {_PG_input_bf = NULL;
            longjmp(io_avail, ERR_FREE);};}

/* check for eof */
    else if (_PG_input_bf[_PG_nc_input-1] == eof)
       {if (_PG_nc_input == 1)
           {_PG_input_bf = NULL;
            longjmp(io_avail, ERR_FREE);}

        else
           {_PG_input_bf[_PG_nc_input-1] = '\0';}
        longjmp(io_avail, ERR_FREE);}

    else if (_PG_input_bf[_PG_nc_input-2] == eof)
       {if (_PG_nc_input == 2)
           {_PG_input_bf = NULL;
            longjmp(io_avail, ERR_FREE);}

        else
           {_PG_input_bf[_PG_nc_input-2] = '\0';
            longjmp(io_avail, ERR_FREE);};};

    longjmp(io_avail, ERR_FREE);

    return;}
        
/*--------------------------------------------------------------------------*/
/*------------------------------------------------------------------------*/

/* PG_WIND_FPRINTF - handle printing in the window environment */

#ifdef PCC

int PG_wind_fprintf(fp, fmt, va_alist)
   FILE *fp;
   char *fmt;
   va_dcl

#endif

#ifdef ANSI

int PG_wind_fprintf(FILE *fp, char *fmt, ...)

#endif

   {int ret;
    char bf[LRG_TXT_BUFFER];
    
    ret = 0;

    if (fp != NULL)
       {if ((fp == stdout) && (PG_console_device != NULL))
	   {if (PG_console_device->gprint_flag)
	       {SC_VA_START(fmt);
		ret = SC_VSPRINTF(bf, fmt);
		SC_VA_END;

		PG_puts(bf);};}

       else
	  {SC_VA_START(fmt);
	   ret = SC_VFPRINTF(fp, fmt);
	   SC_VA_END;};};

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
