/*
 * GSHLS.C - hidden line/surface removal routines for PGS
 *
 * Source Version: 2.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"

#include "pgs.h"

#define COMPLETE 2            /* completely hidden */
#define PARTIAL  1            /* partially obscured */

#define NODE(a, n) ((a)->node[(n)])

#define REMEMBER_TRI(a, lst, n, nx)                                         \
    {if (lst == NULL)                                                       \
        {nx  = 1000;                                                        \
	 n   = 0;                                                           \
	 lst = FMAKE_N(PG_triangle, nx, "GSHLS.C:lst");};                   \
     lst[n++] = *a;                                                         \
     if (n >= nx)                                                           \
	{nx += 1000;                                                        \
	 REMAKE_N(lst, PG_triangle, nx);};}

struct s_PG_node
   {int indx;      /* -1 if the node is artificial */
    REAL x[3];
    REAL val;};

typedef struct s_PG_node PG_node;

struct s_PG_triangle
   {PG_node node[3];};

typedef struct s_PG_triangle PG_triangle;

/* this is stuff for debugging the triangle overlap */

static int
 no_order = FALSE,
 depth_sort = TRUE,
 debug = FALSE;

static double
 xmn = 0.0,
 xmx = 1.1,
 ymn = 0.0,
 ymx = 1.0;

static PG_triangle
 *SC_DECLARE(PG_triangularize, (char *type,
			     REAL *rx, REAL *ry, REAL *rz, REAL *dextr,
			     REAL *f, byte *cnnct,
			     double theta, double phi, double chi, double norm,
			     int *pn, int *pnx, double **orient)),
 *SC_DECLARE(_PG_depth_sort, (PG_triangle *trial, int ntr, double *orient)),
 *SC_DECLARE(PG_triangularize_ac, (PM_mesh_topology *mt,
				REAL *rx, REAL *ry, REAL *rz, REAL *dextr,
				REAL *f, double theta, double phi, double chi,
				double norm, int *pn, int *pnx, double **orient)),
 *SC_DECLARE(PG_triangularize_lr, (int *maxes,
				REAL *rx, REAL *ry, REAL *rz, REAL *dextr,
				REAL *f, double theta, double phi, double chi,
				double norm, int *pn, int *pnx, double **orient));

static void
 SC_DECLARE(_PG_orientation, (PG_node *cn, REAL *pnx, REAL *pny, REAL *pnz));

static int
 SC_DECLARE(_PG_push_triangle, 
            (PG_node *node0, PG_node *node1, PG_node *node2,
	     PG_triangle *a, PG_triangle *tr,
	     int itr)),
 SC_DECLARE(_PG_split_tri, (PG_triangle *a, PG_triangle *b,
			 PG_node *inter, PG_triangle **ptr,
			 int *pi, int *pntr, int *pntrx)),
 SC_DECLARE(_PG_inside, (PG_node *a, PG_triangle *b)),
 SC_DECLARE(_PG_same_point, (PG_node *a, PG_node *b)),
 SC_DECLARE(_PG_unique_points, 
            (PG_node *inter, REAL *px, REAL *py, REAL *pz)),
 SC_DECLARE(_PG_colinear, (PG_node *nodes, int n,
			REAL *px, REAL *py, REAL *pz)),
 SC_DECLARE(_PG_intersect_projection, (PG_triangle *a, PG_triangle *b,
				    PG_node *inter)),
 SC_DECLARE(_PG_intersect, (PG_triangle *a, PG_triangle *b,
			 PG_node *inter, PG_triangle **ptr,
			 int *pntr, int *pntrx)),
 SC_DECLARE(_PG_hides, (PG_triangle *b, PG_triangle *a));

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* NPRINT - print some nodes for diagnostic purposes */

static void nprint(a, n)
   PG_node *a;
   int n;
   {int i;
    PG_node *node;
    REAL *X;

    for (i = 0, node = a; i < n; i++, node++)
        {X = node->x;
         PRINT(stdout, "(%9.2e,%9.2e,%9.2e) %d %9.2e\n",
	       X[0], X[1], X[2], node->indx, node->val);};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* NDRAW - draw some nodes for diagnostic purposes */

static void ndraw(a, n)
   PG_node *a;
   int n;
   {int i;
    PG_node *node;
    REAL *X;
    REAL x1, y1, x2, y2;
    static PG_device *dev = NULL;

    if (dev == NULL)
       {dev = PG_make_device("WINDOW", "COLOR", "Diagnostic");
	PG_white_background(dev, TRUE);
	PG_open_device(dev, 0.7, 0.0, 0.3, 0.3);
	PG_set_window(dev, xmn, xmx, ymn, ymx);};

    if (a == NULL)
       {PG_clear_window(dev);
	PG_set_window(dev, xmn, xmx, ymn, ymx);};

    for (i = 0, node = a+n-1; i < n; i++, node--)
        {if (node->indx == -2)
            continue;
         X = node->x;
         x2 = X[0];
         y2 = X[1];
         break;};

    for (i = 0, node = a; i < n; i++, node++)
        {if (node->indx == -2)
            continue;
	 x1 = x2;
         y1 = y2;
         X = node->x;
         x2 = X[0];
         y2 = X[1];
         PG_draw_line(dev, x1, y1, x2, y2);
         PG_draw_markers(dev, 1, &x2, &y2, 1);};

    PG_update_vs(dev);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_TRIANGULARIZE - build a list of triangles from a PM_mesh_topology */

static PG_triangle *PG_triangularize(type, rx, ry, rz, dextr, f, cnnct, 
				     theta, phi, chi, norm, pn, pnx, orient)
   char *type;
   REAL *rx, *ry, *rz, *dextr, *f;
   byte *cnnct;
   double theta, phi, chi, norm;
   int *pn, *pnx;
   double **orient;
   {PG_triangle *tri;

    if (strcmp(type, PM_MESH_TOPOLOGY_S) == 0)
       tri = PG_triangularize_ac((PM_mesh_topology *) cnnct,
				 rx, ry, rz, dextr, f,
				 theta, phi, chi, norm, pn, pnx, orient);
    else
       tri = PG_triangularize_lr((int *) cnnct, rx, ry, rz, dextr, f,
				 theta, phi, chi, norm, pn, pnx, orient);

    return(tri);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_TRIANGULARIZE_AC - build a list of triangles from an
 *                     - Arbitrarily-Connected description of connectivity
 */

static PG_triangle *PG_triangularize_ac(mt, rx, ry, rz, dextr, f,
					theta, phi, chi, norm, pn, pnx, orient)
   PM_mesh_topology *mt;
   REAL *rx, *ry, *rz, *dextr, *f;
   double theta, phi, chi, norm;
   int *pn, *pnx;
   double **orient;
   {int i, jf, je, je1, je2;
    int of, oe, nf, ne, nn, ni, ntri, ntrix;
    int n0, n1, n2;
    int *nc, *np, n[3], nfp, nep;
    long **cells, *faces, *edges;
    double xmn, xmx, ymn, ymx, zmn, zmx;
    double ct, st, cp, sp;
    double lsx, lsy, lsz;
    double x1, x2, y1, y2, z1, z2, x3, y3, z3;
    double dx1, dy1, dz1, dx2, dy2, dz2;
    double s3x, s3y, s3z, *lorient;
    REAL *sx, *sy, *sz;
    REAL *X;
    PG_node *node;
    PG_triangle *tri, *a;

    tri   = NULL;
    ntri  = 0;
    ntrix = 0;

    cells = mt->boundaries;
    faces = cells[2];
    edges = cells[1];

    nc = mt->n_cells;
    nf = nc[2];
    ne = nc[1];
    nn = 2*ne;

    sx = FMAKE_N(REAL, nn, "PG_TRIANGULARIZE_AC:sx");
    sy = FMAKE_N(REAL, nn, "PG_TRIANGULARIZE_AC:sy");
    sz = FMAKE_N(REAL, nn, "PG_TRIANGULARIZE_AC:sz");

    xmn = dextr[0];
    xmx = dextr[1];
    ymn = dextr[2];
    ymx = dextr[3];
    zmn = dextr[4];
    zmx = dextr[5];

    PG_ADJUST_LIMITS_3D(xmn, xmx, ymn, ymx, zmn, zmx);

    PG_rotate_3d_WC(rx, ry, rz, xmn, xmx, ymn, ymx, zmn, zmx,
                    theta, phi, chi, sx, sy, sz, nc[0], FALSE);

    np  = mt->n_bound_params;
    nfp = np[2];
    nep = np[1];
    switch (mt->n_dimensions)
	{case 2 :
	      ntri = 0;
	      for (jf = 0; jf < nf; jf++)
		  {of  = jf*nfp;
		   je1 = faces[of];
		   je2 = faces[of+1];
                   ntri += (je2 - je1 - 1);};

              ntrix = 1000*(1 + ((int) (ntri/1000)));
              tri   = FMAKE_N(PG_triangle, ntrix,
                              "PG_TRIANGULARIZE_AC:tri");

              *orient = FMAKE_N(REAL, ntrix,
                              "PG_TRIANGULARIZE_AC:orient");
              lorient = *orient;

	      ct = cos(theta);
	      cp = cos(phi);
	      st = sin(theta);
	      sp = sin(phi);

/* NOTE: we rotate by phi from negative y axis! */
	      lsx =  st*sp;
	      lsy = -st*cp;
	      lsz = ct;

              ntri = 0;
	      for (jf = 0; jf < nf; jf++)
		  {of  = jf*nfp;
		   je1 = faces[of];
		   je2 = faces[of+1];
                   n[0] = n0 = edges[je1*nep];
		   for (je = je1+1; je < je2; je++)
		       {oe   = je*nep;
			n[1] = n1 = edges[oe];
			n[2] = n2 = edges[oe+1];

			x1 = rx[n0];
			y1 = ry[n0];
			z1 = rz[n0];
			x2 = rx[n1];
			y2 = ry[n1];
			z2 = rz[n1];
			x3 = rx[n2];
			y3 = ry[n2];
			z3 = rz[n2];
                        if ((x1 < xmn) || (xmx < x1) ||
                            (y1 < ymn) || (ymx < y1) ||
                            (z1 < zmn) || (zmx < z1) ||
			    (x2 < xmn) || (xmx < x2) ||
                            (y2 < ymn) || (ymx < y2) ||
                            (z2 < zmn) || (zmx < z2) ||
			    (x3 < xmn) || (xmx < x3) ||
                            (y3 < ymn) || (ymx < y3) ||
                            (z3 < zmn) || (zmx < z3))
			   continue;

			dx1 = x2 - x1;
			dy1 = y2 - y1;
			dz1 = z2 - z1;
			dx2 = x3 - x2;
			dy2 = y3 - y2;
			dz2 = z3 - z2;
			s3x = dy1*dz2 - dz1*dy2;
			s3y = dz1*dx2 - dx1*dz2;
			s3z = dx1*dy2 - dy1*dx2;

/* don't bother with surfaces which are pointing away from the viewer */
			if (no_order || depth_sort || (s3x*lsx + s3y*lsy + s3z*lsz > 0))
			   {lorient[ntri] = s3x*lsx + s3y*lsy + s3z*lsz;
                            a = tri + ntri++;
			    for (i = 0; i < 3; i++)
			        {node = &NODE(a, i);
				 ni   = n[i];

				 X = node->x;
				 X[0] = sx[ni];
				 X[1] = sy[ni];
				 X[2] = sz[ni];

				 node->indx = ni;
				 if (f != NULL)
				    node->val = f[ni];
				 else
				    node->val = 0.0;};};};};};
    SFREE(sx);
    SFREE(sy);
    SFREE(sz);

    *pn  = ntri;
    *pnx = ntrix;

    return(tri);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_TRIANGULARIZE_LR - build a list of triangles from a Logical-Rectangular
 *                     - description of connectivity
 */

static PG_triangle *PG_triangularize_lr(maxes, rx, ry, rz, dextr, f,
					theta, phi, chi, norm, pn, pnx, orient)
   int *maxes;
   REAL *rx, *ry, *rz, *dextr, *f;
   double theta, phi, chi, norm;
   int *pn, *pnx;
   double **orient;
   {int ntri, ntrix;
    PG_triangle *tri;

    tri   = NULL;
    ntri  = 0;
    ntrix = 0;

    *pn  = ntri;
    *pnx = ntrix;

    *orient = NULL;

    return(tri);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_EDGE_ON - return TRUE iff the triangle is parallel to the 
 *             - line of sight
 */

static int _PG_edge_on(a)
   PG_triangle *a;
   {REAL *X1a, *X2a, *X3a;
    REAL px[3], py[3];

    X1a = NODE(a, 0).x;
    X2a = NODE(a, 1).x;
    X3a = NODE(a, 2).x;

    px[0] = X1a[0];
    py[0] = X1a[1];
    px[1] = X2a[0];
    py[1] = X2a[1];
    px[2] = X3a[0];
    py[2] = X3a[1];
    return(PM_colinear_2d(px, py, 3));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_SAME_POINT - return TRUE iff the nodes A and B are
 *                - at the same point
 */

static int _PG_same_point(a, b)
   PG_node *a, *b;
   {REAL *X1, *X2;
    double x1, y1, x2, y2;

    X1 = a->x;
    X2 = b->x;

    x1 = X1[0];
    y1 = X1[1];

    x2 = X2[0];
    y2 = X2[1];

    return(PM_CLOSETO_REL(x1, x2) &&
	   PM_CLOSETO_REL(y1, y2));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_INSIDE - return TRUE iff the node A is inside the triangle B */

static int _PG_inside(a, b)
   PG_node *a;
   PG_triangle *b;
   {REAL *X, *X1b, *X2b, *X3b, px[4], py[4];
    double x, y;

    X = a->x;
    x = X[0];
    y = X[1];

    X1b = NODE(b, 0).x;
    X2b = NODE(b, 1).x;
    X3b = NODE(b, 2).x;

    px[0] = px[3] = X1b[0];
    py[0] = py[3] = X1b[1];
    px[1] = X2b[0];
    py[1] = X2b[1];
    px[2] = X3b[0];
    py[2] = X3b[1];

    if ((PM_CLOSETO_REL(x, px[0]) && PM_CLOSETO_REL(y, py[0])) ||
        (PM_CLOSETO_REL(x, px[1]) && PM_CLOSETO_REL(y, py[1])) ||
	(PM_CLOSETO_REL(x, px[2]) && PM_CLOSETO_REL(y, py[2])))
       return(FALSE);
    else
       return(PM_contains_2d(x, y, px, py, 4));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_UNIQUE_POINTS - pick out the unique intersection points from
 *                   - the triangle intersection matrix INTER
 *                   - return the number of unique points and put the
 *                   - unique points in the PX, PY, and PZ arrays
 */

static int _PG_unique_points(inter, px, py, pz)
   PG_node *inter;
   REAL *px, *py, *pz;
   {int i, j, nu;
    REAL *Xc;
    double x, y, z;
    PG_node *node;

    node = inter;    
    Xc   = node->x;
    for (i = 0, nu = 0, node++; i < 9; i++, node++)
        {Xc = node->x;
         x = Xc[0];
         y = Xc[1];
         z = Xc[2];
	 if (node->indx == -1)
	    {for (j = 0; j < nu; j++)
                 {if (PM_CLOSETO_REL(px[j], x) &&
		      PM_CLOSETO_REL(py[j], y) &&
		      PM_CLOSETO_REL(pz[j], z))
		     break;};
	     if (j == nu)
	        {px[nu] = x;
		 py[nu] = y;
		 pz[nu] = z;
		 nu++;};};};

    return(nu);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_COLINEAR - return TRUE iff N nodes are colinear */

static int _PG_colinear(nodes, n, px, py, pz)
   PG_node *nodes;
   int n;
   REAL *px, *py, *pz;
   {int i, nn;
    REAL *X;
    PG_node *node;

    for (i = 0, nn = 0, node = nodes; i < n; i++, node++)
        {if (node->indx == -2)
            continue;
         X = node->x;
	 px[nn] = X[0];
	 py[nn] = X[1];
	 pz[nn] = X[2];
         nn++;};

    return(PM_colinear_3d(px, py, pz, nn));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_INTERSECT - return TRUE iff the two triangles have
 *               - a line of intersection contained inside
 *               - the triangles
 */

static int _PG_intersect(a, b, inter, ptr, pntr, pntrx)
   PG_triangle *a, *b;
   PG_node *inter;
   PG_triangle **ptr;
   int *pntr, *pntrx;
   {int i0, i1, nc, ni;
    REAL *X1a, *X2a;
    REAL px[3], py[3], pz[3];
    REAL x0, y0, z0, x1, y1, z1, x2, y2, z2;

    if (_PG_colinear(b->node, 3, px, py, pz))
       return(FALSE);

    for (i0 = 0, nc = 0, ni = 0; i0 < 3; i0++)
        {i1  = (i0+1) % 3;
         X1a = NODE(a, i0).x;
         X2a = NODE(a, i1).x;
         x1  = X1a[0];
         y1  = X1a[1];
         z1  = X1a[2];
         x2  = X2a[0];
         y2  = X2a[1];
         z2  = X2a[2];
         if (PM_cross_line_plane(x1, y1, z1, x2, y2, z2,
				 px, py, pz, &x0, &y0, &z0, 0))
	    {nc++;
	     if (PM_contains_3d(x0, y0, z0, px, py, pz, FALSE))
		ni++;}};

/* GOTCHA: to do this right it will be necessary to save the two
 * intersection points and do something like a _PG_split_tri on A
 * being careful to do the interpolations right
 */

    return(ni);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_INTERSECT_PROJECTION - return TRUE iff the projections onto the
 *                          - plane normal to the line of sight of the
 *                          - two triangles a non-empty intersection
 */

static int _PG_intersect_projection(a, b, inter)
   PG_triangle *a, *b;
   PG_node *inter;
   {int i, is, in, j, jn, k, nc, nu;
    double x1, y1, x2, y2, x3, y3, x4, y4, xc, yc;
    double z3, z4, v3, v4, f, dx;
    REAL *X1a, *X2a, *X1b, *X2b, *Xc;
    REAL px[9], py[9], pz[9];
    PG_node *node;

    for (i = 0; i < 9; i++)
        inter[i].indx = -2;

    for (i = 0, nc = 0; i < 3; i++)
        {in  = (i+1) % 3;
         X1a = NODE(a, i).x;
         X2a = NODE(a, in).x;
         x1  = X1a[0];
         y1  = X1a[1];
         x2  = X2a[0];
         y2  = X2a[1];
         if ((x1 == x2) && (y1 == y2))
            continue;
            
         for (j = 0; j < 3; j++)
	     {jn   = (j+1) % 3;
	      k    = i*3 + j;
	      node = inter + k;

	      X1b = NODE(b, j).x;
	      X2b = NODE(b, jn).x;
	      x3  = X1b[0];
	      y3  = X1b[1];
	      x4  = X2b[0];
 	      y4  = X2b[1];
	      if ((x3 == x4) && (y3 == y4))
		 continue;

              xc = HUGE;
              yc = HUGE;
              is = FALSE;
              if (((x1 == x3) && (y1 == y3)) ||
		  ((x2 == x3) && (y2 == y3)))
		 {xc = x3;
		  yc = y3;
		  is = TRUE;}
	      else if (((x1 == x4) && (y1 == y4)) ||
		       ((x2 == x4) && (y2 == y4)))
		 {xc = x4;
		  yc = y4;
		  is = TRUE;}
	      else
		 is = PM_cross_seg(x1, y1, x2, y2, x3, y3, x4, y4, &xc, &yc);

	      if (is)
		 {Xc = node->x;
                  dx = x4 - x3;

		  z3 = X1b[2];
		  z4 = X2b[2];
		  v3 = NODE(b, j).val;
		  v4 = NODE(b, jn).val;
                  if (dx != 0.0)
		     f = (xc - x3)/dx;
		  else
		     f = (yc - y3)/(y4 - y3);

                  Xc[0] = xc;
                  Xc[1] = yc;
		  Xc[2] = z3 + (z4 - z3)*f;

		  node->indx = -1;
                  node->val  = v3 + (v4 - v3)*f;

		  nc++;};};};

/* find the area of overlap */
    nu = _PG_unique_points(inter, px, py, pz);

/* if there are less than 2 points there is no area of overlap */
    if (nu < 2)
       return(FALSE);

    else if (nu == 2)
       return(TRUE);

/* if the points are colinear there is no area of overlap */
    else if (PM_colinear_3d(px, py, pz, nu))
       return(FALSE);

    return(nc);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_HIDES - return COMPLETE iff the triangle A is completely hidden by
 *           - the triangle B, PARTIAL if B obscures part of A, and FALSE
 *           - otherwise
 */

static int _PG_hides(b, a)
   PG_triangle *b, *a;
   {int i, hide, behind, below;
    REAL *X1, *X2, *X3;
    double x1, y1, z1, x2, y2, z2, x3, y3, z3, xc, yc, zc;
    double dx1, dy1, dz1, dx2, dy2, dz2, dx3, dy3;
    double nz1, nz2, nz3, ns1, ns2, ns3, dp;
    PG_node *node;

    X1 = NODE(b, 0).x;
    X2 = NODE(b, 1).x;
    X3 = NODE(b, 2).x;

    x1  = X1[0];
    y1  = X1[1];
    z1  = X1[2];
    x2  = X2[0];
    y2  = X2[1];
    z2  = X2[2];
    x3  = X3[0];
    y3  = X3[1];
    z3  = X3[2];
    dx1 = x2 - x1;
    dy1 = y2 - y1;
    dz1 = z2 - z1;
    dx2 = x3 - x2;
    dy2 = y3 - y2;
    dz2 = z3 - z2;
    dx3 = x1 - x3;
    dy3 = y1 - y3;

    ns1 = dy1*dz2 - dz1*dy2;
    ns2 = dz1*dx2 - dx1*dz2;
    ns3 = dx1*dy2 - dy1*dx2;

    hide   = TRUE;
    behind = FALSE;
    for (i = 0, node = a->node; i < 3; i++, node++)
        {X1 = node->x;
	 xc = X1[0];
	 yc = X1[1];
	 zc = X1[2];

         nz1 = dx1*(yc - y1) - dy1*(xc - x1);
         nz2 = dx2*(yc - y2) - dy2*(xc - x2);
         nz3 = dx3*(yc - y3) - dy3*(xc - x3);

         dp  = ns1*(xc - x2) + ns2*(yc - y2) + ns3*(zc - z2);

         below = (dp < TOLERANCE);
         behind |= below;
         hide   &= (below &&
		    (((nz1 > -TOLERANCE) &&
		      (nz2 > -TOLERANCE) &&
		      (nz3 > -TOLERANCE)) ||
		     ((nz1 < TOLERANCE) &&
		      (nz2 < TOLERANCE) &&
		      (nz3 < TOLERANCE))));};

    return(hide ? COMPLETE : (behind ? PARTIAL : FALSE));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_ORIENTATION - compute the normal of the 3 given nodes */

static void _PG_orientation(cn, pnx, pny, pnz)
   PG_node *cn;
   REAL *pnx, *pny, *pnz;
   {REAL *X1, *X2, *X3;
    double x1, y1, z1, x2, y2, z2, x3, y3, z3;
    double dx1, dy1, dz1, dx2, dy2, dz2;

    X1 = (cn++)->x;
    X2 = (cn++)->x;
    X3 = (cn++)->x;

    x1  = X1[0];
    y1  = X1[1];
    z1  = X1[2];
    x2  = X2[0];
    y2  = X2[1];
    z2  = X2[2];
    x3  = X3[0];
    y3  = X3[1];
    z3  = X3[2];
    dx1 = x2 - x1;
    dy1 = y2 - y1;
    dz1 = z2 - z1;
    dx2 = x3 - x2;
    dy2 = y3 - y2;
    dz2 = z3 - z2;

    *pnx = dy1*dz2 - dz1*dy2;
    *pny = dz1*dx2 - dx1*dz2;
    *pnz = dx1*dy2 - dy1*dx2;

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_PUSH_TRIANGLE - push a triangle onto the stack TR
 *                   - if it is non-degenerate
 *                   - ensure that the orientation is the same as the
 *                   - given triangle
 */

static int _PG_push_triangle(node0, node1, node2, a, tr, itr)
   PG_node *node0, *node1, *node2;
   PG_triangle *a, *tr;
   int itr;
   {int i;
    REAL *X, px[3], py[3], pz[3], xc, yc, zc;
    REAL nxa, nya, nza, nxc, nyc, nzc;
    double x1, y1, z1;
    PG_node *cn;
    PG_triangle tc;

    cn = tc.node;

    cn[0] = *node0;
    cn[1] = *node1;
    cn[2] = *node2;
    if (!_PG_colinear(cn, 3, px, py, pz))
       {_PG_orientation(a->node, &nxa, &nya, &nza);
	_PG_orientation(cn, &nxc, &nyc, &nzc);
	if (nxa*nxc + nya*nyc + nza*nzc > 0.0)
           {cn[2] = *node1;
            cn[1] = *node2;};

/* make sure that the z value of new nodes is correct */
        for (i = 0; i < 3; i++)
	    {X = NODE(a, i).x;
	     px[i] = X[0];
	     py[i] = X[1];
	     pz[i] = X[2];};
            
        for (i = 0; i < 3; i++)
	    {if (cn[i].indx < 0)
	        {X = cn[i].x;
		 x1 = X[0];
		 y1 = X[1];
		 z1 = X[2];
		 if (PM_cross_line_plane(x1, y1, z1, x1, y1, 0.0,
					 px, py, pz,
					 &xc, &yc, &zc, 2))
		    X[2] = zc;};};

	if (!_PG_colinear(cn, 3, px, py, pz))
	   tr[itr--] = tc;};

    return(itr);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_SPLIT_TRI - partition the triangle A into triangles such that
 *               - at most one side or one point of each new triangle
 *               - is on the line defined by one side of B
 *               - use INTER to recognize the intersections of side of
 *               - A and B
 *               - the new triangles of A go onto the list PTR
 *               - return the new list PTR and its length NPTR
 *               - return TRUE iff there has been a split
 */

static int _PG_split_tri(a, b, inter, ptr, pi, pntr, pntrx)
   PG_triangle *a, *b;
   PG_node *inter;
   PG_triangle **ptr;
   int *pi, *pntr, *pntrx;
   {int i, itr, flag;
    int ntr, ntrx;
    PG_node *node, *cn;
    PG_triangle ta, tc, *tr;

    tr   = *ptr;
    itr  = *pi;
    ntr  = *pntr;
    ntrx = *pntrx;

    ta = *a;

/* try to save space by pushing the new triangles over already processed
 * ones
 */
    flag = itr < 2;
    if (flag)
       {itr = ntr;
        ntr += 3;
        if (ntr >= ntrx)
           {ntrx += 1000;
	    REMAKE_N(tr, PG_triangle, ntrx);};};

    cn = tc.node;

/* if there are nodes of B in the interior of A pick one and divide A
 * into three triangles each of which shares the node of B
 */
    for (i = 0, node = b->node; i < 3; i++, node++)
        {if (_PG_inside(node, a))
            break;};

    if (i < 3)
       {cn[0] = node[0];
	cn[1] = ta.node[0];
	cn[2] = ta.node[1];
	tr[itr--] = tc;

	cn[1] = cn[2];
	cn[2] = ta.node[2];
	tr[itr--] = tc;

	cn[1] = cn[2];
	cn[2] = ta.node[0];
	tr[itr--] = tc;}

/* there are no nodes of B in A if we are here */
    else

/* find the first intersection point which is distinct from the first
 * node of triangle A
 */
       {int n1, n2, k1, k2, oitr;
	PG_node *node1, *node2;

	n1 = 1;
	n2 = 2;
        for (k1 = 0, node1 = inter; k1 < 9; k1++, node1++)
	    {k2    = (k1 + 3) % 9;
             node2 = inter + k2;
	     if (k1 >= 3)
	        {n1 = 2;
		 n2 = 1;};

	     if ((node1->indx != -2) && (node2->indx != -2))
	        {if (!_PG_same_point(node1, ta.node) &&
		     !_PG_same_point(node1, ta.node+1))
		    break;};};

        if (k1 >= 9)
           return(FALSE);

        oitr = itr - 1;

	itr = _PG_push_triangle(ta.node, node1, node2, a, tr, itr);
	itr = _PG_push_triangle(ta.node, node2, ta.node+n2, a, tr, itr);
	itr = _PG_push_triangle(ta.node+n1, node2, node1, a, tr, itr);
        if (itr >= oitr)
           return(FALSE);};

    if (!flag)
       *pi = itr;

    *ptr   = tr;
    *pntr  = ntr;
    *pntrx = ntrx;

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* PG_HLS_REMOVE - process all elements building up a list of visible
 *               - ones
 *               - input is the set of rotated node coordinate vectors
 *               - and their connectivity
 */

int PG_hls_remove(type, rx, ry, rz, dextr, f, theta, phi, chi, cnnct, alst,
		  prx, pry, prz, pf, pn)
   char *type;
   REAL *rx, *ry, *rz, *dextr, *f;
   double theta, phi, chi;
   byte *cnnct;
   pcons *alst;
   REAL **prx, **pry, **prz, **pf;
   int *pn;
   {int i, j, nn, ntr, ntrx, nvi, nvix, rel;
    REAL *tx, *ty, *tz, *tf, *X;
    double norm, *pm, *orient;
    PG_node inter[9], *node;
    PG_triangle *trial, *vis, *a, *b;

    SC_assoc_info(alst,
		  "NORMAL-DIRECTION", &pm,
		  NULL);

    norm = (pm == NULL) ? 1.0 : *pm;

    nvi   = 0;
    nvix  = 0;
    orient = NULL;
    trial = PG_triangularize(type, rx, ry, rz, dextr, f, cnnct,
			     theta, phi, chi, norm, &ntr, &ntrx, &orient);
    if (trial == NULL)
       return(FALSE);

/*    PRINT(stdout, "Process %d trial triangles\n", ntr); */

    memset(inter, 0, 9*sizeof(PG_node));

/* reference nprint to make some warnings go away */
    if (debug)
       nprint(inter, 9);

    if (no_order)
       {vis = trial;
	nvi = ntr;}

    else if(depth_sort)
       {vis = _PG_depth_sort(trial, ntr, orient);
        nvi = ntr;
        if (orient != NULL)
           SFREE(orient);
        SFREE(trial);}

    else

/* process the trial elements */
       {vis = NULL;
	for (i = 0; i < ntr; i++)
	    {a = trial + i;
	     if (_PG_edge_on(a))
	        continue;

	     for (j = 0; j < nvi; j++)
	         {b = vis + j;
		  if (debug)
		     {ndraw((PG_node *) NULL,0);
		      ndraw((PG_node *) a,3);
		      ndraw((PG_node *) b,3);};

		  if (_PG_intersect_projection(a, b, inter))
		     {if (_PG_intersect(a, b, inter, &trial, &ntr, &ntrx))
			 break;

/* this isn't right but it is better than without it */
		      else if (_PG_hides(a, b) == COMPLETE)
			 vis[j--] = vis[--nvi];

		      else if ((rel = _PG_hides(b, a)) == COMPLETE)
			 break;

		      else if (rel == PARTIAL)
			 {if (_PG_split_tri(a, b, inter,
					    &trial, &i, &ntr, &ntrx))
			     break;};};};

	     if (j == nvi)
	        {REMEMBER_TRI(a, vis, nvi, nvix);};};

	SFREE(trial);};

/*    PRINT(stdout, "Found %d visible triangles\n", nvi); */

    nn = 3*nvi;

    tx = FMAKE_N(REAL, nn, "PG_HLS_REMOVE:tx");
    ty = FMAKE_N(REAL, nn, "PG_HLS_REMOVE:ty");
    tz = FMAKE_N(REAL, nn, "PG_HLS_REMOVE:tz");
    tf = FMAKE_N(REAL, nn, "PG_HLS_REMOVE:tf");

    nn = 0;
    for (j = 0; j < nvi; j++)
        {b = vis + j;
	 for (i = 0; i < 3; i++)
	     {node = &NODE(b, i);

	      X = node->x;
	      tx[nn] = X[0];
	      ty[nn] = X[1];
	      tz[nn] = X[2];
              tf[nn] = node->val;
	      nn++;};};

    SFREE(vis);

    *prx = tx;
    *pry = ty;
    *prz = tz;
    *pn  = nvi;

    if (pf != NULL)
       *pf = tf;

    return(TRUE);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _PG_DEPTH_SORT - Sort an array of PG_triangle on z */

static PG_triangle *_PG_depth_sort(trial, ntr, orient)
   PG_triangle *trial;
   int ntr;
   double *orient;
   {PG_triangle *triout, *a;
    int *ind, i, itr;
    REAL *z;
    PG_node *node0, *node1, *node2;
 

    z      = FMAKE_N(REAL, ntr, "_PG_depth_sort:z");
    ind    = FMAKE_N(int, ntr, "_PG_depth_sort:ind");
    triout = FMAKE_N(PG_triangle, ntr, "_PG_depth_sort:triout");

    for (i = 0; i < ntr; i++)
        {a = trial + i;

         node0 = &NODE(a, 0);
         node1 = &NODE(a, 1);
         node2 = &NODE(a, 2);

/*
         z0     = (node0->x[2] < node1->x[2]) ?  node0->x[2]:
                                                 node1->x[2];
         z[i]   = (z0 < node2->x[2] ) ? z0 : node2->x[2];
*/
         z[i] = (node0->x[2] + node1->x[2] + node2->x[2])/3.0;
         ind[i] = i;}

/*
    done = FALSE;
    while (!done)
        {done = TRUE;
         for (i = 0; i < ntr - 1; i++)
             if (z[i] > z[i+1])
                {ztemp = z[i];
                 z[i] = z[i+1];
                 z[i+1] = ztemp;

                 itemp = ind[i];
                 ind[i] = ind[i+1];
                 ind[i+1] = itemp;
                 done = FALSE;};};
*/

    PM_q_sort(z, ind, ntr);

    if (orient != NULL)
       {itr = 0;
        for (i = 0; i < ntr; i++)
            if (orient[ind[i]] <= 0)
               {triout[itr] = trial[ind[i]];
                itr++;}

        for (i = 0; i < ntr; i++)
            if (orient[ind[i]] > 0)
               {triout[itr] = trial[ind[i]];
	       itr++;};}
    else
       for (i = 0; i < ntr; i++)
           triout[i] = trial[ind[i]];
            

    return triout;}
    

