/*
 * SXFUNC.C - basic math functions for SX
 *
 * Source Version: 3.0
 * Software Release #92-0043
 *
 */

#include "cpyright.h"
 
#include "sx.h"

#define GET_DATA_1D(_f, _n, _x, _y)                                         \
    {PM_set *_dm, *_rn;                                                     \
     _dm = (_f)->domain;                                                    \
     _rn = (_f)->range;                                                     \
     _n  = _dm->n_elements;                                                 \
     _x  = PM_array_real(_dm->element_type,                                 \
			 DEREF(_dm->elements), _n, NULL);                   \
     _y  = PM_array_real(_rn->element_type,                                 \
			 DEREF(_rn->elements), _n, NULL);}


extern void
 SC_DECLARE(_SX_shift_set, (PM_set *set, double val)),
 SC_DECLARE(_SX_scale_set, (PM_set *set, double val)),
 SC_DECLARE(_SX_integrate_mapping, (PM_mapping *f));

static object
 SC_DECLARE(*SX_quit, (byte));

/*--------------------------------------------------------------------------*/

/*                          RUDAMENTARY FUNCTIONS                           */

#if 0

/*--------------------------------------------------------------------------*/

/* LABEL - change the label of the given curve */

object *label(argl)
   object *argl;
   {object *s, *tok;
    int j;

    Sprep_arg(argl);
    s = SS_car(argl);
    argl = SS_cdr(argl);
    if (!Scurvep(s))
       SS_error("BAD CURVE ARGUMENT - LABEL", s);

    if (!SS_stringp(tok = SS_car(argl)))
       SS_error("BAD LABEL ARGUMENT - LABEL", tok);

    j = get_curve(s);
    SFREE(dataset[j].text);
    dataset[j].text = SC_strsavef(SS_GET(string, tok)->string,
                      "char*:LABEL:label");

    return(s);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* AVERAGE - return a new curve containing the average of
 *         - the curves specified
 */

object *average(s)
   object *s;
   {int j;
    object *c, *numtoks;
    char *t;

    Sprep_arg(s);
    plot_off();

    numtoks = SS_mk_integer((BIGINT) _SS_length(s));
    c = bc((ifunc) PM_fplus, s);
    if (SS_true(c))
       {c = opyc((ifunc) PM_fdivide, SS_mk_cons(c, mk_cons(numtoks, SS_null)));
        j = get_curve(c);
        SC_strtok(dataset[j].text, " ", t);
        sprintf(pbuffer, "Append %s", SC_strtok(NULL, "\n", t));
        SFREE(dataset[j].text);
        dataset[j].text = SC_strsavef(pbuffer, "char*:AVERAGE:average");
        restore_plot();
        return(c);}
    else
       {return(SS_null);};}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SMOOTH - the n point smoothing routine */

object *smooth(obj, s)
   object *obj, *s;
   {int i, j, l;
    REAL *xp, *yp;
    REAL sum;
    int n;

    n = (int) *SS_GET(BIGINT, s);
    n = ((n % 2) != 1) ? n = n+1 : n;                /* force n odd */
    l = get_curve(obj);

    xp = dataset[l].xp;
    yp = dataset[l].yp;
    buf1x = FMAKE_N(REAL, dataset[l].n, "SMOOTH:buf1x");
    buf1y = FMAKE_N(REAL, dataset[l].n, "SMOOTH:buf1y");
        
    for (i = 0; i+n <= dataset[l].n; i++, xp++, yp++)
        {for (j = 1, sum = 0; j < n; j++)
             {sum += (xp[j]-xp[j-1])*(yp[j]+yp[j-1]);};
         if (sum == 0) 
            {for (j = 0; j < n; j++)
                 sum += yp[j];
             sum /= n;}
         else
            {if (xp[n-1] != *xp)
                sum /= 2*(xp[n-1]-xp[0]);
             else
                sum = yp[0];};
         buf1y[i] = sum;
         buf1x[i] = (xp[0] + xp[n-1])/2;};

    xp = dataset[l].xp;                                /* copy back */
    yp = dataset[l].yp;
    for (j = 0; j < i; j++)
        {xp[j] = buf1x[j];
         yp[j] = buf1y[j];};

    dataset[l].n = i;
    lmt(dataset[l].xp, i, &dataset[l].xmin, &dataset[l].xmax);
    lmt(dataset[l].yp, i, &dataset[l].ymin, &dataset[l].ymax);
    SFREE(buf1x);
    SFREE(buf1y);

    return(obj);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SMOOTH3 - do 3 point smooths on the given curves
 *         - NOTE: instead of bashing 3 onto the end and having bltoc
 *         -       pull do two reverses to get it, why not cons it on from
 *         -       and use bftoc? Same for smooth5.
 */

object *smooth3(s)
   object *s;
   {object *tmp;

    plot_off();
    Sprep_arg(s);
    tmp = _SS_append(s, SS_mk_cons(SS_mk_integer((BIGINT)3), SS_null));
    SS_MARK(tmp);
    SS_MARK(s);
    bltoc((ifunc) smooth, tmp);
    SS_GC(tmp);
    restore_plot();
    return(s);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SMOOTH5 - do 5 point smooths on the given curves */

object *smooth5(s)
   object *s;
   {object *tmp;

    plot_off();
    Sprep_arg(s);
    tmp = _SS_append(s, SS_mk_cons(SS_mk_integer((BIGINT)5), SS_null));
    SS_MARK(tmp);
    SS_MARK(s);
    bltoc((ifunc) smooth, tmp);
    SS_GC(tmp);
    restore_plot();
    return(s);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* COMPOSE - a functional composition of curves */

double compose(a, i)
   double a;
   int i;
   {int j;
    double x1, x2, y1, y2;

    j = lfind(a, dataset[i].xp, dataset[i].n);
    if (j >= 0)
       {x1 = dataset[i].xp[j];
        y1 = dataset[i].yp[j];
        x2 = dataset[i].xp[j+1];
        y2 = dataset[i].yp[j+1];
        return(interp(a, x1, y1, x2, y2));}
    else
       {return(HUGE);};}                    /* case where nothing is found */

/*--------------------------------------------------------------------------*/

#endif

/*--------------------------------------------------------------------------*/

/* SX_QUIT - gracefully exit from this session */

static object *SX_quit()
   {_SX_quit(0);

    return(SS_f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_QUIT - gracefully exit from PDBView */

void _SX_quit(i)
   int i;
   {

    PC_exit_all();

/* check the need to close the command log */
    if (SX_command_log != NULL)
       io_close(SX_command_log);
#if 0
/* check the need to close the PostScript device */
    if (SX_PS_device != NULL)
       PG_close_device(SX_PS_device);

/* check the need to close the CGM device */
    if (SX_CGM_device != NULL)
       PG_close_device(SX_CGM_device);

/* close the cache file and any open data files */
    SX_close_open_files();
#endif
    exit(i);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SYNONYM - make synonyms for the given function */

object *SX_synonym(argl)
   object *argl;
   {object *func;
    char *synname;

    SX_prep_arg(argl);

    func = SS_exp_eval(SS_car(argl));
    if (!SS_procedurep(func))
       SS_error("FIRST ARG MUST BE FUNCTION - SX_SYNONYM", func);

    for (argl = SS_cdr(argl); SS_consp(argl); argl = SS_cdr(argl))
        {synname = SS_get_string(SS_car(argl));
         SC_hash_rem(synname, SS_symtab);
         SC_install(synname, func, SS_OBJECT_S, SS_symtab);};

    return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_TOGGLE_LOG - toggle the command log file */

object *SX_toggle_log(argl)
   object *argl;
   {object *obj;
    static char fname[MAXLINE];
    static char mode[] = "a";

    if (SX_command_log != NULL)
       {io_close(SX_command_log);
	SX_command_log = NULL;};

    if (SS_consp(argl))
       {obj = SS_car(argl);
        strcpy(fname, SS_get_string(obj));
        if (strcmp(fname, "off") ==  0)
           *fname = '\0';
        else if (strcmp(fname, "on") == 0)
           sprintf(fname, SX_command_log_name);}
    else
       sprintf(fname, SX_command_log_name);

    if (*fname != '\0')
       {SX_command_log = io_open(fname, mode);
        if (SX_command_log == NULL)
           SS_error("CANNOT OPEN LOG FILE - SX_TOGGLE_LOG", SS_null);
        else
           SC_setbuf(SX_command_log, NULL);};

    if (SX_command_log == NULL)
       return(SS_f);
    else
       return(SS_t);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GET_DIMENSION - return a cons whose car is the dimension of the
 *                  - domain and whose cdr is the dimension of the range
 *                  - of the given mapping
 */

object *SX_get_dimension(obj)
   object *obj;
   {PG_graph *g;
    PM_mapping *f;
    PM_set *set;
    int ndd, ndr;

    if (SX_GRAPHP(obj))
       {g = SS_GET(PG_graph, obj);
        f = g->f;}
    else if (SX_MAPPINGP(obj))
       f = SS_GET(PM_mapping, obj);
    else
       SS_error("BAD MAPPING - SX_GET_DIMENSION", obj);

    set = f->domain;
    ndd = set->dimension_elem;
    set = f->range;
    ndr = set->dimension_elem;

    return(SS_mk_cons(SS_mk_integer((BIGINT)ndd), SS_mk_integer((BIGINT)ndr)));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GET_DOMAIN - return the domain of the mapping object as a set object */

object *SX_get_domain(arg)
   object *arg;
   {PM_set *set;

    if (SX_MAPPINGP(arg))
       set = MAPPING_DOMAIN(arg);
    else if (SX_GRAPHP(arg))
       set = GRAPH_F(arg)->domain;
    else if (SX_SETP(arg))
       set = SS_GET(PM_set, arg);
    else
       SS_error("BAD SET - SX_GET_DOMAIN", arg);

    return(SX_mk_set(set));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GET_RANGE - return the range of the mapping object as a set object */

object *SX_get_range(arg)
   object *arg;
   {PM_mapping *f;

    f = NULL;
    SS_args(arg,
            G_MAPPING, &f,
	    0);

    if (f == NULL)
       SS_error("BAD MAPPING - SX_GET_RANGE", arg);

    return(SX_mk_set(f->range));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_GET_MAPPING_NAME - return the name of the mapping object
 *                     - as a string object
 */

object *SX_get_mapping_name(arg)
   object *arg;
   {char *name;

    if (!SX_MAPPINGP(arg))
       SS_error("BAD MAPPING - SX_GET_RANGE", arg);

    name = MAPPING_NAME(arg);
    return(SS_mk_string(name));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SET_VOLUME - return the product of the differences of the extrema
 *               - of the given set
 */

object *SX_set_volume(arg)
   object *arg;
   {PM_set *set;
    REAL *extr, vol, mn, mx;
    int i, nd;

    if (!SX_SETP(arg))
       SS_error("BAD SET - SX_SET_VOLUME", arg);

    set = SS_GET(PM_set, arg);
    
    extr = (REAL *) set->extrema;
    nd   = set->dimension;
    vol  = 1.0;
    for (i = 0; i < nd; i++)
        {mn = *extr++;
         mx = *extr++;
         vol *= (mx - mn);};
    
    return(SS_mk_float(vol));}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SHIFT_DOMAIN - shift all components of all elements of the
 *                 - domain of the given mapping
 *                 - return the mapping
 */

object *SX_shift_domain(argl)
   object *argl;
   {object *obj, *ret;
    PM_set *set;
    REAL val;

    ret = SS_null;
    if (SS_consp(argl))
       {SX_GET_OBJECT_FROM_LIST(SX_MAPPINGP(obj), set, MAPPING_DOMAIN(obj),
                                argl, "BAD MAPPING - SX_SHIFT_DOMAIN");
        SS_Assign(ret, obj);};

    if (SS_consp(argl))
       {SX_GET_FLOAT_FROM_LIST(val, argl,
                               "BAD FLOATING POINT VALUE - SX_SHIFT_DOMAIN");}

    else
       SS_error("INSUFFICIENT ARGUMENTS - SX_SHIFT_DOMAIN", argl);

    _SX_shift_set(set, val);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SHIFT_RANGE - shift all components of all elements of the
 *                - range of the given mapping
 *                - return the mapping
 */

object *SX_shift_range(argl)
   object *argl;
   {object *obj, *ret;
    PM_set *set;
    REAL val;

    ret = SS_null;
    if (SS_consp(argl))
       {SX_GET_OBJECT_FROM_LIST(SX_MAPPINGP(obj), set, MAPPING_RANGE(obj),
                                argl, "BAD MAPPING - SX_SHIFT_RANGE");
        SS_Assign(ret, obj);};

    if (SS_consp(argl))
       {SX_GET_FLOAT_FROM_LIST(val, argl,
                               "BAD FLOATING POINT VALUE - SX_SHIFT_RANGE");}

    else
       SS_error("INSUFFICIENT ARGUMENTS - SX_SHIFT_RANGE", argl);

    _SX_shift_set(set, val);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SCALE_DOMAIN - scale all components of all elements of the
 *                 - domain of the given mapping
 *                 - return the mapping
 */

object *SX_scale_domain(argl)
   object *argl;
   {object *obj, *ret;
    PM_set *set;
    REAL val;

    ret = SS_null;
    if (SS_consp(argl))
       {SX_GET_OBJECT_FROM_LIST(SX_MAPPINGP(obj), set, MAPPING_DOMAIN(obj),
                                argl, "BAD MAPPING - SX_SCALE_DOMAIN");
        SS_Assign(ret, obj);};

    if (SS_consp(argl))
       {SX_GET_FLOAT_FROM_LIST(val, argl,
                               "BAD FLOATING POINT VALUE - SX_SCALE_DOMAIN");}
    else
       SS_error("INSUFFICIENT ARGUMENTS - SX_SCALE_DOMAIN", argl);

    _SX_scale_set(set, val);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SCALE_RANGE - scale all components of all elements of the
 *                - range of the given mapping
 *                - return the mapping
 */

object *SX_scale_range(argl)
   object *argl;
   {object *obj, *ret;
    PM_set *set;
    REAL val;

    ret = SS_null;
    if (SS_consp(argl))
       {SX_GET_OBJECT_FROM_LIST(SX_MAPPINGP(obj), set, MAPPING_RANGE(obj),
                                argl, "BAD MAPPING - SX_SCALE_RANGE");
        SS_Assign(ret, obj);};

    if (SS_consp(argl))
       {SX_GET_FLOAT_FROM_LIST(val, argl,
                               "BAD FLOATING POINT VALUE - SX_SCALE_RANGE");}

    else
       SS_error("INSUFFICIENT ARGUMENTS - SX_SCALE_RANGE", argl);

    _SX_scale_set(set, val);

    return(ret);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_SHIFT_SET - shift the components of all elements of the given
 *               - set by the given value
 */

void _SX_shift_set(set, val)
   PM_set *set;
   double val;
   {REAL **elem, *pe;
    int i, j, nde, ne;

    ne   = set->n_elements;
    nde  = set->dimension_elem;
    elem = (REAL **) set->elements;

    for (i = 0; i < nde; i++)
        {pe = elem[i];
         for (j = 0; j < ne; j++)
             *pe++ += val;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_SCALE_SET - scale the components of all elements of the given
 *               - set by the given value
 */

void _SX_scale_set(set, val)
   PM_set *set;
   double val;
   {REAL **elem, *pe;
    int i, j, nde, ne;

    ne   = set->n_elements;
    nde  = set->dimension_elem;
    elem = (REAL **) set->elements;

    for (i = 0; i < nde; i++)
        {pe = elem[i];
         for (j = 0; j < ne; j++)
             *pe++ *= val;};

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_MAP_COUNT - keep track of generated mappings */

int SX_map_count()
   {static int mid = 'a';

    if (mid > 'z')
       mid = 'a';

    return(mid++);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_NORM_MAPPING - build a mapping whose range elements are the Euclidean
 *                 - norm of the input mapping range elements
 *                 - return a new mapping
 */

PM_mapping *SX_norm_mapping(h)
   PM_mapping *h;
   {PM_set *domain, *range;
    PM_mapping *f;
    int i, j, nde, ne;
    REAL **sre, *dre, *pr, *ps, dx;
    char label[MAXLINE];
    byte *elem[1];

/* build the return mapping */
    domain = SX_copy_set(h->domain);
    range  = h->range;

    nde = range->dimension_elem;
    ne  = range->n_elements;
    sre = (REAL **) range->elements;

    dre = FMAKE_N(REAL, ne, "SX_NORM_MAPPING:dre");

    for (j = 0; j < nde; j++)
        {pr = dre;
         ps = sre[j];
         for (i = 0; i < ne; i++)
             {dx     = *ps++;
              *pr++ += dx*dx;};};

    pr = dre;
    for (i = 0; i < ne; i++, pr++)
        *pr = sqrt(*pr);

    elem[0] = (byte *) dre;
    sprintf(label, "||%s||", range->name);
    range = PM_make_set_alt(label, SC_REAL_S, FALSE,
			    range->dimension, range->max_index,
			    nde, elem);

    sprintf(label, "%s->%s", domain->name, range->name);
    f = PM_make_mapping(label, PM_LR_S, domain, range, N_CENT, NULL);

    return(f);}
        
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_COPY_SET - make a copy of the given set */

PM_set *SX_copy_set(in)
   PM_set *in;
   {PM_set *set;
    int i, nd, ne, nde, nec;
    int *mx, *maxes;
    byte **elem, **iel, *pe;

    nd = in->dimension;
    mx = in->max_index;
    ne = in->n_elements;

    maxes = FMAKE_N(int, nd, "SX_COPY_SET:maxes");
    for (i = 0; i < nd; i++)
        maxes[i] = mx[i];

    nde  = in->dimension_elem;
    nec  = ne*sizeof(REAL);
    iel  = (byte **) in->elements;
    elem = FMAKE_N(byte *, nde, "SX_COPY_SET:elem");
    for (i = 0; i < nde; i++)
        {elem[i] = pe = (byte *) FMAKE_N(char, nec, "SX_COPY_SET:elem[]");
         memcpy(pe, iel[i], nec);};

    set = _PM_make_set(in->name, in->element_type, FALSE,
		       ne, nd, nde, maxes, elem,
                       PM_REAL_Opers, NULL,
                       NULL, NULL, NULL, NULL, NULL, NULL,
                       NULL);

    return(set);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_COPY_MAPPING - make a copy of the given mapping */

PM_mapping *SX_copy_mapping(h)
   PM_mapping *h;
   {char name[MAXLINE], *cat;
    PM_set *domain, *range;
    PM_mapping *f;

    domain = SX_copy_set(h->domain);
    range  = SX_copy_set(h->range);
    cat    = h->category;
    sprintf(name, "copy %s", h->name);

    f = PM_make_mapping(name, cat, domain, range, N_CENT, NULL);

    if (strcmp(f->map_type, SC_PCONS_P_S) == 0)
       {SC_free_alist(f->map, 0);
	f->map = SC_copy_alist(h->map);};

    return(f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_INTEGRATE_MAPPING - integrate a mapping and
 *                      - return a new mapping
 */

PM_mapping *SX_integrate_mapping(h)
   PM_mapping *h;
   {PM_set *domain;
    PM_mapping *f;
    char label[MAXLINE];

/* build the return mapping */
    domain = SX_build_common_domain(h);
    f      = SX_build_accumulator_mapping(domain, h);

    sprintf(label, "Integral %s", h->name);

    _SX_integrate_mapping(f);

    SFREE(f->name);
    f->name = SC_strsavef(label, "char*:SX_INTEGRATE_MAPPING:label");
    PM_find_extrema(f->range);

    return(f);}
        
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_INTEGRATE_MAPPING - integrate a mapping in place */

void _SX_integrate_mapping(f)
   PM_mapping *f;
   {PM_set *domain, *range;
    int i, j, jn, jc, is, ndd, ned, ndr;
    int id, lne, dj, ix;
    int *dmx;
    long step, npts;
    REAL vol;
    REAL **re, *dsc, *rn;

/* build the return mapping */
    domain = f->domain;
    range  = f->range;

    dsc = (REAL *) domain->scales;
    ndd = domain->dimension;
    dmx = domain->max_index;
    ned = domain->n_elements;

    ndr = range->dimension_elem;
    re  = (REAL **) range->elements;

/* compute the node volume */
    vol = 1.0;
    for (j = 0; j < ndd; j++)
        vol *= dsc[j];

/* multiply by the volume */
    for (j = 0; j < ndr; j++)
        {rn = re[j];
         for (i = 0; i < ned; i++)
             *rn++ *= vol;};

/* scale each face by one half - corrects volume factor */
/*
    for (i = 0; i < ned; i++)
        {id  = i;
         lne = ned;
         for (j = ndd-1; j >= 0; j--)
             {dj   = dmx[j];
              lne /= dj;
              ix   = id / lne;
              if ((ix == 0) || (ix == dj-1))
                 for (jc = 0; jc < ndr; jc++)
                     re[jc][i] *= 0.5;
              id %= lne;};};
*/
/* sum it up */
    lne = ned;
    for (j = ndd-1; j >= 0; j--)
        {npts = 1L;
         for (jn = 0; jn <= j; jn++)
             npts *= dmx[jn];

         dj   = dmx[j];
         step = lne;
         lne /= dj;
         for (i = 0; i < ned; i++)
             {id = i % step;
              ix = id / lne;
              if (ix == 0)
                 for (jc = 0; jc < ndr; jc++)
                     {rn = re[jc] + i;
                      for (is = lne; is < npts; is += lne)
                          rn[is] += rn[is-lne];};};};

    return;}
        
/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_PLANE - generate the hyper-plane mapping: sum(i = 0,n; ci*xi)
 *          - usage: (hyper-plane c0 (c1 x1min x1max np1) ...)
 */

object *SX_plane(argl)
   object *argl;
   {int i, j, nd, nde, plf;
    int *maxes, *pm;
    long ne;
    double xmn, xmx;
    REAL *extr, *px, *coeff, *pc, *r, *pr, v;
    double **delem;
    char name[MAXLINE];
    object *lst, *mo;
    PM_set *dom, *ran;
    PM_mapping *f;

    plf   = SX_have_display_list();
    nd    = _SS_length(argl);
    coeff = pc = FMAKE_N(REAL, nd--, "SX_PLANE:coeff");

    SS_args(argl,
	    SC_REAL_I, pc++,
            0);
    argl = SS_cdr(argl);

/* organize the args into input to make the domain */
    maxes = pm = FMAKE_N(int, nd, "SX_PLANE:maxes");
    extr  = px = FMAKE_N(REAL, 2*nd, "SX_PLANE:extr");
    for (i = 0, ne = 0L; i < nd; i++, argl = SS_cdr(argl))
        {lst = SS_car(argl);
	 SS_args(lst,
		 SC_REAL_I, pc++,
		 SC_DOUBLE_I, &xmn,
		 SC_DOUBLE_I, &xmx,
		 SC_INTEGER_I, pm++,
		 0);
	 *px++ = xmn;
	 *px++ = xmx;};

    nde = nd;

/* make the domain */
    sprintf(name, "D%d_%d", nd, nde);
    dom = PM_make_lr_domain(name, SC_DOUBLE_S, nd, nde, maxes, extr);

    SFREE(extr);

/* make the range */
    ne    = dom->n_elements;
    delem = (double **) dom->elements;

    r  = pr = FMAKE_N(REAL, ne, "SX_PLANE:r");
    for (i = 0; i < ne; i++)
        {for (j = 0, v = coeff[0]; j < nde; j++)
             {v += coeff[j+1]*delem[j][i];};
	 *pr++ = v;};

    ran = PM_make_set("Plane", SC_REAL_S, FALSE, 1, ne, 1, r);

/* make the mapping */
    sprintf(name, "%s->%s", dom->name, ran->name);
    f = PM_make_mapping(name, PM_LR_S, dom, ran, N_CENT, NULL);

    mo = SX_mk_mapping(f);
    if (plf)
       mo = SX_display_map(mo);

    return(mo);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_DERIVATIVE - take the derivative of a 1d mapping */

PM_mapping *SX_derivative(h)
   PM_mapping *h;
   {int n, m;
    REAL *xp, *yp;
    REAL *bx, *by;
    char labl[MAXLINE];
    PM_mapping *f;
    PM_set *d, *r;

    GET_DATA_1D(h, n, xp, yp);

    m = n + 5;

    bx = FMAKE_N(REAL, m, "SX_DERIVATIVE:bx");
    by = FMAKE_N(REAL, m, "SX_DERIVATIVE:by");

    PM_derivative(n, xp, yp, bx, by);

    sprintf(labl, "d/dx %s", h->name);

    if (n == 2)
       {n = 3;
	bx[0] = xp[0];
	bx[1] = xp[1];
	by[1] = by[0];}
    else if (n > 2)
       n--;

    d = PM_make_set("X", SC_REAL_S, FALSE, 1, n, 1, bx);
    r = PM_make_set("Y", SC_REAL_S, FALSE, 1, n, 1, by);
    f = PM_make_mapping(labl, PM_LR_S, d, r, N_CENT, NULL);

    return(f);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* _SX_FILTER_COEFF - worker for coefficient based filters */

void _SX_filter_coeff(yp, n, arr, ntimes)
   REAL *yp;
   int n;
   C_array *arr;
   int ntimes;
   {int i, ne, nc, nh, ne0;
    REAL *coeff;

    if (arr != NULL)
       {ne    = arr->length;
	coeff = NULL;
	CONVERT(SC_REAL_S, &coeff, arr->type, arr->data, ne, FALSE);
        ne--;

        nc  = coeff[0];
        nh  = nc >> 1;
        ne0 = nc + nh*(3*nh + 1);
        if (ne != ne0)
           SS_error("INCORRECT FILTER SIZE - _SX_FILTER_COEF", SS_null);}

    else
       SS_error("BAD COEFFICIENT ARRAY - _SX_FILTER_COEF", SS_null);

    for (i = 0; i < ntimes; i++)
        PM_filter_coeff(yp, n, coeff + 1, nc);

    SFREE(coeff);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_FILTER_COEF - the n point filter routine */

PM_mapping *SX_filter_coef(h, argl)
   PM_mapping *h;
   object *argl;
   {int n, ntimes;
    REAL *xp, *yp;
    C_array *arr;

    GET_DATA_1D(h, n, xp, yp);

    arr    = NULL;
    ntimes = 1;
    SS_args(argl,
            G_NUM_ARRAY, &arr,
	    SC_INTEGER_I, &ntimes,
	    0);

    _SX_filter_coeff(yp, n, arr, ntimes);

    PM_find_extrema(h->range);

    return(h);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_SMOOTH - the n point smoothing routine */

PM_mapping *SX_smooth(h, argl)
   PM_mapping *h;
   object *argl;
   {int i, n, pts, ntimes;
    REAL *xp, *yp;

    GET_DATA_1D(h, n, xp, yp);

    pts    = 3;
    ntimes = 1;
    SS_args(argl,
	    SC_INTEGER_I, &pts,
	    SC_INTEGER_I, &ntimes,
	    0);

    if (SC_str_icmp(SX_smooth_method, "fft") == 0)
       {for (i = 0; i < ntimes; i++)
	    PM_smooth_fft(xp, yp, n, pts, PM_smooth_filter);}

    else if (SC_str_icmp(SX_smooth_method, "averaging") == 0)
       {for (i = 0; i < ntimes; i++)
	    PM_smooth_int_ave(xp, yp, n, pts);}

    else
       {C_array *arr;
	char bf[MAXLINE];
	object *obj;

        obj = SS_INQUIRE_OBJECT(SX_smooth_method);
        if (obj == NULL)
           {sprintf(bf, "NO FILTER NAMED %s EXISTS - SX_SMOOTH",
		    SX_smooth_method);
	    SS_error(bf, SS_null);};

        SS_args(SS_lk_var_val(obj, SS_Env),
                G_NUM_ARRAY, &arr,
		0);

        if (arr == NULL)
           {sprintf(bf, "%s IS NOT A FILTER - SX_SMOOTH",
		    SX_smooth_method);
	    SS_error(bf, SS_null);};

	_SX_filter_coeff(yp, n, arr, ntimes);};

    PM_find_extrema(h->range);

    return(h);}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/

/* SX_INSTALL_GLOBAL_FUNCS - install some functions that everybody can use */

void SX_install_global_funcs()
   {
/*
    SS_install("change-value!",
               "Special Form: change the value of an object rather than changing its values via set!",
               SS_nargs,
               SX_change_value, SS_UE_MACRO);
*/
    SS_install("command-log",
               "Macro: Control command logging\n     Usage: command-log [on | off | <filename>]",
               SS_znargs,
               SX_toggle_log, SS_UR_MACRO);

    SS_install("end",
               "Macro: End the session\n     Usage: end",
               SS_zargs,
               SX_quit, SS_UR_MACRO);

    SS_install("ld",
               "Macro: Read SCHEME forms from file\n     Usage: ld <file-name>",
               SS_nargs,
               SS_load, SS_UR_MACRO);

    SS_install("pp",
               "Procedure: pretty print a list",
               SS_sargs,
               SX_pp_names, SS_PR_PROC);

    SS_install("synonym",
               "Macro: Define synonyms for the given function\n     Usage: synonym <func> <syn-list>",
               SS_nargs,
               SX_synonym, SS_UR_MACRO);

    SS_install("curve?",
               "Prodedure: Return #t iff the argument is an Ultra curve object\n     Usage: curve? <object>",
               SS_sargs,
               SX_curveobjp, SS_PR_PROC);

    SS_install("ultra-file?",
               "Procedure: Return #t iff the file is a valid ULTRA II file\n     Usage: ultra-file? <file-name>",
               SS_sargs,
               SX_valid_ultra_filep, SS_PR_PROC);

    return;}

/*--------------------------------------------------------------------------*/
/*--------------------------------------------------------------------------*/
