(define-class <message-table> (<table>)
  (name type: <symbol>)
  (id type: <fixnum>)
  (message-index type: <hash-integer-table>))

(define-class <message> (<object>)
  (owner type: <message-table>)
  (id type: <fixnum>)
  (type type: <symbol>)  ;; from (fatal error warning debug notice)
  default-text)

(define-method table-insert! ((self <message-table>) 
			      (key <fixnum>) 
			      (value <message>))
  (table-insert! (message-index self) key value))

(define-method name ((self <message>))
  (format #f "~d-~03d~c"
	  (id (owner self)) 
	  (id self)
	  (case (type self)
	    ((debug) #\D)
	    ((error) #\E)
	    ((fatal) #\F)
	    ((warning) #\W)
	    ((notice) #\N)
	    (else #\?))))

(define (alloc-new-id (tbl <message-table>))
  (table-size (message-index tbl)))

(define (add-message! tbl id text type)
  (let ((m (make <message>
		 owner: tbl
		 id: id
		 type: type
		 default-text: text)))
    (table-insert! tbl id m)
    m))

(define-class <message-out-port> (<output-port>)
  (message-prefix type: <string> init-value: "")
  (underlying-output-port type: <output-port>))

(define-method output-port-write-char ((self <message-out-port>) ch)
  (output-port-write-char (underlying-output-port self) ch)
  (if (eq? ch #\newline)
      (write-string (underlying-output-port self) (message-prefix self))))

(define (message->string (self <message>) (argv <vector>) plc)
  (let ((p (open-output-string)))
    (display-message self p argv plc)
    (close-output-port p)))

(define (display-message (self <message>) (port <output-port>) (argv <vector>)
			 plc)
  (let* ((tmp (open-output-string))
	 (mp (if plc
	          (make <message-out-port>
	         	message-prefix: (format #f "~a " plc)
		        underlying-output-port: tmp)
		  tmp)))
  (if plc (write-string port (message-prefix mp)))
  (write-string mp (name self))
  (write-char #\space mp)
  ;
  (apply format 
	 mp
	 (default-text self)
	 (vector->list argv))
  (newline tmp)
  (write-string port (close-output-port tmp))
  (values)))

;; (alloc-message TYPE ID TEXT)
;; (alloc-message TYPE TEXT)

(with-module compiler
(define-macro (alloc-message . stuff)
  (let ((var (lookup-aliased '*messages* $envt $envt))
	(type (car stuff))
	(text #f)
	(id #f)
	(msgcat #f))
  (if (not var)
      (error "message table not defined"))
  (set! msgcat (value var))
  (if (not (memq type '(fatal error warning debug notice)))
      (error "invalid message type `~s'" type))
  (cond
   ((and (= (length stuff) 2)
	 (string? (cadr stuff)))
    (set! id (alloc-new-id msgcat))
    (set! text (cadr stuff)))
   ((and (= (length stuff) 3)
	 (fixnum? (cadr stuff))
	 (string? (caddr stuff)))
    (set! id (cadr stuff))
    (set! text (caddr stuff)))
   (else
    (error "alloc-message: expected `type id text' or just `type text'\n> ~s" stuff)))
    (list 'quote (add-message! msgcat id text type))))
)
		  
(define (make-message-table (name <symbol>) (id <fixnum>))
  (make <message-table>
	name: name
	id: id
	message-index: (make-table eq? integer->hash)))

(define *message-tables* (make-symbol-table))

(define (get-message-table name)
  (or (table-lookup *message-tables* name)
      (error "message table `~s' not already defined" name)))

(define (setup-message-table name id)
  (let ((t (make-message-table name id)))
    (table-insert! *message-tables* name t)
    t))

;; provide standard RScheme tables
;; (for modules that don't want to use their own)

(setup-message-table 'rs 110)
(setup-message-table 'sys 120)

(define-macro (define-message-table name . r)
  (let ((m (if (pair? r)
	       (setup-message-table name (car r))
	       (get-message-table name))))
    `(define *messages* ',m)))

(define (fmt-msg msg argv . more)
  (message->string msg
		   argv 
		   (if (null? more)
		       *message-prefix*
		       (car more))))

(define-macro (fm . args)
  (bind ((msg args xtra (foo 'notice args))
	 (mn (gensym)))
    `(let ((,mn (alloc-message ,@msg)))
       (',fmt-msg ,mn (vector ,@args) ,@xtra))))

(&module (export fm))
