/* Implementatoin of TLVector class.
   This file is part of TL, Tiggr's Library.
   Written by Tiggr <tiggr@es.ele.tue.nl>
   Copyright (C) 1995, 1996 Pieter J. Schoenmakers
   TL is distributed WITHOUT ANY WARRANTY.
   See the file LICENSE in the TL distribution for details.

   $Id: TLVector.m,v 1.1 1998/01/08 16:12:09 tiggr Exp $  */

#define TLVECTOR_DECLARE_PRIVATE_METHODS
#import "tl/support.h"
#import "tl/TLVector.h"
#import "tl/TLStream.h"
#import "tl/TLSymbol.h"
#import "tl/TLCons.h"
#import "tl/TLGC.h"
#import "tl/subr.h"
#import <string.h>
#if HAVE_STRINGS_H
#import <strings.h>
#endif

#define DEFAULT_CAPACITY  8

@interface TLVectorEnumerator: TLObject <TLEnumerator>
{
  id <TLVector> vector;
  int next;
}

+(TLVectorEnumerator *) enumeratorWithVector: (id <TLVector>) vector;

-initWithVector: (id <TLVector>) vector;

@end

@interface TLVectorReverseEnumerator: TLVectorEnumerator
{
}

@end

@implementation TLVector

/******************** private methods ********************/

-(void) resize
{
  int new = cap ? 2 * cap : DEFAULT_CAPACITY;

  contents = xrealloc (contents, new * sizeof (*contents));
  bzero ((void *) (contents + cap), (new - cap) * sizeof (*contents));
  cap = new;
} /* -resize */

/******************** public methods ********************/

+(TLVector *) vector
{
  return ([(TLVector *) [self gcAlloc] initWithCapacity: 0]);
} /* +vector */

+(TLVector *) vectorByPerforming: (SEL) sel
	    onElementsOfSequence: seq
{
  id <TLEnumerator> e = [seq enumerator];
  TLVector *vector = [self vector];
  id o;

  while ((o = [e nextObject]) || [e notEndP])
    [vector addElement: [o perform: sel]];
  
  return vector;
} /* vectorByPerforming:onElementsOfSequence: */

+(TLVector *) vectorWith: (int) n copies: (id) o
{
  return [(TLVector *) [self gcAlloc] initWith: n copies: o];
} /* +vectorWith:copies: */

+(TLVector *) vectorWithCapacity: (int) c
{
  return ([(TLVector *) [self gcAlloc] initWithCapacity: c]);
} /* +vectorWithCapacity: */

+(TLVector *) vectorWithElements: (int) n, ...;
{
  va_list ap;
  id r;

  va_start (ap, n);
  r = [self vectorWithElements: n : ap];
  va_end (ap);
  return (r);
} /* +vectorWithElements: */

+(TLVector *) vectorWithElements: (int) n : (va_list) ap
{
  return ([[self gcAlloc] vinitWith: n elements: ap]);
} /* +vectorWithElements: */

+(TLVector *) vectorWithEnumerator: (id <TLEnumerator>) e
{
  TLVector *vector = [self vector];
  id o;

  while ((o = [e nextObject]) || [e notEndP])
    [vector addElement: o];
  
  return (vector);
} /* +vectorWithEnumerator: */

+(TLVector *) vectorWithSequence: seq
{
  return ([self vectorWithEnumerator: [seq enumerator]]);
} /* +vectorWithSequence: */

-addElement: e
{
  if (num == cap)
    [self resize];
  contents[num++] = e;
  return (e);
} /* -addElement: */

-(int) addElementsFromEnumerator: (id <TLEnumerator>) e
{
  int i = 0;
  id o;

  while ((o = [e nextObject]) || [e notEndP])
    {
      [self addElement: o];
      i++;
    }

  return i;
} /* -addElementsFromEnumerator: */

-(int) addElementsFromSequence: (id) seq
{
  return [self addElementsFromEnumerator: [seq enumerator]];
} /* -addElementsFromSequence: */

-(int) compare: (id <TLVector>) v
{
  int i, j, n;

  n = [v length];
  if (n != num)
    return n > num ? 1 : -1;

  for (i = 0; i < num; i++)
    {
      id o = [v _elementAtIndex: i];

      if (contents[i])
	{
	  j = [contents[i] compare: o];
	  if (j)
	    return j;
	}
      else if (o)
	return -1;
    }
  return 0;
}

-_elementAtIndex: (int) i
{
  return (i < 0 || i >= num ? nil : contents[i]);
} /* _elementAtIndex: */

-elementAtIndex: i
{
  return ([self _elementAtIndex: [i intValue]]);
} /* -elementAtIndex: */

-(id <TLEnumerator>) enumerator
{
  static id tlvector_enumerator_class;
  if (!tlvector_enumerator_class)
    tlvector_enumerator_class = [TLVectorEnumerator self];

  return ([tlvector_enumerator_class enumeratorWithVector: self]);
}

-(unsigned int) hash
{
  int hash, i;

  /* Hash at most the first 16 objects.  */
#define HASH_NUM_OBJECTS  16

  for (i = hash = 0; i < HASH_NUM_OBJECTS && i < num; i++)
    hash ^= [contents[i] hash];

  return hash;
} /* -hash */

-(int) _indexOfElement: (id) o
{
  return ([self _indexOfElement: o range: tll_full_range]);
} /* -_indexOfElement: */

-(int) _indexOfElement: (id) o range: (id <TLRange>) range
{
  int start = [range _start], length = [range length], end;

  range_intersect (0, num, &start, &length);

  for (end = start + length; start < end; start++)
    if ((!o && !contents[start])
	|| ![o compare: contents[start]])
      break;
  return (start < end ? start : -1);
} /* -_indexOfElement:range: */

-(TLNumber *) indexOfElement: (id) o
{
  int i = [self _indexOfElement: o range: tll_full_range];
  return (i == -1 ? nil : [CO_TLNumber numberWithInt: i]);
} /* -indexOfElement: */

-(TLNumber *) indexOfElement: (id) o range: (id <TLRange>) range
{
  int i = [self _indexOfElement: o range: range];
  return (i == -1 ? nil : [CO_TLNumber numberWithInt: i]);
} /* -indexOfElement:range: */

-(int) _indexOfElementIdenticalTo: (id) o
{
  return ([self _indexOfElementIdenticalTo: o range: tll_full_range]);
} /* -_indexOfElementIdenticalTo: */

-(int) _indexOfElementIdenticalTo: (id) o range: (id <TLRange>) range
{
  int start = [range _start], length = [range length], end;

  range_intersect (0, num, &start, &length);

  for (end = start + length; start < end; start++)
    if (o == contents[start])
      break;
  return (start < end ? start : -1);
} /* -_indexOfElementIdenticalTo:range: */

-(TLNumber *) indexOfElementIdenticalTo: (id) o
{
  int i = [self _indexOfElementIdenticalTo: o range: tll_full_range];
  return (i == -1 ? nil : [CO_TLNumber numberWithInt: i]);
} /* -indexOfElementIdenticalTo: */

-(TLNumber *) indexOfElementIdenticalTo: (id) o
 range: (id <TLRange>) range
{
  int i = [self _indexOfElementIdenticalTo: o range: range];
  return (i == -1 ? nil : [CO_TLNumber numberWithInt: i]);
} /* -indexOfElementIdenticalTo:range: */

-(id) initWith: (int) n copies: (id) o
{
  int i;

  [self initWithCapacity: n];

  for (i = 0, num = n; i < num; i++)
    contents[i] = o;

  return self;
} /* -initWith:copies: */

-initWithCapacity: (int) c
{
  cap = c;
  contents = xcalloc (cap, sizeof (*contents));
  return (self);
} /* -initWithCapacity: */

-(void) _insertElement: o atIndex: (int) i
{
  int j;

  if (i < 0 || i > num)
    [self error: "bad index %d for vector with %d elements", i, num];
  if (num == cap)
    [self resize];
  for (j = num; j > i; j--)
    contents[j] = contents[j - 1];
  num++;
  contents[i] = o;
} /* _insertElement:atIndex: */

-(int) length
{
  return (num);
} /* -length */

-mapcar: (TLSymbol *) sym
{
  /* XXX The class should be overridable.  */
  id retval = nil, last = nil;
  int i;
  GCDECL2;

  GCPRO2 (sym, retval);
  for (i = 0; i < num; i++)
    {
      id e = CONS (EVAL_WITH_ARGS (sym,
				   CONS (CONS (Qquote, CONS (contents[i], nil)),
					 nil)), nil);
      if (last)
	{
	  [last setCdr: e];
	  last = e;
	}
      else
	retval = last = e;
    }
  GCUNPRO;
  return (retval);
} /* -mapcar: */

-memq: (id) o
{
  int i = [self _indexOfElementIdenticalTo: o range: tll_full_range];

  return i >= 0 ? contents[i] : nil;
} /* -memq: */

-objectLength
{
  return ([CO_TLNumber numberWithInt: num]);
} /* -objectLength */

-(void) print: (id <TLMutableStream>) stream quoted: (BOOL) qp
{
  const char *s = class_get_class_name (isa);
  int i;

  [stream writeByte: '('];
  [stream writeBytes: strlen (s) fromBuffer: s];
  for (i = 0; i < num; i++)
    {
      [stream writeByte: ' '];
      print (contents[i], stream, qp);
    }
  [stream writeByte: ')'];
} /* -print:quoted: */

-(void) _removeElementAtIndex: (int) i
{
  [self removeElementsFromIndex: i range: 1];
} /* -_removeElementAtIndex: */

-(id) removeElementAtIndex: (id <TLNumber>) i
{
  int idx = [i integerPIntValue];
  id r = idx >= 0 && idx < num ? contents[idx] : nil;
  [self _removeElementAtIndex: idx];
  return (r);
} /* -removeElementAtIndex: */

-(void) removeElementsFromIndex: (int) i range: (int) n
{
  int j;

  if (i < 0 || i > num || i + n > num)
    [self error: "bad index %d (+ %d) for vector with %d elements", i, n, num];

  for (j = i + n; j < num; j++)
    contents[j - n] = contents[j];
  num -= n;
} /* -removeElementsFromIndex:range: */

-(void) removeElementIdenticalTo: e
{
  int i, j;

  for (i = 0; i < num; i++)
    if (contents[i] == e)
      {
	for (j = i + 1; j < num; j++)
	  contents[j - 1] = contents[j];
	num--;
	break;
      }
} /* -removeElementIdenticalTo: */

-(id) _replaceElementAtIndex: (int) i by: (id) o
{
  id previous;

  if (i >= num)
    [self error: "index %d beyond end %d", i, num];

  previous = contents[i];
  contents[i] = o;
  return (previous);
} /* -_replaceElementAtIndex:by: */

-(id) replaceElementAtIndex: (id <TLNumber>) i by: (id) o
{
  return ([self _replaceElementAtIndex: [i integerPIntValue] by: o]);
} /* -replaceElementAtIndex:by: */

-(id <TLEnumerator>) reverseEnumerator
{
  return ([TLVectorReverseEnumerator enumeratorWithVector: self]);
} /* -reverseEnumerator */

-vinitWith: (int) n elements: (va_list) ap
{
  [self initWithCapacity: n];
  for (num = 0; num < n; num++)
    contents[num] = va_arg (ap, id);
  return (self);
} /* -vinitWith:elements: */

/******************** garbage collection ********************/

-(void) dealloc
{
  xfree (contents);
} /* -dealloc */

-(void) gcReference
{
  int i;

  for (i = 0; i < num; i++)
    if (contents[i])
      MARK (contents[i]);
} /* -gcReference */

@end

@implementation TLVectorEnumerator

+(TLVectorEnumerator *) enumeratorWithVector: (id <TLVector>) vec
{
  return ([[self gcAlloc] initWithVector: vec]);
} /* +enumeratorWithVector: */

-initWithVector: (id <TLVector>) vec
{
  vector = vec;
  return (self);
} /* -initWithVector: */

/******************** TLEnumerator ********************/

-notEndP
{
  return (next > [vector length] ? nil : Qt);
} /* -notEndP */

-nextObject
{
  return (++next <= [vector length] ? [vector _elementAtIndex: next - 1] : nil);
} /* -nextObject */

/******************** garbage collection ********************/

-(void) gcReference
{
  MARK (vector);
} /* -gcReference */

@end

@implementation TLVectorReverseEnumerator

/******************** TLEnumerator ********************/

-nextObject
{
  int l = [vector length];
  return (++next <= l ? [vector _elementAtIndex: l - next] : nil);
} /* -nextObject */

@end
