<copyright> File class.
    Written by <a href="mailto:tiggr@ics.ele.tue.nl">Pieter J. Schoenmakers</a>

    Copyright &copy; 1995-1997 Pieter J. Schoenmakers.

    This file is part of TOM.  TOM is distributed under the terms of the
    TOM License, a copy of which can be found in the TOM distribution; see
    the file LICENSE.

    <id>$Id: File.t,v 1.26 1998/07/06 15:37:44 tiggr Exp $</id>
    </copyright>

<c>
#include <stdio.h>
</c>

<doc> The File class offers an abstraction from files in the filesystem.
    </doc>
implementation class
File: ByteStream, SeekableStream,
	Constants, Conditions
{
  const OPEN_INPUT = 0x100;
  const OPEN_OUTPUT = 0x200;
}

<doc> Return the {filename} removing any directory component, and removing
    the extension {ext} if it matches and is not {nil}.  </doc>
String
	    basename String filename
  without-extension: String ext = nil
{
  int length = [filename length];
  int start, beyond = length;

  if (ext != nil)
    {
      int ext_len = [ext length];

      if (length > ext_len
	  && [ext equal [filename substring (length - ext_len, ext_len)]]
	  && filename[length - ext_len - 1] != '/')
	beyond -= ext_len;
    }

  for (start = beyond - 1; start >= 0; start--)
    if (filename[start] == '/')
      break;

  = [filename substring (start + 1, beyond - (start + 1))];
}

<doc> Return the current directory, as a directory name (tailing slash).
    </doc>
extern String
  current_directory;

<doc> Set the current {directory}.  This raises a {file-error} when
    problems arise.  </doc>
extern void
  set_current_directory String directory;

<doc> Return the directory-name of the directory containing the file named
    {filename}.  </doc>
String
  directory-of-file String name
{
  MutableString dir = [name mutableSubstring (0, -1)];
  int i, n = [dir length];

  for (i = n - 1; i >= 0; i--)
    if (dir[i] == '/')
      break;

  = (i < 0 ? "" : i == 0 ? "/" : ({[dir resize i + 1]; dir;}));
}

<doc> Expand the {filename} relative to the {directory}.  If {directory}
    is {nil}, expansion is relative to the current working directory.
    </doc>
String
  expand-filename String filename
     relative-to: String directory = nil
{
  // ~-expansion and `//' recognition are not yet performed.
  // Fri Aug 23 14:31:23 1996, tiggr@cobra.es.ele.tue.nl
  if (directory != nil && [directory length] != 0 && filename[0] != '/')
    {
      MutableByteString f = [[MutableByteString new] print directory];

      if (f[[f length] - 1] != '/')
	[f print '/'];
      filename = [f print filename];
    }
  = filename;
}

<doc> Express the {filename} in terms relative to the {directory}.  </doc>
String
  express-filename String filename
       relative-to String directory
{
  if (!directory || ![directory length])
    return filename;

  int fl = [filename length];
  if (fl > 0 && filename[0] == '/')
    return filename;

  int dl = [directory length];
  int min = fl > dl ? dl : fl;
  int i, j, last_slash = -1;
  char c, d;

  for (i = 0; i < min; i++)
    {
      c = filename[i];
      d = directory[i];

      if (c != d)
	break;
      if (c == '/')
	last_slash = i;
    }

  if (last_slash == -1)
    // Actually, this string should be of the same class as the filename.
    // Fri Sep 6 11:26:07 1996, tiggr@cobra.es.ele.tue.nl
    return [[MutableByteString new] print (directory, filename)];

  if (last_slash == dl - 1)
    return [filename substring (dl, -1)];

  MutableByteString r = [MutableByteString new];
  for (j = last_slash + 1; j < dl; j++)
    {
      c = directory[j];
      if (c == '/')
	[r print "../"];
    }

  [r print [filename substring (last_slash + 1, -1)]];

  = r;
}

<doc> Return the {filename} as the name of a directory.  </doc>
String
  filename-as-directory String filename
{
  int l = [filename length];

  if (l > 0 && filename[l - 1] != '/')
    {
      filename = [[MutableByteString withCapacity l + 1]
		   print (filename, '/')];
      [filename freeze];
    }

  = filename;
}

<doc> Return the filenames in the directory named {dir_name}.  </doc>
extern MutableArray
  filenames-in-directory String dir_name;

<doc> Return the filename of the {file} somewhere along the {path}.
    Return {nil} if it could not be found.  </doc>
String
  locate-file String file
   along-path Indexed path
{
  int i, n = [path length];

  for (i = 0; i < n; i++)
    {
      MutableByteString s = [MutableByteString new];

      [s print ([File filename-as-directory path[i]], file)];
      if ([self file-exists s])
	return s;
    }
}

<doc> Return {YES} iff the file with the {name} exists.  </doc>
extern boolean
  file-exists String name;

<doc> Return one of the {FILE_TYPE_*} constants.  </doc>
extern int
  type-of-file String name
  follow-link: boolean follow_link = YES;

<doc> Return a new {File}.  </doc>
instance (id)
          open String name
         input: boolean input_p = FALSE
        output: boolean output_p = FALSE
	 flags: int action = 0
{
  = [[self alloc] init name flags ((action & FILE_MASK)
				   | (input_p ? OPEN_INPUT : 0)
				   | (output_p ? OPEN_OUTPUT : 0))];
}

<doc> Search for the file along the {path}.

    For the subdirectories {subdir1} and {subdir2}, when not nil, the
    following attempts are made for a {dir} in the {path}: {dir},
    {dir/subdir2}, {dir/subdir1}, and {dir/subdir1/subdir2}.  </doc>
instance (id)
	open String name
   alongPath Indexed path
     subdir: String subdir1 = nil
  subsubdir: String subdir2 = nil
{
  int i, n = [path length];
  File f;

  for (i = 0; i < n; i++)
    {
      String alt, nm = [self expand-filename name relative-to: path[i]];

      if (![self file-exists nm])
	{
	  nm = nil;

	  if (subdir2 != nil)
	    {
	      alt = [self expand-filename [self expand-filename name
						relative-to: subdir2]
			  relative-to: path[i]];
	      if ([self file-exists alt])
		nm = alt;
	    }

	  if (!nm && subdir1 != nil)
	    {
	      alt = [self expand-filename [self expand-filename name
						relative-to: subdir1]
			  relative-to: path[i]];
	      if ([self file-exists alt])
		nm = alt;
	      else if (subdir2 != nil)
		{
		  alt = [self expand-filename
			      [self expand-filename
				    [self expand-filename name
					  relative-to: subdir2]
				    relative-to: subdir1]
			      relative-to: path[i]];
		  if ([self file-exists alt])
		    nm = alt;
		}
	    }

	  if (!nm)
	    continue;
	}

      f = [[self alloc] init nm flags OPEN_INPUT];
      if (f != nil)
	return f;
    }
}

<doc> Remove the file or directory with the {name}.  </doc>
void
  remove String name
{
  pointer p;
  int len;

  (p, len) = [ByteString (name) byteStringContents];
<c>
  {
    char *s = alloca (len + 1);
    int result;

    memcpy (s, p, len);
    s[len] = '\0';

    result = remove (s);
    if (!result)
      return;
  }
</c>
  [Runtime perror name for self class file-error raise NO];
}

end;

implementation instance
File
{
  <doc> The name of the file.  </doc>
  public String name;

  <doc> If it is to be reopenend, these are the flags indicating how to do
      so.  </doc>
  int flags;
}

<doc> Designated initializer.  </doc>
protected id
   init String n
  flags int f
{
  [super init];

  (name, flags) = (expand-filename (n), f);

  = [self reopen];
}

<doc> Return the name of the directory containing our file.  </doc>
String
  directoryName
{
  = directory-of-file (name);
}

<doc> Reopen the file according to the {flags}.  </doc>
extern id
  reopen;

<doc> Return one of the {FILE_TYPE_*} constants for the receiving File.
    If the file is not open, the file is tested as if the file were open,
    i.e. following links.  </doc>
int
  type-of-file
{
  = (descriptor >= 0 ? [super type-of-file]
     : [isa type-of-file name follow-link: YES]);
}

OutputStream
  writeFields OutputStream s
{
  = [[super writeFields s] print (" filename=", name, " flags=", flags)];
}

<doc> <h4>Positioning</h4> </doc>

<doc> Return the length of the file.  </doc>
long
  length
{
  long p = [self position];

  [self seek 0 relative: STREAM_SEEK_END];
  = [self position];

  [self seek p];
}

<doc> Return the current file position.  </doc>
extern long
  position;

<doc> Position the file `pointer'.  </doc>
extern void
       seek long offset
  relative: int whence = STREAM_SEEK_SET;

end;

// implementation class
// MappedFile: ByteString, Descriptor
// 
// end;
// 
// implementation instance
// MappedFile
// 
// end;
