(**
   A gadget, that displays text offered by a derived class from TextViewModel.
**)

MODULE VOTextView;

(*
    Implements a textviewer gadget based on the text model.

    Copyright (C) 1997  Tim Teulings (rael@edge.ping.de)

    This module is free software; you can redistribute it and/or
    modify it under the terms of the GNU Lesser General Public License
    as published by the Free Software Foundation; either version 2 of
    the License, or (at your option) any later version.

    This module is distributed in the hope that it will be useful, but
    WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public
    License along with VisualOberon. If not, write to the Free Software Foundation,
    59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*)

IMPORT D  := VODisplay,
       F  := VOFrame,
       G  := VOGUIObject,
       O  := VOObject,
       TM := VOTextViewModel,
       V  := VOValue;

TYPE
  TextView*     = POINTER TO TextViewDesc;
  TextViewDesc* = RECORD (G.ImageDesc)
                    frame   : F.Frame;

                    model   : TM.TextModel;
                    top     : TM.Reader;
                    vis-,
                    total-,
                    topPos- : V.ValueModel;
                  END;

  PROCEDURE (t : TextView) Init*;

  BEGIN
    t.Init^;

    t.model:=NIL;
    t.top:=NIL;

    NEW(t.topPos);
    t.topPos.Init;
    t.topPos.SetLongint(1);
    t.AttachModel(t.topPos);
    NEW(t.vis);
    t.vis.Init;
    t.vis.SetLongint(1);
    NEW(t.total);
    t.total.Init;
    t.total.SetLongint(1);

    NEW(t.frame);
    t.frame.Init;
    t.frame.SetFlags({G.horizontalFlex,G.verticalFlex});
    t.frame.SetInternalFrame(F.double3DIn);
  END Init;

  PROCEDURE (t : TextView) SetModel*(model : TM.TextModel);

  BEGIN
    IF t.model#NIL THEN
      t.UnattachModel(t.model);
    END;

    t.model:=model;

    IF t.model#NIL THEN
      t.top:=model.GetReader();
      ASSERT(t.top#NIL);

(*    t.top:=model.Get(1);
    t.topPos.Set(1);*)
      t.AttachModel(model);
    END;
  END SetModel;

  PROCEDURE (t : TextView) CalcSize*(display :D.Display);

  BEGIN
    t.frame.CalcSize(display);

    t.width:=t.frame.leftBorder+10*display.spaceWidth+t.frame.rightBorder;
    t.height:=t.frame.topBorder+10*display.spaceHeight+t.frame.bottomBorder;

    t.CalcSize^(display);
  END CalcSize;

  PROCEDURE (t : TextView) Draw*(x,y : LONGINT; draw : D.DrawInfo);

  VAR
    reader     : TM.Reader;
    font       : D.Font;
    line       : TM.LineDesc;
    lines,
    innerWidth,
    width,i,
    xPos,yPos  : LONGINT;

  BEGIN
    t.Draw^(x,y,draw);

    t.frame.Resize(t.width,t.height);
    t.frame.Draw(x,y,draw);

(*    draw.PushForeground(D.backgroundColor);
    draw.FillRectangle(x+t.frame.leftBorder,y+t.frame.topBorder,
                       t.width-t.frame.leftBorder-t.frame.rightBorder,
                       t.height-t.frame.topBorder-t.frame.bottomBorder);
    draw.PopForeground;*)

    font:=t.display.GetFont(D.fixedFont);

    lines:=(t.height-t.frame.topBorder-t.frame.bottomBorder) DIV font.height;
    xPos:=x+t.frame.leftBorder;
    yPos:=y+t.frame.topBorder;
    innerWidth:=t.width-t.frame.leftBorder-t.frame.rightBorder;

    IF t.model#NIL THEN
      reader:=t.model.GetReader();
      reader.Assign(t.top);

      IF reader.IsValid() THEN
        IF lines>t.model.lines-t.top.pos+1 THEN
          lines:=t.model.lines-t.top.pos+1;
        END;

        t.draw.InstallClip;

        t.draw.AddRegion(x+t.frame.leftBorder,y+t.frame.topBorder,
                         t.width-t.frame.leftBorder-t.frame.rightBorder,
                         t.height-t.frame.topBorder-t.frame.bottomBorder);

        draw.PushFont(D.fixedFont,{});
        draw.PushForeground(D.blackColor);
        draw.PushBackground(D.whiteColor);

        FOR i:=1 TO lines DO

          reader.GetLine(line);

          width:=font.TextWidth(line.chars^,LEN(line.chars^)-1,{});
          draw.DrawFillString(xPos,yPos+font.ascent,line.chars^,LEN(line.chars^)-1);
          IF width<innerWidth THEN
            draw.PushForeground(D.whiteColor);
            draw.FillRectangle(xPos+width,yPos,innerWidth-width,font.height);
            draw.PopForeground;
          END;

          reader.Next;

          INC(yPos,font.height);
        END;

        draw.PopBackground;
        draw.PopForeground;
        draw.PopFont;

        t.draw.FreeLastClip;
      END;

      IF yPos<y+t.frame.topBorder+t.height THEN
        draw.PushForeground(D.whiteColor);
        draw.FillRectangle(xPos,yPos,innerWidth,y+t.frame.topBorder+t.height-yPos);
        draw.PopForeground;
      END;

      t.topPos.SetLongint(t.top.pos);
      t.total.SetLongint(t.model.lines);
      t.vis.SetLongint(lines);

    ELSE
      t.topPos.SetLongint(0);
      t.total.SetLongint(0);
      t.vis.SetLongint(0);
    END;
  END Draw;

  PROCEDURE (t : TextView) Hide*;

  BEGIN
    IF t.visible THEN
      t.frame.Hide;
      t.DrawHide; (* suboptimal *)
      t.Hide^;
    END;
  END Hide;

  PROCEDURE (t : TextView) Resync*(model : O.Model; msg : O.ResyncMsg);

  BEGIN
    IF model=t.model THEN
      IF msg#NIL THEN
        WITH msg : O.Notify DO
          CASE msg.notify OF
            TM.resyncMsg:
              t.top:=t.model.GetReader();
              t.Redraw;
          | TM.resizeMsg:
              t.Redraw;
          ELSE
          END;
        ELSE
          t.Redraw;
        END;
      END;
    ELSIF model=t.topPos THEN
      IF t.model#NIL THEN
        t.top.Position(t.topPos.GetLongint());
        t.Redraw;
      END;
    END;
  END Resync;

  PROCEDURE (t : TextView) ShowBottom*;

  BEGIN
    IF t.model#NIL THEN
      IF t.topPos.GetLongint()+t.vis.GetLongint()<=t.model.lines THEN
        t.topPos.SetLongint(t.model.lines-t.vis.GetLongint()+1);
      END;
    END;
  END ShowBottom;

END VOTextView.