/*
 * Copyright 1998-1999, University of Notre Dame.
 * Authors: Brian W. Barrett, Arun F. Rodrigues, Jeffrey M. Squyres,
 * 	 and Andrew Lumsdaine
 *
 * This file is part of XMPI
 *
 * You should have received a copy of the License Agreement for XMPI 
 * along with the software; see the file LICENSE.  If not, contact 
 * Office of Research, University of Notre Dame, Notre Dame, IN 46556.
 *
 * Permission to modify the code and to distribute modified code is
 * granted, provided the text of this NOTICE is retained, a notice that
 * the code was modified is included with the above COPYRIGHT NOTICE and
 * with the COPYRIGHT NOTICE in the LICENSE file, and that the LICENSE
 * file is distributed with the modified code.
 *
 * LICENSOR MAKES NO REPRESENTATIONS OR WARRANTIES, EXPRESS OR IMPLIED.
 * By way of example, but not limitation, Licensor MAKES NO
 * REPRESENTATIONS OR WARRANTIES OF MERCHANTABILITY OR FITNESS FOR ANY
 * PARTICULAR PURPOSE OR THAT THE USE OF THE LICENSED SOFTWARE COMPONENTS
 * OR DOCUMENTATION WILL NOT INFRINGE ANY PATENTS, COPYRIGHTS, TRADEMARKS
 * OR OTHER RIGHTS.
 *
 * Additional copyrights may follow.

 *
 *	$Id: xmpi_trace.c,v 1.2 1999/11/08 06:20:26 bbarrett Exp $
 * 
 *	Function:	- tracing subwindow
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#define _NO_PROTO

#include <X11/Xlib.h>
#include <Xm/DialogS.h>
#include <Xm/DrawingA.h>
#include <Xm/Form.h>
#include <Xm/RowColumn.h>
#include <Xm/ScrollBar.h>
#include <Xm/ScrolledW.h>
#include <Xm/SeparatoG.h>

#include "lam.h"
#include "xmpi.h"

#include "Bitmaps/tack.xbm"
#include "Bitmaps/zoom.xbm"
#include "Bitmaps/unzoom.xbm"
#include "Bitmaps/vcrplay.xbm"
#include "Bitmaps/vcrrew.xbm"
#include "Bitmaps/vcrfwd.xbm"
#include "Bitmaps/vcrstop.xbm"

/*
 * global functions
 */
void			xmpi_tr_destroy();
void			xmpi_tr_create();
int			xmpi_tr_vcron();
void			xmpi_tr_busy();
void			xmpi_tr_unbusy();

/*
 * local functions
 */
static void		tr_fill();
static void		tr_draw();
static void		zoom_cb();
static void		unzoom_cb();
static void		tack_cb();
static void		destroy_cb();
static void		tr_clear();
static void		draw_cb();
static void		movedial();
static void		resize_cb();
static void		hscroll_cb();
static void		vscroll_cb();
static void		arm_cb();
static void		vcrrew_cb();
static void		vcrstop_cb();
static void		vcrplay_cb();
static void		vcrfwd_cb();
static void		vcradvance();
static void		redraw();
static void		gcinit();
static int		initparams();
static void		showtime();
static void		showscale();
static void		showlabel();
static void		setupdial();
static void		rubber();
static void		dragged_zoom();

/*
 * local macros and constants
 */
#define XMPI_DIALWIDTH	20
#define XMPI_EPSSCALE	0.001

/*
 * global variables
 */
char			tracefile[XMPI_PATHMAX]; /* trace filename */

/*
 * external variables
 */
extern struct _gps	*xmpi_app_procs;	/* appl. GPS array */
extern int		xmpi_do_rerun;		/* re-running application? */
extern int		xmpi_is_express;	/* current appl. trace view? */
extern int		xmpi_do_express;	/* creating current app view */

/*
 * local variables
 */
static Widget		trace_w = 0;		/* trace shell */
static Widget		banner_w;		/* trace label banner */
static Widget		scale_w;		/* scale label */
static Widget		tleft_w;		/* left time label */
static Widget		tright_w;		/* right time label */
static Widget		tdial_w;		/* dial time label */
static Widget		scroll_w;		/* scrolled window manager */
static Widget		draw_w;			/* trace drawing area */
static Widget		zoom_w;			/* zoom button */
static Widget		unzoom_w;		/* unzoom button */
static Widget		hsb_w;			/* horizontal scrollbar */
static Widget		vsb_w;			/* vertical scrollbar */
static Pixmap		pixmap;			/* trace pixmap */
static Pixmap		savedial;		/* saved dial pixmap */
static Pixmap		dial;			/* dial pixmap */
static Pixmap		mask;			/* dial mask */
static GC		dialpen = 0;		/* for drawing the dial */
static GC		fgpen = 0;		/* foreground pen */
static GC		bgpen = 0;		/* background pen */
static GC		misspen = 0;		/* missing pen */
static GC		blkpen = 0;		/* blocked pen */
static GC		syspen = 0;		/* system pen */
static GC		runpen = 0;		/* running pen */
static GC		arwpen = 0;		/* arrow pen */
static GC		rubpen = 0;		/* rubber band pen */
static Display		*disp;			/* drawing area display */
static Drawable		root;			/* root window */
static unsigned int	depth;			/* screen depth */
static XtIntervalId	vcrtimer;		/* VCR timeout timer */
static int		maxwidth;		/* max width */
static int		pixwidth;		/* pixmap width */
static int		pixheight;		/* pixmap height */
static int		drawwidth;		/* drawing width */
static int		drawheight;		/* drawing height */
static int		wextra;			/* extra width pixels */
static int		hextra;			/* extra height pixels */
static int		xdrawpos = 0;		/* horiz. drawing pos. */
static int		ydrawpos = 0;		/* vert. drawing pos. */
static int		xpix;			/* pixmap x position */
static int		xdial = -1;		/* dial x position */
static int		narrows;		/* # arrows */
static int		nprocs;			/* # processes */
static int		ntraces;		/* # traces */
static int		fl_band = 0;		/* rubber banding flag */
static int		fl_resize = 0;		/* resize flag */
static int		fl_power2;		/* scale is power of 2 flag */
static int		fl_vcron = 0;		/* VCR on flag */
static int		fl_zoom;		/* zoom is sensitive flag */
static int		vcrincr;		/* VCR pixel increment */
static unsigned long	vcrdelay;		/* VCR timer delay, msec */
static double		zoomhi;			/* high zoom factor */
static double		zoomlo;			/* low zoom factor */
static double		scale;			/* scale factor, pix/sec */
static double		onescale;		/* one to one scale */
static double		minscale;		/* min scale factor, pix/sec */
static double		maxscale;		/* max scale factor, pix/sec */
static double		tmin;			/* min traces time */
static double		tmax;			/* max traces time */
static struct xmarrow	*arrows = 0;		/* array of arrows */
static struct xmtrace	*traces = 0;		/* array of traces */
static char		fmtbuf[80];		/* formatting buffer */
static int		actionsinit = 0;	/* actions initialized? */

static XtActionsRec	actions[] = {
	{
		"rubber",
		(XtActionProc) rubber
	}
};

static char		*trans =		/* translation table */
	"<Btn1Down>:	DrawingAreaInput()\n\
	 <Btn1Up>:	DrawingAreaInput()\n\
	 <Btn1Motion>:	DrawingAreaInput()\n\
	 <Btn3Down>:	rubber(down)\n\
	 <Btn3Up>:	rubber(up)\n\
	 <Btn3Motion>:	rubber(motion)";

/*
 * button info messages
 */
static char		*quit_msg = "Quit trace window";
static char		*zoom_msg = "Zoom trace view";
static char		*unzoom_msg = "Un-zoom trace view";
static char		*vcrrew_msg = "Rewind VCR to beginning";
static char		*vcrstop_msg = "Stop VCR playback";
static char		*vcrplay_msg = "Play VCR";
static char		*vcrfwd_msg = "Play VCR at faster speed";

/*
 *	xmpi_tr_destroy
 *
 *	Function:	- destroys trace window
 */
void
xmpi_tr_destroy()

{
	Widget		w;

	if (trace_w) {
/*
 * Remove destroy callback since we may create a new trace widget
 * before the current event dispatch is complete.
 */
		XtRemoveCallback(trace_w, XmNdestroyCallback, destroy_cb, NULL);
		XtUnmapWidget(trace_w);
		w = trace_w;
		tr_clear();
		XtDestroyWidget(w);
	}
}

/*
 *	xmpi_tr_create
 *
 *	Function:	- creates widgets for trace window
 */
void
xmpi_tr_create()

{
	Widget		trace_mgr_w;		/* trace manager */
	Widget		mgr_w;			/* prog/appl panel manager */
	Widget		lbl_w;			/* label */
	Widget		rc_w;			/* row of buttons */
	Widget		w;
	Dimension	width, height;		/* window dimensions */
	Dimension	hsbsize;		/* horiz. scrollbar size */
	Dimension	vsbsize;		/* vert. scrollbar size */
/*
 * Initialize.
 */
	if (initparams()) return;

	if (!actionsinit) {
		XtAppAddActions(app, actions, 1);
		actionsinit = 1;
	}

	tr_fill();
/*
 * Create the trace window.
 */
	trace_w = XtVaCreatePopupShell("trace_pop",
			xmDialogShellWidgetClass, xmpi_shell,
			XmNdeleteResponse, XmDESTROY,
			XmNmappedWhenManaged, False,
			XmNtitle, "XMPI Trace",
			NULL);

        XtAddCallback(trace_w, XmNdestroyCallback, destroy_cb, NULL);

	trace_mgr_w = XtVaCreateWidget("trace_mgr",
			xmFormWidgetClass, trace_w,
			XmNmarginHeight, 2,
			XmNmarginWidth, 2,
			NULL);
/*
 * Build the control area.
 */
	mgr_w = XtVaCreateWidget("trace_ctl_mgr",
			xmFormWidgetClass, trace_mgr_w,
			XmNtopAttachment, XmATTACH_FORM,
			XmNleftAttachment, XmATTACH_FORM,
			XmNrightAttachment, XmATTACH_FORM,
			NULL);

	lbl_w = XtVaCreateWidget("trace_lbl",
			xmFormWidgetClass, mgr_w,
			XmNtopAttachment, XmATTACH_FORM,
			XmNrightAttachment, XmATTACH_FORM,
			XmNleftAttachment, XmATTACH_FORM,
			NULL);
	
	w = xmpi_mkpixbutton(lbl_w, tack_cb, NULL,
			arm_cb, quit_msg, showscale, NULL,
			tack_bits, tack_width, tack_height, False);

	XtVaSetValues(w,
			XmNtopAttachment, XmATTACH_FORM,
			XmNleftAttachment, XmATTACH_FORM,
			XmNbottomAttachment, XmATTACH_FORM,
			NULL);

	xmpi_add_pophelp(w, quit_msg);

	banner_w = xmpi_mklabel(lbl_w, "banner", XmALIGNMENT_CENTER, False);
	XtVaSetValues(banner_w,
			XmNtopAttachment, XmATTACH_FORM,
			XmNrightAttachment, XmATTACH_FORM,
			XmNbottomAttachment, XmATTACH_FORM,
			XmNleftAttachment, XmATTACH_WIDGET,
			XmNleftWidget, w,
			XmNleftOffset, 3,
			NULL);
	showlabel();
	XtManageChild(lbl_w);
/*
 * Create the row of control buttons.
 */
	rc_w = XtVaCreateWidget("trace_ctl_buttons",
			xmRowColumnWidgetClass, mgr_w,
			XmNorientation, XmHORIZONTAL,
			XmNnumColumns, 1,
			XmNleftAttachment, XmATTACH_FORM,
			XmNtopAttachment, XmATTACH_WIDGET,
			XmNtopWidget, lbl_w,
			NULL);

	zoom_w = xmpi_mkpixbutton(rc_w, zoom_cb, NULL,
			arm_cb, zoom_msg, showscale, NULL,
			zoom_bits, zoom_width, zoom_height, True);

	xmpi_add_pophelp(zoom_w, zoom_msg);

	unzoom_w = xmpi_mkpixbutton(rc_w, unzoom_cb, NULL,
			arm_cb, unzoom_msg, showscale, NULL,
			unzoom_bits, unzoom_width, unzoom_height, True);

	xmpi_add_pophelp(unzoom_w, unzoom_msg);

	w = xmpi_mkpixbutton(rc_w, vcrrew_cb, NULL,
			arm_cb, vcrrew_msg, showscale, NULL,
			vcrrew_bits, vcrrew_width, vcrrew_height, False);

	xmpi_add_pophelp(w, vcrrew_msg);

	w = xmpi_mkpixbutton(rc_w, vcrstop_cb, NULL,
			arm_cb, vcrstop_msg, showscale, NULL,
			vcrstop_bits, vcrstop_width, vcrstop_height, False);

	xmpi_add_pophelp(w, vcrstop_msg);

	w = xmpi_mkpixbutton(rc_w, vcrplay_cb, NULL,
			arm_cb, vcrplay_msg, showscale, NULL,
			vcrplay_bits, vcrplay_width, vcrplay_height, False);

	xmpi_add_pophelp(w, vcrplay_msg);

	w = xmpi_mkpixbutton(rc_w, vcrfwd_cb, NULL,
			arm_cb, vcrfwd_msg, showscale, NULL,
			vcrfwd_bits, vcrfwd_width, vcrfwd_height, False);

	xmpi_add_pophelp(w, vcrfwd_msg);

	XtManageChild(rc_w);
/*
 * Create scale label.
 */
	scale_w = xmpi_mklabel(mgr_w, " ", XmALIGNMENT_BEGINNING, False);
	XtVaSetValues(scale_w,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, lbl_w,
		XmNleftAttachment, XmATTACH_WIDGET,
		XmNleftWidget, rc_w,
		XmNleftOffset, 10,
		XmNbottomAttachment, XmATTACH_OPPOSITE_WIDGET,
		XmNbottomWidget, rc_w,
		NULL);
/*
 * Create the time labels.
 */
	w = XtVaCreateWidget("ts_labels",
		xmFormWidgetClass, mgr_w,
		XmNfractionBase, 3,
                XmNleftAttachment, XmATTACH_FORM,
		XmNtopAttachment, XmATTACH_WIDGET,
		XmNtopWidget, rc_w,
                XmNrightAttachment, XmATTACH_FORM,
		NULL);

	tleft_w = xmpi_mklabel(w, " ", XmALIGNMENT_BEGINNING, False);
	xmpi_formattach(tleft_w, 0, 3, 0, 1);

	tright_w = xmpi_mklabel(w, " ", XmALIGNMENT_END, False);
	xmpi_formattach(tright_w, 0, 3, 2, 3);

	tdial_w = xmpi_mklabel(w, " ", XmALIGNMENT_CENTER, False);
	xmpi_formattach(tdial_w, 0, 3, 1, 2);

	XtManageChild(w);
	XtManageChild(mgr_w);
/*
 * Add a separator.
 */
	mgr_w = XtVaCreateManagedWidget("focus_sep",
			xmSeparatorGadgetClass, trace_mgr_w,
			XmNtopAttachment, XmATTACH_WIDGET,
			XmNtopWidget, mgr_w,
			XmNleftAttachment, XmATTACH_FORM,
			XmNleftOffset, 1,
			XmNrightAttachment, XmATTACH_FORM,
			XmNrightOffset, 1,
			NULL);
/*
 * Build the trace viewing area.
 */
	mgr_w = XtVaCreateWidget("trace_view_mgr",
			xmFormWidgetClass, trace_mgr_w,
			XmNleftAttachment, XmATTACH_FORM,
			XmNleftOffset, 1,
			XmNrightAttachment, XmATTACH_FORM,
			XmNrightOffset, 1,
			XmNtopAttachment, XmATTACH_WIDGET,
			XmNtopWidget, mgr_w,
			XmNtopOffset, 1,
			XmNbottomAttachment, XmATTACH_FORM,
			NULL);
/*
 * Build the scrolled window.
 */
	scroll_w = XtVaCreateManagedWidget("trace_view",
			xmScrolledWindowWidgetClass, mgr_w,
			XmNscrollingPolicy, XmAPPLICATION_DEFINED,
			XmNvisualPolicy, XmVARIABLE,
			XmNscrolledWindowMarginHeight, 4,
			XmNscrolledWindowMarginWidth, 4,
			XmNtraversalOn, False,
			XmNtopAttachment, XmATTACH_FORM,
			XmNbottomAttachment, XmATTACH_FORM,
			XmNleftAttachment, XmATTACH_FORM,
			XmNrightAttachment, XmATTACH_FORM,
			NULL);
/*
 * Build the drawing area.
 */
	draw_w = XtVaCreateManagedWidget("trace_draw",
			xmDrawingAreaWidgetClass, scroll_w,
			XmNtranslations, XtParseTranslationTable(trans),
			XmNwidth, drawwidth,
			XmNheight, drawheight,
			XmNshadowThickness, 2,
			XmNtopShadowPixmap, NULL,
			XmNbottomShadowPixmap, NULL,
			NULL);

	XtAddCallback(draw_w, XmNinputCallback, draw_cb, NULL);
	XtAddCallback(draw_w, XmNexposeCallback, draw_cb, NULL);
	XtAddCallback(draw_w, XmNresizeCallback, resize_cb, NULL);

	XtManageChild(draw_w);
/*
 * Build the scrollbars.
 */
	vsb_w = XtVaCreateManagedWidget("vertscrollbar",
			xmScrollBarWidgetClass, scroll_w,
			XmNorientation, XmVERTICAL,
			XmNminimum, 0,
			XmNmaximum, pixheight,
			XmNsliderSize, drawheight,
			XmNpageIncrement, drawheight - 1,
			XmNincrement, 1,
			NULL);

	hsb_w = XtVaCreateManagedWidget("horizscrollbar",
			xmScrollBarWidgetClass, scroll_w,
			XmNorientation, XmHORIZONTAL,
			XmNminimum, 0,
			XmNmaximum, maxwidth,
			XmNsliderSize, drawwidth,
			XmNpageIncrement, drawwidth - 1,
			XmNincrement, 1,
			NULL);

	XmScrolledWindowSetAreas(scroll_w, hsb_w, vsb_w, draw_w);

	XtAddCallback(vsb_w, XmNvalueChangedCallback, vscroll_cb, NULL);
	XtAddCallback(vsb_w, XmNdragCallback, vscroll_cb, NULL);

	XtAddCallback(hsb_w, XmNvalueChangedCallback, hscroll_cb, NULL);
	XtAddCallback(hsb_w, XmNdragCallback, hscroll_cb, NULL);

	XtManageChild(scroll_w);

	XtManageChild(mgr_w);

	XtManageChild(trace_mgr_w);
/*
 * Initialize graphic contexts.
 */
	gcinit();
/*
 * Create the trace drawing area and the dial pixmaps.
 */
	pixmap = XCreatePixmap(disp, root, pixwidth, pixheight, depth);
	setupdial();

	tr_draw();
/*
 * Pop up the window.
 */
	XtPopup(trace_w, XtGrabNone);
/*
 * Set the scrolled window dimensions.
 */
	XtVaGetValues(hsb_w, XmNheight, &hsbsize, NULL);
	XtVaGetValues(vsb_w, XmNwidth, &vsbsize, NULL);

	XtVaSetValues(scroll_w, XmNwidth, drawwidth + vsbsize,
			XmNheight, drawheight + hsbsize, NULL);
/*
 * Compute the extra pixels between drawing area and popup.
 * Set the max/min window sizes.
 */
	XtVaGetValues(trace_w, XmNheight, &height, XmNwidth, &width, NULL);

	hextra = height - drawheight;
	wextra = width - drawwidth;

	XtVaSetValues(trace_w,
			XmNminHeight, XMPI_TRDISP + hextra,
			XmNmaxHeight, pixheight + hextra,
			XmNminWidth, XMPI_TRMINWIDTH + wextra,
			XmNmaxWidth, maxwidth + wextra,
			NULL);

	fl_resize = 1;
}

/*
 *	xmpi_tr_vcron
 *
 *	Function:	- vcr forward motion state
 *	Returns:	- 1 if vcr is in play or ffwd mode, else 0
 */
int
xmpi_tr_vcron()

{
	return(fl_vcron);
}

/*
 *	tr_fill
 *
 *	Function:	- fill the tracing information from the database
 */
static void
tr_fill()

{
	double		t1, t2;
/*
 * Get the trace information.
 */
	if (traces) free((char *) traces);
	if (arrows) free((char *) arrows);

	t1 = tmin + (xpix / scale);
	t2 = tmin + ((xpix + pixwidth) / scale);

	ntraces = xmpi_db_gettraces(t1, t2, &traces);
	narrows = xmpi_db_getarrows(t1, t2, &arrows);
}

/*
 *	tr_draw
 *
 *	Function:	- draw the tracing information
 */
static void
tr_draw()

{
	int		maxrank;		/* max. rank */
	int		x, y;			/* coordinates */
	int		x2, y2;			/* other coordinates */
	int		width;			/* rectangle width */
	int		i;			/* favourite index */
	GC		pen;			/* colouring pen */
	struct xmtrace	*ptrace;		/* ptr trace */
	struct xmarrow	*parrow;		/* ptr arrow */
/*
 * Set the zoom/unzoom button sensitivity.
 */
	fl_zoom = scale < maxscale;
	XtVaSetValues(zoom_w, XmNsensitive, (fl_zoom) ? True : False, NULL);

	XtVaSetValues(unzoom_w, XmNsensitive,
		((drawwidth == maxwidth) || (scale < minscale))
		? False : True, NULL);

	maxrank = xmpi_db_getnprocs() - 1;
	XFillRectangle(disp, pixmap, bgpen, 0, 0, pixwidth, pixheight);
/*
 * Loop drawing the traces.
 */
	for (i = 0, ptrace = traces; i < ntraces; ++i, ++ptrace) {

		if (ptrace->xmt_state == XMPI_SUNDEF) continue;
			
		y = ((ptrace->xmt_rank + 1) * XMPI_TRDISP) - XMPI_TRHEIGHT;

		x = ((ptrace->xmt_time - tmin) * scale) - xpix;
		width = (ptrace->xmt_lapse * scale) + 1;

		if (x < 0) {
			width += x;
			if (width <= 0) continue;
			x = 0;
		}
		width = min(width, pixwidth);
		
		switch(ptrace->xmt_state) {
			case XMPI_SMISSING:	pen = misspen; break;
			case XMPI_SRUN:		pen = runpen; break;
			case XMPI_SBLOCK:	pen = blkpen; break;
			case XMPI_SSYSTEM:	pen = syspen; break;
		}

		XFillRectangle(disp, pixmap, pen, x, y, width, XMPI_TRHEIGHT);
	}
/*
 * Loop drawing the arrows.
 */
	for (i = 0, parrow = arrows; i < narrows; ++i, ++parrow) {

		if (parrow->xma_srank == -1) {
/*
 * Unmatched receive, draw a stub.
 */
			x = ((parrow->xma_rtime - tmin) * scale) - xpix;
			y = (parrow->xma_rrank + 1) * XMPI_TRDISP;

			if (parrow->xma_rrank == maxrank) {
				y -= XMPI_TRHEIGHT;
				y2 = y - XMPI_TRHEIGHT;
			} else {
				--y;
				y2 = y + XMPI_TRHEIGHT;
			}
			XDrawLine(disp, pixmap, arwpen, x, y,
					x - XMPI_TRHEIGHT, y2);
		}
		else if (parrow->xma_rrank == -1) {
/*
 * Unmatched send, draw a stub.
 */
			x = ((parrow->xma_stime - tmin) * scale) - xpix;
			y = (parrow->xma_srank + 1) * XMPI_TRDISP + 1;
			
			if (parrow->xma_srank == maxrank) {
				y -= XMPI_TRHEIGHT;
				y2 = y - XMPI_TRHEIGHT;
			} else {
				--y;
				y2 = y + XMPI_TRHEIGHT;
			}
			XDrawLine(disp, pixmap, arwpen, x, y,
					x + XMPI_TRHEIGHT, y2);
		}
		else {
/*
 * Matched send and receive.
 */
			x = ((parrow->xma_stime - tmin) * scale) - xpix;
			x2 = ((parrow->xma_rtime - tmin) * scale) - xpix;

			y = (parrow->xma_srank + 1) * XMPI_TRDISP;
			y2 = (parrow->xma_rrank + 1) * XMPI_TRDISP;
/*
 * If same sender/receiver, draw a half-ellipse
 * (Angle is in 1/64th of a degree, i.e. 11520 = 180 * 64).
 */
			if (parrow->xma_srank == parrow->xma_rrank) {
				XDrawArc(disp, pixmap, arwpen,
					x, y - 2 * XMPI_TRHEIGHT - 1,
					x2 - x, 2 * XMPI_TRHEIGHT, 0, 11520);
			}
/*
 * Different processes, draw a straight line.
 */
			else {
				if (parrow->xma_rrank > parrow->xma_srank) {
					++y;
					y2 -= XMPI_TRHEIGHT + 1;
				} else {
					y -= XMPI_TRHEIGHT + 1;
					++y2;
				}

				XDrawLine(disp, pixmap, arwpen, x, y, x2, y2);
			}
		}
	}
/*
 * Draw the dial, first saving what is under it.
 */
	if ( (xdial >= xpix) && (xdial < (xpix + pixwidth)) ) {

		XCopyArea(disp, pixmap, savedial, fgpen, xdial - xpix, 0,
			XMPI_DIALWIDTH, pixheight, 0, 0);
		XSetClipOrigin(disp, dialpen, xdial - xpix, 0);
		XCopyArea(disp, dial, pixmap, dialpen, 0, 0,
			XMPI_DIALWIDTH, pixheight, xdial - xpix, 0);
	}

	showtime();
	showscale();
	redraw();
}

/*
 *	zoom_cb
 *
 *	Function:	- trace zooming callback
 */
static void
zoom_cb()

{
	int		xdraw;			/* draw position */
	double		newscale;		/* new scale */
	double		tdial;			/* dial time */
/*
 * If dial is outside view, center it.
 */
	xdraw = xpix + xdrawpos;

	if ( (xdial < xdraw) || (xdial >= (xdraw + drawwidth)) ) {
		xdial = xdraw + (drawwidth / 2);
		showtime();
		xmpi_vw_update(!fl_vcron);
		xmpi_kv_update();
	}

	tdial = tmin + xdial / scale;
/*
 * Zoom around dial location.
 */
	if (fl_power2) {
		if (zoomhi > 1.0 || zoomlo == 1.0) {
			zoomhi *= 2.0;
			newscale = onescale * zoomhi;
		} else {
			zoomlo /= 2.0;
			newscale = onescale / zoomlo;
		}
	} else {
		if (scale > onescale) {
			zoomhi = pow(2.0, ceil(log(scale/onescale)/log(2.0)));
			zoomlo = 1.0;
			newscale = onescale * zoomhi;
		} else {
			zoomlo = pow(2.0, floor(log(onescale/scale)/log(2.0)));
			zoomhi = 1.0;
			newscale = onescale / zoomlo;
		}

		fl_power2 = 1;
	}

	if (newscale > maxscale + XMPI_EPSSCALE) {
		scale = maxscale;
		fl_power2 = 0;
		zoomhi = maxscale / onescale;
		zoomlo = 1.0;
	} else {
		scale = newscale;
	}

	xdial = (tdial - tmin) * scale;
	maxwidth = (scale * (tmax - tmin)) + 0.5;

	xdrawpos = (pixwidth - drawwidth) / 2;
	xpix = xdial - pixwidth / 2;

	if (xpix < 0) {
		xdrawpos = max(xdrawpos + xpix, 0);
		xpix = 0;
	}

	if ((xpix + xdrawpos + drawwidth) > maxwidth) {
		xdrawpos = 0;
		xpix = maxwidth - drawwidth;
	}

	if ((xpix + xdrawpos) < 0) {
		xpix = xdrawpos = 0;
	}

	XtVaSetValues(hsb_w,
		XmNvalue, xpix + xdrawpos,
		XmNmaximum, max(maxwidth, drawwidth), NULL);
/*
 * Set the new max width.
 */
	XtVaSetValues(trace_w,
		XmNmaxWidth, max(maxwidth, drawwidth) + wextra, NULL);
/*
 * Refresh screen.
 */
	tr_fill();
	tr_draw();
	movedial(xdial, 1);
}

/*
 *	unzoom_cb
 *
 *	Function:	- trace unzooming callback
 */
static void
unzoom_cb()

{
	int		xdraw;			/* draw position */
	double		newscale;		/* new scale */
	double		tdial;			/* dial time */
/*
 * If dial is outside view, center it.
 */
	xdraw = xpix + xdrawpos;

	if ( (xdial < xdraw) || (xdial >= (xdraw + drawwidth)) ) {
		xdial = xdraw + (drawwidth / 2);
		showtime();
		xmpi_vw_update(!fl_vcron);
		xmpi_kv_update();
	}

	tdial = tmin + xdial / scale;
/*
 * Unzoom around dial location.
 */
	if (fl_power2) {
		if (zoomlo > 1.0 || zoomhi == 1.0) {
			zoomlo *= 2.0;
			newscale = onescale / zoomlo;
		} else {
			zoomhi /= 2.0;
			newscale = onescale * zoomhi;
		}
	} else {
		if (scale > onescale) {
			zoomhi = pow(2.0, floor(log(scale/onescale)/log(2.0)));
			zoomlo = 1.0;
			newscale = onescale * zoomhi;
		} else {
			zoomlo = pow(2.0, ceil(log(onescale/scale)/log(2.0)));
			zoomhi = 1.0;
			newscale = onescale / zoomlo;
		}

		fl_power2 = 1;
	}

	if (newscale < minscale - XMPI_EPSSCALE) {
		scale = minscale;
		fl_power2 = 0;
		zoomhi = 1.0;
		zoomlo = onescale / minscale;
	} else {
		scale = newscale;
	}

	xdial = (tdial - tmin) * scale;
	maxwidth = (scale * (tmax - tmin)) + 0.5;

	xdrawpos = (pixwidth - drawwidth) / 2;
	xpix = xdial - pixwidth / 2;

	if (xpix < 0) {
		xdrawpos = max(xdrawpos + xpix, 0);
		xpix = 0;
	}
	
	if ((xpix + xdrawpos + drawwidth) > maxwidth) {
		xdrawpos = 0;
		xpix = maxwidth - drawwidth;
	}

	if ((xpix + xdrawpos) < 0) {
		xpix = xdrawpos = 0;
	}

	XtVaSetValues(hsb_w,
		XmNvalue, xpix + xdrawpos,
		XmNmaximum, max(maxwidth, drawwidth), NULL);
/*
 * Set the new max width.
 */
	XtVaSetValues(trace_w,
		XmNmaxWidth, max(maxwidth, drawwidth) + wextra, NULL);
/*
 * Refresh screen.
 */
	tr_fill();
	tr_draw();
	movedial(xdial, 1);
}

/*
 *	tack_cb
 *
 *	Function:	- tack button callback
 */
static void
tack_cb()

{
	Widget		w;

	XtRemoveCallback(trace_w, XmNdestroyCallback, destroy_cb, NULL);
	XtUnmapWidget(trace_w);
	w = trace_w;
	tr_clear();
	XtDestroyWidget(w);
	xmpi_db_free();
}

/*
 *	destroy_cb
 *
 *	Function:	- trace window destroy callback
 */
static void
destroy_cb()

{
	tr_clear();
	xmpi_db_free();
}

/*
 *	tr_clear
 *
 *	Function:	- clear trace and associated windows
 *			- freeing data structures
 */
static void
tr_clear()

{
	Widget		w;

	if (!trace_w) return;

	w = trace_w;
	trace_w = 0;
/*
 * Stop the VCR.
 */
	if (fl_vcron) {
		XtRemoveTimeOut(vcrtimer);
		fl_vcron = 0;
	}
/*
 * Free information storage space.
 */
	if (traces) free((char *) traces);
	if (arrows) free((char *) arrows);

	traces = 0;
	arrows = 0;
	fl_resize = 0;
/*
 * Clear the current display.
 */
	xmpi_ctl_setsensitive(XMPI_BSNAPSHOT, False);
	xmpi_ctl_setsensitive(XMPI_BKIVIAT, False);
	xmpi_ctl_setsensitive(XMPI_BMATRIX, False);

	xmpi_dt_clear();
	xmpi_kv_popdown();

	if (xmpi_app_procs) {
/*
 * A process is running (may have finalized).
 */
		if (xmpi_is_express) {
			if (!xmpi_do_rerun && !xmpi_do_express) {
				xmpi_vw_dbsetmode(0);
				xmpi_vw_update(1);
			}
		}
		else {
			xmpi_mat_destroy();
			xmpi_vw_dbsetmode(0);
			xmpi_vw_init();
		}
		
		xmpi_ctl_setsensitive(XMPI_BSNAPSHOT, True);
	}
	else {
		xmpi_mat_destroy();
		xmpi_vw_clear();
	}
/*
 * Destroy pixmaps and dial pen.
 */
	XmUpdateDisplay(w);
	XFreePixmap(disp, pixmap);
	XFreePixmap(disp, savedial);
	XFreePixmap(disp, dial);
	XFreePixmap(disp, mask);
	XFreeGC(disp, dialpen);
}

/*
 *	draw_cb
 *
 *	Function:	- callback function for drawing traces
 *			- handles mouse clicks and expose events
 *	Accepts:	- widget
 *			- client data
 *			- ptr callback struct
 */
static void
draw_cb(w, cdata, p_callback)

Widget			w;
XtPointer		cdata;
XmDrawingAreaCallbackStruct
			*p_callback;

{
	int		x, y;			/* mouse position */
	XEvent		*event;			/* activating event */

	if (!trace_w || fl_band) return;
	
	event = p_callback->event;
/*
 * Move the dial and update views.
 */
	if (p_callback->reason == XmCR_INPUT) {

		x = event->xbutton.x;
		y = event->xbutton.y;
/*
 * If mouse inside window, move the dial.
 * Update views if button is released.
 */
		if ((x >= 0) && (x < drawwidth) &&
				(y >= 0) && (y < drawheight)) {
			movedial(x + xdrawpos + xpix,
					event->xany.type == ButtonRelease);
		}
	}
/*
 * Handle the expose event.
 */
	else if (p_callback->reason == XmCR_EXPOSE) {
		redraw();
	}
}

/*
 *	movedial
 *
 *	Function:	- move dial to a given position
 *			- update all view if needed
 *	Accepts:	- new dial x position
 *			- update views flag
 */
static void
movedial(x, fl_update)

int			x;
int			fl_update;

{
/*
 * If still in the same pixmap, restore previous dial location.
 */
	if ( (xdial >= xpix) && (xdial < (xpix + pixwidth)) ) {
		XCopyArea(disp, savedial, pixmap, fgpen, 0, 0,
			XMPI_DIALWIDTH, pixheight, xdial - xpix, 0);
	}
/*
 * Draw new dial, first saving what is under it.
 */
	XCopyArea(disp, pixmap, savedial, fgpen, x - xpix, 0,
		XMPI_DIALWIDTH, pixheight, 0, 0);
	XSetClipOrigin(disp, dialpen, x - xpix, 0);
	XCopyArea(disp, dial, pixmap, dialpen, 0, 0,
		XMPI_DIALWIDTH, pixheight, x - xpix, 0);
	
	xdial = x;
	showtime();

	redraw();
/*
 * Update views if needed.
 */
	if (fl_update) {
		xmpi_vw_update(!fl_vcron);
		xmpi_kv_update();
	}
}

/*
 *	resize_cb
 *
 *	Function:	- callback function for resizing trace view
 *	Accepts:	- widget
 *			- client data
 *			- ptr callback struct
 */
static void
resize_cb(w, cdata, p_callback)

Widget			w;
XtPointer		cdata;
XmDrawingAreaCallbackStruct
			*p_callback;

{
	Dimension	width, height;		/* drawing area dimensions */
	
	if (!fl_resize) return;
/*
 * Determine new dimensions and parameters.
 */
	XtVaGetValues(draw_w, XmNwidth, &width, XmNheight, &height, NULL);

	drawwidth = width;
	drawheight = height;

	pixwidth = drawwidth * XMPI_TRNVIEW;
	maxwidth = (scale * (tmax - tmin)) + 0.5;
/*
 * Allocate new pixmap.
 */
	if (pixmap) XFreePixmap(disp, pixmap);
	pixmap = XCreatePixmap(disp, root, pixwidth, pixheight, depth);
/*
 * Change the vertical scrollbar parameters.
 */
	if ((ydrawpos + drawheight) > pixheight) {
		ydrawpos = pixheight - drawheight;
	}

	if (ydrawpos < 0) ydrawpos = 0;

	XtVaSetValues(vsb_w,
		XmNvalue, ydrawpos,
		XmNsliderSize, drawheight,
		XmNpageIncrement, drawheight - 1,
		NULL);
/*
 * Change the horizontal scrollbar parameters.
 */
	if ((xpix + xdrawpos + drawwidth) > maxwidth) {
		xdrawpos = 0;
		xpix = maxwidth - drawwidth;
	}

	if ((xpix + xdrawpos) < 0)
		xpix = xdrawpos = 0;

	if ((xdrawpos + drawwidth) > pixwidth)
		xdrawpos = pixwidth - drawwidth;

	XtVaSetValues(hsb_w,
		XmNvalue, xpix + xdrawpos,
		XmNsliderSize, drawwidth,
		XmNpageIncrement, drawwidth - 1,
		NULL);
/*
 * Refresh screen. 
 */
	tr_fill();
	tr_draw();
}

/*
 *	hscroll_cb
 *
 *	Function:	- callback function for horizontal scrollbar
 *	Accepts:	- widget
 *			- client data
 *			- ptr callback struct
 */
static void
hscroll_cb(w, cdata, p_callback)

Widget			w;
XtPointer		cdata;
XmScrollBarCallbackStruct
			*p_callback;

{
	int		x;

	x = p_callback->value;
/*
 * If the view moved outside pixmap cache, refill and center the view.
 */
	if ( (x < xpix) || ((x + drawwidth) > (xpix + pixwidth)) ) {

		xpix = x - drawwidth;
		if (xpix < 0) {
			xpix = 0;
			xdrawpos = x;
		} else {
			xdrawpos = drawwidth;
		}

		tr_fill();
		tr_draw();
	}
/*
 * Otherwise just set view offset and redraw.
 */
	else {
		xdrawpos = x - xpix;
		showtime();
		redraw();
	}
}

/*
 *	vscroll_cb
 *
 *	Function:	- callback function for vertical scrollbar
 *	Accepts:	- widget
 *			- client data
 *			- ptr callback struct
 */
static void
vscroll_cb(w, cdata, p_callback)

Widget			w;
XtPointer		cdata;
XmScrollBarCallbackStruct
			*p_callback;

{
	ydrawpos = p_callback->value;
	redraw();
}

/*
 *	arm_cb
 *
 *	Function:	- print armed button info message
 *	Accepts:	- widget
 *			- info message
 */
static void
arm_cb(w, mesg)

Widget			w;
char			*mesg;

{
/*	xmpi_setlabel(scale_w, mesg);
	XmUpdateDisplay(scale_w);
*/	
}

/*
 *	vcrrew_cb
 *
 *	Function:	- rewind VCR
 */
static void
vcrrew_cb()

{
	vcrstop_cb();

	xpix = xdrawpos = 0;
	XtVaSetValues(hsb_w, XmNvalue, 0, NULL);
	movedial(0, 1);

	tr_fill();
	tr_draw();
}

/*
 *	vcrstop_cb
 *
 *	Function:	- stop VCR and do a final update
 */
static void
vcrstop_cb()

{
	if (fl_vcron) {
		XtRemoveTimeOut(vcrtimer);
		fl_vcron = 0;
		XtMapWidget(tleft_w);
		XtMapWidget(tright_w);
		movedial(xdial, 1);
	}
}

/*
 *	vcrplay_cb
 *
 *	Function:	- play VCR
 */
static void
vcrplay_cb()

{
	vcrdelay = XMPI_TRVCRPLAY;
	vcrincr = 1;

	if (!fl_vcron) {
		vcrtimer = XtAppAddTimeOut(app, vcrdelay,
				(XtTimerCallbackProc) vcradvance, NULL);
		fl_vcron = 1;
		XtUnmapWidget(tleft_w);
		XtUnmapWidget(tright_w);
	}
}

/*
 *	vcrfwd_cb
 *
 *	Function:	- play VCR faster
 */
static void
vcrfwd_cb()

{
	vcrdelay = XMPI_TRVCRFWD;
	vcrincr = 2;

	if (!fl_vcron) {
		vcrtimer = XtAppAddTimeOut(app, vcrdelay,
				(XtTimerCallbackProc) vcradvance, NULL);
		fl_vcron = 1;
		XtUnmapWidget(tleft_w);
		XtUnmapWidget(tright_w);
	}
}

/*
 *	vcradvance
 *
 *	Function:	- move VCR one frame forward
 */
static void
vcradvance()

{
	int		val;
	int		size;
	int		inc;
	int		pinc;
	int		x;

	if (xdial >= maxwidth) {
		vcrstop_cb();
		return;
	}

	x = xdial + vcrincr;
	if (x > maxwidth) x = maxwidth;

	if (x > (xpix + xdrawpos + (drawwidth / 2))) {
		XmScrollBarGetValues(hsb_w, &val, &size, &inc, &pinc);

		val += vcrincr;
		if (val > (maxwidth - size)) {
			val = maxwidth - size;
		}

		XmScrollBarSetValues(hsb_w, val, size, inc, pinc, True);
	}

	movedial(x, 1);

	vcrtimer = XtAppAddTimeOut(app, vcrdelay,
				(XtTimerCallbackProc) vcradvance, NULL);
}

/*
 *	redraw
 *
 *	Function:	- redraw the trace view
 */
static void
redraw()

{
	XCopyArea(disp, pixmap, XtWindow(draw_w), fgpen,
			xdrawpos, ydrawpos, drawwidth, drawheight, 0, 0);
}

/*
 *	gcinit
 *
 *	Function:	- allocate coloured pens
 */
static void
gcinit()

{
	Pixel		fg, bg;			/* drawing area colours */
	XGCValues	gcval;			/* graphic context info */
	XColor		col_def;		/* colour definition */
	XColor		dev_null;		/* unused functionality */

	if (fgpen) return;

	disp = XtDisplay(draw_w);
	root = RootWindowOfScreen(XtScreen(draw_w));
	depth = DefaultDepthOfScreen(XtScreen(draw_w));
/*
 * Create the graphic contexts.
 */
	XtVaGetValues(draw_w, XmNforeground, &fg, XmNbackground, &bg, NULL);

	gcval.foreground = bg;
	gcval.background = fg;
	gcval.font = app_res.ap_rankfont->fid;

	fgpen = XCreateGC(disp, root,
			(GCFont | GCForeground | GCBackground), &gcval);

	gcval.foreground = fg;
	gcval.background = bg;
	bgpen = XCreateGC(disp, root, GCForeground, &gcval);

	XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(draw_w)),
					"yellow", &col_def, &dev_null);

	gcval.foreground = col_def.pixel;
	syspen = XCreateGC(disp, root, GCForeground, &gcval);

	XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(draw_w)),
					"red", &col_def, &dev_null);

	gcval.foreground = col_def.pixel;
	blkpen = XCreateGC(disp, root, GCForeground, &gcval);

	XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(draw_w)),
					"green", &col_def, &dev_null);

	gcval.foreground = col_def.pixel;
	runpen = XCreateGC(disp, root, GCForeground, &gcval);

	XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(draw_w)),
					"white", &col_def, &dev_null);

	gcval.foreground = col_def.pixel;
	arwpen = XCreateGC(disp, root, GCForeground, &gcval);

	XAllocNamedColor(disp, DefaultColormapOfScreen(XtScreen(draw_w)),
					"gray", &col_def, &dev_null);

	gcval.foreground = col_def.pixel;
	misspen = XCreateGC(disp, root, GCForeground, &gcval);

	gcval.foreground = app_res.ap_bandcol;
	gcval.line_width = app_res.ap_bandwidth;
	gcval.line_style = (app_res.ap_banddash) ? LineOnOffDash : LineSolid;
	rubpen = XCreateGC(disp, root, (GCForeground | GCLineStyle
					| GCLineWidth), &gcval);
}

/*
 *	initparams
 *
 *	Function:	- initialize trace viewing parameters
 */
static int
initparams()

{
	int		n;
/*
 * Get the # processes and min/max times.
 */
	nprocs = xmpi_db_getnprocs();
	if (nprocs == 0) return(LAMERROR);
	
	tmin = xmpi_db_getmintime();
	tmax = xmpi_db_getmaxtime();
/*
 * Determine the view sizes and scaling factor.
 */
	drawwidth = XMPI_TRWIDTH;
	pixwidth = drawwidth * XMPI_TRNVIEW;

	n = ((nprocs > XMPI_TRNPROCS) ? XMPI_TRNPROCS : nprocs);
	drawheight = ((n + 1) * XMPI_TRDISP) - XMPI_TRHEIGHT;
	pixheight = ((nprocs + 1) * XMPI_TRDISP) - XMPI_TRHEIGHT;

	minscale = drawwidth / (tmax - tmin);
	maxscale = 20.0 / xmpi_db_getminlapse();

	if (maxscale < minscale) {
		maxscale = minscale;
	}

	n = xmpi_db_getmaxtraces();

	scale = minscale * max(n / 20.0, 1.0);
	if (scale > maxscale) {
		scale = maxscale;
	}
	onescale = scale;
	zoomhi = zoomlo = 1.0;
	fl_power2 = 1;

	maxwidth = (scale * (tmax - tmin)) + 0.5;

	xdrawpos = ydrawpos = 0;
	xpix = 0;

	xdial = drawwidth / 2;

	return(0);
}

/*
 *	showtime
 *
 *	Function:	- show time information
 */
static void
showtime()

{
	static double		tleft = -1.0;
	static double		tright = -1.0;
	double			newleft;
	double			newright;

	xmpi_db_settime(tmin + (xdial / scale));

	sprintf(fmtbuf, "%.6f", tmin + (xdial / scale));
	xmpi_setlabel(tdial_w, fmtbuf);

	newleft = tmin + ((xpix + xdrawpos) / scale);
	newright = tmin + ((xpix + xdrawpos + drawwidth) / scale);

	if (newleft != tleft) {
		tleft = newleft;
		sprintf(fmtbuf, "%.6f", tleft);
		xmpi_setlabel(tleft_w, fmtbuf);
	}

	if (newright != tright) {
		tright = newright;
		sprintf(fmtbuf, "%.6f", tright);
		xmpi_setlabel(tright_w, fmtbuf);
	}
}

/*
 *	showscale
 *
 *	Function:	- show scale information
 */
static void
showscale()

{
	sprintf(fmtbuf, "Scale: %-.4g x %-.4g", zoomhi, zoomlo);
	xmpi_setlabel(scale_w, fmtbuf);
}

/*
 *	showlabel
 *
 *	Function:	- show the label information
 */
static void
showlabel()

{
	char		*label = "<Running Application>";

	if (*tracefile) {
		label = strrchr(tracefile, STRDIR);
		label = label ? label + 1 : tracefile;
	}

	sprintf(fmtbuf, "Timeline  (%s)", label);
	xmpi_setlabel(banner_w, fmtbuf);
}

/*
 *	setupdial
 *
 *	Function:	- set up the dial pixmap and its save pixmap
 */
static void
setupdial()

#define XMPI_DTXTOFF 3
#define XMPI_DPROCOFF 18

{
	Pixel		fg, bg;		/* drawing area colours */
	XGCValues	gcval;		/* graphic context info */
	int		i, y;
	char		numstr[8];
	GC		maskgc;		/* used in creating the mask */
	
	XtVaGetValues(draw_w, XmNforeground, &fg, XmNbackground, &bg, NULL);

	dial = XCreatePixmap(disp, root, XMPI_DIALWIDTH, pixheight, depth);
	savedial = XCreatePixmap(disp, root, XMPI_DIALWIDTH, pixheight, depth);
	mask = XCreatePixmap(disp, root, XMPI_DIALWIDTH, pixheight, 1);

       	gcval.foreground = 0;
	gcval.background = 0;
	gcval.font = app_res.ap_rankfont->fid;

	maskgc = XCreateGC(disp, mask, (GCFont | GCForeground |
			GCBackground), &gcval);

	XFillRectangle(disp, mask, maskgc, 0, 0, XMPI_DIALWIDTH, pixheight);
	XSetForeground(disp, maskgc, 1);
/*
 * Draw process numbers into the dial and mask.
 */
	y = XMPI_DPROCOFF;
	for (i=0; i<nprocs; i++) {
		sprintf(numstr, "%d", i);
		XDrawString(disp, dial, fgpen, XMPI_DTXTOFF, y,
			numstr, strlen(numstr));
		XDrawString(disp, mask, maskgc, XMPI_DTXTOFF, y,
			numstr, strlen(numstr));
		y += XMPI_TRDISP;
	}
/*
 * Draw the dial 'cursor'.
 */
	XDrawLine(disp, dial, fgpen, 0, 0, 0, pixheight);
	XDrawLine(disp, mask, maskgc, 0, 0, 0, pixheight);

	XFreeGC(disp, maskgc);

	gcval.foreground = fg;
	gcval.background = bg;
	gcval.clip_mask = mask;
	dialpen = XCreateGC(disp, root, (GCForeground | GCBackground |
					GCClipMask), &gcval);
}

/*
 *	xmpi_tr_busy
 *
 *	Function:	- set the busy cursor for the Trace window
 */
void
xmpi_tr_busy()

{
	if (trace_w) xmpi_busy_widget(trace_w);
}

/*
 *	xmpi_tr_unbusy
 *
 *	Function:	- set the normal cursor for the Trace window
 */
void
xmpi_tr_unbusy()

{
	if (trace_w) xmpi_unbusy_widget(trace_w);
}

#define RUBBER_MARGIN	3

/*
 *	rubber
 *
 *	Function:	- zoom selection rubber banding action
 *	Accepts:	- Xt action parameters
 */
static void
rubber(widget, event, args, num_args)

Widget			widget;
XButtonEvent		*event;
String			*args;
int			*num_args;

{
	static Position	init_x;			/* x start of selection */
	static Position	x;			/* current x position */
	static unint	w;			/* width */
	static int	erase = 0;		/* 1 if box drawn, else 0 */

	if (!fl_zoom || fl_vcron) return;

	if (strcmp(args[0], "down") == 0) {

		x = init_x = event->x;
		w = 0;
		erase = 0;
		fl_band = 1;
	} else {
		if (erase) {
			erase = 0;
			redraw();
		}

		if (event->x < 0) {
			event->x = 0;
		} else if (event->x >= drawwidth) {
			event->x = drawwidth - 1;
		}

		if (event->x < init_x) {
			w = init_x - event->x;
			x = event->x;
		} else {
			w = event->x - x;
			x = init_x;
		}

		if (strcmp(args[0], "motion") == 0) {

			if (w > 1 && event->y >= 0 && event->y <= drawheight) {

				erase = 1;
				XDrawRectangle(disp, XtWindow(draw_w), rubpen,
					x, RUBBER_MARGIN,
					w, drawheight - 2 * RUBBER_MARGIN - 1);
			}
		} else {
/*
 * button up
 */
			fl_band = 0;
			
			if (w > 1 && event->y >= 0 && event->y <= drawheight) {
				dragged_zoom(x, (int) w);
			}
		}
	}
}

/*
 *	dragged_zoom
 *
 *	Function:	- zoom of selected timeline segment
 *	Accepts:	- x offset in window of left side of segment
 *			- width of segment
 */
static void
dragged_zoom(x, w)

int			x;
int			w;

{
	double		t;			/* time at selection leftside */
	double		factor;			/* scale factor */
	double		power2;			/* powers of 2 */

	if (!fl_zoom) return;

	if (xpix + x >= maxwidth) return;

	if (xpix + x + w > maxwidth) {
		w = maxwidth - xpix - x;
	}

	t = ((double) (xpix + xdrawpos + x)) / scale + tmin;

	scale *= (double) drawwidth / (double) w;
	if (scale > maxscale) {
		scale = maxscale;
	}
/*
 * If scale is close enough to a power of two then use that.
 */
	factor = scale / onescale;
	
	power2 = pow(2.0, floor(log(factor)/log(2.0)));

	if (factor > power2 - XMPI_EPSSCALE
			&& factor < power2 + XMPI_EPSSCALE) {

		fl_power2 = 1;

		if (power2 >= 1.0) {
			zoomhi = power2;
			zoomlo = 1.0;
			scale = onescale * power2;
		} else {
			zoomlo = 1.0 / power2;
			zoomhi = 1.0;
			scale = onescale / zoomlo;
		}

	} else {
		power2 = pow(2.0, ceil(log(factor)/log(2.0)));

		if (factor > power2 - XMPI_EPSSCALE
				&& factor < power2 + XMPI_EPSSCALE) {

			fl_power2 = 1;
			if (power2 >= 1.0) {
				zoomhi = power2;
				zoomlo = 1.0;
				scale = onescale * power2;
			} else {
				zoomlo = 1.0 / power2;
				zoomhi = 1.0;
				scale = onescale / zoomlo;
			}
		} else {
			fl_power2 = 0;
			if (scale >= onescale) {
				zoomhi = scale / onescale;
				zoomlo = 1.0;
			} else {
				zoomlo = onescale / scale;
				zoomhi = 1.0;
			}
		}
	}
/*
 * Determine new drawing positions.
 */
	maxwidth = (scale * (tmax - tmin)) + 0.5;
	xdrawpos = (pixwidth - drawwidth) / 2;
	xpix = ((int) ((t - tmin) * scale)) - xdrawpos;

	if (xpix < 0) {
		xdrawpos = max(xdrawpos + xpix, 0);
		xpix = 0;
	}

	if ((xpix + xdrawpos + drawwidth) > maxwidth) {
		xdrawpos = 0;
		xpix = maxwidth - drawwidth;
	}

	if ((xpix + xdrawpos) < 0) {
		xpix = xdrawpos = 0;
	}
/*
 * Center the dial.
 */
	xdial = xpix + xdrawpos + drawwidth / 2;

	XtVaSetValues(hsb_w,
		XmNvalue, xpix + xdrawpos,
		XmNmaximum, max(maxwidth, drawwidth), NULL);

	XtVaSetValues(trace_w,
		XmNmaxWidth, max(maxwidth, drawwidth), NULL);

	tr_fill();
	tr_draw();
}
