/* vi:set ts=8 sts=0 sw=8:
 * $Id: htmlview.c,v 1.14 2000/02/19 06:57:54 kahn Exp kahn $
 *
 * Copyright (C) 1998 Andy C. Kahn
 *
 *     This program is free software; you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation; either version 2 of the License, or
 *     (at your option) any later version.
 *
 *     This program is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with this program; if not, write to the Free Software
 *     Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <sys/types.h>
#include "main.h"
#include <glib.h>
#include <gtk/gtk.h>
#include "doc.h"
#include "file.h"
#include "dialog.h"
#include "misc.h"
#include "prefs.h"
#include "htmlview.h"
#include "gnpintl.h"


#if defined(USE_GTKXMHTML) || defined(USE_GTKHTML)
static void html_view_close_cb(GtkWidget *wgt, gpointer cbdata);
static void html_view_in_new_win_by_text(char *fname, char *buf, htmlview_t);
#endif
static void html_view_browser(doc_t *d);


#ifdef USE_GNOME
/*** we need these individual callbacks because we can't specify a callback
     action as part of the Gnome menus ***/
/*
 * PUBLIC: html_view_xmhtml_win_cb
 *
 * view in a separate gtk-XmHtml window
 */
void
html_view_xmhtml_win_cb(GtkWidget *wgt, gpointer cbdata)
{
	html_view_cb(wgt, cbdata, (guint)HtmlViewXmHtml);
} /* html_view_xmhtml_win_cb */


/*
 * PUBLIC: html_view_gtkhtml_win_cb
 *
 * view in a separate gtk-XmHtml window
 */
void
html_view_gtkhtml_win_cb(GtkWidget *wgt, gpointer cbdata)
{
	html_view_cb(wgt, cbdata, (guint)HtmlViewGtkHtml);
} /* html_view_gtkhtml_win_cb */


/*
 * PUBLIC: html_view_doc_xmhtml_cb
 *
 * view as a document inside gnotepad+
 */
void
html_view_xmhtml_doc_cb(GtkWidget *wgt, gpointer cbdata)
{
	html_view_cb(wgt, cbdata, (guint)HtmlViewXmHtmlDoc);
} /* html_view_xmhtml_doc_cb */


/*
 * PUBLIC: html_view_doc_gtkhtml_cb
 *
 * view as a document inside gnotepad+
 */
void
html_view_gtkhtml_doc_cb(GtkWidget *wgt, gpointer cbdata)
{
	html_view_cb(wgt, cbdata, (guint)HtmlViewGtkHtmlDoc);
} /* html_view_gtkhtml_doc_cb */


/*
 * PUBLIC: html_view_browser_cb
 *
 * view in netscape
 */
void
html_view_browser_cb(GtkWidget *wgt, gpointer cbdata)
{
	html_view_cb(wgt, cbdata, (guint)HtmlViewBrowser);
} /* html_view_browser_cb */
#endif	/* USE_GNOME */


/*
 * PUBLIC: html_view_cb
 *
 * menu callback to view a document as HTML with different methods (as a
 * separate popup window, as a document, or under a browser).
 */
void
html_view_cb(GtkWidget *wgt, gpointer cbdata, guint cbaction)
{
	win_t *w = (win_t *)cbdata;
	htmlview_t hviewtype = (htmlview_t)cbaction;
	doc_t *d;
#ifdef USE_GTKXMHTML
	char *buf;
#endif

	g_assert(w != NULL);
	d = DOC_CURRENT(w);
	g_assert(d != NULL);

	if (!GTK_IS_TEXT(d->data)) {
		(void)do_dialog_error( _("Not a text file!"),
				      _(" Can't view non-text docs as HTML! "));
		return;
	}

	switch (hviewtype) {
#ifdef USE_GTKXMHTML
	case HtmlViewXmHtml:
		buf = gtk_editable_get_chars(
				GTK_EDITABLE(d->data), 0,
				gtk_text_get_length(GTK_TEXT(d->data)));
		html_view_in_new_win_by_text(d->fname, buf, hviewtype);
		g_free(buf);
		break;
	case HtmlViewXmHtmlDoc:
		doc_new(w, d->fname, DocXmHtml,
			(UPDATE_MSGBAR | DO_LOAD | UPDATE_TITLE));
		break;
#endif
#ifdef USE_GTKHTML
	case HtmlViewGtkHtml:
		html_view_in_new_win_by_text(d->fname, NULL, hviewtype);
		break;
	case HtmlViewGtkHtmlDoc:
		doc_new(w, d->fname, DocGtkHtml,
			(UPDATE_MSGBAR | DO_LOAD | UPDATE_TITLE));
		break;
#endif
	case HtmlViewBrowser:
		/*
		 * if the document has been changed, then write to a temporary
		 * file and let the browser view that temporary file.  if it
		 * hasn't been changed, then let the browser view the file
		 * directly from disk.
		 */
		html_view_browser(d);
		break;
	default:
		g_warning( _("unknown HTML view type = %d\n"), hviewtype);
	}
} /* html_view_cb */


/*
 * PRIVATE: html_view_browser
 *
 * views a text document by launching a browser.
 */
static void
html_view_browser(doc_t *d)
{
	pid_t pid;

	if (d->changed)
		(void)do_dialog_ok( _("File updated!"),
				   _(" Warning: new changes in the file \nhave not been saved yet. \n"));

	if ((pid = fork()) == -1) {
		perror("html_view_browser_exec: could not fork");
		(void)do_dialog_error( _("Fork failed!"),
				      _(" Could not fork child process! "));
	} else if (pid == 0) {
		char *cmd, *full;

		full = file_full_pathname_make(d->fname);
		cmd = g_new(char, (2 * strlen(full)) + 64);
		sprintf(cmd,
			"netscape -remote openURL\\(file://%s\\) || "
			"netscape file://%s", full, full);
		GNPDBG_HTMLVIEW(("html_view_browser_exec: cmd = '%s'\n", cmd));
		if (execl("/bin/sh", "sh", "-c", cmd, NULL) == -1) {
			perror("html_view_browser_exec: child exec failed");
			(void)do_dialog_error( _("Exec failed!"),
					      _(" Child could not exec! "));
		}
		exit(127);
	}
} /* html_view_browser */


#ifdef USE_GTKHTML
void
html_view_url_requested(GtkHTML *html, const char *url,
			GtkHTMLStreamHandle handle, gpointer cbdata)
{
	FILE *fp;
	int nread;
	char buf[8192];

	if ((fp = fopen(url, "r")) == NULL) {
		gtk_html_end(html, handle, GTK_HTML_STREAM_ERROR);
		return;
	}

	while (!feof(fp)) {
		nread = fread(buf, 1, sizeof(buf), fp);
		gtk_html_write(html, handle, buf, nread);
	}
	gtk_html_end(html, handle, GTK_HTML_STREAM_OK);
	fclose(fp);
} /* html_view_url_requested */
#endif


#if defined(USE_GTKXMHTML) || defined(USE_GTKHTML)
/*
 * PRIVATE: html_view_in_new_win_by_text
 *
 * views a text document by creating a new popup window.  this routine requires
 * the gtk-xmhtml library.
 */
static void
html_view_in_new_win_by_text(char *fname, char *textbuf, htmlview_t hviewtype)
{
	GtkWidget *toplev, *tmp;
	char *titlebuf;
#ifdef USE_GTKHTML
	GtkWidget *html = NULL;
#endif

	if (fname) {
		titlebuf = g_new(char, strlen(APP_NAME) +
				       strlen(" HTMLView: ") +
				       (fname ? strlen(fname) : 0) + 1);
		sprintf(titlebuf, "%s HTMLView: %s", APP_NAME, fname);
	} else {
		titlebuf = g_new(char, strlen(APP_NAME) +
				       strlen(" HTMLView") + 1);
		sprintf(titlebuf, "%s HTMLView", APP_NAME);
	}

	toplev = gtk_dialog_new();
	gtk_window_set_title(GTK_WINDOW(toplev), titlebuf);
	g_free(titlebuf);
	gtk_widget_set_name(toplev, "gnp+");
	gtk_window_set_wmclass(GTK_WINDOW(toplev), "gnp+", "gnp+");
	gtk_window_set_policy(GTK_WINDOW(toplev), TRUE, TRUE, FALSE);
#ifdef GTK_HAVE_FEATURES_1_1_0
	gtk_window_set_default_size(GTK_WINDOW(toplev),
				    prefs.html_view_w, prefs.html_view_h);
#else
	gtk_widget_set_usize(toplev, 320, 200);
#endif

#ifdef USE_GTKXMHTML
	if (hviewtype == HtmlViewXmHtml) {
		tmp = gtk_xmhtml_new();
		/* "activate" and "frame" event */
		gtk_xmhtml_source(GTK_XMHTML(tmp), textbuf);
		gtk_box_pack_start(GTK_BOX(GTK_DIALOG(toplev)->vbox), tmp,
				   TRUE, TRUE, 10);
	}
#endif
#ifdef USE_GTKHTML
	else {
		GtkWidget *sb;
		
		tmp = gtk_hbox_new(FALSE, 0);
		gtk_box_pack_start(GTK_BOX(GTK_DIALOG(toplev)->vbox), tmp,
				   TRUE, TRUE, 10);

		html = gtk_html_new(NULL, NULL);
		gtk_signal_connect(GTK_OBJECT(html), "url_requested",
				   GTK_SIGNAL_FUNC(html_view_url_requested),
				   NULL);
		gtk_box_pack_start_defaults(GTK_BOX(tmp), html);
		sb = gtk_vscrollbar_new(GTK_LAYOUT(html)->vadjustment);
		gtk_box_pack_start(GTK_BOX(tmp), sb, FALSE, TRUE, 0);
# if 0
		/* doesn't work right now */
		sb = gtk_vscrollbar_new(GTK_LAYOUT(html)->hadjustment);
		gtk_box_pack_start(GTK_BOX(GTK_DIALOG(toplev)->vbox), sb,
				   FALSE, FALSE, 0);
# endif
	}
#endif

	(void)misc_button_new_w_label(_("Ok"), GNOME_STOCK_BUTTON_OK,
				      html_view_close_cb, toplev,
				      GTK_DIALOG(toplev)->action_area,
				      PACK_START | PACK_EXPAND | PACK_FILL, 2);
	gtk_widget_show_all(toplev);

#ifdef USE_GTKHTML
	if (hviewtype == HtmlViewGtkHtml) {
		gtk_html_begin(GTK_HTML(html), fname);
		gtk_html_parse(GTK_HTML(html));
	}
#endif
} /* html_view_in_new_win_by_text */


/*
 * PRIVATE: html_view_close_cb
 *
 * callback for the "Ok" button to close the popup window created in
 * html_view_in_new_win_by_text().
 */
static void
html_view_close_cb(GtkWidget *wgt, gpointer cbdata)
{
	gdk_window_get_size(GTK_WIDGET(cbdata)->window,
			    &prefs.html_view_w, &prefs.html_view_h);
	GNPDBG_HTMLVIEW(("html_view_close_cb: width = %d, height = %d\n",
			 prefs.html_view_w, prefs.html_view_h));
	gtk_widget_destroy(GTK_WIDGET(cbdata));
} /* html_view_close_cb */
#endif


/* the end */
