/*
 * LEVEE, or Captain Video;  A vi clone
 *
 * Copyright (c) 1982-1997 David L Parsons
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms are permitted
 * provided that the above copyright notice and this paragraph are
 * duplicated in all such forms and that any documentation,
 * advertising materials, and other materials related to such
 * distribution and use acknowledge that the software was developed
 * by David L Parsons (orc@pell.chi.il.us).  My name may not be used
 * to endorse or promote products derived from this software without
 * specific prior written permission.  THIS SOFTWARE IS PROVIDED
 * AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
 * WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF MERCHANTIBILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE.
 */
#include <stdlib.h>
#include <string.h>
#include <curses.h>
#include <term.h>
#include <unistd.h>

#include "levee.h"
#include "extern.h"

/* do a gotoXY -- allowing -1 for same row/column */

#if !TERMCAP

#define MAXCOLS COLS

#else /* TERMCAP */

#define MAXCOLS 160

#if TERMCAP_EMULATION
/*
 * Termcap handlers
 *
 * Routines included:
 *   tc_init() -- set up all the terminal stuff levee will need.
 *  *xtract() -- get a field out of the termcap entry.
 *  *parseit() -- parse a termcap field.
 *   tgoto()   -- put a gotoXY string into a buffer.
 *  * -> internal routine.
 */

/* parse a termcap field */
char *parseit(char *ptr, char **savearea)
{
    char *p = *savearea;
    char *op = *savearea;
    int tmp;

    while (*ptr && *ptr != ':') {
	if (*ptr == '\\' && ptr[1]) {
	    ++ptr;
	    switch (*ptr) {
	      case 'E':
		*p++ = '\033';
		break;
	      case '0': case '1': case '2': case '3': case '4':
	      case '5': case '6': case '7': case '8': case '9':
		  tmp = 0;
		  while (isdigit(*ptr))
		      tmp = (tmp*8)+(*ptr++ - '0');
		  *p++ = tmp;
		  --ptr;
	      default:
		  *p++ = *ptr;
	    }
	}
	else *p++ = *ptr;
	++ptr;
    }
    *p++ = 0;
    *savearea = p;
    return op;
} /* parseit */

/* get something from the termcap
 *
 * arguments: tcentry -- the termcap entry
 *            what    -- the field we want to get (NULL means first field)
 *            savearea-- pointer to static buffer for parsed fields.
 */
char *xtract(char *ptr, char name[], char **savearea)
{
    int size;

    if (!ptr)
	return NULL;

    if (!name)	/* return first field of entry -- terminal name? */
	return parseit(ptr,savearea);

    size = strlen(name);
    /* always skip the first (terminal name) field */
    while (*ptr) {
	while (*ptr && *ptr != ':') {
	    if (*ptr == '\\')
		ptr++;
	    ptr++;
	}
	if (*ptr)
	    ptr++;
	if (*ptr && strncmp(name,ptr,size) == 0 && ptr[size] == '=')
	    return parseit(&ptr[1+size],savearea);
	puts("\r");
    }
    return NULL;
} /* xtract */

/* get a character field from the termcap */
char charext(char *tc, char *what, char **savearea)
{
    char *p = xtract(tc,what,savearea);
    if (p)
	return *p;
    return 0;
} /* charext */

/* internal variables just for termcap */
static int _Xfirst, _xpad, _ypad;

/* get the termcap stuff and go berserk parsing it */
/* if anything horrid goes wrong, levee will crash */
void tc_init()
{
    char *getenv();
    char *p = getenv("TERMCAP");
    char *lp, *ptr;

    if (!p) {
	puts("lv: no termcap\n");
	exit(1);
    }
    lp = malloc(strlen(p)+1);
    if (!lp) {
	puts("lv: out of memory\n");
	exit(1);
    }

    TERMNAME = xtract(p,NULL,&lp);
    CM   = xtract(p,"CM",&lp);
    HO   = xtract(p,"HO",&lp);
    UP   = xtract(p,"UP",&lp);
    CE   = xtract(p,"CE",&lp);
    CL   = xtract(p,"CL",&lp);
    BELL = xtract(p,"BELL",&lp);
    if (!BELL)
	BELL = "\007";
    OL   = xtract(p,"OL",&lp);
    UpS  = xtract(p,"UpS",&lp);
    cur_on= xtract(p,"cur_on",&lp);
    cur_off=xtract(p,"cur_off",&lp);

    FkL  = charext(p,"FkL",&lp);
    CurRT= charext(p,"CurR",&lp);
    CurLT= charext(p,"CurL",&lp);
    CurUP= charext(p,"CurU",&lp);
    CurDN= charext(p,"CurD",&lp);

    canUPSCROLL = (UpS != NULL);
    CA = (CM != NULL);

    if ((LINES=atoi(ptr=xtract(p,"LINES",&lp))) <= 0) {
	puts("lv: bad termcap");
	exit(1);
    }
    dofscroll = LINES/2;
    if ((COLS=atoi(ptr=xtract(p,"COLS",&lp))-1) <= 0 || COLS >= MAXCOLS) {
	puts("lv: bad termcap");
	exit(1);
    }

    _ypad = _xpad = 0;
    _Xfirst = 1;

    p = CM;

    while (*p && *p != ',')
	p++;
    if (!*p)
	return;
    *p++ = 0;
    if (*p != ',')
	_Xfirst = (*p++ == 'X');
    if (!*p)
	return;
    ++p;
    while (*p && *p != ',')
	_xpad = (_xpad*10) + (*p++ - '0');
    if (!*p)
	return;
    ++p;
    while (*p)
	_ypad = (_ypad*10) + (*p++ - '0');
}

#define tgoto(s,y,x)	(_Xfirst?sprintf(s,CM,x+_xpad,y+_ypad):\
				sprintf(s,CM,y+_ypad,x+_xpad))
#else /* TERMCAP_EMULATION */

/* read in the termcap entry for this terminal.  */
void tc_init()
{
    static char tcbuf[2048+1024];
    char *bufp;
    char *term = getenv("TERM");

    if (( term ? tgetent(tcbuf, term) : 0) != 0) {
	TERMNAME = term;
	bufp = tcbuf + 2048;
	CM = tgetstr("cm", &bufp);
	UP = tgetstr("up", &bufp);
	HO = tgetstr("ho", &bufp);
	if (!HO) {
	    char *goto0 = tgoto(CM, 0, 0);

	    if (goto0)
		HO = strdup(goto0);
	}

	CL = tgetstr("cl", &bufp);
	CE = tgetstr("ce", &bufp);
	BELL = tgetstr("vb", &bufp);
	if (!BELL)
	    BELL = "\007";
	OL = tgetstr("al", &bufp);
	UpS = tgetstr("sr", &bufp);
	cur_on = tgetstr("ve", &bufp);
	cur_off = tgetstr("vi", &bufp);

	LINES = tgetnum("li");
	COLS  = tgetnum("co");

	dofscroll = LINES/2;

	/* set cursor movement keys to zero for now */
	FkL = CurRT = CurLT = CurUP = CurDN = EOF;

	canUPSCROLL = (UpS != NULL);
	CA = (CM != NULL);
    }
}
#endif /* TERMCAP_EMULATION */

#endif /* TERMCAP */

void levee_mvcur(int y, int x)
{
#if TERMCAP_EMULATION || ANSI
    static char gt[30];
#endif

    if (y == -1)
	y = curpos.y;
    else
	curpos.y = y;
    if (y >= LINES)
	y = LINES-1;
    if (x == -1)
	x = curpos.x;
    else
	curpos.x = x;
    if (x >= COLS)
	x = COLS-1;

#if TERMCAP
#if TERMCAP_EMULATION
    tgoto(gt,y,x);
    strput(gt);
#else
    strput( tgoto(CM, x, y) );
#endif /* TERMCAP_EMULATION */
#elif ZTERM
    zgoto(x,y);
#elif ANSI
    {	char *p = gt;		/* make a ansi gotoXY string */
	*p++ = 033;
	*p++ = '[';
	numtoa(p,1+y); p += strlen(p);
	*p++ = ';';
	numtoa(p,1+x); p += strlen(p);
	*p++ = 'H';
	WRITE_TEXT(1, gt, (p-gt));
    }
#elif VT52
    CM[2] = y+32;
    CM[3] = x+32;
    strput(CM);
#endif
}

void numtoa(char *str, int num)
{
    int i = 10;			/* I sure hope that str is 10 bytes long... */
    boolean neg = (num < 0);

    if (neg)
	num = -num;

    str[--i] = 0;
    do{
	str[--i] = (num % 10) + '0';
	num /= 10;
    }while (num > 0);
    if (neg)
	str[--i] = '-';
    moveleft(&str[i], str, 10 - i);
}

void printi(int num)
{
    char nb[10];
    int size;
    
    numtoa(nb,num);
    size = min(strlen(nb),COLS-curpos.x);
    if (size > 0) {
	nb[size] = 0;
	zwrite(nb, size);
	curpos.x += size;
    }
}

void println()
{
    zwrite("\r\n", 2);
    curpos.x = 0;
    curpos.y = min(curpos.y+1, LINES-1);
}

/* print a character out in a readable form --
 *    ^<x> for control-<x>
 *    spaces for <tab>
 *    normal for everything else
 */

static char hexdig[] = "0123456789ABCDEF";

/* format: put a displayable version of c into out */
/* FIXME: is c an unsigned char? */
int format(char *out, unsigned short c)
{
    if (c >= ' ' && c < '') {
    	out[0] = c;
    	return 1;
    }
    else if (c == '\t' && !list) {
	int i;
	int size;

	for (i = size = tabsize - (curpos.x % tabsize);i > 0;)
	    out[--i] = ' ';
	return size;
    }
    else if (c < 128) {
    	out[0] = '^';
    	out[1] = c^64;
    	return 2;
    }
    else {
	out[0] = '\\';
	out[1] = hexdig[(c>>4)&017];
	out[2] = hexdig[c&017];
	return 3;
    }
}

void printch(char c)
{
    int size;
    char buf[MAXCOLS];

    size = min(format(buf,c),COLS-curpos.x);
    if (size > 0) {
	buf[size] = 0;
	zwrite(buf, size);
	curpos.x += size;
    }
}

void prints(char *s)
{
    int size, oxp = curpos.x;
    char buf[MAXCOLS + 1];
    int bi = 0;

    while (*s && curpos.x < COLS) {
    	size = format(&buf[bi], *s++);
    	bi += size;
    	curpos.x += size;
    }
    size = min(bi,COLS-oxp);
    if (size > 0) {
	buf[size] = 0;
	zwrite(buf, size);
    }
}

void writeline(int y, int x, int start)
{
    int endd,oxp;
    int size;
    char buf[MAXCOLS+1];
    int bi = 0;
    
    endd = fseekeol(start);
    if (start==0 || core[start-1] == EOL)
	levee_mvcur(y, 0);
    else
	levee_mvcur(y, x);

    oxp = curpos.x;

    while (start < endd && curpos.x < COLS) {
    	size = format(&buf[bi],core[start++]);
    	bi += size;
    	curpos.x += size;
    }
    if (list) {
    	buf[bi++] = '$';
    	curpos.x++;
    }
    size = min(bi,COLS-oxp);
    if (size > 0) {
	buf[size] = 0;
	zwrite(buf, size);
    }
    if (curpos.x < COLS)
	strput(CE);
}

/* redraw && refresh the screen */
void levee_refresh(int y, int x, int start, int endd, boolean rest)
{
    while (start <= endd) {
	writeline(y, x, start);
	start = 1 + fseekeol(start);
	y++;
	x = 0;
    }
    if (rest && start >= bufmax)
	while (y < LINES - 1) { /* fill screen with ~ */
	    levee_mvcur(y, 0);
	    printch('~'); strput(CE);
	    y++;
	}
}

/* redraw everything */

void redisplay(boolean flag)
{
    if (flag)
	clrprompt();
    levee_refresh(0, 0, ptop, pend, TRUE);
}
    
void scrollback(int curr)
{
    levee_mvcur(0, 0);		/* move to the top line */
    do {
	ptop = bseekeol(ptop-1);
	strput(UpS);
	writeline(0, 0, ptop);
    } while (ptop > curr);
    setend();
}

void scrollforward(int curr)
{
    do {
	writeline(LINES-1, 0, pend+1);
	zwrite("\n", 1);
	pend = fseekeol(pend+1);
	ptop = fseekeol(ptop)+1;
    } while (pend < curr);
}

/* find if the number of lines between top && bottom is less than dofscroll */

boolean ok_to_scroll(int top, int bottom)
{
    int nl, i;
    
    nl = dofscroll;
    i = top;

    do
	i += 1+scan(bufmax-i,'=',EOL, &core[i]);
    while (--nl > 0 && i < bottom);

    return(nl > 0);
}

void clrprompt()
{
    levee_mvcur(LINES - 1,0);
    strput(CE);
}

void prompt(boolean toot, char *s)
{
    if (toot)
	error();
    clrprompt();
    prints(s);
}
