/*
 * multilingual support for nvi
 * Copyright(c) 1996, 1997 by Jun-ichiro Itoh.  All rights reserved.
 * Author contact: <itojun@mt.cs.keio.ac.jp>
 * $Id: multi_sjis.c,v 1.1.2.1 1997/09/23 00:43:52 itojun Exp $
 *
 * Freely redistributable, reusable, unless otherwise noted in accompanying
 * document. (for example, redistribution is prohibited during alpha-test
 * period)
 * Absolutely no warranty.
 *
 * The code is based on:
 *	jelvis japanization patch by Jun-ichiro Itoh
 *	nvi 1.03 japanization patch by Yoshitaka Tokugawa <toku@dit.co.jp>
 */

#include "config.h"

#ifdef MULTIBYTE

#include <sys/types.h>
#include <sys/queue.h>

#include <bitstring.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "../common/common.h"
#include "multibyte.h"

static void jistosjis __P((CHAR_T *, CHAR_T *));
static void sjistojis __P((CHAR_T *, CHAR_T *));

static void
jistosjis(dst, src)
	CHAR_T *dst;
	CHAR_T *src;
{
	CHAR_T high;
	CHAR_T low;
	
	high = src[0];
	low = src[1];
	if (high & 1)
		low += 0x1f;
	else
		low += 0x7d;
	if (low >= 0x7f)
		low++;
	high = (high - 0x21 >> 1) + 0x81;
	if (high > 0x9f)
		high += 0x40;
	dst[0] = high;
	dst[1] = low;
}

static void
sjistojis(dst, src)
	CHAR_T *dst;
	CHAR_T *src;
{
	CHAR_T high;
	CHAR_T low;
	
	high = src[0];
	low = src[1];
	if (high <= 0x9f)
		high -= 0x71;
	else
		high -= 0xb1;
	high = high * 2 + 1;
	if (low > 0x7f)
		low--;
	if (low >= 0x9e) {
		low -= 0x7d;
		high++;
	} else
		low -= 0x1f;
	dst[0] = high;
	dst[1] = low;
}

int
sjis_to_int(sp, e, dst, pdlen, src, slen, pflags, state)
	SCR *sp;
	ENCODING const *e;
	CHAR_T *dst;
	size_t *pdlen;
	CHAR_T *src;
	size_t slen;
	int *pflags;
	ISO2022STATE *state;
{
	CHAR_T *p;
	CHAR_T *q;
	size_t i;
	u_short ch;

	q = dst;
	p = src;
	if (pflags)
		FL_INIT(*pflags, 0);

	for (i = 0; i < slen; i++) {
		if (i + 1 < slen && issjiskanji1(p[0]) && issjiskanji2(p[1])) {
			/*
			 * supposedly sjis.
			 */
			if (dst) {
				q[0] = CS_JISX0208_1983;
				sjistojis(&q[1], &p[0]);
				q += 3;
				p += 2;
			} else {
				q += 3;
				p += 2;
			}
			i++;		/*plus one at for loop*/
			if (pflags)
				FL_SET(*pflags, MB_MULTIBYTE);
			continue;
		}
		if (isjishankana(p[0])) {
			if (dst) {
				*q++ = CS_JISX0201_RIGHT;
				*q++ = *p++ & 0x7f;
			} else {
				q += 2;
				p += 1;
			}
			if (pflags)
				FL_SET(*pflags, MB_MULTIBYTE);
			continue;
		}

		/*
		 * ASCII and other binary chars.
		 */
		if (p[0] & 0x80) {
			if (dst) {
				*q++ = CS_RAW0 + v_key_len(sp, p[0]);
				*q++ = *p++ & 0x7f;
			} else {
				q += 2;
				p++;
			}
			if (pflags)
				FL_SET(*pflags, MB_MULTIBYTE|MB_RAW);
			continue;
		}

		if (dst)
			*q++ = *p++;
		else {
			q++;
			p++;
		}
	}

	if (pdlen)
		*pdlen = q - dst;
	return 0;
}

int
int_to_sjis(sp, e, dst, pdlen, src, slen, pflags, state)
	SCR *sp;
	ENCODING const *e;
	CHAR_T *dst;
	size_t *pdlen;
	CHAR_T *src;
	size_t slen;
	int *pflags;
	ISO2022STATE *state;
{
	CHAR_T *p;
	CHAR_T *q;
	size_t i;
	u_short ch;

	if (pflags)
		FL_INIT(*pflags, 0);
	q = dst;
	p = src;

	for (i = 0; i < slen; i++) {
		if (CS_RAW(p[0])) {
			if (dst) {
				*q++ = p[1] | 0x80;
				p += 2;
			} else {
				q++;
				p += 2;
			}
			i++;		/*plus one at for loop*/
			if (pflags)
				FL_SET(*pflags, MB_MULTIBYTE);
			continue;
		}
		if (p[0] == CS_JISX0208_1990 || p[0] == CS_JISX0208_1983
		 || p[0] == CS_JISX0208_1978) {
			/*
			 * If we're using sjis, there's no distinction
			 * between JIS0208 variants.
			 */
			if (dst) {
				jistosjis(&q[0], &p[1]);
				q += 2;
				p += 3;
			} else {
				q += 2;
				p += 3;
			}
			i += 2;		/*plus one at for loop*/
			if (pflags)
				FL_SET(*pflags, MB_MULTIBYTE);
			continue;
		}
		if (p[0] == CS_JISX0201_RIGHT) {
			if (dst) {
				p++;
				*q++ = 0x80 | *p++;
			} else {
				q += 1;
				p += 2;
			}
			i += 1;		/*plus one at for loop*/
			if (pflags)
				FL_SET(*pflags, MB_MULTIBYTE);
			continue;
		}
		if (p[0] & 0x80) {
			/*
			 * Unknown encoding.  Simply skip them.
			 */
			int t;

			t = charset(p[0]).blen;
			p += t;
			i += t - 1;		/*plus one at for loop*/
			if (pflags)
				FL_SET(*pflags, MB_MULTIBYTE|MB_SKIP);
			continue;
		}

		if (dst)
			*q++ = *p++;
		else {
			q++;
			p++;
		}
	}

	if (pdlen)
		*pdlen = q - dst;
	return 0;
}

void
sjis_keyinput(sp, e, kbuf, kbuflen, intbuf, pintbuflen, pstate, pconsumed)
	SCR *sp;
	ENCODING const *e;
	CHAR_T *kbuf;
	size_t kbuflen;
	CHAR_T *intbuf;
	size_t *pintbuflen;
	int *pstate;
	size_t *pconsumed;
{
	size_t i;
	size_t j;
	size_t consumed;
	u_short ch;

	i = j = consumed = 0;
	while (i < kbuflen) {
		/*
		 * Kanji cases.
		 */
		if (i + 1 < kbuflen
		 && issjiskanji1(kbuf[i]) && issjiskanji2(kbuf[i + 1])) {
			intbuf[j] = CS_JISX0208_1983;
			sjistojis(&intbuf[j + 1], &kbuf[i]);
			j += 3;
			i += 2;
			*pstate = 0;
			consumed += 2;
		} else if (i + 1 == kbuflen && issjiskanji1(kbuf[i])) {
			/*
			 * Only the firstbyte is available.
			 * Keep this kanji char till next round.
			 */
			i++;
			*pstate = 1;
		}

		/*
		 * Non-kanji cases.
		 */
		else if (kbuf[i] & 0x80) {
			intbuf[j++] = CS_RAW0 + v_key_len(sp, kbuf[i]);
			intbuf[j++] = kbuf[i] & 0x7f;
			i++;
			*pstate = 0;
			consumed++;
		} else {
			intbuf[j++] = kbuf[i++];
			*pstate = 0;
			consumed++;
		}
	}
	*pintbuflen = j;
	*pconsumed = consumed;
}

int
sjis_display(sp, e, name, p)
	SCR *sp;
	ENCODING const *e;
	CHAR_T *name;
	CHAR_T *p;
{
	u_short ch;

	if (p[0] == CS_JISX0208_1990 || p[0] == CS_JISX0208_1983
	 || p[0] == CS_JISX0208_1978) {
		jistosjis(&name[0], &p[1]);
		name[2] = '\0';
		return 1;
	}
	if (p[0] == CS_JISX0201_RIGHT) {
		name[0] = p[1] | 0x80;
		name[1] = '\0';
		return 1;
	}
	return 0;
}
#endif /*MULTIBYTE*/
