/************************************************************************/
/*									*/
/*  Read an RTF text file into a BufferDocument				*/
/*									*/
/************************************************************************/

#   include	"config.h"

#   include	<stdlib.h>
#   include	<string.h>
#   include	<stdio.h>
#   include	<ctype.h>

#   include	<debugon.h>

#   include	<psFont.h>
#   include	"docRtf.h"

/************************************************************************/
/*									*/
/*  Forward declarations.						*/
/*									*/
/************************************************************************/

static int docRtfReadPntext(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	);
static int docRtfSkipPn(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	);
static int docRtfReadListtext(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	);
static int docRtfReadInfo(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	);

/************************************************************************/
/*									*/
/*  Consume a font table.						*/
/*									*/
/************************************************************************/

static int docRtfFontName(	RtfReadingContext *	rrc,
				const unsigned char *	name,
				int			len	)
    {
    DocumentFont *	df;

    static char *	scratch;
    char *		fresh;

    fresh= (char *)realloc( scratch, len+ 1 );
    if  ( ! fresh )
	{ XDEB(fresh); return -1;	}
    scratch= fresh;
    strncpy( scratch, (const char *)name, len )[len]= '\0';

    if  ( len > 0 && scratch[len-1] == ';' )
	{ scratch[len-1]= '\0';	}

    if  ( ! scratch[0] )
	{ return 0;	}

    df= docInsertFont( &(rrc->rrcBd->bdFontList),
		    rrc->rrcCurrentFont.dfDocFamilyNumber,
		    rrc->rrcCurrentFont.dfFamilyStyle, scratch );

    if  ( ! df )

	{ SSDEB(rrc->rrcCurrentFont.dfFamilyStyle,scratch); return -1; }

    df->dfCharset= rrc->rrcCurrentFont.dfCharset;
    df->dfPitch= rrc->rrcCurrentFont.dfPitch;

    return 0;
    }

static int docRtfFontNumber(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    docCleanFont( &(rrc->rrcCurrentFont) );
    docInitFont( &(rrc->rrcCurrentFont) );

    rrc->rrcCurrentFont.dfDocFamilyNumber= arg;

    return 0;
    }

static int docRtfFontFamily(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    rrc->rrcCurrentFont.dfFamilyStyle= strdup( rcw->rcwWord );

    if  ( ! rrc->rrcCurrentFont.dfFamilyStyle )
	{ XDEB(rrc->rrcCurrentFont.dfFamilyStyle); return -1;	}

    return 0;
    }

static int docRtfFontProperty(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    switch( rcw->rcwId )
	{
	case RTFidFCHARSET:
	    rrc->rrcCurrentFont.dfCharset= arg;
	    break;

	case RTFidFPRQ:
	    rrc->rrcCurrentFont.dfPitch= arg;
	    break;

	default:
	    SLDEB(rcw->rcwWord,arg);
	}

    return 0;
    }

static RtfControlWord	docRtfFontGroupWords[]=
    {
	{ "fnil",	RTFidFNIL,	DOClevDOC, docRtfFontFamily, },
	{ "froman",	RTFidFROMAN,	DOClevDOC, docRtfFontFamily, },
	{ "fswiss",	RTFidFSWISS,	DOClevDOC, docRtfFontFamily, },
	{ "fmodern",	RTFidFMODERN,	DOClevDOC, docRtfFontFamily, },
	{ "fscript",	RTFidFSCRIPT,	DOClevDOC, docRtfFontFamily, },
	{ "fdecor",	RTFidFDECOR,	DOClevDOC, docRtfFontFamily, },
	{ "ftech",	RTFidFTECH,	DOClevDOC, docRtfFontFamily, },

	{ "fcharset",	RTFidFCHARSET,	DOClevDOC, docRtfFontProperty, },
	{ "fprq",	RTFidFPRQ,	DOClevDOC, docRtfFontProperty, },

	{ 0, 0, 0 }
    };

static int docRtfFontGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    docCleanFont( &(rrc->rrcCurrentFont) );
    docInitFont( &(rrc->rrcCurrentFont) );

    rrc->rrcCurrentFont.dfDocFamilyNumber= arg;

    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfFontGroupWords, docRtfEmptyTable,
				    docRtfFontName ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    docCleanFont( &(rrc->rrcCurrentFont) );
    docInitFont( &(rrc->rrcCurrentFont) );

    return 0;
    }

static RtfControlWord	docRtfFontTableWords[]=
    {
	{ "f",		RTFidF,		DOClevDOC, docRtfFontNumber, },

	{ "fnil",	RTFidFNIL,	DOClevDOC, docRtfFontFamily, },
	{ "froman",	RTFidFROMAN,	DOClevDOC, docRtfFontFamily, },
	{ "fswiss",	RTFidFSWISS,	DOClevDOC, docRtfFontFamily, },
	{ "fmodern",	RTFidFMODERN,	DOClevDOC, docRtfFontFamily, },
	{ "fscript",	RTFidFSCRIPT,	DOClevDOC, docRtfFontFamily, },
	{ "fdecor",	RTFidFDECOR,	DOClevDOC, docRtfFontFamily, },
	{ "ftech",	RTFidFTECH,	DOClevDOC, docRtfFontFamily, },

	{ "fcharset",	RTFidFCHARSET,	DOClevDOC, docRtfFontProperty, },
	{ "fprq",	RTFidFPRQ,	DOClevDOC, docRtfFontProperty, },

	{ 0, 0, 0 }
    };

static RtfControlWord	docRtfFontTableGroups[]=
    {
	{ "f",		RTFidF,		DOClevDOC,  docRtfFontGroup, },

	{ 0, 0, 0 }
    };

static int docRtfFontTable(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfFontTableWords, docRtfFontTableGroups,
				docRtfFontName ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Read a color table.							*/
/*									*/
/************************************************************************/
static int docRtfSaveColor(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len )
    {
    RGB8Color *	fresh;

    if  ( ! rrc->rrcGotComponent )
	{
	DocumentProperties *	dp= &(rrc->rrcBd->bdProperties);

	if  ( dp->dpDefaultColor < 0 )
	    { dp->dpDefaultColor= rrc->rrcBd->bdColorCount;	}
	}

    fresh= (RGB8Color *)realloc( rrc->rrcBd->bdColors,
		( rrc->rrcBd->bdColorCount+ 1) * sizeof( RGB8Color ) );
    if  ( ! fresh )
	{ LXDEB(rrc->rrcBd->bdColorCount,fresh); return -1;	}

    rrc->rrcBd->bdColors= fresh;
    fresh[rrc->rrcBd->bdColorCount++]= rrc->rrcColor;

    rrc->rrcGotComponent= 0;
    return 0;
    }

static int docRtfColorComp(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    switch( rcw->rcwId )
	{
	case	RTFidRED:
	    rrc->rrcGotComponent= 1;
	    rrc->rrcColor.rgb8Red= arg;
	    break;
	case	RTFidGREEN:
	    rrc->rrcGotComponent= 1;
	    rrc->rrcColor.rgb8Green= arg;
	    break;
	case	RTFidBLUE:
	    rrc->rrcGotComponent= 1;
	    rrc->rrcColor.rgb8Blue= arg;
	    break;
	default:
	    /* SLDEB(rcw->rcwWord,arg); */
	    break;
	}

    return 0;
    }

static RtfControlWord	docRtfColorTableWords[]=
    {
	{ "red",	RTFidRED,	DOClevDOC, docRtfColorComp, },
	{ "green",	RTFidGREEN,	DOClevDOC, docRtfColorComp, },
	{ "blue",	RTFidBLUE,	DOClevDOC, docRtfColorComp, },
	{ 0, 0, 0 }
    };

static int docRtfColorTable(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    rrc->rrcGotComponent= 0;

    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfColorTableWords, docRtfEmptyTable,
				    docRtfSaveColor ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Consume the 'stylesheet' of a document				*/
/*									*/
/************************************************************************/

static int docRtfCharacterStyle(	SimpleInputStream *	sis,
					const RtfControlWord *	rcw,
					int			arg,
					RtfReadingContext *	rrc	);

static int docRtfStyleName(	RtfReadingContext *	rrc,
				const unsigned char *	name,
				int			len	)
    {
    DocumentStyle *	ds;

    static char *	scratch;
    char *		fresh;

    fresh= (char *)realloc( scratch, len+ 1 );
    if  ( ! fresh )
	{ XDEB(fresh); return -1;	}
    scratch= fresh;
    strncpy( scratch, (const char *)name, len )[len]= '\0';

    if  ( len > 0 && scratch[len-1] == ';' )
	{ scratch[len-1]= '\0';	}

    ds= docInsertStyle( rrc->rrcBd, rrc->rrcStyleNumber, scratch );
    if  ( ! ds )
	{ XDEB(ds); return -1;	}

    docCopyParagraphProperties( &(ds->dsParagraphProperties),
					    &(rrc->rrcParagraphProperties) );
    ds->dsTextAttribute= rrc->rrcCurrentTextAttribute;

    rrc->rrcStyleNumber= 0;
    rrc->rrcStyleBasedOn= 0;
    rrc->rrcStyleNext= 0;

    docCleanParagraphProperties( &(rrc->rrcParagraphProperties) );
    docInitParagraphProperties( &(rrc->rrcParagraphProperties) );
    docInitTextAttribute( &(rrc->rrcCurrentTextAttribute) );
    rrc->rrcInDeletedText= 0;

    return 0;
    }

static int docRtfStyleProperty(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    switch( rcw->rcwId )
	{
	case RTFidSBASEDON:
	    rrc->rrcStyleBasedOn= arg;
	    break;
	case RTFidSNEXT:
	    rrc->rrcStyleNext= arg;
	    break;
	case RTFidS:
	    rrc->rrcStyleNumber= arg;
	    break;
	case RTFidSAUTOUPD:
	    break;
	case RTFidCS:
	    break;
	case RTFidADDITIVE:
	    break;

	default:
	    SDEB(rcw->rcwWord); break;
	}

    return 0;
    }

static RtfControlWord	docRtfStylesheetGroups[]=
    {
	{ "cs",		RTFidCS,	DOClevDOC, docRtfCharacterStyle, },

	{ 0, 0, 0 }
    };

static RtfControlWord	docRtfStylesheetWords[]=
    {
	{ "sbasedon",	RTFidSBASEDON,	DOClevDOC, docRtfStyleProperty, },
	{ "snext",	RTFidSNEXT,	DOClevDOC, docRtfStyleProperty, },
	{ "s",		RTFidS,		DOClevDOC, docRtfStyleProperty, },
	{ "sautoupd",	RTFidSAUTOUPD,	DOClevDOC, docRtfStyleProperty, },
	{ "additive",	RTFidADDITIVE,	DOClevDOC, docRtfStyleProperty, },
	{ "cs",		RTFidCS,	DOClevDOC, docRtfStyleProperty, },

	{ 0, 0, 0 }
    };

static int docRtfCharacterStyle(	SimpleInputStream *	sis,
					const RtfControlWord *	rcw,
					int			arg,
					RtfReadingContext *	rrc	)
    {
    rrc->rrcStyleNumber= arg;

    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfStylesheetWords, docRtfEmptyTable,
				    docRtfStyleName ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

static int docRtfStylesheet(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    rrc->rrcStyleNumber= 0;
    rrc->rrcStyleBasedOn= 0;
    rrc->rrcStyleNext= 0;

    docCleanParagraphProperties( &(rrc->rrcParagraphProperties) );
    docInitParagraphProperties( &(rrc->rrcParagraphProperties) );
    docInitTextAttribute( &(rrc->rrcCurrentTextAttribute) );
    rrc->rrcInDeletedText= 0;

    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfStylesheetWords, docRtfStylesheetGroups,
				docRtfStyleName ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Special actions for section properties.				*/
/*									*/
/************************************************************************/

static int docRtfSectProp(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    return 0;
    }

static int docRtfTextSpecial(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    switch( rcw->rcwId )
	{
	case	RTFidTAB:
	    if  ( docSaveTab( rrc->rrcBi, rrc->rrcCurrentTextAttribute ) )
		{ LDEB(1); return -1;	}
	    break;
	case	RTFidPAGE:
	case	RTFidCHFTN:
	case	RTFidU:
	    break;
	default:
	    SLDEB(rcw->rcwWord,rcw->rcwId);
	    break;
	}

    return 0;
    }

static int docRtfBeginBorder(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    docInitBorderProperties( &(rrc->rrcBorderProperties) );
    rrc->rrcBorderProperties.bpIsSet= 1;
    return 0;
    }

static int docRtfBrdrProperty(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    switch( rcw->rcwId )
	{
	case RTFidBRDRS:
	    rrc->rrcBorderProperties.bpThickness= 1;
	    break;
	case RTFidBRDRTH:
	    rrc->rrcBorderProperties.bpThickness= 2;
	    break;
	case RTFidBRDRW:
	    if  ( arg > 255 )
		{ LDEB(arg); arg= 255;	}

	    rrc->rrcBorderProperties.bpWidthTwips= arg;
	    break;
	case RTFidBRDRCF:
	    if  ( arg > 255 )
		{ LDEB(arg); arg= 255;	}

	    rrc->rrcBorderProperties.bpColor= arg;
	    break;
	case RTFidBRSP:
	    rrc->rrcBorderProperties.bpSpacingTwips= arg;
	    break;
	case RTFidBRDRSH:
	    rrc->rrcBorderProperties.bpStyle= DOCbsSHADOWED;
	    break;
	case RTFidBRDRDB:
	    rrc->rrcBorderProperties.bpStyle= DOCbsDOUBLE;
	    break;
	case RTFidBRDRDOT:
	    rrc->rrcBorderProperties.bpStyle= DOCbsDOT;
	    break;
	case RTFidBRDRDASH:
	    rrc->rrcBorderProperties.bpStyle= DOCbsDASH;
	    break;
	case RTFidBRDRHAIR:
	    rrc->rrcBorderProperties.bpStyle= DOCbsHAIR;
	    break;
	case RTFidBRDRDASHSM:
	    rrc->rrcBorderProperties.bpStyle= DOCbsDASHSM;
	    break;
	case RTFidBRDRDASHD:
	    rrc->rrcBorderProperties.bpStyle= DOCbsDASHD;
	    break;
	case RTFidBRDRDASHDD:
	    rrc->rrcBorderProperties.bpStyle= DOCbsDASHDD;
	    break;
	case RTFidBRDRTRIPLE:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTRIPLE;
	    break;
	case RTFidBRDRTNTHSG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTNTHSG;
	    break;
	case RTFidBRDRTHTNSG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTHTNSG;
	    break;
	case RTFidBRDRTNTHTNSG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTNTHTNSG;
	    break;
	case RTFidBRDRTNTHMG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTNTHMG;
	    break;
	case RTFidBRDRTHTNMG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTHTNMG;
	    break;
	case RTFidBRDRTNTHTNMG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTNTHTNMG;
	    break;
	case RTFidBRDRTNTHLG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTNTHLG;
	    break;
	case RTFidBRDRTHTNLG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTHTNLG;
	    break;
	case RTFidBRDRTNTHTNLG:
	    rrc->rrcBorderProperties.bpStyle= DOCbsTNTHTNLG;
	    break;
	case RTFidBRDRWAVY:
	    rrc->rrcBorderProperties.bpStyle= DOCbsWAVY;
	    break;
	case RTFidBRDRWAVYDB:
	    rrc->rrcBorderProperties.bpStyle= DOCbsWAVYDB;
	    break;
	case RTFidBRDRDASHDOTSTR:
	    rrc->rrcBorderProperties.bpStyle= DOCbsDASHDOTSTR;
	    break;
	case RTFidBRDREMBOSS:
	    rrc->rrcBorderProperties.bpStyle= DOCbsEMBOSS;
	    break;
	case RTFidBRDRENGRAVE:
	    rrc->rrcBorderProperties.bpStyle= DOCbsENGRAVE;
	    break;

	default:
	    SLDEB(rcw->rcwWord,rcw->rcwId);
	    break;
	}

    return 0;
    }

static int docRtfSetProperty(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    switch( rcw->rcwId )
	{
	case	RTFidMARGL:
	    rrc->rrcBd->bdGeometry.dgLeftMarginTwips= arg;
	    break;
	case	RTFidMARGT:
	    rrc->rrcBd->bdGeometry.dgTopMarginTwips= arg;
	    break;
	case	RTFidMARGR:
	    rrc->rrcBd->bdGeometry.dgRightMarginTwips= arg;
	    break;
	case	RTFidMARGB:
	    rrc->rrcBd->bdGeometry.dgBottomMarginTwips= arg;
	    break;
	case	RTFidPAPERW:
	    rrc->rrcBd->bdGeometry.dgPaperWideTwips= arg;
	    break;
	case	RTFidPAPERH:
	    rrc->rrcBd->bdGeometry.dgPaperHighTwips= arg;
	    break;
	case	RTFidDEFTAB:
	    rrc->rrcBd->bdProperties.dpTabIntervalTwips= arg;
	    break;
	case	RTFidDEFF:
	    rrc->rrcDefaultFont= arg;
	    rrc->rrcCurrentTextAttribute.taFontNumber= arg;
	    break;
	case	RTFidDEFLANG:
	case	RTFidDEFLANGFE:
	case	RTFidUC:
	    break;

	case RTFidNOWIDCTLPAR:
	case RTFidWIDCTLPAR:
	    break;

	case RTFidADJUSTRIGHT:
	case RTFidADJUSTLEFT:
	    break;
	default:
	    /*SLDEB(rcw->rcwWord,rcw->rcwId);*/
	    break;

	case RTFidANSICPG:
	    switch( arg )
		{
		case 1252:
		    rrc->rrcBd->bdItem.biDocAnsiCodePage= arg;
		    break;

		case 1250:
		    memcpy( rrc->rrcInputMapping, docWIN1250_to_ISO2, 256 );
		    rrc->rrcBd->bdItem.biDocAnsiCodePage= arg;
		    break;

		default:
		    SLDEB(rcw->rcwWord,arg); break;
		}
	}

    return 0;
    }

static int docRtfHierarchy(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {

    switch( rcw->rcwId )
	{
	case	RTFidSECT:
	    if  ( rrc->rrcSplitLevel < DOClevSECT )
		{
		if  ( docRtfAdjustLevel( rrc, DOClevPARA, DOClevTEXT ) )
		    { SDEB(rcw->rcwWord); return -1; }
		}

	    if  ( docRtfAdjustLevel( rrc, DOClevSECT- 1, DOClevTEXT ) )
		{ SDEB(rcw->rcwWord); return -1; }
	    rrc->rrcSplitLevel= DOClevSECT- 1;
	    break;

	case	RTFidROW:
	    if  ( rrc->rrcSplitLevel < DOClevROW )
		{
		if  ( docRtfAdjustLevel( rrc, DOClevPARA, DOClevTEXT ) )
		    { SDEB(rcw->rcwWord); return -1; }
		}

	    if  ( docRtfAdjustLevel( rrc, DOClevROW- 1, DOClevTEXT ) )
		{ SDEB(rcw->rcwWord); return -1; }
	    rrc->rrcSplitLevel= DOClevROW- 1;
	    break;

	case	RTFidCELL:
	    if  ( rrc->rrcSplitLevel < DOClevCELL )
		{
		if  ( docRtfAdjustLevel( rrc, DOClevPARA, DOClevTEXT ) )
		    { SDEB(rcw->rcwWord); return -1; }
		}

	    if  ( docRtfAdjustLevel( rrc, DOClevCELL- 1, DOClevTEXT ) )
		{ SDEB(rcw->rcwWord); return -1; }
	    rrc->rrcSplitLevel= DOClevCELL- 1;

	    break;

	case	RTFidPAR:
	    if  ( docRtfAdjustLevel( rrc, rcw->rcwLevel- 1, DOClevTEXT ) )
		{ SDEB(rcw->rcwWord); return -1; }
	    break;

	case	RTFidTROWD:
	    if  ( docRtfAdjustLevel( rrc, rcw->rcwLevel+ 1, DOClevTEXT ) )
		{ SDEB(rcw->rcwWord); return -1; }

	    docCleanRowProperties( &(rrc->rrcBi->biRowProperties) );
	    docInitRowProperties(  &(rrc->rrcBi->biRowProperties) );

	    docCleanRowProperties( &(rrc->rrcRowProperties) );
	    docInitRowProperties(  &(rrc->rrcRowProperties) );

	    docInitCellProperties( &(rrc->rrcCellProperties) );
	    break;

	case	RTFidLINE:
	    break;

	default:
	    SLDEB(rcw->rcwWord,rcw->rcwId); break;
	}

    return 0;
    }

/************************************************************************/
/*									*/
/*  Consume Headers and Footers.					*/
/*									*/
/************************************************************************/

int docRtfReadHeaderFooter(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    BufferItem **	pBi;
    BufferItem *	bi;

    BufferItem *	savedBi;
    int			savedLevel;

    int			res;

    savedBi= rrc->rrcBi;
    savedLevel= rrc->rrcLevel;
    if  ( ! savedBi )
	{ XDEB(savedBi); return -1;	}

    switch( rcw->rcwId )
	{
	case RTFidHEADER:
	    pBi= &(rrc->rrcBi->biSectHeader); break;
	case RTFidHEADERF:
	    pBi= &(rrc->rrcBi->biSectFirstPageHeader); break;
	case RTFidHEADERL:
	    pBi= &(rrc->rrcBi->biSectLeftPageHeader); break;
	case RTFidHEADERR:
	    pBi= &(rrc->rrcBi->biSectRightPageHeader); break;

	case RTFidFOOTER:
	    pBi= &(rrc->rrcBi->biSectFooter); break;
	case RTFidFOOTERF:
	    pBi= &(rrc->rrcBi->biSectFirstPageFooter); break;
	case RTFidFOOTERL:
	    pBi= &(rrc->rrcBi->biSectLeftPageFooter); break;
	case RTFidFOOTERR:
	    pBi= &(rrc->rrcBi->biSectRightPageFooter); break;
	default:
	    SDEB(rcw->rcwWord); return -1;
	}

    bi= *pBi;
    if  ( ! bi )
	{
	bi= (BufferItem *)malloc( sizeof(BufferItem) );
	if  ( ! bi )
	    { XDEB(bi); return -1;	}

	docInitItem( bi, (BufferItem *)0, 0, DOClevSECT );

	*pBi= bi;
	}

    rrc->rrcBi= bi;
    rrc->rrcLevel= DOClevSECT;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfDocumentGroups,
				docRtfTextParticule );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    rrc->rrcLevel= savedLevel;
    rrc->rrcBi= savedBi;

    return res;
    }

/************************************************************************/
/*									*/
/*  Control words to remember border properties.			*/
/*									*/
/************************************************************************/

static RtfControlWord	docRtfBorderWords[]=
    {
	{ "brdrcf",	RTFidBRDRCF,	DOClevPARA, docRtfBrdrProperty, },
	{ "brdrdash ",	RTFidBRDRDASH ,	DOClevPARA, docRtfBrdrProperty, },
	{ "brdrdot",	RTFidBRDRDOT,	DOClevPARA, docRtfBrdrProperty, },
	{ "brdrhair",	RTFidBRDRHAIR,	DOClevPARA, docRtfBrdrProperty, },
	{ "brdrsh",	RTFidBRDRSH,	DOClevPARA, docRtfBrdrProperty, },
	{ "brdrth",	RTFidBRDRTH,	DOClevPARA, docRtfBrdrProperty, },
	{ "brdrw",	RTFidBRDRW,	DOClevPARA, docRtfBrdrProperty, },
	{ "brsp",	RTFidBRSP,	DOClevPARA, docRtfBrdrProperty, },

	{ "brdrs",	RTFidBRDRS,	DOClevPARA, docRtfBrdrProperty, },
	{ "brdrdb",	RTFidBRDRDB,	DOClevPARA, docRtfBrdrProperty, },
	{ "brdrdot",	RTFidBRDRDOT,	DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrhair",	RTFidBRDRHAIR,	DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrdashsm",	RTFidBRDRDASHSM,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrdashd",	RTFidBRDRDASHD,	DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrdashdd",	RTFidBRDRDASHDD,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrtriple",	RTFidBRDRTRIPLE,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrtnthsg",	RTFidBRDRTNTHSG,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrthtnsg",	RTFidBRDRTHTNSG,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrtnthtnsg",
			RTFidBRDRTNTHTNSG,
					DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrtnthmg",	RTFidBRDRTNTHMG,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrthtnmg",	RTFidBRDRTHTNMG,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrtnthtnmg",
			RTFidBRDRTNTHTNMG,
					DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrtnthlg",	RTFidBRDRTNTHLG,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrthtnlg",	RTFidBRDRTHTNLG,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrtnthtnlg",
			RTFidBRDRTNTHTNLG,
					DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrwavy",	RTFidBRDRWAVY,	DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrwavydb",	RTFidBRDRWAVYDB,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrdashdotstr",
			RTFidBRDRDASHDOTSTR,
					DOClevPARA, docRtfBrdrProperty,	},
	{ "brdremboss",	RTFidBRDREMBOSS,DOClevPARA, docRtfBrdrProperty,	},
	{ "brdrengrave",RTFidBRDRENGRAVE,
					DOClevPARA, docRtfBrdrProperty,	},

	{ 0, 0, 0 }
    };

RtfControlWord	docRtfDocumentWords[]=
    {
				/****************************************/
				/*  Document Formatting properties.	*/
				/****************************************/
	{ "allprot",	RTFidALLPROT,	DOClevDOC, docRtfSetProperty, },
	{ "annotprot",	RTFidANNOTPROT,	DOClevDOC, docRtfSetProperty, },
	{ "brkfrm",	RTFidBRKFRM,	DOClevDOC, docRtfSetProperty, },
	{ "cvmme",	RTFidCVMME,	DOClevDOC, docRtfSetProperty, },
	{ "defformat",	RTFidDEFFORMAT,	DOClevDOC, docRtfSetProperty, },
	{ "deflang",	RTFidDEFLANG,	DOClevDOC, docRtfSetProperty, },
	{ "deftab",	RTFidDEFTAB,	DOClevDOC, docRtfSetProperty, },
	{ "doctemp",	RTFidDOCTEMP,	DOClevDOC, docRtfSetProperty, },
	{ "enddoc",	RTFidENDDOC,	DOClevDOC, docRtfSetProperty, },
	{ "endnotes",	RTFidENDNOTES,	DOClevDOC, docRtfSetProperty, },
	{ "facingp",	RTFidFACINGP,	DOClevDOC, docRtfSetProperty, },
	{ "fet",	RTFidFET,	DOClevDOC, docRtfSetProperty, },
	{ "fldalt",	RTFidFLDALT,	DOClevDOC, docRtfSetProperty, },
	{ "formdisp",	RTFidFORMDISP,	DOClevDOC, docRtfSetProperty, },
	{ "formprot",	RTFidFORMPROT,	DOClevDOC, docRtfSetProperty, },
	{ "formshade",	RTFidFORMSHADE,	DOClevDOC, docRtfSetProperty, },
	{ "fracwidth",	RTFidFRACWIDTH,	DOClevDOC, docRtfSetProperty, },
	{ "gutter",	RTFidGUTTER,	DOClevDOC,  docRtfSetProperty, },
	{ "hyphauto",	RTFidHYPHAUTO,	DOClevDOC, docRtfSetProperty, },
	{ "hyphcaps",	RTFidHYPHCAPS,	DOClevDOC, docRtfSetProperty, },
	{ "hyphconsec",	RTFidHYPHCONSEC, DOClevDOC, docRtfSetProperty, },
	{ "hyphhotz",	RTFidHYPHHOTZ,	DOClevDOC, docRtfSetProperty, },
	{ "landscape",	RTFidLANDSCAPE,	DOClevDOC, docRtfSetProperty, },
	{ "linestart",	RTFidLINESTART,	DOClevDOC,  docRtfSetProperty, },
	{ "linkstyles",	RTFidLINKSTYLES, DOClevDOC, docRtfSetProperty, },
	{ "ltrdoc",	RTFidLTRDOC,	DOClevDOC, docRtfSetProperty, },
	{ "makebackup",	RTFidMAKEBACKUP, DOClevDOC, docRtfSetProperty, },
	{ "margl",	RTFidMARGL,	DOClevANY, docRtfSetProperty,	},
	{ "margr",	RTFidMARGR,	DOClevANY, docRtfSetProperty,	},
	{ "margt",	RTFidMARGT,	DOClevANY, docRtfSetProperty,	},
	{ "margb",	RTFidMARGB,	DOClevANY, docRtfSetProperty,	},
	{ "margmirror",	RTFidMARGMIRROR, DOClevDOC, docRtfSetProperty, },
	{ "nocolbal",	RTFidNOCOLBAL,	DOClevDOC, docRtfSetProperty, },
	{ "noextrasprl",RTFidNOEXTRASPRL, DOClevDOC, docRtfSetProperty, },
	{ "notabind",	RTFidNOTABIND,	DOClevDOC, docRtfSetProperty, },
	{ "otblrul",	RTFidOTBLRUL,	DOClevDOC, docRtfSetProperty, },
	{ "paperh",	RTFidPAPERH,	DOClevANY, docRtfSetProperty, },
	{ "paperw",	RTFidPAPERW,	DOClevANY, docRtfSetProperty, },
	{ "psz",	RTFidPSZ,	DOClevANY, docRtfSetProperty, },
	{ "pgnstart",	RTFidPGNSTART,	DOClevDOC, docRtfSetProperty, },
	{ "prcolbl",	RTFidPRCOLBL,	DOClevDOC, docRtfSetProperty, },
	{ "printdata",	RTFidPRINTDATA,	DOClevDOC, docRtfSetProperty, },
	{ "psover",	RTFidPSOVER,	DOClevDOC, docRtfSetProperty, },
	{ "revbar",	RTFidREVBAR,	DOClevDOC, docRtfSetProperty, },
	{ "revisions",	RTFidREVISIONS,	DOClevDOC, docRtfSetProperty, },
	{ "revprop",	RTFidREVPROP,	DOClevDOC, docRtfSetProperty, },
	{ "revprot",	RTFidREVPROT,	DOClevDOC, docRtfSetProperty, },
	{ "rtldoc",	RTFidRTLDOC,	DOClevDOC, docRtfSetProperty, },
	{ "sprslnsp",	RTFidSPRSLNSP,	DOClevDOC, docRtfSetProperty, },
	{ "sprsspbf",	RTFidSPRSSPBF,	DOClevDOC, docRtfSetProperty, },
	{ "sprstsp",	RTFidSPRSTSP,	DOClevDOC, docRtfSetProperty, },
	{ "subfontbysize", RTFidSUBFONTBYSIZE,	DOClevDOC, docRtfSetProperty, },
	{ "swpbdr",	RTFidSWPBDR,	DOClevDOC, docRtfSetProperty, },
	{ "transmf",	RTFidTRANSMF,	DOClevDOC, docRtfSetProperty, },
	{ "ight",	RTFidIGHT,	DOClevDOC, docRtfSetProperty, },
	{ "widowctrl",	RTFidWIDOWCTRL,	DOClevDOC, docRtfSetProperty, },
	{ "wraptrsp",	RTFidWRAPTRSP,	DOClevDOC, docRtfSetProperty, },
	{ "viewzk",	RTFidVIEWZK,	DOClevDOC, docRtfSetProperty, },
					/********************************/
					/*  Document Properties.	*/
					/********************************/
	{ "ansi",	RTFidANSI,	DOClevDOC, docRtfSetProperty,	},
	{ "mac",	RTFidMAC,	DOClevDOC, docRtfSetProperty,	},
	{ "pc",		RTFidPC,	DOClevDOC, docRtfSetProperty,	},
	{ "ansicpg",	RTFidANSICPG,	DOClevDOC, docRtfSetProperty,	},

	{ "viewkind",	RTFidVIEWKIND,	DOClevDOC, docRtfSetProperty, },
	{ "viewscale",	RTFidVIEWSCALE,	DOClevDOC, docRtfSetProperty, },
	{ "pgbrdrhead",	RTFidPGBRDRHEAD,DOClevDOC, docRtfSetProperty, },
	{ "pgbrdrfoot",	RTFidPGBRDRFOOT,DOClevDOC, docRtfSetProperty, },
	{ "lytprtmet",	RTFidLYTPRTMET,	DOClevDOC, docRtfSetProperty, },

	{ "ogutter",	RTFidOGUTTER,	DOClevDOC, docRtfSetProperty,	},
	{ "deff",	RTFidDEFF,	DOClevDOC, docRtfSetProperty,	},
	{ "deflangfe",	RTFidDEFLANGFE,	DOClevDOC, docRtfSetProperty,	},
	{ "uc",		RTFidUC,	DOClevDOC, docRtfSetProperty,	},
				/****************************************/
				/*  Section Formatting Properties	*/
				/****************************************/
	{ "binfsxn",	RTFidBINFSXN,	DOClevSECT,   docRtfSectProp, },
	{ "binsxn",	RTFidBINSXN,	DOClevSECT,   docRtfSectProp, },
	{ "colno",	RTFidCOLNO,	DOClevSECT,   docRtfSectProp, },
	{ "cols",	RTFidCOLS,	DOClevSECT,   docRtfSectProp, },
	{ "colsr",	RTFidCOLSR,	DOClevSECT,   docRtfSectProp, },
	{ "colsx",	RTFidCOLSX,	DOClevSECT,   docRtfSectProp, },
	{ "colw",	RTFidCOLW,	DOClevSECT,   docRtfSectProp, },
	{ "ds",		RTFidDS,	DOClevSECT,   docRtfSectProp, },
	{ "endnhere",	RTFidENDNHERE,	DOClevSECT,   docRtfSectProp, },
	{ "footery",	RTFidFOOTERY,	DOClevSECT,   docRtfSectProp, },
	{ "guttersxn",	RTFidGUTTERSXN,	DOClevSECT,   docRtfSectProp, },
	{ "headery",	RTFidHEADERY,	DOClevSECT,   docRtfSectProp, },
	{ "linebetcol",	RTFidLINEBETCOL, DOClevSECT,  docRtfSectProp, },
	{ "linecont",	RTFidLINECONT,	DOClevSECT,   docRtfSectProp, },
	{ "linemod",	RTFidLINEMOD,	DOClevSECT,   docRtfSectProp, },
	{ "lineppage",	RTFidLINEPPAGE,	DOClevSECT,   docRtfSectProp, },
	{ "linerestart",RTFidLINERESTART, DOClevSECT, docRtfSectProp, },
	{ "linestarts",	RTFidLINESTARTS, DOClevSECT,  docRtfSectProp, },
	{ "linex",	RTFidLINEX,	DOClevANY,    docRtfSectProp, },
	{ "lndscpsxn",	RTFidLNDSCPSXN,	DOClevSECT,   docRtfSectProp, },
	{ "ltrsect",	RTFidLTRSECT,	DOClevSECT,   docRtfSectProp, },
	{ "margbsxn",	RTFidMARGBSXN,	DOClevSECT,   docRtfSectProp, },
	{ "marglsxn",	RTFidMARGLSXN,	DOClevSECT,   docRtfSectProp, },
	{ "margrsxn",	RTFidMARGRSXN,	DOClevSECT,   docRtfSectProp, },
	{ "margtsxn",	RTFidMARGTSXN,	DOClevSECT,   docRtfSectProp, },
	{ "pghsxn",	RTFidPGHSXN,	DOClevSECT,   docRtfSectProp, },
	{ "pgncont",	RTFidPGNCONT,	DOClevSECT,   docRtfSectProp, },
	{ "pgndec",	RTFidPGNDEC,	DOClevSECT,   docRtfSectProp, },
	{ "pgnhn",	RTFidPGNHN,	DOClevSECT,   docRtfSectProp, },
	{ "pgnhnsc",	RTFidPGNHNSC,	DOClevSECT,   docRtfSectProp, },
	{ "pgnhnsh",	RTFidPGNHNSH,	DOClevSECT,   docRtfSectProp, },
	{ "pgnhnsm",	RTFidPGNHNSM,	DOClevSECT,   docRtfSectProp, },
	{ "pgnhnsn",	RTFidPGNHNSN,	DOClevSECT,   docRtfSectProp, },
	{ "pgnhnsp",	RTFidPGNHNSP,	DOClevSECT,   docRtfSectProp, },
	{ "pgnlcltr",	RTFidPGNLCLTR,	DOClevSECT,   docRtfSectProp, },
	{ "pgnlcrm",	RTFidPGNLCRM,	DOClevSECT,   docRtfSectProp, },
	{ "pgnrestart",	RTFidPGNRESTART, DOClevSECT,  docRtfSectProp, },
	{ "pgnstarts",	RTFidPGNSTARTS,	DOClevSECT,   docRtfSectProp, },
	{ "pgnucltr",	RTFidPGNUCLTR,	DOClevSECT,   docRtfSectProp, },
	{ "pgnucrm",	RTFidPGNUCRM,	DOClevSECT,   docRtfSectProp, },
	{ "pgnx",	RTFidPGNX,	DOClevSECT,   docRtfSectProp, },
	{ "pgny",	RTFidPGNY,	DOClevSECT,   docRtfSectProp, },
	{ "pgwsxn",	RTFidPGWSXN,	DOClevSECT,   docRtfSectProp, },
	{ "rtlsect",	RTFidRTLSECT,	DOClevSECT,   docRtfSectProp, },
	{ "sbkcol",	RTFidSBKCOL,	DOClevSECT,   docRtfSectProp, },
	{ "sbkeven",	RTFidSBKEVEN,	DOClevSECT,   docRtfSectProp, },
	{ "sbknone",	RTFidSBKNONE,	DOClevANY,    docRtfSectProp, },
	{ "sbkodd",	RTFidSBKODD,	DOClevSECT,   docRtfSectProp, },
	{ "sbkpage",	RTFidSBKPAGE,	DOClevSECT,   docRtfSectProp, },
	{ "sectd",	RTFidSECTD,	DOClevSECT,   docRtfSectProp, },
	{ "sectunlocked", RTFidSECTUNLOCKED, DOClevSECT, docRtfSectProp, },
	{ "titlepg",	RTFidTITLEPG,	DOClevSECT,   docRtfSectProp, },
	{ "vertalb",	RTFidVERTALB,	DOClevSECT,   docRtfSectProp, },
	{ "vertalc",	RTFidVERTALC,	DOClevSECT,   docRtfSectProp, },
	{ "vertalj",	RTFidVERTALJ,	DOClevSECT,   docRtfSectProp, },
	{ "vertalt",	RTFidVERTALT,	DOClevSECT,   docRtfSectProp, },
				/****************************************/
				/*  Paragraph Formatting Properties	*/
				/****************************************/
	{ "level",	RTFidLEVEL,	DOClevANY,  docRtfSetProperty, },
	{ "ltrpar",	RTFidLTRPAR,	DOClevANY,  docRtfSetProperty, },
	{ "noline",	RTFidNOLINE,	DOClevANY,  docRtfSetProperty, },
	{ "rtlpar",	RTFidRTLPAR,	DOClevANY,  docRtfSetProperty, },
	{ "sbys",	RTFidSBYS,	DOClevANY,  docRtfSetProperty, },
	{ "subdocument", RTFidSUBDOCUMENT, DOClevANY, docRtfSetProperty, },

	{ "jclisttab",	RTFidJCLISTTAB,	DOClevPARA,docRtfSetProperty, },
					/********************************/
					/*  Text Numeric.		*/
					/********************************/
	{ "fc",		RTFidFC,	DOClevANY,  docRtfRememberProperty, },
				/****************************************/
				/*  Special characters.			*/
				/****************************************/
	{ "bullet",	RTFidBULLET,	DOClevTEXT, docRtfTextSpecial, },
	{ "chatn",	RTFidCHATN,	DOClevTEXT, docRtfTextSpecial, },
	{ "chdate",	RTFidCHDATE,	DOClevTEXT, docRtfTextSpecial, },
	{ "chdpa",	RTFidCHDPA,	DOClevTEXT, docRtfTextSpecial, },
	{ "chdpl",	RTFidCHDPL,	DOClevTEXT, docRtfTextSpecial, },
	{ "chftn",	RTFidCHFTN,	DOClevTEXT, docRtfTextSpecial, },
	{ "chftnsep",	RTFidCHFTNSEP,	DOClevTEXT, docRtfTextSpecial, },
	{ "chftnsepc",	RTFidCHFTNSEPC,	DOClevTEXT, docRtfTextSpecial, },
	{ "chpgn",	RTFidCHPGN,	DOClevTEXT, docRtfTextSpecial, },
	{ "chtime",	RTFidCHTIME,	DOClevTEXT, docRtfTextSpecial, },
	{ "column",	RTFidCOLUMN,	DOClevTEXT, docRtfTextSpecial, },
	{ "emdash",	RTFidEMDASH,	DOClevTEXT, docRtfTextSpecial, },
	{ "emspace",	RTFidEMSPACE,	DOClevTEXT, docRtfTextSpecial, },
	{ "endash",	RTFidENDASH,	DOClevTEXT, docRtfTextSpecial, },
	{ "enspace",	RTFidENSPACE,	DOClevTEXT, docRtfTextSpecial, },
	{ "ldblquote",	RTFidLDBLQUOTE,	DOClevTEXT, docRtfTextSpecial, },
	{ "lquote",	RTFidLQUOTE,	DOClevTEXT, docRtfTextSpecial, },
	{ "ltrmark",	RTFidLTRMARK,	DOClevTEXT, docRtfTextSpecial, },
	{ "page",	RTFidPAGE,	DOClevTEXT, docRtfTextSpecial, },
	{ "rdblquote",	RTFidRDBLQUOTE,	DOClevTEXT, docRtfTextSpecial, },
	{ "rquote",	RTFidRQUOTE,	DOClevTEXT, docRtfTextSpecial, },
	{ "rtlmark",	RTFidRTLMARK,	DOClevTEXT, docRtfTextSpecial, },
	{ "sectnum",	RTFidSECTNUM,	DOClevTEXT, docRtfTextSpecial, },
	{ "softcol",	RTFidSOFTCOL,	DOClevTEXT, docRtfTextSpecial, },
	{ "softlheight",RTFidSOFTLHEIGHT, DOClevTEXT, docRtfTextSpecial, },
	{ "softline",	RTFidSOFTLINE,	DOClevTEXT, docRtfTextSpecial, },
	{ "softpage",	RTFidSOFTPAGE,	DOClevTEXT, docRtfTextSpecial, },
	{ "tab",	RTFidTAB,	DOClevTEXT, docRtfTextSpecial, },
	{ "zwj",	RTFidZWJ,	DOClevTEXT, docRtfTextSpecial, },
	{ "zwnj",	RTFidZWNJ,	DOClevTEXT, docRtfTextSpecial, },

	{ "u",		RTFidU,		DOClevTEXT, docRtfTextSpecial, },
				/****************************************/
				/*  Hierarchy tags.			*/
				/****************************************/
	{ "line",	RTFidLINE,	DOClevPARA, docRtfHierarchy, },
	{ "par",	RTFidPAR,	DOClevPARA, docRtfHierarchy, },
	{ "cell",	RTFidCELL,	DOClevANY,  docRtfHierarchy, },
	{ "row",	RTFidROW,	DOClevANY,  docRtfHierarchy, },
	{ "sect",	RTFidSECT,	DOClevANY,  docRtfHierarchy, },

	{ 0, 0, 0 }
    };

RtfControlWord	docRtfDocumentGroups[]=
    {
	{ "fonttbl",	RTFidFONTTBL,	DOClevDOC,  docRtfFontTable,  },
	{ "colortbl",	RTFidCOLORTBL,	DOClevDOC,  docRtfColorTable, },
	{ "stylesheet",	RTFidSTYLESHEET,DOClevDOC,  docRtfStylesheet,  },
	{ "info",	RTFidINFO,	DOClevDOC,  docRtfReadInfo,  },
	{ "pn",		RTFidPN,	DOClevANY,  docRtfSkipPn,   },
	{ "pntext",	RTFidPNTEXT,	DOClevPARA, docRtfReadPntext, },
	{ "pict",	RTFidPICT,	DOClevPARA, docRtfReadPict, },
	{ "nonshppict",	RTFidNONSHPPICT,DOClevPARA, docRtfReadNonshppict, },
	{ "shppict",	RTFidSHPPICT,	DOClevPARA, docRtfReadShppict, },
	{ "object",	RTFidOBJECT,	DOClevPARA, docRtfReadObject, },
	{ "field",	RTFidFIELD,	DOClevPARA, docRtfReadField, },
	{ "shp",	RTFidSHP,	DOClevPARA, docRtfReadShape, },
	{ "shpgrp",	RTFidSHPGRP,	DOClevPARA, docRtfReadShape, },
	{ "xe",		RTFidXE,	DOClevPARA, docRtfReadLookupEntry, },
	{ "tc",		RTFidTC,	DOClevPARA, docRtfReadLookupEntry, },
	{ "tcn",	RTFidTCN,	DOClevPARA, docRtfReadLookupEntry, },
				/****************************************/
				/*  Headers and Footers.		*/
				/****************************************/
	{ "footer",	RTFidFOOTER,	DOClevSECT, docRtfReadHeaderFooter, },
	{ "footerf",	RTFidFOOTERF,	DOClevSECT, docRtfReadHeaderFooter, },
	{ "footerl",	RTFidFOOTERL,	DOClevSECT, docRtfReadHeaderFooter, },
	{ "footerr",	RTFidFOOTERR,	DOClevSECT, docRtfReadHeaderFooter, },
	{ "header",	RTFidHEADER,	DOClevSECT, docRtfReadHeaderFooter, },
	{ "headerf",	RTFidHEADERF,	DOClevSECT, docRtfReadHeaderFooter, },
	{ "headerl",	RTFidHEADERL,	DOClevSECT, docRtfReadHeaderFooter, },
	{ "headerr",	RTFidHEADERR,	DOClevSECT, docRtfReadHeaderFooter, },
				/****************************************/
				/*  Lists.				*/
				/****************************************/
	{ "listtext",	RTFidLISTTEXT,	DOClevPARA, docRtfReadListtext, },
				/****************************************/
				/*  Annotation.				*/
				/****************************************/
	{ "chatn",	RTFidCHATN,	DOClevPARA, docRtfSkipGroup, },
				/****************************************/
				/*  Bookmarks				*/
				/****************************************/
	{ "bkmkstart",	RTFidBKMKSTART,	DOClevPARA, docRtfGetBookmark, },
	{ "bkmkend",	RTFidBKMKEND,	DOClevPARA, docRtfGetBookmark, },

	{ 0, 0, 0 }
    };

/************************************************************************/
/*									*/
/*  Read a document as a whole.						*/
/*									*/
/************************************************************************/

int docRtfTextParticule(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    {
    if  ( rrc->rrcInDeletedText )
	{ return 0;	}

    if  ( rrc->rrcJustAfterPntext )
	{
	if  ( docCopyParagraphProperties( &(rrc->rrcBi->biParaProperties),
					 &(rrc->rrcParagraphProperties) ) )
	    { LDEB(1); return -1;	}

	rrc->rrcJustAfterPntext= 0;
	}

    return docSaveParticules( rrc->rrcBi,
	    rrc->rrcCurrentTextAttribute, rrc->rrcInputMapping, text, len );
    }

static int docRtfReadDoc(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int		res;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfDocumentGroups,
				docRtfTextParticule );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

static int docRtfReadListtext(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int		res;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfDocumentGroups,
				docRtfTextParticule );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

static int docRtfIgnorePnGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int		res;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfEmptyTable, docRtfEmptyTable,
				docRtfIgnoreText );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

static RtfControlWord	docRtfPnGroups[]=
    {
	{ "pntxta",	RTFidPNTXTA,	DOClevANY, docRtfIgnorePnGroup, },
	{ "pntxtb",	RTFidPNTXTB,	DOClevANY, docRtfIgnorePnGroup, },

	{ 0, 0, 0 }
    };


static RtfControlWord	docRtfPnWords[]=
    {
				/****************************************/
				/*  Bullets and Numbering.		*/
				/*  (No destinations)			*/
				/****************************************/
	{ "pnacross",	RTFidPNACROSS,	DOClevANY, docRtfIgnoreWord, },
	{ "pnb",	RTFidPNB,	DOClevANY, docRtfIgnoreWord, },
	{ "pncaps",	RTFidPNCAPS,	DOClevANY, docRtfIgnoreWord, },
	{ "pncard",	RTFidPNCARD,	DOClevANY, docRtfIgnoreWord, },
	{ "pncf",	RTFidPNCF,	DOClevANY, docRtfIgnoreWord, },
	{ "pndec",	RTFidPNDEC,	DOClevANY, docRtfIgnoreWord, },
	{ "pnf",	RTFidPNF,	DOClevANY, docRtfIgnoreWord, },
	{ "pnfs",	RTFidPNFS,	DOClevANY, docRtfIgnoreWord, },
	{ "pnhang",	RTFidPNHANG,	DOClevANY, docRtfIgnoreWord, },
	{ "pni",	RTFidPNI,	DOClevANY, docRtfIgnoreWord, },
	{ "pnindent",	RTFidPNINDENT,	DOClevANY, docRtfIgnoreWord, },
	{ "pnlcltr",	RTFidPNLCLTR,	DOClevANY, docRtfIgnoreWord, },
	{ "pnlcrm",	RTFidPNLCRM,	DOClevANY, docRtfIgnoreWord, },
	{ "pnlvl",	RTFidPNLVL,	DOClevANY, docRtfIgnoreWord, },
	{ "pnlvlblt",	RTFidPNLVLBLT,	DOClevANY, docRtfIgnoreWord, },
	{ "pnlvlbody",	RTFidPNLVLBODY,	DOClevANY, docRtfIgnoreWord, },
	{ "pnlvlcont",	RTFidPNLVLCONT,	DOClevANY, docRtfIgnoreWord, },
	{ "pnnumonce",	RTFidPNNUMONCE,	DOClevANY, docRtfIgnoreWord, },
	{ "pnord",	RTFidPNORD,	DOClevANY, docRtfIgnoreWord, },
	{ "pnordt",	RTFidPNORDT,	DOClevANY, docRtfIgnoreWord, },
	{ "pnprev",	RTFidPNPREV,	DOClevANY, docRtfIgnoreWord, },
	{ "pnqc",	RTFidPNQC,	DOClevANY, docRtfIgnoreWord, },
	{ "pnql",	RTFidPNQL,	DOClevANY, docRtfIgnoreWord, },
	{ "pnqr",	RTFidPNQR,	DOClevANY, docRtfIgnoreWord, },
	{ "pnrestart",	RTFidPNRESTART,	DOClevANY, docRtfIgnoreWord, },
	{ "pnscaps",	RTFidPNSCAPS,	DOClevANY, docRtfIgnoreWord, },
	{ "pnsp",	RTFidPNSP,	DOClevANY, docRtfIgnoreWord, },
	{ "pnstart",	RTFidPNSTART,	DOClevANY, docRtfIgnoreWord, },
	{ "pnstrike",	RTFidPNSTRIKE,	DOClevANY, docRtfIgnoreWord, },
	{ "pnucltr",	RTFidPNUCLTR,	DOClevANY, docRtfIgnoreWord, },
	{ "pnucrm",	RTFidPNUCRM,	DOClevANY, docRtfIgnoreWord, },
	{ "pnul",	RTFidPNUL,	DOClevANY, docRtfIgnoreWord, },
	{ "pnuld",	RTFidPNULD,	DOClevANY, docRtfIgnoreWord, },
	{ "pnuldb",	RTFidPNULDB,	DOClevANY, docRtfIgnoreWord, },
	{ "pnulnone",	RTFidPNULNONE,	DOClevANY, docRtfIgnoreWord, },
	{ "pnulw",	RTFidPNULW,	DOClevANY, docRtfIgnoreWord, },

	{ "ls",		RTFidLS,	DOClevANY, docRtfIgnoreWord, },
	{ "ilvl",	RTFidILVL,	DOClevANY, docRtfIgnoreWord, },
	{ "pnrnot",	RTFidPNRNOT,	DOClevANY, docRtfIgnoreWord, },

	{ 0, 0, 0 }
    };

static int docRtfSkipPn(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int		res;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfPnWords, docRtfPnGroups,
				docRtfIgnoreText );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

/************************************************************************/
/*									*/
/*  Read the info group.						*/
/*									*/
/************************************************************************/

static int docRtfInfoText(	RtfReadingContext *	rrc,
				unsigned char **	pText,
				const unsigned char *	text,
				int			len	)
    {
    char *	saved= (char *)malloc( len+ 1 );
    char *	to;

    if  ( ! saved )
	{ XDEB(saved); return -1;	}

    to= saved;
    while( len > 0 )
	{ *(to++)= rrc->rrcInputMapping[*(text++)]; len--;	}
    *to= '\0';

    if  ( *pText )
	{ free( *pText );	}

    *pText= (unsigned char *)saved;

    return 0;
    }

static int docRtfTitleText(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    { return docRtfInfoText( rrc, &(rrc->rrcBd->bdTitle), text, len ); }

static int docRtfAuthorText(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    { return docRtfInfoText( rrc, &(rrc->rrcBd->bdAuthor), text, len ); }

static int docRtfSubjectText(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    { return docRtfInfoText( rrc, &(rrc->rrcBd->bdSubject), text, len ); }

static int docRtfKeywordsText(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    { return docRtfInfoText( rrc, &(rrc->rrcBd->bdKeywords), text, len ); }

static int docRtfCommentText(	RtfReadingContext *	rrc,
				const unsigned char *	text,
				int			len	)
    { return docRtfInfoText( rrc, &(rrc->rrcBd->bdComment), text, len ); }

static int docRtfTitleGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfEmptyTable, docRtfEmptyTable,
				    docRtfTitleText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

static int docRtfAuthorGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfEmptyTable, docRtfEmptyTable,
				    docRtfAuthorText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

static int docRtfSubjectGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfEmptyTable, docRtfEmptyTable,
				    docRtfSubjectText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

static int docRtfKeywordsGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfEmptyTable, docRtfEmptyTable,
				    docRtfKeywordsText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

static int docRtfCommentGroup(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    if  ( docRtfReadGroup( sis, rcw->rcwLevel,
				    (RtfControlWord *)0, 0, 0, rrc,
				    docRtfEmptyTable, docRtfEmptyTable,
				    docRtfCommentText ) )
	{ SLDEB(rcw->rcwWord,arg); return -1;	}

    return 0;
    }

static RtfControlWord	docRtfInfoGroups[]=
    {
	{ "title",	RTFidTITLE,	DOClevDOC, docRtfTitleGroup, },
	{ "author",	RTFidAUTHOR,	DOClevDOC, docRtfAuthorGroup, },
	{ "subject",	RTFidSUBJECT,	DOClevDOC, docRtfSubjectGroup, },
	{ "keywords",	RTFidKEYWORDS,	DOClevDOC, docRtfKeywordsGroup, },
	{ "comment",	RTFidCOMMENT,	DOClevDOC, docRtfCommentGroup, },

	{ "operator",	RTFidOPERATOR,	DOClevDOC, docRtfSkipGroup, },
	{ "creatim",	RTFidCREATIM,	DOClevDOC, docRtfSkipGroup, },
	{ "revtim",	RTFidREVTIM,	DOClevDOC, docRtfSkipGroup, },
	{ "printim",	RTFidPRINTIM,	DOClevDOC, docRtfSkipGroup, },
	{ "version",	RTFidVERSION,	DOClevDOC, docRtfSkipGroup, },
	{ "edmins",	RTFidEDMINS,	DOClevDOC, docRtfSkipGroup, },
	{ "nofpages",	RTFidNOFPAGES,	DOClevDOC, docRtfSkipGroup, },
	{ "nofwords",	RTFidNOFWORDS,	DOClevDOC, docRtfSkipGroup, },
	{ "nofchars",	RTFidNOFCHARS,	DOClevDOC, docRtfSkipGroup, },
	{ "nofcharsws",	RTFidNOFCHARSWS,DOClevDOC, docRtfSkipGroup, },
	{ "vern",	RTFidVERN,	DOClevDOC, docRtfSkipGroup, },

	{ 0, 0, 0 }
    };

static int docRtfReadInfo(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int		res;

    res= docRtfReadGroup( sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfEmptyTable, docRtfInfoGroups,
				docRtfIgnoreText );
    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

/************************************************************************/
/*									*/
/*  Read the result of a bullet.					*/
/*									*/
/*  For one reason or another, Word resets the paragraph properties	*/
/*  inside the result. So save and restore them.			*/
/*									*/
/************************************************************************/

static int docRtfReadPntext(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    int				res;
    ParagraphProperties		pp;

    docInitParagraphProperties( &pp );
    docCopyParagraphProperties( &pp, &(rrc->rrcBi->biParaProperties) );

    res= docRtfReadGroup(	sis, DOClevPARA,
				(RtfControlWord *)0, 0, 0, rrc,
				docRtfDocumentWords, docRtfEmptyTable,
				docRtfTextParticule );

    docCopyParagraphProperties( &(rrc->rrcBi->biParaProperties), &pp );
    docCleanParagraphProperties( &pp );

    rrc->rrcJustAfterPntext= 1;

    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

static RtfControlWord	docRtfOutsideGroups[]=
    {
	{ "rtf",	RTFidRTF,	DOClevDOC, docRtfReadDoc,	},
	{ 0, 0, 0 }
    };

BufferDocument *	docRtfReadFile(	SimpleInputStream *	sis )
    {
    BufferDocument *		bd;
    RtfReadingContext		rrc;

    int				res;
    const RtfControlWord *	rcw;

    char			controlWord[TEDszRTFCONTROL+1];
    int				gotArg;
    int				arg= -1;
    int				c;

    bd= (BufferDocument *)malloc( sizeof(BufferDocument) );
    if  ( ! bd )
	{ XDEB(bd); return bd;	}

    docInitDocument( bd );

    docRtfInitReadingContext( &rrc );

    rrc.rrcBd= bd;
    rrc.rrcBi= &(bd->bdItem);
    rrc.rrcLevel= DOClevDOC;
    rrc.rrcSplitLevel= DOClevDOC;

    res= docRtfFindControl( sis, &rrc, &c, controlWord, &gotArg, &arg );

    if  ( res != RTFfiCTRLGROUP )
	{ docFreeDocument( bd ); return (BufferDocument *)0; }

    rcw= docRtfFindWord( controlWord, docRtfOutsideGroups, 1 );
    if  ( ! rcw )
	{ LDEB(1); docFreeDocument( bd ); return (BufferDocument *)0; }

    res= docRtfApplyControlWord( sis, rcw, gotArg, arg, &rrc );
    if  ( res )
	{ LDEB(1); docFreeDocument( bd ); return (BufferDocument *)0; }

    docRtfCleanReadingContext( &rrc );

    return bd;
    }

/************************************************************************/
/*									*/
/*  Just read a ruler from RTF.						*/
/*									*/
/************************************************************************/
static int docRtfReadRulerGroup(	SimpleInputStream *	sis,
					const RtfControlWord *	rcw,
					int			arg,
					RtfReadingContext *	rrc	)
    {
    int		res;

    res= docRtfConsumeGroup( sis, DOClevPARA, rrc,
					    docRtfEmptyTable, NULL, NULL );

    if  ( res )
	{ SLDEB(rcw->rcwWord,res);	}

    return res;
    }

static RtfControlWord	docRtfRulerOutsideGroups[]=
    {
	{ "ruler",	RTFidRULER,	DOClevSECT, docRtfReadRulerGroup, },
	{ 0, 0, 0 }
    };

int docRtfReadRuler(	SimpleInputStream *	sis,
			ParagraphProperties *	pp )
    {
    BufferItem			bi;
    RtfReadingContext		rrc;

    int				res;
    const RtfControlWord *	rcw;

    char			controlWord[TEDszRTFCONTROL+1];
    int				gotArg;
    int				arg= -1;
    int				c;

    docInitItem( &bi, (BufferItem *)0, 0, DOClevCELL );
    docRtfInitReadingContext( &rrc );

    rrc.rrcBi= &bi;
    rrc.rrcLevel= bi.biLevel;
    rrc.rrcSplitLevel= bi.biLevel;

    res= docRtfFindControl( sis, &rrc, &c, controlWord, &gotArg, &arg );

    if  ( res != RTFfiCTRLGROUP )
	{ LDEB(1); docCleanItem( (BufferDocument *)0, &bi ); return -1; }

    rcw= docRtfFindWord( controlWord, docRtfRulerOutsideGroups, 1 );
    if  ( ! rcw )
	{ LDEB(1); docCleanItem( (BufferDocument *)0, &bi ); return -1; }

    res= docRtfApplyControlWord( sis, rcw, gotArg, arg, &rrc );
    if  ( res )
	{ LDEB(1); docCleanItem( (BufferDocument *)0, &bi ); return -1; }

    docCopyParagraphProperties( pp, &(rrc.rrcParagraphProperties) );

    docCleanItem( (BufferDocument *)0, &bi );
    docRtfCleanReadingContext( &rrc );

    return 0;
    }

/************************************************************************/
/*									*/
/*  Remember properties to be used subsequently.			*/
/*									*/
/*  The level should be DOClevANY.					*/
/*									*/
/*  As this is the way Word parses RTF, and this makes an RTF reader	*/
/*  insensitive to dubious RTF, this is the way to go in further	*/
/*  extensions.								*/
/*									*/
/************************************************************************/

# define rrcBd do not use
# define rrcBi do not use
# define rrcInsertedObject do not use
# define rrcFieldNumber do not use

# define	RTF_REMEMBER(s,id) \
    { s,      id,         DOClevANY,  docRtfRememberProperty, }

# define	RTF_BORDER(s,id) \
    { s,	id,	DOClevANY,  docRtfRememberProperty, \
				    docRtfBeginBorder, docRtfBorderWords, }

RtfControlWord	docRtfPropertyWords[]=
{
				/****************************************/
				/*  Text properties.			*/
				/*  Character Formatting Properties	*/
				/****************************************/
    RTF_REMEMBER( "b",			RTFidB ),
    RTF_REMEMBER( "caps",		RTFidCAPS ),
    RTF_REMEMBER( "cf",			RTFidCF ),
    RTF_REMEMBER( "f",			RTFidF ),
    RTF_REMEMBER( "fs",			RTFidFS ),
    RTF_REMEMBER( "i",			RTFidI ),
    RTF_REMEMBER( "lang",		RTFidLANG ),
    RTF_REMEMBER( "plain",		RTFidPLAIN ),
    RTF_REMEMBER( "ul",			RTFidUL ),
    RTF_REMEMBER( "ulnone",		RTFidULNONE ),
    RTF_REMEMBER( "up",			RTFidUP ),
    RTF_REMEMBER( "cgrid",		RTFidCGRID ),

    RTF_REMEMBER( "cb",			RTFidCB ),
    RTF_REMEMBER( "cchs",		RTFidCCHS ),
    RTF_REMEMBER( "cs",			RTFidCS ),
    RTF_REMEMBER( "deleted",		RTFidDELETED ),
    RTF_REMEMBER( "dn",			RTFidDN ),
    RTF_REMEMBER( "ltrch",		RTFidLTRCH ),
    RTF_REMEMBER( "nosupersub",		RTFidNOSUPERSUB ),
    RTF_REMEMBER( "outl",		RTFidOUTL ),
    RTF_REMEMBER( "revauth",		RTFidREVAUTH ),
    RTF_REMEMBER( "revauthdel",		RTFidREVAUTHDEL ),
    RTF_REMEMBER( "revdttm",		RTFidREVDTTM ),
    RTF_REMEMBER( "revdttmdel",		RTFidREVDTTMDEL ),
    RTF_REMEMBER( "revised",		RTFidREVISED ),
    RTF_REMEMBER( "rtlch",		RTFidRTLCH ),
    RTF_REMEMBER( "shad",		RTFidSHAD ),

    RTF_REMEMBER( "scaps",		RTFidSCAPS ),
    RTF_REMEMBER( "strike",		RTFidSTRIKE ),
    RTF_REMEMBER( "sub",		RTFidSUB ),
    RTF_REMEMBER( "super",		RTFidSUPER ),
    RTF_REMEMBER( "uld",		RTFidULD ),
    RTF_REMEMBER( "uldb",		RTFidULDB ),
    RTF_REMEMBER( "ulw",		RTFidULW ),
    RTF_REMEMBER( "v",			RTFidV ),
    RTF_REMEMBER( "kerning",		RTFidKERNING ),
    RTF_REMEMBER( "expnd",		RTFidEXPND ),
    RTF_REMEMBER( "expndtw",		RTFidEXPNDTW ),
    RTF_REMEMBER( "charscalex",		RTFidCHARSCALEX ),
    RTF_REMEMBER( "highlight",		RTFidHIGHLIGHT ),

    RTF_REMEMBER( "chbghoriz",		RTFidCHBGHORIZ ),
    RTF_REMEMBER( "chbgvert",		RTFidCHBGVERT ),
    RTF_REMEMBER( "chbgfdiag",		RTFidCHBGFDIAG ),
    RTF_REMEMBER( "chbgbdiag",		RTFidCHBGBDIAG ),
    RTF_REMEMBER( "chbgcross",		RTFidCHBGCROSS ),
    RTF_REMEMBER( "chbgdcross",		RTFidCHBGDCROSS ),
    RTF_REMEMBER( "chbgdkhoriz",	RTFidCHBGDKHORIZ ),
    RTF_REMEMBER( "chbgdkvert",		RTFidCHBGDKVERT ),
    RTF_REMEMBER( "chbgdkfdiag",	RTFidCHBGDKFDIAG ),
    RTF_REMEMBER( "chbgdkbdiag",	RTFidCHBGDKBDIAG ),
    RTF_REMEMBER( "chbgdkcross",	RTFidCHBGDKCROSS ),
    RTF_REMEMBER( "chbgdkdcross",	RTFidCHBGDKDCROSS ),

    RTF_BORDER(   "chbrdr",		RTFidCHBRDR ),

    RTF_REMEMBER( "chcbpat",		RTFidCHCBPAT ),
    RTF_REMEMBER( "chcfpat",		RTFidCHCFPAT ),
    RTF_REMEMBER( "chshdng",		RTFidCHSHDNG ),
				/****************************************/
				/*  Paragraph properties.		*/
				/****************************************/
    RTF_REMEMBER( "pard",		RTFidPARD ),
    RTF_REMEMBER( "s",			RTFidS ),
    RTF_REMEMBER( "fi",			RTFidFI ),
    RTF_REMEMBER( "li",			RTFidLI ),
    RTF_REMEMBER( "ri",			RTFidRI ),
    RTF_REMEMBER( "ql",			RTFidQL ),
    RTF_REMEMBER( "qr",			RTFidQR ),
    RTF_REMEMBER( "qj",			RTFidQJ ),
    RTF_REMEMBER( "qc",			RTFidQC ),
    RTF_REMEMBER( "sa",			RTFidSA ),
    RTF_REMEMBER( "sb",			RTFidSB ),
    RTF_REMEMBER( "intbl",		RTFidINTBL ),
    RTF_REMEMBER( "hyphpar",		RTFidHYPHPAR ),

    RTF_REMEMBER( "pagebb",		RTFidPAGEBB ),

    RTF_REMEMBER( "keep",		RTFidKEEP ),
    RTF_REMEMBER( "keepn",		RTFidKEEPN ),
    RTF_REMEMBER( "outlinelevel",	RTFidOUTLINELEVEL ),

    RTF_REMEMBER( "sl",			RTFidSL ),
    RTF_REMEMBER( "slmult",		RTFidSLMULT ),
    RTF_REMEMBER( "nowidctlpar",	RTFidNOWIDCTLPAR ),
    RTF_REMEMBER( "widctlpar",		RTFidWIDCTLPAR ),
    RTF_REMEMBER( "adjustright",	RTFidADJUSTRIGHT ),
    RTF_REMEMBER( "adjustleft",		RTFidADJUSTLEFT ),

    RTF_REMEMBER( "tql",		RTFidTQL ), /*?*/

				/****************************************/
				/*  Paragraph Shading.			*/
				/****************************************/
    RTF_REMEMBER( "bgbdiag",		RTFidBGBDIAG ),
    RTF_REMEMBER( "bgcross",		RTFidBGCROSS ),
    RTF_REMEMBER( "bgdcross",		RTFidBGDCROSS ),
    RTF_REMEMBER( "bgdkbdiag",		RTFidBGDKBDIAG ),
    RTF_REMEMBER( "bgdkcross",		RTFidBGDKCROSS ),
    RTF_REMEMBER( "bgdkdcross",		RTFidBGDKDCROSS ),
    RTF_REMEMBER( "bgdkfdiag",		RTFidBGDKFDIAG ),
    RTF_REMEMBER( "bgdkhoriz",		RTFidBGDKHORIZ ),
    RTF_REMEMBER( "bgdkvert",		RTFidBGDKVERT ),
    RTF_REMEMBER( "bgfdiag",		RTFidBGFDIAG ),
    RTF_REMEMBER( "bghoriz",		RTFidBGHORIZ ),
    RTF_REMEMBER( "bgvert",		RTFidBGVERT ),
    RTF_REMEMBER( "cbpat",		RTFidCBPAT ),
    RTF_REMEMBER( "cfpat",		RTFidCFPAT ),
    RTF_REMEMBER( "shading",		RTFidSHADING ),
				/****************************************/
				/*  Paragraph Borders.			*/
				/****************************************/
    RTF_BORDER( "brdrt",		RTFidBRDRT ),
    RTF_BORDER( "brdrb",		RTFidBRDRB ),
    RTF_BORDER( "brdrl",		RTFidBRDRL ),
    RTF_BORDER( "brdrr",		RTFidBRDRR ),
    RTF_BORDER( "brdrbtw",		RTFidBRDRBTW ),
    RTF_BORDER( "brdrbar",		RTFidBRDRBAR ),
    RTF_BORDER( "box",			RTFidBOX ),
				/****************************************/
				/*  Positioned Objects and Frames:	*/
				/*  ( Paragraph properties. )		*/
				/****************************************/
    RTF_REMEMBER( "absh",		RTFidABSH ),
    RTF_REMEMBER( "abslock",		RTFidABSLOCK ),
    RTF_REMEMBER( "absw",		RTFidABSW ),
    RTF_REMEMBER( "dfrmtxtx",		RTFidDFRMTXTX ),
    RTF_REMEMBER( "dfrmtxty",		RTFidDFRMTXTY ),
    RTF_REMEMBER( "dropcapli",		RTFidDROPCAPLI ),
    RTF_REMEMBER( "dropcapt",		RTFidDROPCAPT ),
    RTF_REMEMBER( "dxfrtext",		RTFidDXFRTEXT ),
    RTF_REMEMBER( "nowrap",		RTFidNOWRAP ),
    RTF_REMEMBER( "phcol",		RTFidPHCOL ),
    RTF_REMEMBER( "phmrg",		RTFidPHMRG ),
    RTF_REMEMBER( "phpg",		RTFidPHPG ),
    RTF_REMEMBER( "posnegx",		RTFidPOSNEGX ),
    RTF_REMEMBER( "posnegy",		RTFidPOSNEGY ),
    RTF_REMEMBER( "posx",		RTFidPOSX ),
    RTF_REMEMBER( "posxc",		RTFidPOSXC ),
    RTF_REMEMBER( "posxi",		RTFidPOSXI ),
    RTF_REMEMBER( "posxl",		RTFidPOSXL ),
    RTF_REMEMBER( "posxo",		RTFidPOSXO ),
    RTF_REMEMBER( "posxr",		RTFidPOSXR ),
    RTF_REMEMBER( "posy",		RTFidPOSY ),
    RTF_REMEMBER( "posyb",		RTFidPOSYB ),
    RTF_REMEMBER( "posyc",		RTFidPOSYC ),
    RTF_REMEMBER( "posyil",		RTFidPOSYIL ),
    RTF_REMEMBER( "posyt",		RTFidPOSYT ),
    RTF_REMEMBER( "pvmrg",		RTFidPVMRG ),
    RTF_REMEMBER( "pvpara",		RTFidPVPARA ),
    RTF_REMEMBER( "pvpg",		RTFidPVPG ),
				/****************************************/
				/*  Bullets and Numbering.		*/
				/*  (No destinations)			*/
				/****************************************/
    RTF_REMEMBER( "ls",			RTFidLS ),
    RTF_REMEMBER( "ilvl",		RTFidILVL ),
				/****************************************/
				/*  Revision marks for paragraph	*/
				/*  numbers.				*/
				/****************************************/
    RTF_REMEMBER( "pnrauth",		RTFidPNRAUTH ),
    RTF_REMEMBER( "pnrdate",		RTFidPNRDATE ),
    RTF_REMEMBER( "pnrnfc",		RTFidPNRNFC ),
    RTF_REMEMBER( "pnrpnbr",		RTFidPNRPNBR ),
    RTF_REMEMBER( "pnrrgb",		RTFidPNRRGB ),
    RTF_REMEMBER( "pnrstart",		RTFidPNRSTART ),
    RTF_REMEMBER( "pnrstop",		RTFidPNRSTOP ),
    RTF_REMEMBER( "pnrxst",		RTFidPNRXST ),
				/****************************************/
				/*  Tabs.				*/
				/****************************************/
    RTF_REMEMBER( "tx",			RTFidTX ),
    RTF_REMEMBER( "tb",			RTFidTB ),
    RTF_REMEMBER( "tldot",		RTFidTLDOT ),
    RTF_REMEMBER( "tleq",		RTFidTLEQ ),
    RTF_REMEMBER( "tlhyph",		RTFidTLHYPH ),
    RTF_REMEMBER( "tlth",		RTFidTLTH ),
    RTF_REMEMBER( "tlul",		RTFidTLUL ),
    RTF_REMEMBER( "tqc",		RTFidTQC ),
    RTF_REMEMBER( "tqdec",		RTFidTQDEC ),
    RTF_REMEMBER( "tqr",		RTFidTQR ),
				/****************************************/
				/*  Associated Character Properties	*/
				/****************************************/
    RTF_REMEMBER( "ab",			RTFidAB ),
    RTF_REMEMBER( "acaps",		RTFidACAPS ),
    RTF_REMEMBER( "acf",		RTFidACF ),
    RTF_REMEMBER( "adn",		RTFidADN ),
    RTF_REMEMBER( "aexpnd",		RTFidAEXPND ),
    RTF_REMEMBER( "af",			RTFidAF ),
    RTF_REMEMBER( "afs",		RTFidAFS ),
    RTF_REMEMBER( "ai",			RTFidAI ),
    RTF_REMEMBER( "alang",		RTFidALANG ),
    RTF_REMEMBER( "aoutl",		RTFidAOUTL ),
    RTF_REMEMBER( "ascaps",		RTFidASCAPS ),
    RTF_REMEMBER( "ashad",		RTFidASHAD ),
    RTF_REMEMBER( "astrike",		RTFidASTRIKE ),
    RTF_REMEMBER( "aul",		RTFidAUL ),
    RTF_REMEMBER( "auld",		RTFidAULD ),
    RTF_REMEMBER( "auldb",		RTFidAULDB ),
    RTF_REMEMBER( "aulnone",		RTFidAULNONE ),
    RTF_REMEMBER( "aulw",		RTFidAULW ),
    RTF_REMEMBER( "aup",		RTFidAUP ),
    RTF_REMEMBER( "loch",		RTFidLOCH ),
    RTF_REMEMBER( "hich",		RTFidHICH ),
    RTF_REMEMBER( "dbch",		RTFidDBCH ),
				/****************************************/
				/*  Row, Cell/Column properties.	*/
				/****************************************/
    RTF_REMEMBER( "trowd",		RTFidTROWD ),

    RTF_REMEMBER( "ltrrow",		RTFidLTRROW ),
    RTF_REMEMBER( "rtlrow",		RTFidRTLROW ),

    RTF_REMEMBER( "trgaph",		RTFidTRGAPH ),
    RTF_REMEMBER( "trhdr",		RTFidTRHDR ),
    RTF_REMEMBER( "trkeep",		RTFidTRKEEP ),
    RTF_REMEMBER( "trleft",		RTFidTRLEFT ),
    RTF_REMEMBER( "trqc",		RTFidTRQC ),
    RTF_REMEMBER( "trql",		RTFidTRQL ),
    RTF_REMEMBER( "trqr",		RTFidTRQR ),
    RTF_REMEMBER( "trrh",		RTFidTRRH ),

    RTF_REMEMBER( "cellx",		RTFidCELLX ),

    RTF_REMEMBER( "celld",		RTFidCELLD ),
    RTF_REMEMBER( "cltxlrtb",		RTFidCLTXLRTB ),
    RTF_REMEMBER( "cltxtbrl",		RTFidCLTXTBRL ),
    RTF_REMEMBER( "clvertalt",		RTFidCLVERTALT ),
    RTF_REMEMBER( "clvertalc",		RTFidCLVERTALC ),
    RTF_REMEMBER( "clvertalb",		RTFidCLVERTALB ),
    RTF_REMEMBER( "clbgbdiag",		RTFidCLBGBDIAG ),
    RTF_REMEMBER( "clbgcross",		RTFidCLBGCROSS ),
    RTF_REMEMBER( "clbgdcross",		RTFidCLBGDCROSS ),
    RTF_REMEMBER( "clbgdkbdiag",	RTFidCLBGDKBDIAG ),
    RTF_REMEMBER( "clbgdkcross",	RTFidCLBGDKCROSS ),
    RTF_REMEMBER( "clbgdkdcross",	RTFidCLBGDKDCROSS ),
    RTF_REMEMBER( "clbgdkfdiag",	RTFidCLBGDKFDIAG ),
    RTF_REMEMBER( "clbgdkhor",		RTFidCLBGDKHOR ),
    RTF_REMEMBER( "clbgdkvert",		RTFidCLBGDKVERT ),
    RTF_REMEMBER( "clbgfdiag",		RTFidCLBGFDIAG ),
    RTF_REMEMBER( "clbghoriz",		RTFidCLBGHORIZ ),
    RTF_REMEMBER( "clbgvert",		RTFidCLBGVERT ),
    RTF_REMEMBER( "clmgf",		RTFidCLMGF ),
    RTF_REMEMBER( "clmrg",		RTFidCLMRG ),
    RTF_REMEMBER( "clcbpat",		RTFidCLCBPAT ),
    RTF_REMEMBER( "clcfpat",		RTFidCLCFPAT ),
    RTF_REMEMBER( "clshdng",		RTFidCLSHDNG ),
				/****************************************/
				/*  Row/Cell borders.			*/
				/****************************************/
    RTF_BORDER( "trbrdrb",		RTFidTRBRDRB ),
    RTF_BORDER( "trbrdrh",		RTFidTRBRDRH ),
    RTF_BORDER( "trbrdrl",		RTFidTRBRDRL ),
    RTF_BORDER( "trbrdrr",		RTFidTRBRDRR ),
    RTF_BORDER( "trbrdrt",		RTFidTRBRDRT ),
    RTF_BORDER( "trbrdrv",		RTFidTRBRDRV ),

    RTF_BORDER( "clbrdrb",		RTFidCLBRDRB ),
    RTF_BORDER( "clbrdrl",		RTFidCLBRDRL ),
    RTF_BORDER( "clbrdrr",		RTFidCLBRDRR ),
    RTF_BORDER( "clbrdrt",		RTFidCLBRDRT ),
				/****************************************/
				/*  Document formatting.		*/
				/****************************************/
    RTF_REMEMBER( "pgbrdrhead",		RTFidPGBRDRHEAD ),
    RTF_REMEMBER( "pgbrdrfoot",		RTFidPGBRDRFOOT ),
    RTF_REMEMBER( "pgbrdrt",		RTFidPGBRDRT ),
    RTF_REMEMBER( "pgbrdrb",		RTFidPGBRDRB ),
    RTF_REMEMBER( "pgbrdrl",		RTFidPGBRDRL ),
    RTF_REMEMBER( "pgbrdrr",		RTFidPGBRDRR ),
    RTF_REMEMBER( "brdrart",		RTFidBRDRART ),
    RTF_REMEMBER( "pgbrdropt",		RTFidPGBRDROPT ),
    RTF_REMEMBER( "pgbrdrsnap",		RTFidPGBRDRSNAP ),

    RTF_REMEMBER( "margl",		RTFidMARGL ),
    RTF_REMEMBER( "margr",		RTFidMARGR ),
    RTF_REMEMBER( "margt",		RTFidMARGT ),
    RTF_REMEMBER( "margb",		RTFidMARGB ),
    RTF_REMEMBER( "paperh",		RTFidPAPERH ),
    RTF_REMEMBER( "paperw",		RTFidPAPERW ),
    RTF_REMEMBER( "psz",		RTFidPSZ ),
				/****************************************/
				/*  Document properties.		*/
				/****************************************/
    RTF_REMEMBER( "noxlattoyen",	RTFidNOXLATTOYEN ),
    RTF_REMEMBER( "expshrtn",		RTFidEXPSHRTN ),
    RTF_REMEMBER( "noultrlspc",		RTFidNOULTRLSPC ),
    RTF_REMEMBER( "dntblnsbdb",		RTFidDNTBLNSBDB ),
    RTF_REMEMBER( "nospaceforul",	RTFidNOSPACEFORUL ),

				/****************************************/
				/*  Section properties.			*/
				/****************************************/
    RTF_REMEMBER( "sectdefaultcl",	RTFidSECTDEFAULTCL ),

				/****************************************/
				/*  Document properyies: (Notes)	*/
				/****************************************/
    RTF_REMEMBER( "aftnbj",		RTFidAFTNBJ ),
    RTF_REMEMBER( "aftnnalc",		RTFidAFTNNALC ),
    RTF_REMEMBER( "aftnnar",		RTFidAFTNNAR ),
    RTF_REMEMBER( "aftnnauc",		RTFidAFTNNAUC ),
    RTF_REMEMBER( "aftnnchi",		RTFidAFTNNCHI ),
    RTF_REMEMBER( "aftnnrlc",		RTFidAFTNNRLC ),
    RTF_REMEMBER( "aftnnruc",		RTFidAFTNNRUC ),
    RTF_REMEMBER( "aftnrestart",	RTFidAFTNRESTART ),
    RTF_REMEMBER( "aftnrstcont",	RTFidAFTNRSTCONT ),
    RTF_REMEMBER( "aftnstart",		RTFidAFTNSTART ),
    RTF_REMEMBER( "aftntj",		RTFidAFTNTJ ),

    RTF_REMEMBER( "aenddoc",		RTFidAENDDOC ),
    RTF_REMEMBER( "aendnotes",		RTFidAENDNOTES ),

    RTF_REMEMBER( "ftnalt",		RTFidFTNALT ),
    RTF_REMEMBER( "ftnbj",		RTFidFTNBJ ),
    RTF_REMEMBER( "ftnnalc",		RTFidFTNNALC ),
    RTF_REMEMBER( "ftnnar",		RTFidFTNNAR ),
    RTF_REMEMBER( "ftnnauc",		RTFidFTNNAUC ),
    RTF_REMEMBER( "ftnnchi",		RTFidFTNNCHI ),
    RTF_REMEMBER( "ftnnrlc",		RTFidFTNNRLC ),
    RTF_REMEMBER( "ftnnruc",		RTFidFTNNRUC ),
    RTF_REMEMBER( "ftnrestart",		RTFidFTNRESTART ),
    RTF_REMEMBER( "ftnrstcont",		RTFidFTNRSTCONT ),
    RTF_REMEMBER( "ftnrstpg",		RTFidFTNRSTPG ),
    RTF_REMEMBER( "ftnstart",		RTFidFTNSTART ),
    RTF_REMEMBER( "ftntj",		RTFidFTNTJ ),

    { 0, 0, 0 }
};

int docRtfRememberProperty(	SimpleInputStream *	sis,
				const RtfControlWord *	rcw,
				int			arg,
				RtfReadingContext *	rrc	)
    {
    switch( rcw->rcwId )
	{
	case	RTFidMARGL:
	    rrc->rrcDocumentGeometry.dgLeftMarginTwips= arg;
	    break;
	case	RTFidMARGT:
	    rrc->rrcDocumentGeometry.dgTopMarginTwips= arg;
	    break;
	case	RTFidMARGR:
	    rrc->rrcDocumentGeometry.dgRightMarginTwips= arg;
	    break;
	case	RTFidMARGB:
	    rrc->rrcDocumentGeometry.dgBottomMarginTwips= arg;
	    break;
	case	RTFidPAPERW:
	    rrc->rrcDocumentGeometry.dgPaperWideTwips= arg;
	    break;
	case	RTFidPAPERH:
	    rrc->rrcDocumentGeometry.dgPaperHighTwips= arg;
	    break;
	case	RTFidDEFTAB:
	    rrc->rrcDocumentProperties.dpTabIntervalTwips= arg;
	    break;
	case	RTFidDEFF:
	    rrc->rrcDefaultFont= arg;
	    rrc->rrcCurrentTextAttribute.taFontNumber= arg;
	    break;
	case	RTFidDEFLANG:
	case	RTFidDEFLANGFE:
	case	RTFidUC:
	    break;

	case	RTFidDELETED:
	    rrc->rrcInDeletedText= 1;
	    break;

	case RTFidFI:
	    rrc->rrcParagraphProperties.ppFirstIndentTwips= arg;
	    break;
	case RTFidLI:
	    rrc->rrcParagraphProperties.ppLeftIndentTwips= arg;
	    break;
	case RTFidRI:
	    rrc->rrcParagraphProperties.ppRightIndentTwips= arg;
	    break;
	case RTFidSB:
	    rrc->rrcParagraphProperties.ppSpaceBeforeTwips= arg;
	    break;
	case RTFidSA:
	    rrc->rrcParagraphProperties.ppSpaceAfterTwips= arg;
	    break;
	case RTFidS:
	    rrc->rrcParagraphProperties.ppStyle= arg;
	    break;

	case RTFidSL:
	    if  ( arg > -1000 && arg < 1000 )
		{ rrc->rrcParagraphProperties.ppLineSpacingTwips= arg;	}
	    else{ rrc->rrcParagraphProperties.ppLineSpacingTwips= 0;	}
	    break;

	case RTFidSLMULT:
	    rrc->rrcParagraphProperties.ppLineSpacingIsMultiple= ( arg != 0 );
	    break;

	case RTFidTX:
	    rrc->rrcTabStop.tsTwips= arg;
	    if  ( docParaAddTab( &(rrc->rrcParagraphProperties),
						    &(rrc->rrcTabStop) ) )
		{ LDEB(arg); return -1;	}
	    docInitTabStop( &(rrc->rrcTabStop) );
	    break;

	case RTFidTQR:		rrc->rrcTabStop.tsKind= DOCtkRIGHT;	break;
	case RTFidTQC:		rrc->rrcTabStop.tsKind= DOCtkCENTRE;	break;
	case RTFidTQDEC:	rrc->rrcTabStop.tsKind= DOCtkDECIMAL;	break;
	case RTFidTLDOT:	rrc->rrcTabStop.tsLeader= DOCtlDOTS;	break;
	case RTFidTLHYPH:	rrc->rrcTabStop.tsLeader= DOCtlHYPH;	break;
	case RTFidTLUL:		rrc->rrcTabStop.tsLeader= DOCtlUNDERLINE; break;
	case RTFidTLTH:		rrc->rrcTabStop.tsLeader= DOCtlTHICK;	break;
	case RTFidTLEQ:		rrc->rrcTabStop.tsLeader= DOCtlEQUAL;	break;

	case	RTFidPARD:
	    docCleanParagraphProperties( &(rrc->rrcParagraphProperties) );
	    docInitParagraphProperties(  &(rrc->rrcParagraphProperties) );

	    break;

	case RTFidQL:
	    rrc->rrcParagraphProperties.ppAlignment= DOCiaLEFT;
	    break;
	case RTFidQR:
	    rrc->rrcParagraphProperties.ppAlignment= DOCiaRIGHT;
	    break;
	case RTFidQC:
	    rrc->rrcParagraphProperties.ppAlignment= DOCiaCENTERED;
	    break;
	case RTFidQJ:
	    rrc->rrcParagraphProperties.ppAlignment= DOCiaJUSTIFIED;
	    break;

	case RTFidKEEP:
	    rrc->rrcParagraphProperties.ppKeepOnPage= arg != 0;
	    break;
	case RTFidKEEPN:
	    rrc->rrcParagraphProperties.ppKeepWithNext= arg != 0;
	    break;

	case RTFidPAGEBB:
	    rrc->rrcParagraphProperties.ppStartsOnNewPage= arg != 0;
	    break;

	case RTFidSHADING:
	    rrc->rrcParagraphProperties.ppShadingLevel= arg != 0;
	    break;

	case RTFidOUTLINELEVEL:
	    rrc->rrcParagraphProperties.ppOutlineLevel= arg != 0;
	    break;

	case RTFidINTBL:
	    rrc->rrcParagraphProperties.ppInTable= arg != 0;
	    rrc->rrcDocumentProperties.dpContainsTables= 1;
	    break;

	case RTFidBRDRT:
	    rrc->rrcParagraphProperties.ppTopBorder=
						    rrc->rrcBorderProperties;
	    break;
	case RTFidBRDRL:
	    rrc->rrcParagraphProperties.ppLeftBorder=
						    rrc->rrcBorderProperties;
	    break;
	case RTFidBRDRR:
	    rrc->rrcParagraphProperties.ppRightBorder=
						    rrc->rrcBorderProperties;
	    break;
	case RTFidBRDRB:
	    rrc->rrcParagraphProperties.ppBottomBorder=
						    rrc->rrcBorderProperties;
	    break;
	case RTFidBRDRBTW:
	    rrc->rrcParagraphProperties.ppBetweenBorder=
						    rrc->rrcBorderProperties;
	    break;

	case RTFidBOX:
	    rrc->rrcParagraphProperties.ppBoxBorder= rrc->rrcBorderProperties;
	    break;

	case RTFidBRDRBAR:
	    rrc->rrcParagraphProperties.ppBar= rrc->rrcBorderProperties;
	    break;

	case RTFidNOWIDCTLPAR:
	case RTFidWIDCTLPAR:
	    break;

	case RTFidTRGAPH:
	    rrc->rrcRowProperties.rpHalfGapWidthTwips= arg;
	    break;

	case RTFidTRRH:
	    rrc->rrcRowProperties.rpHeightTwips= arg;
	    break;

	case RTFidTRLEFT:
	    rrc->rrcRowProperties.rpLeftIndentTwips= arg;
	    break;

	case RTFidTRHDR:
	    rrc->rrcRowProperties.rpIsTableHeader= ( arg != 0 );
	    break;

	case RTFidTRKEEP:
	    rrc->rrcRowProperties.rpKeepOnPage= ( arg != 0 );
	    break;

	case RTFidTRBRDRT:
	    rrc->rrcRowProperties.rpTopBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidTRBRDRL:
	    rrc->rrcRowProperties.rpLeftBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidTRBRDRB:
	    rrc->rrcRowProperties.rpBottomBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidTRBRDRR:
	    rrc->rrcRowProperties.rpRightBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidTRBRDRH:
	    rrc->rrcRowProperties.rpHasHorizontalBorders= ( arg != 0 );
	    rrc->rrcRowProperties.rpHorizontalBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidTRBRDRV:
	    rrc->rrcRowProperties.rpHasVerticalBorders= ( arg != 0 );
	    rrc->rrcRowProperties.rpVerticalBorder= rrc->rrcBorderProperties;
	    break;

	case RTFidTRQL:
	    rrc->rrcRowProperties.rpAlignment= DOCiaLEFT;
	    break;
	case RTFidTRQC:
	    rrc->rrcRowProperties.rpAlignment= DOCiaCENTERED;
	    break;
	case RTFidTRQR:
	    rrc->rrcRowProperties.rpAlignment= DOCiaRIGHT;
	    break;

	case RTFidCLMGF:
	    rrc->rrcCellProperties.cpFirstInMergedRange= ( arg != 0 );
	    break;

	case RTFidCLMRG:
	    rrc->rrcCellProperties.cpMergedWithPrevious= ( arg != 0 );
	    break;

	case RTFidCLBRDRT:
	    rrc->rrcCellProperties.cpTopBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidCLBRDRL:
	    rrc->rrcCellProperties.cpLeftBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidCLBRDRR:
	    rrc->rrcCellProperties.cpRightBorder= rrc->rrcBorderProperties;
	    break;
	case RTFidCLBRDRB:
	    rrc->rrcCellProperties.cpBottomBorder= rrc->rrcBorderProperties;
	    break;

	case RTFidCELLD:
	    docCleanCellProperties( &(rrc->rrcCellProperties) );
	    docInitCellProperties( &(rrc->rrcCellProperties) );
	    break;
	case RTFidCLVERTALT:
	    rrc->rrcCellProperties.cpVerticalTextAlignment= DOCvtaTOP;
	    break;
	case RTFidCLVERTALC:
	    rrc->rrcCellProperties.cpVerticalTextAlignment= DOCvtaCENTERED;
	    break;
	case RTFidCLVERTALB:
	    rrc->rrcCellProperties.cpVerticalTextAlignment= DOCvtaBOTTOM;
	    break;
	case RTFidCLTXLRTB:
	    rrc->rrcCellProperties.cpVerticalTextAlignment= DOCvtaVERT_LEFT;
	    break;
	case RTFidCLTXTBRL:
	    rrc->rrcCellProperties.cpVerticalTextAlignment= DOCvtaVERT_RIGHT;
	    break;

	case RTFidCLCFPAT:
	    rrc->rrcCellProperties.cpForegroundColor= arg;
	    break;
	case RTFidCLCBPAT:
	    rrc->rrcCellProperties.cpBackgroundColor= arg;
	    break;

	case RTFidCLSHDNG:
	    rrc->rrcCellProperties.cpShadingLevel= arg;
	    break;

	case RTFidCLBGHORIZ:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspHORIZONTAL;
	    break;
	case RTFidCLBGVERT:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspVERTICAL;
	    break;
	case RTFidCLBGFDIAG:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspDIAGONAL_DOWN;
	    break;
	case RTFidCLBGBDIAG:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspDIAGONAL_UP;
	    break;
	case RTFidCLBGCROSS:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspCROSSED;
	    break;
	case RTFidCLBGDCROSS:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspDIAGONALCROSSED;
	    break;
	case RTFidCLBGDKHOR:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspDK_HORIZONTAL;
	    break;
	case RTFidCLBGDKVERT:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspDK_VERTICAL;
	    break;
	case RTFidCLBGDKFDIAG:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspDK_DIAGONAL_DOWN;
	    break;
	case RTFidCLBGDKBDIAG:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspDK_DIAGONAL_UP;
	    break;
	case RTFidCLBGDKCROSS:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspDK_CROSSED;
	    break;
	case RTFidCLBGDKDCROSS:
	    rrc->rrcCellProperties.cpShadingPattern= DOCspDK_DIAGONALCROSSED;
	    break;

	case RTFidADJUSTRIGHT:
	case RTFidADJUSTLEFT:
	    break;
	case	RTFidF:
	    rrc->rrcCurrentTextAttribute.taFontNumber= arg;
	    break;
	case	RTFidFS:
	    rrc->rrcCurrentTextAttribute.taFontSizeHalfPoints= arg;
	    break;
	case	RTFidCF:
	    /*  foreground color: index in color table or 0	*/
	    break;
	case	RTFidFC:
	    /*  Does not appear in the RTF spec.		*/
	    /*  It can however be found in NextStep files.	*/
	    break;
	case	RTFidUP:
	    break;
	case	RTFidB:
	    rrc->rrcCurrentTextAttribute.taFontIsBold= arg != 0;
	    break;
	case	RTFidI:
	    rrc->rrcCurrentTextAttribute.taFontIsSlanted= arg != 0;
	    break;
	case	RTFidUL:
	    rrc->rrcCurrentTextAttribute.taIsUnderlined= arg != 0;
	    break;
	case	RTFidULNONE:
	    rrc->rrcCurrentTextAttribute.taIsUnderlined= 0;
	    break;
	case	RTFidPLAIN:
	    docInitTextAttribute( &(rrc->rrcCurrentTextAttribute) );
	    rrc->rrcCurrentTextAttribute.taFontNumber= rrc->rrcDefaultFont;
	    rrc->rrcCurrentTextAttribute.taFontSizeHalfPoints= 24;
	    rrc->rrcInDeletedText= 0;
	    break;
	case	RTFidSUB:
	    rrc->rrcCurrentTextAttribute.taSuperSub= DOCfontSUBSCRIPT;
	    break;
	case	RTFidSUPER:
	    rrc->rrcCurrentTextAttribute.taSuperSub= DOCfontSUPERSCRIPT;
	    break;
	case	RTFidNOSUPERSUB:
	    rrc->rrcCurrentTextAttribute.taSuperSub= DOCfontREGULAR;
	    break;

	case RTFidCHCFPAT:
	    rrc->rrcTextForeground= arg;
	    break;
	case RTFidCHCBPAT:
	    rrc->rrcTextBackground= arg;
	    break;

	case RTFidCHSHDNG:
	    rrc->rrcTextShadingLevel= arg;
	    break;

	case RTFidCHBGHORIZ:
	    rrc->rrcTextShadingPattern= DOCspHORIZONTAL;
	    break;
	case RTFidCHBGVERT:
	    rrc->rrcTextShadingPattern= DOCspVERTICAL;
	    break;
	case RTFidCHBGFDIAG:
	    rrc->rrcTextShadingPattern= DOCspDIAGONAL_DOWN;
	    break;
	case RTFidCHBGBDIAG:
	    rrc->rrcTextShadingPattern= DOCspDIAGONAL_UP;
	    break;
	case RTFidCHBGCROSS:
	    rrc->rrcTextShadingPattern= DOCspCROSSED;
	    break;
	case RTFidCHBGDCROSS:
	    rrc->rrcTextShadingPattern= DOCspDIAGONALCROSSED;
	    break;
	case RTFidCHBGDKHORIZ:
	    rrc->rrcTextShadingPattern= DOCspDK_HORIZONTAL;
	    break;
	case RTFidCHBGDKVERT:
	    rrc->rrcTextShadingPattern= DOCspDK_VERTICAL;
	    break;
	case RTFidCHBGDKFDIAG:
	    rrc->rrcTextShadingPattern= DOCspDK_DIAGONAL_DOWN;
	    break;
	case RTFidCHBGDKBDIAG:
	    rrc->rrcTextShadingPattern= DOCspDK_DIAGONAL_UP;
	    break;
	case RTFidCHBGDKCROSS:
	    rrc->rrcTextShadingPattern= DOCspDK_CROSSED;
	    break;
	case RTFidCHBGDKDCROSS:
	    rrc->rrcTextShadingPattern= DOCspDK_DIAGONALCROSSED;
	    break;

	case RTFidCHBRDR:
	    rrc->rrcTextBorderProperties= rrc->rrcBorderProperties;
	    break;

	case	RTFidTROWD:
	    docCleanRowProperties( &(rrc->rrcRowProperties) );
	    docInitRowProperties(  &(rrc->rrcRowProperties) );

	    docInitCellProperties( &(rrc->rrcCellProperties) );
	    break;

	case	RTFidCELLX:
	    rrc->rrcCellProperties.cpRightBoundaryTwips= arg;

	    if  ( docInsertRowColumn( &(rrc->rrcRowProperties), -1,
						&rrc->rrcCellProperties ) )
		{ LDEB(rrc->rrcRowProperties.rpCellCount); return -1; }

	    docInitBorderProperties( &(rrc->rrcCellProperties.cpTopBorder) );
	    docInitBorderProperties( &(rrc->rrcCellProperties.cpLeftBorder) );
	    docInitBorderProperties( &(rrc->rrcCellProperties.cpRightBorder) );
	    docInitBorderProperties( &(rrc->rrcCellProperties.cpBottomBorder) );

	    break;

	case	RTFidCGRID:
	case	RTFidLANG:
	    break;
	default:
	    /*SLDEB(rcw->rcwWord,rcw->rcwId);*/
	    break;
	}

    return 0;
    }
