" Vim syntax file
" Language:     ANSI C
" Maintainer:   Mike Williams <mikew@harlequin.co.uk>
" Filenames:    *.c,*.h
" Last change:  9th October 1998
" URL:          N/A
"
" Change history
" 1. Better handling of conditional operator, plus choice of highlighting for it
" 2. Allow multibyte character and string constants (i.e. start with L) apart from #include
" 3. Rework character constants to allow for character sequences
" 4. Add functions - catches macros, including defines like #define myconst (0)
" 5. Add macro functions and correct #define to use Define, not Macro
" 6. Add missing hi link for cComment2String 
" 7. Add cComment[2]String to exclude cluster for parens

" Remove any old syntax stuff hanging around
syn clear

" C is case sensitive
syn case match

" Use default keyword chars
set isk&

" Yer TODO highlighter
syn keyword	cTodo		contained TODO FIXME XXX

" C statements
syn keyword	cStatement	goto break return continue
syn keyword	cLabel		case default
syn keyword	cConditional	if else switch 
syn keyword	cRepeat		while for do

" C operators
syn keyword	cOperator	sizeof

" C predefined types
syn keyword	cType		int long short char void signed unsigned float double
if exists("c_typedefs")
  syn keyword   cType           size_t wchar_t ptrdiff_t sig_atomic_t fpos_t div_t ldiv_t
  syn keyword   cType           clock_t time_t va_list jmp_buf FILE
endif

" C language structures
syn keyword	cStructure	struct union enum typedef

" C storage modifiers
syn keyword	cStorageClass	static register auto volatile extern const

" C operators
syn match cMathOperator         "[-+\*/%=]"
syn match cPointerOperator      "\(->\|\.\)"
syn match cLogicalOperator      "[!<>][=]\="
syn match cLogicalOperator      "=="
syn match cBinaryOperator       "\(&\||\|^\|\~\|<<\|>>\)[=]\="
syn match cLogicalOperator      "\(&&\|||\)"
syn match cLogicalOperatorError "\(&&\|||\)="
syn cluster cOperator           contains=cMathOperator,cPointerOperator,cLogicalOperator,cBinaryOperator,cLogicalOperatorError

" C identifiers - function, macro, variable
syn match       cIdentifier	"\<\h\w*\>"
syn match       cFunction       "\<\h\w*\>\s*("me=e-1

if exists("c_constants")
" C constants
  syn match     cConstant       "\<__\(LINE\|FILE\|DATE\|TIME\|STDC\)__\>"
  syn keyword   cConstant       CHAR_BIT MB_LEN_MAX MB_CUR_MAX 
  syn match     cConstant       "\<U\(CHAR\|INT\|LONG\|SHRT\)_MAX\>"
  syn match     cConstant       "\<\([S]\=CHAR\|INT\|LONG\|SHRT\)_\(MIN\|MAX\)\>"
  syn match     cConstant       "\<FLT_R\(ADIX\|OUNDS\)\>"
  syn match     cConstant       "\<\(FLT\|[L]\=DBL\)_\(\(MANT_\)\=DIG\|EPSILON\)\>" 
  syn match     cConstant       "\<\(FLT\|[L]\=DBL\)_\(M\(IN\|AX\)\|M\(IN\|AX\)_\(10_\)\=EXP\)\>"
  syn keyword   cConstant       HUGE_VAL EDOM ERANGE CLOCKS_PER_SEC NULL
  syn match     cConstant       "\<LC_\(ALL\|COLLATE\|CTYPE\|MONETARY\|NUMERIC\|TIME\)\>"
  syn match     cConstant       "\<SIG\(_\(DFL\|ERR\|IGN\)\|ABRT\|FPE\|ILL\|INT\|SEGV\|TERM\)\>"
  syn keyword   cConstant       _IOFBF _IOLBF _IONBF BUFSIZ EOF FOPEN_MAX FILENAME_MAX L_tmpnam
  syn match     cConstant       "\<SEEK_\(CUR\|END\|SET\)\>"
  syn keyword   cConstant       TMP_MAX stderr stdin stdout EXIT_FAILURE EXIT_SUCCESS RAND_MAX
endif

" Escaped characters
syn match       cEscapeCharError contained "\\[^'\"?\\abfnrtv]"
syn match       cEscapeChar     contained "\\['\"?\\abfnrtv]"
" Octal characters
syn match       cOctalChar      contained "\\\o\{1,3}"
" Hex characters
syn match       cHexChar        contained "\\x\x\+"
" Useful groupings of character types
syn cluster     cSpecialChar    contains=cEscapeCharError,cEscapeChar,cOctalChar,cHexChar
syn cluster     cSpecialCharNoError contains=cEscapeChar,cOctalChar,cHexChar
" Character constants
syn match	cCharacterError "[L]\='\($\|['\n]\)"
syn match	cCharacter    	"[L]\='.\{-}'" contains=@cSpecialChar
syn match	cCharacterNoError contained "[L]\='.\{-}'" contains=@cSpecialCharNoError
" String constants
syn region	cString		start=+[L]\="+ skip=+\\"+ end=+"+ contains=@cSpecialChar,cPPLineJoin

" Numbers
syn case ignore
" Integer 
syn match	cNumber		"\<\d\+\(u\=l\=\|lu\)\>"
" Hex integer 
syn match	cHex   		"\<0x\x\+\(u\=l\=\|lu\)\>"
" Octal integers
if exists("c_no_octal")
  syn match     cOctalError     "\<0\o\+\>"
else
  syn match     cOctal          "\<0\o\+\>"
endif
syn match	cOctalError	"\<0\o*[89]"
syn cluster     cInteger        contains=cNumber,cHex,cOctal
" Fp with no dp or exponent
syn match	cFloat		"\<\d\+f\>"
" Fp with dot, optional exponent
syn match	cFloat		"\<\d\+\.\d*\(e[-+]\=\d\+\)\=[fl]\=\>"
" FP starting with a dot, optional exponent
syn match	cFloat		"\.\d\+\(e[-+]\=\d\+\)\=[fl]\=\>"
" FP without dot, with exponent
syn match	cFloat		"\<\d\+e[-+]\=\d\+[fl]\=\>"
syn case match

" When wanted, highlight trailing white space
if exists("c_no_trailing_space")
  syn match	cSpaceError	"\s*$"
endif

" When wanted, highlight trailing spaces amongst tabs
if exists("c_no_mixed_space_tabs")
  syn match	cSpaceError	" \+\t"me=e-1
endif

" C comments
if exists("c_comment_strings")
  " A comment can contain cString, cCharacter and cInteger.
  " But a "*/" inside a cString in a cComment DOES end the comment!  So we
  " need to use a special type of cString: cCommentString, which also ends on
  " "*/", and sees a "*" at the start of the line as comment again.
  syntax match	cCommentSkip	contained "^\s*\*\($\|\s\+\)"
  syntax region cCommentString	contained start=+[L]\="+ skip=+\\"+ end=+"+ end=+\*/+me=s-1 contains=cCharacterNoError,@cSpecialCharNoError,cCommentSkip
  syntax region cComment	start="/\*" end="\*/" contains=cTodo,cSpaceError,cCommentString,cCharacterNoError,@cInteger,cFloat,cConstant,cType
else
  " Bog ordinary comments
  syn region	cComment	start="/\*" end="\*/" contains=cTodo,cSpaceError
endif
syntax match	cCommentError	"\*/"

" Either you want cpp comments or you don't
if exists("c_cpp_comments")
  if exists("c_comment_strings")
    " Unfortunately this doesn't very well work for // type of comments :-(
    syntax region cComment2String contained start=+[L]\="+ skip=+\\"+ end=+"+ end="$" contains=cCharacterNoError,@cSpecialCharNoError,cCommentSkip
    syntax match cComment	"//.*" contains=cTodo,cSpaceError,cComment2String,cCharacterNoError,@cInteger,cFloat
  else
    syn match   cComment        "//.*$" contains=cTodo,cSpaceError
  endif
else
  syn match     cCommentError   "//.*$"
endif

" C pre-processor commands
syn cluster	cPPCommands	contains=cInclude,cDefine,cPreCondit,cPreProc,cPPEmptyLine
" Highlight cpp joined lines
syn match       cPPLineJoin     contained "\\$"
syn match       cPPLineJoinError contained "\\\s\+$"
" Empty cpp lines
syn match       cPPTokenOperator contained "#\{1,2}"
syn match       cPPEmptyLine    "^\s*#\s*$"
" File inclusion
syn match       cPPInclude      contained "#\s*include\>"
syn region	cPPIncludeFile	contained start=+"+ skip=+\\"+ end=+"+ contains=cPPLineJoin,cPPLineJoinError
syn match	cPPIncludeFile	contained "<[^>]*>"
syn match	cInclude	"^\s*#\s*include\>\s*["<]" contains=cPPInclude,cPPIncludeFile
" Conditional code
syn keyword     cPPOperator     contained defined
syn match       cPPIf           "^\s*#\s*\(if\|ifdef\|ifndef\|elif\|else\|endif\)\>" 
syn region      cPreCondit      start="^\s*#\s*\(if\|ifdef\|ifndef\|elif\|else\|endif\)\>" skip="\\$" end="$" contains=ALLBUT,@cPPCommands,cUserLabel,cInParen,cConditionalOperator
" Macros
syn match       cPPDefine       contained "#\s*\(define\|undef\)\>"
syn region      cDefine         start="^\s*#\s*\(define\|undef\)\>" skip="\\$" end="$" contains=ALLBUT,@cPPCommands,cUserLabel,cInParen,cConditionalOperator
syn match       cMacro          "[[:upper:]_][[:upper:][:digit:]_]*\s*("me=e-1
" Everything else
syn match       cPPMisc         contained "#\s*\(pragma\|line\|error\)\>"
syn region      cPreProc        start="^\s*#\s*\(pragma\|line\|error\)\>" skip="\\$" end="$" contains=ALLBUT,@cPPCommands,cUserLabel,cInParen,cConditionalOperator

" Highlight User Labels
syn cluster	cMultiGroup	contains=cPPInclude,cPPIncludeFile,@cSpecialChar,cTodo,cUserCont,cUserLabel,cBitField,cInParen,cParenError
if exists("c_conditional_is_operator")
  syn match     cConditionalOperator contained "[?:]"
else
  syn match     cConditional    contained "[?:]"
endif
syn region	cConditional    transparent start="?" end=":\{-}" contains=ALLBUT,@cMultiGroup
syn match	cUserCont	"^\s*\I\i*\s*:\s*" contains=cUserLabel
syn match	cUserCont	";\s*\I\i*\s*:\s*" contains=cUserLabel
syn match	cUserLabel	contained "\I\i*"

" Avoid recognizing most bitfields as labels
syn match	cBitField	"^\s*\h\w*\s*:\s*[1-9]"me=e-1 contains=cIdentifier
syn match	cBitField	";\s*\h\w*\s*:\s*[1-9]"me=e-1 contains=cIdentifier

" Catch errors caused by wrong parenthesis
syn cluster     cParenGroup	contains=cParenError,cTodo,cUserCont,cUserLabel,cBitField,cPPLineJoin,cPPLineJoinError,cPPIncludeFile,cPPOperator,cPPTokenOperator,cCommentString,cComment2String
syn region      cParen		transparent start='(' end=')' contains=ALLBUT,@cParenGroup
syn match       cParenError	")"
syn match       cInParen	contained "[{}]"


if !exists("c_minlines")
  let c_minlines = 15
endif
exec "syn sync ccomment cComment minlines=" . c_minlines

if !exists("did_mrwc_syntax_inits")
  let did_mrwc_syntax_inits = 1
  " The default methods for highlighting.  Can be overridden later
  hi link cTodo         Todo

  hi link cStatement    Statement
  hi link cLabel	Label
  hi link cUserLabel	cLabel
  hi link cConditional	Conditional
  hi link cRepeat	Repeat

  hi link cOperator	Operator
  hi link cMathOperator	cOperator
  hi link cPointerOperator cOperator
  hi link cLogicalOperator cOperator
  hi link cBinaryOperator cOperator
  hi link cConditionalOperator cOperator

  hi link cType		Type
  hi link cStructure	Structure
  hi link cStorageClass	StorageClass
  hi link cIdentifier	Identifier
  hi link cFunction	Function

  hi link cCharacter	Character
  hi link cCharacterNoError cCharacter
  hi link cSpecialChar  SpecialChar
  hi link cEscapeChar	cSpecialChar
  hi link cOctalChar	cSpecialChar
  hi link cHexChar	cSpecialChar
  hi link cString	String

  hi link cInteger      Number
  hi link cNumber	cInteger
  hi link cOctal      	cInteger
  hi link cHex      	cInteger
  hi link cFloat	Float

  hi link cConstant     Constant

  hi link cComment	Comment
  hi link cCommentString cString
  hi link cComment2String cString
  hi link cCommentSkip	cComment

  hi link cError	Error
  hi link cOctalError	cError
  hi link cParenError	cError
  hi link cInParen	cError
  hi link cCommentError	cError
  hi link cSpaceError	cError
  hi link cCharacterError cError
  hi link cEscapeCharError cError
  hi link cLogicalOperatorError cError

  hi link cPreProc	PreProc
  hi link cPPEmptyLine  cPreProc
  hi link cPPLineJoin   cPreProc
  hi link cPPMisc       cPreProc
  hi link cInclude      Include
  hi link cPPInclude    cInclude
  hi link cPPIncludeFile cString
  hi link cDefine	Define
  hi link cPPDefine	cDefine
  hi link cMacro        Macro
  hi link cPreCondit	PreCondit
  hi link cPPIf         cPreCondit
  hi link cPPError      cError
  hi link cPPLineJoinError cPPError
  hi link cPPOperator   cOperator
  hi link cPPTokenOperator cPPOperator

endif

if exists("my_c_file")
  if file_readable(expand(my_c_file))
    execute "source " . my_c_file
  endif
endif

let b:current_syntax = "mrwc"

" vim: ts=8
