;;; x-symbol-mule.el --- XEmacs/Mule support for package x-symbol

;; Copyright (C) 1997-1999 Free Software Foundation, Inc.
;;
;; Author: Christoph Wedler <wedler@fmi.uni-passau.de>
;; Maintainer: (Please use `M-x x-symbol-package-bug' to contact the maintainer)
;; Version: $Id: x-symbol-mule.el,v 3.3 1999/01/18 14:16:12 wedler Exp wedler $
;; Keywords: WYSIWYG, LaTeX, HTML, wp, math, internationalization, Mule
;; X-URL: http://www.fmi.uni-passau.de/~wedler/x-symbol/

;; This program is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; This program is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with this program; if not, write to the Free Software
;; Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

;;; Commentary:

;; If you want to use package x-symbol, please visit the URL (use
;; \\[x-symbol-package-web]) and read the info (use \\[x-symbol-package-info]).

;;; Code:

(eval-and-compile
  (unless (featurep 'mule)
    (error "This file is meant to be used with XEmacs/Mule")))
(provide 'x-symbol-mule)
(require 'x-symbol-hooks)

(unless (eq x-symbol-default-coding 'iso-8859-1)
  (warn "Package x-symbol under XEmacs/Mule has only been tested with default coding `iso-8859-1'"))

;; Provide easier XEmacs-21/Mule bug workaround:
(defvar x-symbol-mule-default-charset 'latin-iso8859-1
  ;; the value should change if the bug is fixed:
  ;; (or (coding-system-charset x-symbol-default-coding 1) 'latin-iso8859-1)
  ;; the line above is by Jan Vroonhof, I had this in v3.1:
  ;;(coding-system-property x-symbol-default-coding 'charset-g1)
  "Mule charset corresponding to `x-symbol-default-coding'.")


;;;===========================================================================
;;;  Function aliases and internal variables
;;;===========================================================================

(defalias 'x-symbol-make-cset 'x-symbol-mule-make-cset)
(defalias 'x-symbol-make-char 'x-symbol-mule-make-char)
(defalias 'x-symbol-init-charsym-syntax 'x-symbol-mule-init-charsym-syntax)
(defalias 'x-symbol-charsym-after 'x-symbol-mule-charsym-after)
(defalias 'x-symbol-string-to-charsyms 'x-symbol-mule-string-to-charsyms)
(defalias 'x-symbol-match-before 'x-symbol-mule-match-before)
(defalias 'x-symbol-encode-lisp 'x-symbol-mule-encode-lisp)
(defalias 'x-symbol-pre-command-hook 'x-symbol-mule-pre-command-hook)
(defalias 'x-symbol-post-command-hook 'x-symbol-mule-post-command-hook)

(defvar x-symbol-mule-char-table nil
  "Internal.  Map characters to charsyms.")
(defvar x-symbol-mule-pre-command nil
  "Internal.  Used for pre- and post-command handling.")


;;;===========================================================================
;;;  Init code
;;;===========================================================================

(defun x-symbol-mule-make-charset (definition graphic registry)
  "Define new charset according to DEFINITION.
DEFINITION looks like nil or (NAME) or (NAME DOCSTRING CHARS FINAL), see
`x-symbol-init-cset'.  GRAPHIC and REGISTRY are charset properties, see
`make-charset' for details."
  (and definition
       (null (find-charset (car definition)))
       (make-charset (car definition) (cadr definition)
		     (list 'registry registry
			   'dimension 1
			   'chars (caddr definition)
			   'final (cadddr definition)
			   'graphic graphic))))

(defun x-symbol-mule-make-cset (cset fonts)
  "Define new charsets according to CSET using FONTS.
See `x-symbol-init-cset'.  Return (NORMAL SUBSCRIPT SUPERSCIPT).  Each
element is a face or nil if the corresponding font in FONTS could not be
found.  Return nil, if no default font for that registry could be found."
  (let ((first (if noninteractive
		   (caar fonts)
		 (x-symbol-try-font-name (car fonts)))))
    (when (or first
	      (and x-symbol-latin-force-use (x-symbol-cset-coding cset))
	      (and (find-charset (car (x-symbol-cset-left cset)))
		   (find-charset (car (x-symbol-cset-right cset)))))
      (let ((default (eq (x-symbol-cset-coding cset) x-symbol-default-coding))
	    (registry (x-symbol-cset-registry cset))
	    (left (x-symbol-cset-left cset))
	    (right (x-symbol-cset-right cset)))
	(x-symbol-mule-make-charset left 0 registry)
	(x-symbol-mule-make-charset right 1 registry)
	(or default
	    (null first)
	    noninteractive
	    (and (null x-symbol-mule-change-default-face)
		 (face-property-matching-instance 'default 'font
						  (or (car left)
							(car right))
						  nil nil t))
	    (let ((origfont (face-property-instance 'default 'font)))
	      (set-face-property 'default 'font (car fonts) nil
				 '(mule-fonts) 'prepend)
	      ;; Edit Faces should show the font for the "normal" charset
	      (set-face-property 'default 'font origfont)))
	(if noninteractive
	    (list nil)
	  (let ((faces '(x-symbol-face x-symbol-sub-face x-symbol-sup-face))
		(docstrings x-symbol-face-docstrings)
		font)
	    (while faces
	      (if (setq font (x-symbol-try-font-name (car fonts)))
		  (progn
		    (make-face (car faces) (car docstrings))
		    (set-face-property (car faces) 'font font nil
				       '(mule-fonts) 'prepend))
		(warn "X-Symbol cannot find font in %s" (car fonts)))
	      (setq fonts (cdr fonts)
		    faces (cdr faces)
		    docstrings (cdr docstrings))))
	(if first '(x-symbol-face) '(default)))))))

(defun x-symbol-mule-make-char (cset encoding charsym face coding)
  "Define character in CSET with ENCODING, represented by CHARSYM.
The character is considered to be a 8bit character in CODING.  Use FACE
when character is presented in the grid."
  (unless (char-table-p x-symbol-mule-char-table)
    (setq x-symbol-mule-char-table (make-char-table 'generic))
    (put-char-table t nil x-symbol-mule-char-table))
  (let* ((char (if (< encoding 128)
		   (make-char (caadr cset) encoding)
		 (make-char (caddr cset) (- encoding 128))))
	 (cstring (char-to-string char)))
    (put-char-table char charsym x-symbol-mule-char-table)
    (x-symbol-set-cstrings
     charsym coding cstring
     (and coding (>= encoding 160)
	  (char-to-string
	   (make-char x-symbol-mule-default-charset (- encoding 128))))
     face)
    cstring))

(defun x-symbol-mule-init-charsym-syntax (charsyms)
  "Initialize the syntax for the characters represented by CHARSYMS.
See `x-symbol-init-cset' and `x-symbol-group-syntax-alist'."
  (dolist (charsym charsyms)
    (when (get charsym 'x-symbol-cstring)
      (let ((syntax (get charsym 'x-symbol-syntax)))
	(when syntax
	  (let ((opposite (and (cdr syntax)
			       (get (cddr syntax) 'x-symbol-cstring))))
	    (modify-syntax-entry (aref (get charsym 'x-symbol-cstring) 0)
				 (if opposite
				     (format (cadr syntax) opposite)
				   (car syntax)))))))))


;;;===========================================================================
;;;  Character recognition
;;;===========================================================================

(defun x-symbol-mule-charsym-after (&optional pos)
  "Return x-symbol charsym for character at POS.
POS defaults to point.  If POS is out of range, return nil.  Otherwise,
return (POS . CHARSYM) where CHARSYM is the x-symbol charsym for the
character at POS or nil otherwise."
  (or pos (setq pos (point)))
  (and (char-after pos)
       (cons pos (get-char-table (char-after pos) x-symbol-mule-char-table))))

(defun x-symbol-mule-string-to-charsyms (string)
  "Return list of charsyms for the characters in STRING.
If a character is not represented as a charsym, use the character itself
if is an ascii in the range \\040-\\176, otherwise nil."
  (let ((chars (nreverse (append string nil)))
	result after)
    (while chars
      (setq after (pop chars))
      (push (or (get-char-table after x-symbol-mule-char-table)
		(and (<= ?\040 after) (< after ?\177) after))
	    result))
    result))

(defun x-symbol-mule-match-before (atree pos &optional case-fn)
  "Return association in ATREE for longest match before POS.
Return (START . VALUE) where the buffer substring between START and
point is the key to the association VALUE in ATREE.  If optional CASE-FN
is non-nil, convert characters before the current position with CASE-FN.
See `x-symbol-atree-push'."
  (let ((result nil))
    (while (setq atree (cdr (assoc (if case-fn
				       (funcall case-fn (char-after (decf pos)))
				     (char-after (decf pos)))
				   (cdr atree))))
      (and (car atree)
	   (setq result (cons pos (car atree)))))
    result))


;;;===========================================================================
;;;  Command hooks
;;;===========================================================================

;; Functions in these hooks are run twice (and more) when pressing a key which
;; runs a keyboard macro, e.g., if [backspace] runs [delete] and [delete] runs
;; `delete-backward-char'.

(defun x-symbol-mule-pre-command-hook ()
  "Function used in `pre-command-hook' when `x-symbol-mode' is turned on.
Hide revealed characters, see `x-symbol-hide-revealed-at-point'.
Provide input method TOKEN, see `x-symbol-token-input'."
  (x-symbol-hide-revealed-at-point)
  (when (and x-symbol-mode (null x-symbol-mule-pre-command))
    (setq x-symbol-mule-pre-command t)
    (x-symbol-token-input)))

(defun x-symbol-mule-post-command-hook ()
  "Function used in `post-command-hook' when `x-symbol-mode' is turned on.
Provide input method ELECTRIC, see `x-symbol-electric-input'.  Start
idle timer for info in echo area and revealing invisible characters, see
`x-symbol-start-itimer-once'."
  (when (and x-symbol-mode x-symbol-mule-pre-command)
    (x-symbol-electric-input)
    (x-symbol-start-itimer-once))
  (setq x-symbol-mule-pre-command nil))


;;;===========================================================================
;;;  Conversion
;;;===========================================================================

;; Since I don't know much about Mule coding systems, this is probably slow
(defun x-symbol-mule-encode-lisp (alist coding language)
  "Encode all characters in buffer to tokens according to ALIST.
Do not encode 8bit characters in CODING.  Characters are converted to
tokens in LANGUAGE.  Character aliases are not encoded!"
  ;; Notation: lN = in charset latin-N, dN = in default-coding with same octet
  ;;   Case 1: standard-coding = latin-1, file-coding = latin-5
  ;;    1a) l3, l5:     delay l3 -> d5
  ;;    1b) l3, no l5:  l3 -> TeX
  ;;	1c) l5:		delay l5 -> d5
  ;;	1d) l1 = d5	-
  ;;	1e) l1 <> d5	l1 -> l5, delay l5 -> d5
  ;;	1f) l1, no l5:	l1 -> TeX
  ;;   Case 2: standard-coding = latin-1, file-coding = latin-1
  ;;    2a) l3, no l1:  l3 -> TeX
  ;;    2b) o1:		-
  (let* ((token-shape (cdr (x-symbol-language-value 'x-symbol-token-shape
						    language)))
	 (token-regexp (car token-shape))
	 (letter-regexp (cdr token-shape))
	 (case-fold-search nil)		; encoding always case-sensitive
	 delays
	 from charsym to normal char)
    (while alist
      (if (setq from (caar alist)
		charsym (cdar alist)
		alist (cdr alist)
		to (plist-get (get charsym 'x-symbol-file-cstrings) coding))
	  ;; Cases 1a, 1c, 1d, 1e, 2b
	  (unless (string-equal from to)
	    ;; Cases 1a, 1c, 1e
	    (when (plist-get (get charsym 'x-symbol-file-cstrings)
			     x-symbol-default-coding)
	      ;; Case 1e
	      (setq normal (plist-get (get charsym 'x-symbol-buffer-cstrings)
				      coding))
	      (goto-char (point-min))
	      (while (search-forward from nil 'limit)
		(replace-match normal t t))
	      (setq from normal))
	    (push (cons from to) delays))
	;; Cases 1b, 1f, 2a
	(goto-char (point-min))
	(setq to (plist-get (get charsym 'x-symbol-tokens) language)
	      normal (and token-regexp
			  (string-match token-regexp to)))
	(while (search-forward from nil 'limit)
	  (replace-match to t t)
	  (and normal
	       (setq char (char-after (point)))
	       (string-match letter-regexp (char-to-string char))
	       (insert " ")))))
    (while delays
      (setq from (caar delays)
	    to (cdar delays)
	    delays (cdr delays))
      (goto-char (point-min))
      (while (search-forward from nil 'limit)
	(replace-match to t t)))))

;;; Local IspellPersDict: .ispell_xsymb
;;; x-symbol-mule.el ends here
