//  UCache.h version 1.5
//  yudit package - Unicode Editor for the X Window System (and Linux) 
//
//  Author: gsinai@iname.com (Gaspar Sinai)
//  GNU Copyright (C) 1997,1998,1999  Gaspar Sinai
// 
//  yudit version 1.5  Copyright(C) 30 November, 1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.4  Copyright(C) 25 November, 1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.3  Copyright(C)  5 April,    1999, Tokyo Japan  Gaspar Sinai
//  yudit version 1.2  Copyright(C) 10 December, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.1  Copyright(C) 23 August,   1998, Tokyo Japan  Gaspar Sinai
//  yudit version 1.0  Copyright(C) 17 May,      1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.99 Copyright(C)  4 April,    1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.97 Copyright(C)  4 February, 1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.95 Copyright(C) 10 January,  1998, Tokyo Japan  Gaspar Sinai
//  yudit version 0.94 Copyright(C) 17 December, 1997, Tokyo Japan  Gaspar Sinai
//  yudit version 0.9 Copyright (C)  8 December, 1997, Tokyo Japan  Gaspar Sinai
//  yutex version 0.8 Copyright (C)  5 November, 1997, Tokyo Japan  Gaspar Sinai
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
//
//
// UCache.h
// Gaspar Sinai <gsinai@gol.com>
// This template class is for classes that try to share a common resource.
// Tokyo 24 Aug 1997
//

#ifndef __uchache_h__
#define __uchache_h__

#include "UCommon.h"
#include <stdio.h>


template <class _UCacheItem>
class UCache
{
	typedef struct _UCacheStruct
	{
		char		*name;  // the name of item in cache
		_UCacheItem	item;	// the item in cache
		int		usage;	// number of clients
		int		purge;	// the time when it became unused
	} UCacheStruct;

public:
enum	UStatus { OK=0, DUPLICATE, ERROR };

//
// Default constructor UCache
// if free items > cacheBatchSizeIn automatic purge will happen.

UCache(const int cacheSizeIn=0, const int cacheBatchSizeIn=0)
{
	itemCount=0;
	items = 0;
	freeCount = 0;
	purgeTime = 0;
	cacheBatchSize=cacheBatchSizeIn;
	cacheSize=cacheSizeIn;
	arraySize = 0;
}

//
// Destructor
//
~UCache()
{
	int		i;
	for (i=0; i<itemCount; i++)
	{
		delete (items[i]->name);
		delete (items[i]->item);
		delete (items[i]);
	}
	delete items;
}

//
// Add an item.  If this returns OK then the item will be owned by the cache.
//
UStatus
addItem (const char *name, _UCacheItem item)
{
	int			i;
	UCacheStruct**		newItems;

	for (i=0; i<itemCount; i++)
	{
		if (strcmp (items[i]->name, name) == 0) 
			return DUPLICATE;
	}

	// autoresize
	if (itemCount % GRANUALITY == 0 && arraySize <= itemCount)
	{
		newItems = new UCacheStruct*[itemCount + GRANUALITY];
		memset (newItems, 0, itemCount * sizeof (UCacheStruct*));
		CHECKNULL (newItems);
		if (itemCount)
		{
			memcpy (newItems, items,
				itemCount * sizeof (UCacheStruct*));
			delete items;
		}
		items = newItems;
		arraySize = itemCount + GRANUALITY;
	}
	items[itemCount] = new UCacheStruct[1];
	items[itemCount]->name = new char [strlen (name) + 1];
	CHECKNULL (items[itemCount]->name);
	strcpy (items[itemCount]->name, name);
	items[itemCount]->item = item;
	items[itemCount]->purge = -1;
	items[itemCount]->usage = 1;
	itemCount++;
	return OK;
}

//
// Drop an item from cache
//
UStatus
dropItem (const char *name)
{
	int	i;
	for (i=0; i<itemCount; i++)
	{
		if (strcmp (name, items[i]->name)!=0) continue;
		if (items[i]->usage != 0)
		{
			cerr << "error: attempted to delete used "
				<< "cache item '" << name << "'.\n";
			return ERROR;
		}
		delete items[i]->name;
		delete items[i]->item;
		delete items[i];
		while (i<itemCount-1)
		{
			items[i] = items[i+1];
			i++;
		}
		itemCount--;
		freeCount--;
		return OK;
	}
	return ERROR;
}

//
// Drop items, the oldest  from cache
// Faster version.
//
UStatus
purge (int leave=0)
{
	int		from;
	int		to;
	int		oldCount;

	oldCount = itemCount;
	for (from=0, to=0; from < oldCount; from++, to++)
	{
		while (from < oldCount && items[from]->usage==0)
		{
			// leave items of cacheSize in the cache
			if (items[from]->purge >= purgeTime - cacheSize)
			{
				break;
			}
#ifdef DEBUG
			cerr << "info: purging '" << items[from]->name 
				<< "' from cache.\n";
#endif /* DEBUG */
			delete items[from]->name;
			delete items[from]->item;
			delete items[from];
			itemCount--;
			from++;
		}
		if (from >=  oldCount) break;
		if (from==to) continue;
		items[to] = items[from];
	}
	if (from==to) return OK;
	// renumber purge times.
	purgeTime=0;
	for (from = 0; from < itemCount; from++)
	{
		if (items[from]->usage == 0)
		{
			items[from]->purge -= cacheSize;
		}
	}
	purgeTime = cacheSize;
	freeCount = 0;
	return OK;
}

//
// Retrieve the item from cache.
//
_UCacheItem
getItem (const char *name)
{
	int	i;
	for (i=0; i<itemCount; i++)
	{
		if (strcmp (name, items[i]->name) ==0)
		{
			return items[i]->item;
		}
	}
	return 0;
}

//
// increment usage flag
//
void
useItem (const char *name)
{
	int	i;
	for (i=0; i<itemCount; i++)
	{
		if (strcmp (name, items[i]->name)==0)
		{
			break;
		}
	}
	if (i == itemCount) return;
	if (items[i]->usage==0)
	{
		freeCount--;
	}
	items[i]->usage++;
}
//
// decrement usage flag
//
void
unuseItem (const char *name)
{
	int	i;
	for (i=0; i<itemCount; i++)
	{
		if (strcmp (name, items[i]->name)==0)
		{
			break;
		}
	}
	if (i == itemCount) return;
	if (items[i]->usage==0)
	{
		cerr << "warn: cached '" << name << "' is already unused.\n";
		return;
	}
	items[i]->usage--;
	if (items[i]->usage==0)
	{
		items[i]->purge = purgeTime;
		purgeTime++;
		freeCount++;
	}
	if (freeCount > cacheBatchSize+cacheSize)
	{
		purge (cacheSize);
	}
}

inline void
tune (int cacheSizeIn=0, int cacheBatchSizeIn=0)
{
	cacheSize=cacheSizeIn;
	cacheBatchSize=cacheBatchSizeIn;
}

inline int
getItemCount() { return itemCount ; }

private:
	enum	UGranual {	GRANUALITY=32 };
	int			cacheSize; 
	int			cacheBatchSize;

	int			freeCount;
	int			purgeTime;

	UCacheStruct**		items;
	int			arraySize; // the size of items array
	int			itemCount; // the number of valid items in items

};

#endif /* __uchache_h__ */
