// Copyright (c) 1996-1999 The University of Cincinnati.  
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF 
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey	phil.wilsey@uc.edu
//          Dale E. Martin	dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu
//          Malolan Chetlur     mal@ece.uc.edu
//          Krishnan Subramani  skrish@ece.uc.edu
//          Radharamanan Radhakrishnan  ramanan@ece.uc.edu
//          Narayanan Thondugulam nthondug@ece.uc.edu
//          Swaminathan Subramanian ssubrama@ececs.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: IIRScram_ArchitectureDeclaration.cc,v 1.14 1999/10/10 19:08:57 dmartin Exp $
// 
//---------------------------------------------------------------------------
#include "IIR_ArchitectureDeclaration.hh"
#include "IIR_AliasDeclaration.hh"
#include "IIR_AssociationElement.hh"
#include "IIR_Attribute.hh"
#include "IIR_BlockStatement.hh"
#include "IIR_ConcurrentStatement.hh"
#include "IIR_ComponentDeclaration.hh"
#include "IIR_ComponentInstantiationStatement.hh"
#include "IIR_ConcurrentGenerateIfStatement.hh"
#include "IIR_ConcurrentGenerateForStatement.hh"
#include "IIR_ConfigurationSpecification.hh"
#include "IIR_Designator.hh"
#include "IIR_EntityDeclaration.hh"
#include "IIR_FileDeclaration.hh"
#include "IIR_TextLiteral.hh"
#include "IIR_Label.hh"
#include "IIR_Name.hh"
#include "IIR_ProcessStatement.hh"
#include "IIR_SignalInterfaceDeclaration.hh"
#include "IIR_SignalDeclaration.hh"
#include "IIR_TypeDefinition.hh"
#include "IIR_UseClause.hh"
#include "IIR_WaitStatement.hh"
#include "error_func.hh"
#include "resolution_func.hh"
#include "symbol_table.hh"

extern symbol_table *cgen_sym_tab_ptr;

#ifdef PROCESS_COMBINATION
#include "ProcessCombiner.hh"
extern int num_partitions;
extern char *partition_algorithm;
extern void cone_partition( IIR_ArchitectureDeclaration*, 
			    IIR_ConcurrentStatementList&, ProcessCombiner* );
#include <stdlib.h> // for rand()
#endif

#ifdef PROCESS_GRAPH
extern bool signal_graph;
#endif

IIRScram_ArchitectureDeclaration::IIRScram_ArchitectureDeclaration() {
  cgen_sym_tab_ptr = NULL;
}


IIRScram_ArchitectureDeclaration::~IIRScram_ArchitectureDeclaration() { }


void 
IIRScram_ArchitectureDeclaration::_publish_vhdl_decl(ostream &_vhdl_out) {
  PublishedUnit oldUnit = _get_currently_publishing_vhdl_unit();

  _set_currently_publishing_vhdl_unit(IIRScram::ARCHITECTURE_DECL);
  
  context_items._publish_vhdl(_vhdl_out);

  _vhdl_out << "architecture ";
  get_declarator()->_publish_vhdl(_vhdl_out);
  _vhdl_out  << " of ";
  get_entity()->_publish_vhdl(_vhdl_out);
  _vhdl_out << " is\n";
  architecture_declarative_part._publish_vhdl_decl(_vhdl_out);
  _vhdl_out << "\nbegin\n";
  architecture_statement_part._publish_vhdl(_vhdl_out);
  _vhdl_out << "end architecture ";
  get_declarator()->_publish_vhdl(_vhdl_out);
  _vhdl_out << ";\n\n";

  _set_currently_publishing_vhdl_unit(oldUnit);
}


void 
IIRScram_ArchitectureDeclaration::_publish_vhdl(ostream &_vhdl_out) {
  get_declarator()->_publish_vhdl(_vhdl_out);
}


void
IIRScram_ArchitectureDeclaration::_publish_vhdl_binding_name(ostream &_vhdl_out){
  _vhdl_out << " entity ";
  get_entity()->_publish_vhdl(_vhdl_out);
  _vhdl_out << "(";
  get_declarator()->_publish_vhdl(_vhdl_out);
  _vhdl_out << ") ";
}


void
IIRScram_ArchitectureDeclaration::_publish_vhdl_with_library_name(ostream &_vhdl_out){
  _vhdl_out << " entity ";
  ASSERT(_get_declarative_region() != NULL);
  ASSERT(_get_declarative_region()->get_kind() == IIR_LIBRARY_DECLARATION);
  _get_declarative_region()->_publish_vhdl(_vhdl_out);
  _vhdl_out << ".";
  get_entity()->_publish_vhdl(_vhdl_out);
  _vhdl_out << "(";
  get_declarator()->_publish_vhdl(_vhdl_out);
  _vhdl_out << ")";
}


void 
IIRScram_ArchitectureDeclaration::_type_check( ){
  architecture_declarative_part._type_check_configuration_specifications( architecture_statement_part );
  architecture_declarative_part._type_check_disconnection_specifications(  );
  architecture_declarative_part._type_check_attribute_specifications( architecture_statement_part );
  architecture_statement_part._type_check_instantiate_statements();
}

IIR_PortList *
IIRScram_ArchitectureDeclaration::_get_port_list(){
  ASSERT( get_entity() != NULL );
  return get_entity()->_get_port_list();
}


IIR_GenericList *
IIRScram_ArchitectureDeclaration::_get_generic_list(){
  ASSERT( get_entity() != NULL );
  return get_entity()->_get_generic_list();
}

IIR_List *
IIRScram_ArchitectureDeclaration::_get_statement_list(){
  return &architecture_statement_part;
}

IIR_EntityDeclaration*
IIRScram_ArchitectureDeclaration::_get_entity() {
  return get_entity();
}


void 
IIRScram_ArchitectureDeclaration::_publish_cc() {
  IIR* temp = _current_publish_node;
  ostrstream ent_arch;
  IIR_Char* design_unit_name;
  IIR_Char* old_current_publish_name = _current_publish_name;
  symbol_table cgen_sym_tab( 4093, false );
  
  cgen_sym_tab_ptr = &cgen_sym_tab;

  _set_currently_publishing_unit(ARCHITECTURE_DECL);
  _current_publish_node = this;

  ent_arch << *_get_entity()->_get_declarator() << "_"
	   << *_get_declarator() << ends;
  design_unit_name = ent_arch.str();

  _current_entity_name = get_entity()->_get_declarator()->_convert_to_c_string();
  _current_architecture_name = _get_declarator()->_convert_to_c_string();

  cerr << "Publishing design unit: " << _current_entity_name
       << "(" << _current_architecture_name << ")\n";

  _current_publish_name = "SEA";

  // Check and group component instantiation statements for large designs
  _group_component_instantiations(&architecture_statement_part, 300);
  
  architecture_statement_part._publish_cc();
  cgen_sym_tab_ptr = &cgen_sym_tab;
  _publish_cc_declarations();
  
  _current_publish_name = old_current_publish_name;
  _current_publish_node = temp;
  
  _set_currently_publishing_unit(NONE);
  
  delete [] design_unit_name;
  cgen_sym_tab_ptr = NULL;
}


void 
IIRScram_ArchitectureDeclaration::_publish_cc_declarations() {
  IIR_Char* design_unit_name;
  ostrstream ent_arch;

   ent_arch << *get_entity()->_get_declarator() << "_"
	   << *_get_declarator() << ends;
  design_unit_name = ent_arch.str();

  _cc_out.set_file(TRUE, design_unit_name, "_decls.hh");
  _cc_out << "#ifndef " << design_unit_name << "_DECLS_HH\n";
  _cc_out << "#define " << design_unit_name << "_DECLS_HH\n\n";
  _cc_out << "#include \"";


  get_entity()->_get_declarator()->_publish_cc();
  _cc_out << "_decls.hh\"\n\n";
  
  context_items._publish_cc();
  architecture_declarative_part._publish_cc();
  architecture_declarative_part._publish_cc_extern_type_info();
  _cc_out.set_file(TRUE, design_unit_name, "_decls.hh");
  _cc_out << "#endif\n";

  _cc_out.set_file(TRUE, design_unit_name, "_decls.cc");
  _cc_out << "#include \"" << design_unit_name << "_decls.hh\"\n\n";
  architecture_declarative_part._publish_cc_decl();

  delete [] design_unit_name;
}  


IIRScram_Declaration::declaration_type 
IIRScram_ArchitectureDeclaration::_get_type(){
  return ARCHITECTURE;
}


void
IIRScram_ArchitectureDeclaration::_publish_cc_elaborate() {
  _current_configuration_name = NULL;

  _current_entity_name = get_entity()->_get_declarator()->_convert_to_c_string();
  _current_architecture_name = _get_declarator()->_convert_to_c_string();
  _current_publish_name = "SEA";
  
  _publish_cc_headerfile();
  _publish_cc_ccfile();

  _publish_cc_blocks_elaborate();
}


void
IIRScram_ArchitectureDeclaration::_publish_cc_blocks_elaborate() {
  IIR_ConcurrentStatement* conc_stmt;
  char *temp = _current_publish_name;

  _current_publish_name = "SEA";
  
  conc_stmt = architecture_statement_part.first();
  while (conc_stmt != NULL) {
    if (conc_stmt->get_kind() == IIR_BLOCK_STATEMENT) {
      conc_stmt->_publish_cc_elaborate();
    }
    if (conc_stmt->get_kind() == IIR_CONCURRENT_GENERATE_FOR_STATEMENT) {
      conc_stmt->_publish_cc_elaborate();
    }
    if (conc_stmt->get_kind() == IIR_CONCURRENT_GENERATE_IF_STATEMENT) {
      conc_stmt->_publish_cc_elaborate();
    }
    
    conc_stmt = architecture_statement_part.successor(conc_stmt);
  }
  _current_publish_name = temp;
}


void
IIRScram_ArchitectureDeclaration::_publish_cc_headerfile() {
  ostrstream filename;
  char *fname;

  filename << "SEA";
  get_entity()->_get_declarator()->_print(filename);
  filename << "_";
  _get_declarator()->_print(filename);
  filename << ends;
  fname = filename.str();
  
  // Publish the .hh file
  _cc_out.set_file(TRUE, fname, "_elab.hh");
  _cc_out << "#ifndef " << fname <<"_HH\n";
  _cc_out << "#define " << fname <<"_HH\n";
  delete [] fname;

  _cc_out << "// Header files needed for this architecture\n";
  // #include the entity_elab header file
  _cc_out << "#include \"SE";
  get_entity()->_get_declarator()->_publish_cc_elaborate();
  _cc_out  <<"_elab.hh\"\n";
  // Just include the decls file which has all the required includes
  _cc_out << "#include \"";
  _cc_out << *_get_entity()->_get_declarator() << "_"
	  << *_get_declarator() << "_decls.hh\"" << endl;
  architecture_declarative_part._publish_cc_extern_type_info();
  architecture_statement_part._publish_cc_extern_type_info();
  _cc_out << "\n// Forward reference classes needed for this architecture\n";
  _publish_cc_class_includes(&architecture_statement_part);


  _cc_out << "\n// Include any configuration specifications specified\n\n";
  IIR_Declaration* decl = architecture_declarative_part.first();
  while(decl != NULL) {
    if(decl->get_kind() == IIR_CONFIGURATION_SPECIFICATION) {
      if (((IIR_ConfigurationSpecification*)decl)->get_entity_aspect() != NULL) {
	_cc_out << "#include \"";
	((IIR_ConfigurationSpecification*)decl)->get_entity_aspect()->_publish_cc_binding_name();
	_cc_out << "_elab.hh\"\n";
      }
    }
    decl =  architecture_declarative_part.successor(decl);
  }
  
  _cc_out << "\n// Elaboration Class for the this architecture\n";
  
  _publish_cc_class();
  _cc_out << "#endif\n";
}

IIR_DeclarationList*
IIRScram_ArchitectureDeclaration::_get_declaration_list() {
  return &architecture_declarative_part;
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_class() {
  _cc_out << "class ";
  _publish_cc_binding_name();
  _cc_out << "_elab : public ";
  get_entity()->_publish_cc_binding_name();
  _cc_out << "_elab {\n";
  _cc_out << "public:\n";

  // Publish the constructor and destructor
  this->_publish_cc_binding_name();
  _cc_out << "_elab();\n";

  IIR_EntityDeclaration* entitydecl = get_entity();
  if(entitydecl->generic_clause.num_elements() > 0) {
    _publish_cc_binding_name();    
    _cc_out << "_elab(\n";
    entitydecl->generic_clause._publish_generic_parameter_list();
    _cc_out << " );\n";
  }

  _cc_out << "~";
  this->_publish_cc_binding_name();
  _cc_out << "_elab();\n";
  
  _cc_out << "void instantiate();\n";
  _cc_out << "void createNetInfo();\n";
  _cc_out << "void connect(int, int, ...);\n";
  _cc_out << "void partition() {}\n";

  // Publish any constants declaraed here so that the other declarations
  // can use them without any trouble.
  architecture_declarative_part._publish_cc_constants();
  
  //Publish the signal info data members
  _publish_cc_signals(&architecture_declarative_part);
  
  // Since the implicit declarations for the signals declared in the port
  // clause in the entity is NULL. But get_entity()->port_clause in the
  // architecture declaration gives implicit declartions for the signals
  // in the port clause of the entity also. we are going to publish the
  // implicit declarations for those signals here in the architecture
  _publish_cc_implicit_signals(&(get_entity()->port_clause));
  
  // Publish the pointers to processes and objects used in this architecture
  _publish_cc_object_pointers(&architecture_statement_part);
  
  architecture_declarative_part._publish_cc_file_objects();
  _cc_out << "};\n";
}

void
IIRScram_ArchitectureDeclaration::_publish_cc_ccfile() {
  ostrstream filename;
  char *fname;
  IIR_ConcurrentStatementList* conc_stmt_list = &architecture_statement_part;
  IIR_ConcurrentStatement *conc_stmt = NULL;
  filename << "SEA";
  (get_entity()->_get_declarator())->_print(filename);
  filename << "_";
  _get_declarator()->_print(filename);
  filename << ends;
  fname = filename.str();
  
  // Publish the .hh file
  _cc_out.set_file(TRUE, fname, "_elab.cc");
  _cc_out << "#include \"" << fname << "_elab.hh\"\n";
  
  conc_stmt = conc_stmt_list->first();
  while(conc_stmt != NULL) {
    if (conc_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT)  {
      IIR_ComponentInstantiationStatement* compInstStmt = (IIR_ComponentInstantiationStatement *) conc_stmt;
      if ((compInstStmt->_get_configuration() != NULL) &&
	  (compInstStmt->_get_configuration()->get_kind() ==
	   IIR_COMPONENT_CONFIGURATION) &&
	  (compInstStmt->_get_configuration()->_get_entity_aspect() != NULL)) {
	_cc_out << "#include \""; 
	((IIR_ComponentInstantiationStatement*)conc_stmt)->_get_configuration()->_get_entity_aspect()->_publish_cc_binding_name();
	_cc_out << "_elab.hh\"" << endl;
      }
    }
    conc_stmt = conc_stmt_list->successor(conc_stmt);
    
  }
  
  delete [] fname;

  _publish_cc_headerfiles_for_cc();
  _cc_out << "extern OBJTYPE* proc_array[];\n";
#ifdef PROCESS_GRAPH
  if (signal_graph == TRUE) {
    _cc_out << "extern ofstream fp;\n";
  }
#endif
  //Publish the typeInfo's for the scalar types Moved to the .hh file
  architecture_declarative_part._publish_cc_type_info();
  architecture_statement_part._publish_cc_type_info();
  _publish_cc_constructor();
  _publish_cc_destructor(&architecture_statement_part);
  _publish_cc_instantiate();
  _publish_cc_createNetInfo();
  _publish_cc_connect();
}


void
IIRScram_ArchitectureDeclaration::_publish_cc_headerfiles_for_cc() {
  IIR_EntityDeclaration *entitydecl = get_entity();
  // Publish the header files for the binding architectures specified
  // in configuration specification

  // Publish the forward references for the entity statement part
  IIR_ConcurrentStatement *conc_stmt = architecture_statement_part.first();
  while (conc_stmt != NULL) {
    //###currently the process name doesn't include its entity and
    //architecture name
    switch( conc_stmt->get_kind() ){
    case IIR_PROCESS_STATEMENT:
      _cc_out << "#include \"";
      conc_stmt->_publish_cc_binding_name( _cc_out );
      _cc_out << ".hh\"\n";      
      break;

    case IIR_CONCURRENT_GENERATE_FOR_STATEMENT:
    case IIR_CONCURRENT_GENERATE_IF_STATEMENT:
      conc_stmt->_publish_cc_headerfiles_for_cc_default();
      break;

    case IIR_BLOCK_STATEMENT:
      _cc_out << "#include \"";
      _cc_out << "SB" << _current_entity_name
	      << "_" << _current_architecture_name << "_"
	      << *(conc_stmt->_get_label()) << "_elab.hh\"\n";
      break;
      
    case IIR_COMPONENT_INSTANTIATION_STATEMENT:
      IIR_ComponentInstantiationStatement *as_component_instantiation_stmt;
      as_component_instantiation_stmt = (IIR_ComponentInstantiationStatement *)conc_stmt;
      IIR_AssociationElement *asselem;

      asselem = as_component_instantiation_stmt->port_map_aspect.first();
      while( asselem != NULL ){
	if (asselem->_get_actual() != NULL) {
	  ASSERT( asselem->get_formal() != NULL );
	  // Okay. The actual is not open
	  if ((asselem->get_formal()->get_kind() == IIR_FUNCTION_CALL) ||
	      (asselem->_get_actual()->get_kind() == IIR_FUNCTION_CALL)) {
	    _cc_out << "#include \"TypeConvert.hh\"\n";
	    // We don't need to publish this multiple times.
	    break;
	  }
	}
	asselem = as_component_instantiation_stmt->port_map_aspect.successor(asselem);
      }
      break;
    default:
      //Nothing to be done for other statements
      ;
    }
    conc_stmt = architecture_statement_part.successor(conc_stmt);
  }

  // Publish the header files for the processes defined in the entity
  conc_stmt = entitydecl->entity_statement_part.first();
  while (conc_stmt != NULL) {
    if (conc_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      // What about processes with no labels?  Also, if there is going to
      // be one more "get_kind" checked, go to a case statment.  Actually,
      // it's not even possible to have a process in the statements of an
      // entity, is it?  Check the LRM.
      _cc_out << "#include \"" << *(conc_stmt->_get_label()) 
	      << ".hh\"\n";
    }
    else {
      cerr << "IIR_ArchitectureDeclaration::_publish_cc_headefiles_for_cc() - "
	   << "Un-handled concurrent statement (" << conc_stmt->get_kind()
	   << ") in entity\n";
    }

    conc_stmt = entitydecl->entity_statement_part.successor(conc_stmt);
  }

  // Include the architecture decls header file

  _cc_out << "#include \"";
  _cc_out << *(get_entity()->_get_declarator()) << "_"
	  << *(_get_declarator())
    	  << "_decls.hh\"\n";
}


void
IIRScram_ArchitectureDeclaration::_publish_cc_constructor() {
  _publish_cc_constructor_with_no_arguments();
  if (get_entity()->generic_clause.num_elements() > 0) {
    _publish_cc_constructor_with_arguments();
  }
}


void
IIRScram_ArchitectureDeclaration::_publish_cc_constructor_with_no_arguments() {
  int firstFlag = 1;
  this->_publish_cc_binding_name();
  _cc_out << "_elab::";
  this->_publish_cc_binding_name();
  _cc_out << "_elab()";

  if (architecture_declarative_part._publish_cc_constants_init() == TRUE) {
    firstFlag = 0;
  }
  
  _publish_cc_signal_objects_init(firstFlag);
  
  _cc_out << "  {\n";
  architecture_declarative_part._publish_cc_global_constants_assignments();
  architecture_declarative_part._publish_cc_file_objects_init();
  _publish_cc_object_pointers_init();
  _cc_out << "}\n";
}


void
IIRScram_ArchitectureDeclaration::_publish_cc_constructor_with_arguments() {
  IIR_EntityDeclaration* entitydecl = get_entity();
  int numGenericClause = entitydecl->generic_clause.num_elements();

  this->_publish_cc_binding_name();
  _cc_out << "_elab::";
  this->_publish_cc_binding_name();
  _cc_out << "_elab(\n";
  entitydecl->generic_clause._publish_generic_parameter_list();
  _cc_out << " )";

  if(numGenericClause > 0) {
    _cc_out << ":\n";
    entitydecl->_publish_cc_binding_name();
    _cc_out << "_elab(";
    entitydecl->generic_clause._publish_generic_parameters_notypes();
    _cc_out << ") ";
    architecture_declarative_part._publish_cc_constants_init(FALSE);
  }
  else {
    architecture_declarative_part._publish_cc_constants_init(TRUE);
  }
  
  _publish_cc_signal_objects_init(0);
  _cc_out << " {\n";
  architecture_declarative_part._publish_cc_global_constants_assignments();
  architecture_declarative_part._publish_cc_file_objects_init();
  _publish_cc_object_pointers_init();
  _cc_out << "}\n";
}


void
IIRScram_ArchitectureDeclaration::_publish_cc_object_pointers_init() {
  // we are setting the _current_publish_node to the architecture declaration
  // so that if during the publish_cc_object_pointers_init for architecture declaration
  // the node needs to get the archicture declaration. it can use the _current
  // _publish_node variable.
  
  IIR *tempNode = _current_publish_node;
  get_entity()->_publish_cc_object_pointers_init();

  _current_publish_node = this;
  IIRScram::_publish_cc_object_pointers_init(&architecture_statement_part, &architecture_declarative_part);
  _current_publish_node = tempNode;
}



void
IIRScram_ArchitectureDeclaration::_publish_cc_instantiate() {
  IIR_ConcurrentStatement* conc_stmt;
  IIR_EntityDeclaration* entitydecl = get_entity();

  _cc_out << "void\n";
  this->_publish_cc_binding_name();
  _cc_out << "_elab::instantiate() {\n";

  // Publish instantiate stmts from entity statement part
  conc_stmt = entitydecl->entity_statement_part.first();
  while (conc_stmt!=NULL) {
    conc_stmt->_publish_cc_instantiate_call();
    conc_stmt = entitydecl->entity_statement_part.successor(conc_stmt);
  }

  // Publish instantiate stmts from architecture statement part
  conc_stmt = architecture_statement_part.first();
  while (conc_stmt != NULL) {
    conc_stmt->_publish_cc_instantiate_call();
    conc_stmt = architecture_statement_part.successor(conc_stmt);
  }
  _cc_out << "createNetInfo();\n";
  _cc_out << "}\n";
}


void
IIRScram_ArchitectureDeclaration::_publish_cc_createNetInfo() {
  IIR_ConcurrentStatement* conc_stmt;
  IIR_EntityDeclaration* entitydecl = get_entity();
  IIR_Boolean found = false;
  int wanted_instantiation = 1;

  IIR_Char* tmp = _current_elab_name;
  IIR_Char* tmp2 = _current_publish_name;
  IIR* tmpNode = _current_publish_node;

  _cc_out << "void\n";
  this->_publish_cc_binding_name();
  _cc_out << "_elab::createNetInfo() {\n";

  // Publish createNetInfo from entity statement part
  entitydecl->_publish_cc_createNetInfo();

  _publish_cc_anonymous_drivers(&(get_entity()->port_clause));
  _publish_cc_anonymous_drivers(&architecture_declarative_part);
  
  // Publish createNetinfo from architecture statement part
  conc_stmt = architecture_statement_part.first();
  while (conc_stmt!=NULL) {
    ostrstream objectname;
    _current_publish_name = NULL;
    _current_publish_node = NULL;

    switch( conc_stmt->get_kind() ){
    case IIR_PROCESS_STATEMENT:{
      if(conc_stmt->_get_label() != NULL) {
	objectname << *(conc_stmt->_get_label());
      }
      else {
	objectname << "ANON_PROCESS" << conc_stmt;
      }
      objectname << "_elab_obj" << ends;
      _current_elab_name = objectname.str();
      // No break here as we want the conc_stmt->_publish_createNetInfo() here.
    }
    case IIR_BLOCK_STATEMENT:
    case IIR_COMPONENT_INSTANTIATION_STATEMENT:{
      conc_stmt->_publish_createNetInfo();
      break;    
    }
    case IIR_CONCURRENT_GENERATE_FOR_STATEMENT:
    case IIR_CONCURRENT_GENERATE_IF_STATEMENT:
      break;
    default:
      cerr << "ERROR! IIRScram_ArchitectureDeclaration::"
	   << "_publish_cc_createNetInfo(): unknown conc_statement "
	   << "type |" << conc_stmt->get_kind_text() << "| in arch\n";
    }

    delete [] _current_elab_name;
    _current_elab_name = NULL;
    found = FALSE;
    wanted_instantiation = 1;
    conc_stmt = architecture_statement_part.successor(conc_stmt);
  }
  _cc_out << "}\n";
  _current_elab_name = tmp;
  _current_publish_name = tmp2;
  _current_publish_node = tmpNode;
}


void
IIRScram_ArchitectureDeclaration::_publish_cc_connect() {
  _cc_out << "void\n";
  this->_publish_cc_binding_name();
  _cc_out << "_elab::connect(int inputsignals, int outputsignals, ...) {\n";
  _cc_out << "int NoofSignals = inputsignals + outputsignals;";
  _cc_out << "va_list ap;\n";
  _cc_out << "opFanoutinfo = (VHDLType**)new char[NoofSignals * sizeof(VHDLType*)];\n";
  _cc_out << "va_start(ap, outputsignals);\n";
  _cc_out << "for(int i=0; i <NoofSignals; i++) {\n";
  _cc_out << "  opFanoutinfo[i] = va_arg(ap, VHDLType*);\n";
  _cc_out << "}\n";
  _cc_out << "va_end(ap);\n";

  //Pass on the output connection  inforamtion to its output signals
  IIR_EntityDeclaration* entitydecl = get_entity();
  IIR_SignalInterfaceDeclaration* portelement;
  IIR_Mode mode;
  int index = 0;

  _cc_out << "if(inputsignals > 0) {\n";

  portelement = entitydecl->port_clause.first();
  for(; portelement != NULL; ) {
    mode = portelement->get_mode();
    if(mode == IIR_IN_MODE ) {
      _cc_out << "setSourceInfo(";
      portelement->_publish_cc_elaborate(); 
      _cc_out << ",*(opFanoutinfo[";
      _cc_out << index << "]));\n";
      index++;
    }
    portelement = entitydecl->port_clause.successor(portelement);
  }

  _cc_out << "}\n";

  _cc_out << "if(outputsignals > 0) {\n";

  portelement = entitydecl->port_clause.first();
  for(; portelement != NULL; ) {
    mode = portelement->get_mode();
    if((mode == IIR_INOUT_MODE)) {
      _cc_out << "Add(";
      portelement->_publish_cc_elaborate(); 
      _cc_out << ",*(opFanoutinfo[";
      _cc_out << index << "]));\n";

      _cc_out << "setSourceInfo(";
      portelement->_publish_cc_elaborate(); 
      _cc_out << ",*(opFanoutinfo[";
      _cc_out << index << "]));\n";

      index++;
    } else if ( mode == IIR_OUT_MODE ) {
      _cc_out << "Add(";
      portelement->_publish_cc_elaborate(); 
      _cc_out << ",*(opFanoutinfo[";
      _cc_out << index << "]));\n";
      index++;
    } else if ( mode != IIR_IN_MODE ) {
      cerr << "IIRScram_ArchitectureDeclaration::_publish_cc_connect Invalid mode" << endl;
    }
    portelement = entitydecl->port_clause.successor(portelement);
  }

  _cc_out << "}\n";

  IIR_ConcurrentStatement* conc_stmt;
  // Publish output  connections for entity statement part
  conc_stmt = entitydecl->entity_statement_part.first();
  while (conc_stmt != NULL) {
    ostrstream objectname;
    if (conc_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT) {
      //###yet to check for label
      objectname << *(conc_stmt->_get_label()) << "_elab_obj" << ends;
      _current_elab_name = objectname.str();
      ((IIR_ComponentInstantiationStatement*)conc_stmt)->_publish_connect();
    }
    else {
      //NO need to do anything for other type of statements
    }
    //delete [] _current_elab_name;
    conc_stmt = entitydecl->entity_statement_part.successor(conc_stmt);
  }

  // Publish output connections for  architecture statement part
  conc_stmt = architecture_statement_part.first();
  while (conc_stmt != NULL) {
    ostrstream objectname;
    if (conc_stmt->get_kind() == IIR_COMPONENT_INSTANTIATION_STATEMENT) {
      //###yet to check for label
      objectname << *(conc_stmt->_get_label()) << "_elab_obj" << ends;
      _current_elab_name = objectname.str();
      ((IIR_ComponentInstantiationStatement*)conc_stmt)->_publish_connect();
#ifdef PROCESS_GRAPH
      if (signal_graph == TRUE) {
	((IIR_ComponentInstantiationStatement*)conc_stmt)->_publish_cc_driver_info();
      }
#endif
    }
    else if (conc_stmt->_is_concurrent_generate_statement() == TRUE) {
      conc_stmt->_get_label()->_publish_cc_elaborate();
      _cc_out << "_elab_obj->connect(0, 0);\n";
    }
    else if (conc_stmt->_is_block_statement() == TRUE) {
      ((IIR_BlockStatement *) conc_stmt)->_publish_connect();
    }
    else { 
      //No need to do anything for other type of statements
    }
    //delete [] _current_elab_name;
    conc_stmt = architecture_statement_part.successor(conc_stmt);
  }
#ifdef PROCESS_GRAPH
  if (signal_graph == TRUE) {
    _publish_cc_driver_info();
  }
#endif
  _cc_out << "}\n";

}


void
IIRScram_ArchitectureDeclaration::_publish_cc_binding_name(ostream& outstream){
  outstream << "SEA";
  get_entity()->_get_declarator()->_print(outstream);
  outstream << "_";
  _get_declarator()->_print(outstream);
}


void
IIRScram_ArchitectureDeclaration::_publish_cc_signal_objects_init(int flag) {
  IIR_Declaration* decl;
  int first = 0;
  decl = architecture_declarative_part.first();
  while (decl != NULL) {
    if(decl->get_kind() == IIR_SIGNAL_DECLARATION) {
      if((first == 0) && (flag == 1)) {
	_cc_out << ":\n";
	first = 1;
      }
      else {
	_cc_out << ",\n";
      }
      decl->_publish_cc_elaborate();
      _cc_out << "(ObjectBase::SIGNAL_NETINFO";
      if (decl->_get_subtype()->_is_scalar_type() == TRUE &&
	  decl->_get_subtype()->_is_kernel_type() == FALSE &&
	  decl->_get_subtype()->_is_anonymous() == FALSE){
	decl->_get_subtype()->_publish_cc_object_type_info();
      }
      if(decl->_get_subtype()->_is_array_type() &&
	 decl->_get_subtype()->_is_anonymous() == TRUE) {
	_cc_out << ", ";
	decl->_get_subtype()->_publish_cc_range();
      } else if(decl->_get_subtype()->_is_unconstrained_array_type()){
	ASSERT(decl->get_value() != NULL);
	if(decl->get_value()->get_kind() != IIR_CONCATENATION_OPERATOR){
	  _cc_out << ", ";
	  decl->get_value()->_publish_cc_range();
	}
      }
      _cc_out << ")";
      //If the Signals has implicit signals
      //They are also needed for elaboration info
      if( decl->_get_implicit_declarations() != NULL
	  && decl->_get_implicit_declarations()->num_elements() != 0) {
	IIR_Declaration* imp_decl = decl->_get_implicit_declarations()->get_element();
	while(imp_decl != NULL) {
	  if(imp_decl->_is_signal() == TRUE) {
	    _cc_out << ",\n";
	    imp_decl->_publish_cc_elaborate();
	    _cc_out << "(ObjectBase::SIGNAL_NETINFO";
	    IIR_TypeDefinition* type_def = imp_decl->_get_attribute_name()->_get_explicit_signal_type();
	    if(type_def->_is_array_type() &&
	       type_def->_is_anonymous() == TRUE) {
	      _cc_out << ", ";
	      type_def->_publish_cc_range();
	    }
	    _cc_out << ")";
	  }
	  imp_decl = decl->_get_implicit_declarations()->get_next_element();
	}
      }
    }
    else if(decl->get_kind() == IIR_ALIAS_DECLARATION) {
      if(((IIR_AliasDeclaration*)decl)->get_name()->_is_signal()) {
	if((first == 0) && (flag == 1)) {
	  _cc_out << ":\n";
	  first = 1;
	}
	else {
	  _cc_out << ",\n";
	}
	((IIR_AliasDeclaration*)decl)->_publish_cc_elaborate_alias_init();
      }
    }
    decl = architecture_declarative_part.successor(decl);    
  }
  
  // Publish the initializations for the implicit_declarations of the signals
  // in the port clause of the entity declartion.
  IIR_DeclarationList *port_clause_list = &(get_entity()->port_clause);
  decl = port_clause_list->first();
  while ( decl != NULL ) {
    IIR_Declaration *nextdecl = port_clause_list->successor(decl);
#ifdef PROCESS_COMBINATION
    // This should retrieve the previously built clone for the interface
    // signal.  The attributes have been attached to the clone, not the
    // original port.
    //decl = (IIR_Declaration*)decl->_clone();
#endif
    if( decl->_get_implicit_declarations() != NULL
	&& decl->_get_implicit_declarations()->num_elements() != 0) {
      IIR_Declaration* imp_decl = decl->_get_implicit_declarations()->get_element();
      while(imp_decl != NULL) {
	if(imp_decl->_is_signal() == TRUE) {
	  if ( first == 0 && flag == 1 ){
	    _cc_out << ":\n";
	    first = 1;
	  } else {
	    _cc_out << ",\n";
	  }
	  
	  imp_decl->_publish_cc_elaborate();
	  _cc_out << "(ObjectBase::SIGNAL_NETINFO";
	  IIR_TypeDefinition* type_def = imp_decl->_get_attribute_name()->_get_explicit_signal_type();
	  if(type_def->_is_array_type() &&
	     type_def->_is_anonymous() == TRUE) {
	    _cc_out << ", ";
	    type_def->_publish_cc_range();
	  }
	  _cc_out << ")";
	}
	imp_decl = decl->_get_implicit_declarations()->get_next_element();
      }
    }
    decl = nextdecl;
  }
  
}


void
IIRScram_ArchitectureDeclaration::_publish_cc_scoping_prefix(IIR *, IIR * ,ostream &)
{
  // Nothing to be done
}

#ifdef PROCESS_GRAPH
void
IIRScram_ArchitectureDeclaration::_publish_cc_driver_info(){
  IIR_EntityDeclaration* entitydecl = get_entity();
  IIR_SignalInterfaceDeclaration* portelement;
  IIR_ConcurrentStatement* conc_stmt;
  
  portelement = entitydecl->port_clause.first();
  for(; portelement != NULL; ) {
    if (portelement->_get_subtype()->_is_scalar_type() == TRUE ||
	portelement->_get_subtype()->_is_array_type() == TRUE) {
      _cc_out << "fp << \"Signal:";
      portelement->_publish_cc_elaborate();
      _cc_out << "\" << endl;" << endl;
      portelement->_publish_cc_elaborate();
      _cc_out << ".dump_connectivity_info(fp);" << endl;
    }
    portelement = entitydecl->port_clause.successor(portelement);
  }

  //dump_info for concurrent statements in the architecture body
  conc_stmt = architecture_statement_part.first();
  while (conc_stmt!=NULL) {
    if (conc_stmt->get_kind() == IIR_PROCESS_STATEMENT) {
      conc_stmt->_publish_cc_driver_info();
    }
    conc_stmt = architecture_statement_part.successor(conc_stmt);
  }
}
#endif

set<IIR_Declaration> *
IIRScram_ArchitectureDeclaration::_find_declarations( IIR_Name *to_find ){
  set<IIR_Declaration> *retval = new set<IIR_Declaration>;
  set<IIR_Declaration> *current_set = architecture_declarative_part._find_declarations( to_find );
  if( current_set != NULL ){
    retval->add( current_set );
    delete current_set;
  }

  ASSERT( get_entity() != NULL );
  current_set = get_entity()->_find_declarations( to_find );
  if( current_set != NULL ){
    retval->add( current_set );
    delete current_set;
  }
  
  if( retval->num_elements() == 0 ){
    delete retval;
    retval = NULL;
  }

  return retval;
}

void 
IIRScram_ArchitectureDeclaration::_make_interface_visible( symbol_table *sym_tab ){
  // This isn't _my_ interface...
  //   if( get_entity() != NULL ){
  //     get_entity()->_make_interface_visible( sym_tab );
  //   }

  sym_tab->make_visible( &architecture_declarative_part );
  IIR_ConcurrentStatement *current_statement;
  current_statement = architecture_statement_part.first();
  while( current_statement != NULL ){
    if( current_statement->get_label() != NULL ){
      sym_tab->make_visible( current_statement->get_label() );
    }
    current_statement = architecture_statement_part.successor( current_statement );
  }
}


void
IIRScram_ArchitectureDeclaration::_publish_cc_implicit_signals(IIR_DeclarationList *decl_list){
  ASSERT(decl_list != NULL);
  ASSERT((this->_is_concurrent_statement() == TRUE) ||
	 (this->_is_entity_decl() == TRUE) ||
	 (this->_is_component_decl() == TRUE) ||
	 (this->get_kind() == IIR_ARCHITECTURE_DECLARATION));
  
  IIR_Declaration* decl = decl_list->first();

  while (decl != NULL) {
    IIR_Declaration *nextdecl = decl_list->successor(decl);
#ifdef PROCESS_COMBINATION
    // This should retrieve the previously built clone for the interface
    // signal.  The attributes have been attached to the clone, not the
    // original port.
    //decl = (IIR_Declaration*)decl->_clone();
#endif
    if(decl->_is_signal() == TRUE) {
      //If the Signals has implicit signals
      //They are also needed for elaboration info
      if( decl->_get_implicit_declarations() != NULL &&
	  decl->_get_implicit_declarations()->num_elements() != 0) {
	IIR_Declaration* imp_decl = decl->_get_implicit_declarations()->get_element();
	while(imp_decl != NULL) {
	  if(imp_decl->_is_signal() == TRUE) {
	    imp_decl->_publish_cc_implicit_signal_type();
	    _cc_out << " ";
	    imp_decl->_publish_cc_elaborate();
	    _cc_out << ";\n";
	  }
	  imp_decl = decl->_get_implicit_declarations()->get_next_element();
	}
      }
    }
    else if(decl->get_kind() == IIR_ALIAS_DECLARATION) {
      if(decl->_is_signal() == TRUE) {
	((IIR_AliasDeclaration*)decl)->_publish_cc_elaborate_alias_definition();
      }
    }
    decl = nextdecl;
  }
}


#ifdef PROCESS_COMBINATION
void
IIRScram_ArchitectureDeclaration::_combine() {
  IIR_ConcurrentStatementList onewaitlist, uncondwaitlist;
  IIR_ConcurrentStatement *next;
  IIR_ProcessStatement *to_combine;
  ProcessCombiner *cproc;

  cerr << "The elaborated design has "
       << architecture_statement_part.num_elements() << " processes: ";

  // move all 1-wait processes from arch_stmt_part to onewaitlist
  to_combine = (IIR_ProcessStatement*)architecture_statement_part.first();
  while ( to_combine != NULL ) {
    next = architecture_statement_part.successor(to_combine);
    ASSERT( IIR_PROCESS_STATEMENT == to_combine->get_kind() );
    
    to_combine->_build_wait_list();
    if ( to_combine->_wait_stmt_list.num_elements() == 1 ) {
      IIR_WaitStatement *waitstmt = to_combine->_wait_stmt_list.first();
      if (waitstmt->get_timeout_clause() == NULL
	  && waitstmt->sensitivity_list.num_elements() == 0
	  && waitstmt->get_condition_clause() == NULL ) {
	// Put processes with one uncinditional wait in uncondwaitlist
	uncondwaitlist.append( to_combine );
      }
      else {
	onewaitlist.append( to_combine );
      }
      architecture_statement_part.remove( to_combine );
    }
    to_combine = (IIR_ProcessStatement*)next;
  }
  
  cerr << uncondwaitlist.num_elements() 
       << " with a single unconditional wait,\nand "
       << onewaitlist.num_elements()
       << " with a single combinable wait statement.\n";
  cerr << "Partitioning design into " << num_partitions << " partitions";

  // combine processes in unconditional wait list--put into arch_stmt_part
  if ( NULL != uncondwaitlist.first() ) {
    ProcessCombiner uncondwaitcombiner;
    to_combine = (IIR_ProcessStatement*)uncondwaitlist.first();

    while ( NULL != to_combine ) {
      next = uncondwaitlist.successor( to_combine );
      uncondwaitcombiner.combine_uncondwait( to_combine );
      uncondwaitlist.remove( to_combine );
      to_combine = (IIR_ProcessStatement*)next;
    }
    architecture_statement_part.append( uncondwaitcombiner.get_process() );
  }

  // OK, this check probably isn't necessary...
  if ( NULL != onewaitlist.first() ) {
    int partition = 0;
    cproc = new ProcessCombiner[num_partitions];
    to_combine = (IIR_ProcessStatement*)onewaitlist.first();

    while ( NULL != to_combine ) {
      next = onewaitlist.successor( to_combine );
      if ( !to_combine->_convert_to_TWF() ) {
	// If the process can't be converted to TWF, remove it from the
	// onewaitlist and add it back to the architecture
	architecture_statement_part.append( to_combine );
	onewaitlist.remove( to_combine );
      }
      to_combine = (IIR_ProcessStatement*)next;
    }
    
    
    // Partitoning algorithms start here.  Probably should be elsewhere.
    to_combine = (IIR_ProcessStatement*)onewaitlist.first();
    if ( partition_algorithm != NULL 
	 && !strncmp( partition_algorithm, "ra", 2 )) {
      cerr << " with random partitioning.\n";
      while ( NULL != to_combine ) {
	next = onewaitlist.successor( to_combine );
	partition = (int)(((float)num_partitions * (float)rand()) 
			  / (RAND_MAX + 1.0));
	cproc[partition].combine( to_combine );
	onewaitlist.remove( to_combine );
	to_combine = (IIR_ProcessStatement*)next;
      }
    }
    else if ( partition_algorithm != NULL
	      && !strncmp( partition_algorithm, "c", 1 )) {
      cerr << " with output-cone partitioning.\n";
      cone_partition( (IIR_ArchitectureDeclaration*)this, onewaitlist, cproc );
    }
    else { // Round Robin partitioning is the default
      cerr << " with round-robin partitioning.\n";
      
      while ( NULL != to_combine ) {
	next = onewaitlist.successor( to_combine );
	cproc[partition].combine( to_combine );
	partition = (partition + 1)%num_partitions;
	onewaitlist.remove( to_combine );
	to_combine = (IIR_ProcessStatement*)next;
      }
    }

    // add combined processes back to architecture
    for (partition = 0; partition < num_partitions; partition++ ) {
      architecture_statement_part.append( cproc[partition].get_process() );
    }
    delete [] cproc;
  }
}


void
IIRScram_ArchitectureDeclaration::_static_elaborate(IIR_ArchitectureDeclaration *arch, 
						    IIR_DeclarationList *cfglist,
						    char *hier_location) {
  IIR_Declaration *decl, *declclone, *impdecl;
  IIR_Identifier *new_signal_id, *old_id;

  decl = architecture_declarative_part.first();
  while (decl != NULL) {
    if (decl->get_kind() == IIR_COMPONENT_DECLARATION 
	|| decl->get_kind() == IIR_CONFIGURATION_SPECIFICATION) {
      declclone = decl;
    }
    else {
      declclone = (IIR_Declaration*)((IIR*)decl)->_clone();
      // mangle names for certain declarations
      if (declclone->get_kind() == IIR_SIGNAL_DECLARATION 
	  || declclone->get_kind() == IIR_FILE_DECLARATION
          || declclone->get_kind() == IIR_ALIAS_DECLARATION) {
	declclone->_static_elaborate_decl(hier_location);
      }
    }

    if( declclone->_get_implicit_declarations() != NULL ){
      // mangle names for all implicit decls
      impdecl = declclone->_get_implicit_declarations()->get_element();
      while (impdecl != NULL) {
	impdecl->_static_elaborate_decl(hier_location);
	impdecl = declclone->_get_implicit_declarations()->get_next_element();
      }
    }

    arch->architecture_declarative_part.append(declclone);
    decl = architecture_declarative_part.successor(decl);
  }

  decl = context_items.first();
  while ( decl != NULL ) {
    ASSERT( IIR_USE_CLAUSE == decl->get_kind() );
    if (arch->context_items.get_position( decl ) == -1 ) {
      arch->context_items.append( decl );
    }
    decl = context_items.successor( decl );
  }

  IIR_ConcurrentStatement *cstmt;
  IIR *clone;
  get_entity()->_static_elaborate(arch, cfglist, hier_location);

  cstmt = architecture_statement_part.first();
  while (cstmt != NULL) {
    if (cstmt->get_kind() == IIR_PROCESS_STATEMENT ||
	cstmt->get_kind() == IIR_SENSITIZED_PROCESS_STATEMENT ||
	cstmt->get_kind() == IIR_CONCURRENT_SELECTED_SIGNAL_ASSIGNMENT ||
	cstmt->get_kind() == IIR_CONCURRENT_CONDITIONAL_SIGNAL_ASSIGNMENT) {
      clone = cstmt->_clone();
      clone->_static_elaborate(arch, cfglist, hier_location);
      arch->architecture_statement_part.append(clone);
    }
    else {
      cstmt->_static_elaborate(arch, cfglist, hier_location);
    }
    cstmt = architecture_statement_part.successor(cstmt);
  }

  decl = architecture_declarative_part.first();
  while (decl != NULL) {
    switch (decl->get_kind()) {
    case IIR_SIGNAL_DECLARATION:
      ((IIRScram_SignalDeclaration*)decl)->_my_clone = NULL;
      break;
    case IIR_FILE_DECLARATION:
      ((IIRScram_FileDeclaration*)decl)->_my_clone = NULL;
      break;
    case IIR_ALIAS_DECLARATION:
      ((IIRScram_AliasDeclaration*)decl)->_my_clone = NULL;
      break;
    case IIR_COMPONENT_DECLARATION:
    case IIR_CONFIGURATION_SPECIFICATION:
      arch->architecture_declarative_part.remove(decl);
      break;
    default:
      break;
    }
    decl = architecture_declarative_part.successor(decl);
  }
}
#endif
