
// Copyright (c) 1993-1999 The University of Cincinnati.
// All rights reserved.

// UC MAKES NO REPRESENTATIONS OR WARRANTIES ABOUT THE SUITABILITY OF
// THE SOFTWARE, EITHER EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED
// TO THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
// PARTICULAR PURPOSE, OR NON-INFRINGEMENT.  UC SHALL NOT BE LIABLE
// FOR ANY DAMAGES SUFFERED BY LICENSEE AS A RESULT OF USING,
// RESULT OF USING, MODIFYING OR DISTRIBUTING THIS SOFTWARE OR ITS
// DERIVATIVES.

// By using or copying this Software, Licensee agrees to abide by the
// intellectual property laws, and all other applicable laws of the
// U.S., and the terms of this license.


// You may modify, distribute, and use the software contained in this package
// under the terms of the "GNU LIBRARY GENERAL PUBLIC LICENSE" version 2,
// June 1991. A copy of this license agreement can be found in the file
// "LGPL", distributed with this archive.

// Authors: Philip A. Wilsey    phil.wilsey@uc.edu
//          Dale E. Martin      dmartin@ece.uc.edu
//          Timothy J. McBrayer tmcbraye@ece.uc.edu

//---------------------------------------------------------------------------
// 
// $Id: resolution_func.cc,v 1.4 1999/10/07 14:05:18 dmartin Exp $
// 
//---------------------------------------------------------------------------

#include "resolution_func.hh"
#include "error_func.hh"
#include "set.hh"
#include "IIR_InterfaceDeclaration.hh"
#include "IIR_AssociationElementByExpression.hh"
#include "IIR_AssociationList.hh"
#include "IIR_FunctionCall.hh"
#include "IIR_SubprogramDeclaration.hh"
#include "IIR_EnumerationTypeDefinition.hh"
#include "IIR_IndexedName.hh"
#include "IIR_TypeDeclaration.hh"
#include "IIR_Identifier.hh"
#include "IIR_Signature.hh"
#include "IIR_Designator.hh"
#include "IIR_DesignatorExplicit.hh"
#include "symbol_table.hh"
#include "IIR_AccessTypeDefinition.hh"
#include "IIR_StringLiteral.hh"
#include "IIR_EnumerationLiteral.hh"
#include "IIR_ArchitectureDeclaration.hh"
#include "IIR_Aggregate.hh"
#include <string.h>
#include "IIR_CaseStatementAlternativeByChoices.hh"
#include "IIR_CaseStatementAlternativeByExpression.hh"
#include "IIR_CaseStatementAlternativeByOthers.hh"
#include "IIR_OthersInitialization.hh"
#include "IIR_Choice.hh"
#include "IIR_Label.hh"
#include "StandardPackage.hh"

IIR_Declaration *
resolve_if_one( set<IIR_Declaration> *set_to_resolve,
		IIR_Name *name_for_error ){

  IIR_Declaration *retval = NULL;
  switch ( set_to_resolve->num_elements() ){
  case 0:{
    if( name_for_error != NULL ){
      report_undefined_symbol( name_for_error );
    }
    break;
  }
  case 1:{
    retval = set_to_resolve->get_element();
    
    break;
  }
  default:{
    break;
  }
  }

  return retval;
}


int 
check_lval_rval_sets( set<IIR_TypeDefinition> *lval_set, 
		      set<IIR_TypeDefinition> *rval_set,
		      IIR *error_info){

  set<IIR_TypeDefinition> *intersect = set_intersect( lval_set, rval_set );
  int retval = intersect->num_elements();
    
  switch ( retval ){
  case 0:{
    if( error_info != NULL ){
      ostrstream err;
      
      err << "Incompatible types in assignment" << ends;
      if( error_info ){
	report_error( error_info, err );
      }
      else{
	report_error( err );
      }
      break;
    }
  }

  case 1:{
    
    break;
  }

  default:{
    if( error_info != NULL ){
      ostrstream err;
      
      err << "Ambiguous assignment" << ends;
      report_error( error_info, err );
    }
  }
  }

  return retval;
}


set<IIR_TypeDefinition> *
decl_set_to_typedef_set(set<IIR_Declaration> *decl_set,IIR_Boolean (IIR::*constraint_function)() ){
  set<IIR_TypeDefinition> *typedef_set = NULL;

  if( decl_set != NULL ){
    int num_elements = decl_set->num_elements();
    
    if( num_elements > 0 ){
      typedef_set = new set<IIR_TypeDefinition>;
      
      if( constraint_function != 0 ){
	decl_set->reduce_set( constraint_function );
      }

      IIR_TypeDefinition *current_rval = NULL;
      IIR_Declaration *current_decl = decl_set->get_element();
      while( current_decl != NULL ){
	current_rval = current_decl->_get_subtype();
	if( current_rval != NULL ){
	  typedef_set->add( current_rval );
	}
	current_decl = decl_set->get_next_element();
      }
    }
    // else nothing..
  }
  // else nothing..

  return typedef_set;
}


void 
reconcile_sets( set<IIR_TypeDefinition> *lval_set, 
		set<IIR_TypeDefinition> *rval_set ){
  
  ASSERT( lval_set != NULL );
  ASSERT( rval_set != NULL );
  
  set<IIR_TypeDefinition> original_set( *lval_set );
  set<IIR_TypeDefinition> new_rval_set;

  bool one_matched = false;
  IIR_TypeDefinition *current_lval = lval_set->get_element();
  current_lval = lval_set->get_element();
  while( current_lval != NULL ){
    IIR_TypeDefinition *current_rval = rval_set->get_element();
    one_matched = false;
    while( current_rval != NULL ){
      IIR_TypeDefinition *compatible_subtype = current_lval->_is_compatible( current_rval );
      if( compatible_subtype != NULL ){
	one_matched = true;
	new_rval_set.add( current_rval );
      }
      current_rval = rval_set->get_next_element();
    }
    if( one_matched == false ){
      lval_set->remove( current_lval );
    }
    current_lval = lval_set->get_next_element();
  }

  // So we have our matches in lval_set.  Let's build our rval_set.
  int num_elements = lval_set->num_elements();

  rval_set->reset( num_elements );
  *rval_set = new_rval_set;

  // Nothing to return - the sets are built!
}


void 
reconcile_sets( set<IIR_Declaration> *decls, 
		set<IIR_TypeDefinition> *rval_set ){

  ASSERT( decls != NULL );
  ASSERT( rval_set != NULL );
  
  set<IIR_Declaration> original_set( *decls );

  bool one_matched = false;
  // First try to do an exact subtype match...
  IIR_Declaration *current_decl = decls->get_element();
  while( current_decl != NULL ){
    if( current_decl->_get_subtype() != NULL ){
      if( rval_set->in_set( current_decl->_get_subtype() ) ){
	one_matched = true;
      }
      else {
	decls->remove( current_decl );
      }
    }

    current_decl = decls->get_next_element();
  }
  
  if( one_matched == false ){
    *decls = original_set;

    IIR_Declaration *current_decl = decls->get_element();
    while( current_decl != NULL ){
      if( current_decl->_get_subtype() != NULL ){
	one_matched = false;
	IIR_TypeDefinition *current_rval = rval_set->get_element();
	while( current_rval != NULL ){
	  if( current_decl->_get_subtype()->_is_compatible( current_rval ) != NULL ){
	    one_matched = TRUE;
	  }
	
	  current_rval = rval_set->get_next_element();
	} // inner loop

	if( one_matched == FALSE ){
	  decls->remove( current_decl );
	}
      }
      current_decl = decls->get_next_element();
    } // outer loop
  } // Check for simple compatibility

  // So we have our matches in decls.  Let's build our rval_set.
  int i;
  int num_elements = decls->num_elements();

  rval_set->reset( num_elements );
  
  current_decl = decls->get_element();
  if( current_decl != NULL ){
    if(  current_decl->_get_subtype() != NULL ){
      rval_set->add( current_decl->_get_subtype() );
    }
  }
  for( i = 1; i < num_elements;  i++ ){
    current_decl = decls->get_next_element();
    if(  current_decl->_get_subtype() != NULL ){
      rval_set->add( current_decl->_get_subtype() );
    }
  }

  // Nothing to return - the sets are built!
}


void 
resolve_subprogram_decls(set<IIR_Declaration> *possible_subprogram_decls,
			 IIR_AssociationList *argument_list,
			 const set<IIR_TypeDefinition> *possible_return_types ){

  ASSERT( possible_subprogram_decls != NULL );
  ASSERT( argument_list != NULL );

  // Consider each function, one at a time...
  bool arguments_match;

  IIR_Declaration *current_decl = possible_subprogram_decls->get_element();
  while( current_decl != NULL ){
    ASSERT( current_decl->_is_subprogram() == TRUE );
    IIR_SubprogramDeclaration *as_subprog = (IIR_SubprogramDeclaration *)current_decl;

    IIR_AssociationList temp_list;
    temp_list = *argument_list;

    arguments_match =
      temp_list._check_valid_arguments( &as_subprog->interface_declarations, NULL );

    if( arguments_match == FALSE ){
      possible_subprogram_decls->remove( current_decl );
    }
    current_decl = possible_subprogram_decls->get_next_element();
  }

  // So now we have all of the subprogram declarations whose arguments
  // match those passed in...  We need need to now check against the
  // return types...
 if( possible_return_types != NULL ){
   // This function (resolve_subprogram_decls) isn't supposed to
   // modify the set of possible return types.
   set<IIR_TypeDefinition> temp_return_types( (set<IIR_TypeDefinition> &)*possible_return_types );
   reconcile_sets( possible_subprogram_decls, &temp_return_types );
 }
}


void 
resolve_subprogram_decls(set<IIR_Declaration> *possible_subprogram_decls,
			 dl_list<set<IIR_TypeDefinition> > *parameter_rval_list,
			 set<IIR_TypeDefinition> *possible_return_types ){

  // Consider each function, one at a time...
  IIR_Declaration *current_subprog_decl; 
  bool one_matched;

  current_subprog_decl = possible_subprogram_decls->get_element();
  while( current_subprog_decl != NULL ){
    // Go through all of the parameters, and see if one matches..
    int i = 0;
    
    set<IIR_TypeDefinition> *current_parameter_rvals;

    current_parameter_rvals = parameter_rval_list->first();
    while( current_parameter_rvals != NULL ){
      // All of the possible r_vals for this argument...
      IIR_TypeDefinition *current_parameter_rval;
      current_parameter_rval = current_parameter_rvals->get_element();
      while( current_parameter_rval != NULL ){	
	if(current_subprog_decl->_check_param(current_parameter_rval, i)
	   == true){
	  one_matched = true;
	  break;
	}
	
	current_parameter_rval = current_parameter_rvals->get_next_element();
      }
    
      if( one_matched == false ){
	// Then this subprogram declaration COULD NOT be correct... (Given
	// positional association ONLY...
	possible_subprogram_decls->remove( current_subprog_decl );
	break;
      }
    
    
      current_parameter_rvals = 
	parameter_rval_list->successor( current_parameter_rvals );
      i++;
    }
    
    // Even though all of the arguments so far have matched, the subprog
    // might require more...
    if(((IIR_SubprogramDeclaration *)
	current_subprog_decl)->_num_required_args() >
       parameter_rval_list->num_elements() ){
      
      possible_subprogram_decls->remove( current_subprog_decl );
    }
    
    // So now we have all of the subprogram declarations whose arguments
    // match those passed in...  We need need to now check against the
    // return types...
    if( possible_return_types != NULL ){
      // This function (resolve_subprogram_decls) isn't supposed to
      // modify the set of possible return types.
      set<IIR_TypeDefinition> temp_return_types( *possible_return_types );
      reconcile_sets( possible_subprogram_decls, &temp_return_types );
    }
    
    current_subprog_decl = possible_subprogram_decls->get_next_element();
  }
}

void 
resolve_subprogram_decls( set<IIR_Declaration> *possible_subprogram_decls,
			  IIR_Signature *signature ){

  // So, we just need to build a dl_list of sets of r_vals for the
  // signature of this function, and one for the return type...
  IIR_TypeDefinition *return_type = signature->get_return_type();

  IIR_DesignatorExplicit *desig_explicit;

  dl_list<IIR_TypeDefinition> *type_def_list = new dl_list<IIR_TypeDefinition>;
  IIR_Designator *current_desig = signature->argument_type_list.first();
  while( current_desig != NULL ){
    ASSERT( current_desig->get_kind() == IIR_DESIGNATOR_EXPLICIT );
    desig_explicit = (IIR_DesignatorExplicit *)current_desig;
    ASSERT( desig_explicit->get_signature() == NULL );

    type_def_list->append( (IIR_TypeDefinition *)desig_explicit->get_name() );
    current_desig = signature->argument_type_list.successor( current_desig );
  }

  // Now we need to generate sets for all of the data we have, and pass
  // it on to the other function...
  set<IIR_TypeDefinition> return_set( return_type );
  dl_list<set<IIR_TypeDefinition> > signature_list;

  IIR_TypeDefinition *current_type_def;
  current_type_def = type_def_list->first();
  while( current_type_def != NULL ){
    set<IIR_TypeDefinition> *current_typedef_set;

    current_typedef_set = new set<IIR_TypeDefinition>( current_type_def );
    signature_list.append( current_typedef_set );

    current_type_def = type_def_list->successor( current_type_def );
  }

  resolve_subprogram_decls( possible_subprogram_decls,
			    &signature_list,
			    &return_set );
}

IIR_EnumerationLiteral *
get_true_literal( symbol_table *sym_tab ){
  return (IIR_EnumerationLiteral *)StandardPackage::boolean_type->get_right();
}

IIR_EnumerationLiteral *
get_false_literal( symbol_table *sym_tab ){
  return (IIR_EnumerationLiteral *)StandardPackage::boolean_type->get_left();
}

IIR *
reduce_aggregate( IIR_Aggregate *aggregate ){
  IIR *retval = aggregate;

  // If the aggregate only has one element, there's a good chance we can
  // optimize it down to it's one element, or at least the association.
  if( aggregate->element_association_list.num_elements() == 1 ){
    IIR_AssociationElement *first_element = aggregate->element_association_list.first();
    if( first_element->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION ){
      IIR_AssociationElementByExpression *by_expression;
      by_expression = (IIR_AssociationElementByExpression *)first_element;
      if( by_expression->get_formal() == NULL &&
	  ( by_expression->get_actual() != NULL && 
	    by_expression->get_actual()->get_kind() != IIR_OTHERS_INITIALIZATION ) ){
	retval = by_expression->get_actual();
	delete by_expression;
	delete aggregate;
      }
    }
  }

  return retval;
}


IIR *
reduce_association_list( IIR_AssociationList *list ){
  IIR *retval = list;

  if( list->num_elements() == 1 ){
    IIR_AssociationElement *first_element = list->first();
    if( first_element->get_kind() == IIR_ASSOCIATION_ELEMENT_BY_EXPRESSION ){
      IIR_AssociationElementByExpression *by_expr;
      by_expr = (IIR_AssociationElementByExpression *)first_element;
      if( by_expr->get_formal() == NULL ){
	retval = by_expr->get_actual();
	delete list;
	delete by_expr;
      }
    }
  }

  return retval;
}

void 
process_aggregate_with_choices( IIR *formal_part, IIR *actual_part, IIR_Aggregate *retval ){
  IIR_AssociationElementByExpression *new_association;
  if( formal_part == NULL ){
    ASSERT( actual_part->get_kind() == IIR_CASE_STATEMENT_ALTERNATIVE_BY_EXPRESSION );
    actual_part = ((IIR_CaseStatementAlternativeByExpression *)actual_part)->get_choice();
    ASSERT( actual_part->get_kind() == IIR_CHOICE );
    actual_part = ((IIR_Choice *)actual_part)->get_value();
    
    new_association = new IIR_AssociationElementByExpression();
    new_association->set_actual( actual_part );
    IIRScram::copy_location( actual_part, new_association );
    retval->element_association_list.append( new_association );  
  }
  else{
    IIR *new_formal = NULL;
    switch( formal_part->get_kind() ){
    case IIR_CASE_STATEMENT_ALTERNATIVE_BY_CHOICES:{
      IIR_CaseStatementAlternativeByChoices *as_choices;
      as_choices = (IIR_CaseStatementAlternativeByChoices *)formal_part;

      IIR_Choice *current_choice = as_choices->choices.first();
      while( current_choice != NULL ){
	new_association = new IIR_AssociationElementByExpression();

	new_association->set_formal( current_choice->get_value() );
	new_association->set_actual( actual_part );
	IIRScram::copy_location( actual_part, new_association );
	retval->element_association_list.append( new_association );  
	current_choice = as_choices->choices.successor( current_choice );
      }
      break;
    }
    case IIR_CASE_STATEMENT_ALTERNATIVE_BY_EXPRESSION:{
      new_association = new IIR_AssociationElementByExpression();
      new_formal = ((IIR_CaseStatementAlternativeByExpression *)formal_part)->get_choice();
      if( new_formal->get_kind() == IIR_CHOICE ){
	IIR *temp = ((IIR_Choice *)new_formal)->get_value();
	delete new_formal;
	new_formal = temp;
      }
      delete (IIR_CaseStatementAlternativeByExpression *)formal_part;
      new_association->set_formal( new_formal );
      new_association->set_actual( actual_part );
      IIRScram::copy_location( actual_part, new_association );
      retval->element_association_list.append( new_association );  
      break;
    }
    case IIR_CASE_STATEMENT_ALTERNATIVE_BY_OTHERS:{
      new_association = new IIR_AssociationElementByExpression();
      IIR_OthersInitialization *others = new IIR_OthersInitialization();
      IIR::copy_location( actual_part, others );
      IIR::copy_location( actual_part, new_association );
      others->set_expression( actual_part );
      delete (IIR_CaseStatementAlternativeByOthers *)formal_part;
      actual_part = others;
      formal_part = NULL;
      new_association->set_formal( new_formal );
      new_association->set_actual( actual_part );
      retval->element_association_list.append( new_association );  
      break;
    }
    default:{
      ostrstream err;
      err << "Internal error in VHDLParser::aggregate - unknown formal type "
	  << formal_part->get_kind_text() << ends;
      report_error( formal_part, err );
      abort();
    }
    }
  }
}

void 
check_name( IIR_TextLiteral *id, IIR_Name *name, char *declaration_type ){
  if( IIR_TextLiteral::_cmp( id, name ) != 0 ) {
    ostrstream err;
    err << "end |" << *name << "|"
       << " does not match " << declaration_type <<" name |" 
       << *id << "|." << ends;
    report_error( name, err );
  }
}

void 
check_name( IIR_TextLiteral *id, IIR_TextLiteral *name, char *declaration_type ){
  if( IIR_TextLiteral::_cmp( id, name ) != 0 ) {
    ostrstream err;
    err << "end |" << *name << "|"
       << " does not match " << declaration_type <<" name |" 
       << *id << "|." << ends;
    report_error( name, err );
  }
}


void 
check_name( IIR_Label *label, IIR_Name *name, char *declaration_type ){
  if( label == NULL && name != NULL ){
    ostrstream err;
    err << "This " << declaration_type << " does not have a label, "
	<< "and therefore |end " << *name << "| does not apply." << ends;
    report_error( name, err );
  }
  else{
    ASSERT( label != NULL && name != NULL );
    check_name( label->get_declarator(), name, declaration_type );
  }
}


// This function takes as its input three interface lists--the formal and
// actual lists to be associated together, and the default map to be
// created.
void
build_default_map( IIR_InterfaceList &formal_list,
		   IIR_InterfaceList &actual_list,
		   IIR_AssociationList &default_map) {

  ASSERT( (formal_list.get_kind()==IIR_PORT_LIST && actual_list.get_kind() == IIR_PORT_LIST)
	  || 
	  (formal_list.get_kind()==IIR_GENERIC_LIST&&actual_list.get_kind() == IIR_GENERIC_LIST));
  
  IIR_InterfaceDeclaration *formal, *actual;
  formal = formal_list.first();
  while ( NULL != formal ) {
    IIR_TextLiteral *formal_name = formal->_get_declarator();

    // Find matching actual for this formal
    actual = actual_list.first();
    while ( NULL != actual ) {
      IIR_TextLiteral *actual_name = actual->_get_declarator();
      if (0 == IIRScram_TextLiteral::_cmp(formal_name, actual_name)) {
	break; // found the matching name
      }
      actual = actual_list.successor( actual );
    }
    
    IIR_AssociationElementByExpression *new_association;
    new_association = new IIR_AssociationElementByExpression();

    ASSERT( formal != actual );

    new_association->set_formal( formal );
    
    if ( actual == NULL ) { // matching name not found
      // This is implementing section 1.1.1.1 of the LRM.
      if( formal->get_value() != NULL ){
	new_association->set_actual( formal->get_value() );	
      }
      else{
	ostrstream err;
	err << "No local " 
	    << ((formal_list.get_kind() == IIR_PORT_LIST)?"port":"generic")
	    << " found with name |" << *formal_name << "|, and no default expression "
	    << "specified for formal." << ends;
	report_error(&default_map, err);
      }
    }
    else{
      new_association->set_actual( actual );	
    }

    default_map.append( new_association );

    formal = formal_list.successor( formal );
  }
}
