/*C* -*-c++-*-
 *
 * Hatman - The Game of Kings
 * Copyright (C) 1997 James Pharaoh & Timothy Fisken
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 *C*/

#include "Background.h"
#include "BkCtrl.h"
#include "Data.h"
#include "files.h"
#include "Game.h"
#include "hatlvl.h"
#include "options.h"
#include "SpriteEditor.h"
#include "fonts.h"
#include "../control/ButtonCtrl.h"
#include "../control/ControlApp.h"
#include "../control/Events.h"
#include "../control/MenuCtrl.h"
#include "../control/TextCtrl.h"
#include "../util/debug.h"
#include "../util/error.h"
#include "../util/File.h"
#include "../gl/Console.h"
#include "../gl/Font.h"
#include "../gl/Keyboard.h"
#include "../gl/Mouse.h"
#include "../gl/Sprite.h"
#include "../gl/VgaBlur.h"
#include "../gl/keycodes.h"
#include "../viewer/viewer.h"
#include <dirent.h>
#include <stdio.h>
#include <stdlib.h>

//--------------------------------------------------------------------------------------------------------------------------------

static char startupMessage[] =
 "Hatman version " VERSION " - The Game of Kings\n"
 "Copyright (c) 1997-1999 by James Pharaoh & Timothy Fisken\n"
 "Hatman comes with ABSOLUTELY NO WARRANTY\n"
 "This is free software, and you are welcome to redistribute it under certain\n"
 "conditions; See the file COPYING for more details, which should have been\n"
 "distributed with the program.\n\n";

//--------------------------------------------------------------------------------------------------------------------------------

const int gotoExit      = 0;
const int gotoMainMenu  = 1;
const int gotoGamesMenu = 2;
const int gotoPlay      = 3;
const int gotoLevelEdit = 4;

//--------------------------------------------------------------------------------------------------------------------------------

const int idOk         = 1;
const int idExit       = 2;
const int idBack       = 3;
const int idPlay       = 4;
const int idHelp       = 5;
const int idLevelEdit  = 6;
const int idSpriteEdit = 7;
const int idRuntime    = 8;
 
//--------------------------------------------------------------------------------------------------------------------------------

class MainMenuCtrl : public Control
{
private:
 MenuCtrl* menuCtrl;

public:
 MainMenuCtrl(Events* _events)
  : Control(emptyRect, _events)
  {
   events->controls->add(new BkCtrl(events));
   events->controls->add(new TextCtrl(Rect(0, 0, vgaW, menuFont->h()), events, menuFont, "hatman "VERSION" - main menu"));
   events->controls->add(menuCtrl = new MenuCtrl(Rect(160, 120, 320, 240), events, this, menuFont));
   menuCtrl->add(idPlay, "play");
   menuCtrl->add(idHelp, "instructions");
   menuCtrl->add(idLevelEdit, "level editor");
   menuCtrl->add(idSpriteEdit, "sprite editor");
   menuCtrl->add(idExit, "exit");
   events->controls->add(new ButtonCtrl(Rect(0, 430, 320, 50), events, KEY_RETURN, this, idOk, menuFont, "ok"));
   events->controls->add(new ButtonCtrl(Rect(320, 430, 320, 50), events, KEY_ESCAPE, this, idExit, menuFont, "exit"));
  }
 
 virtual void evCommand(int id)
  {
   if(id == idOk) id = menuCtrl->itemId();
   switch(id)
    {
    case idExit: events->quit(gotoExit); break;
    case idPlay: events->quit(gotoGamesMenu); break;
    case idLevelEdit: events->quit(gotoLevelEdit); break;
    default: fatal("MainMenuCtrl::evCommand(%d): don't know what to do\n", id);
    }
  }
};

//--------------------------------------------------------------------------------------------------------------------------------

class GamesMenuItem : public Object
{
private:
 int pId;
 String pName, pDescrip, pFilename;

public:
 GamesMenuItem(int _id, const char* _name, const char* _descrip, const char* _filename);

 int id() { return pId; }
 const char* filename() { return pFilename; }
 const char* descrip() { return pDescrip; }
};

GamesMenuItem::GamesMenuItem(int _id, const char* _name, const char* _descrip, const char* _filename)
{
 pId = _id;
 pName = _name;
 pDescrip = _descrip;
 pFilename = _filename;
}

template Collection<GamesMenuItem>;

class GamesMenuCtrl : public Control
{
private:
 MenuCtrl* menuCtrl;
 Collection<GamesMenuItem>* items;
 TextCtrl* descripCtrl;

public:
 GamesMenuCtrl(Events* _events);
 ~GamesMenuCtrl();
 bool addGames();
 virtual void evCommand(int id);
 virtual void evContext(int id);
};

GamesMenuCtrl::GamesMenuCtrl(Events* _events)
 : Control(emptyRect, _events)
{
 items = new Collection<GamesMenuItem>;
 events->controls->add(new BkCtrl(events));
 events->controls->add(new TextCtrl(Rect(0, 0, vgaW, menuFont->h()), events, menuFont, "hatman "VERSION" - games menu"));
 events->controls->add(menuCtrl = new MenuCtrl(Rect(160, 120, 320, 240), events, this, menuFont));
 events->controls->add(descripCtrl = new TextCtrl(Rect(0, 430 - menuFont->h(), 640, menuFont->h()), events, menuFont, ""));
 events->controls->add(new ButtonCtrl(Rect(0, 430, 320, 50), events, KEY_RETURN, this, idOk, menuFont, "ok"));
 events->controls->add(new ButtonCtrl(Rect(320, 430, 320, 50), events, KEY_ESCAPE, this, idBack, menuFont, "back"));
}

GamesMenuCtrl::~GamesMenuCtrl()
{
 delete items;
}

bool GamesMenuCtrl::addGames()
{
 DIR* dp;
 dirent* ep;
 int itemId;

 if((dp = opendir(libFileName("games"))) == NULL) { setError("%s", errnoStr()); goto error; }

 itemId = idRuntime;
 while((ep = readdir(dp)))
  {
   if(strlen(ep->d_name) < 6 || strcmp(ep->d_name + strlen(ep->d_name) - 5, ".game")) continue;
   String filename(String("games/") + ep->d_name);
   VPRINTF("<menu> scanning %s\n", (const char*)filename);

   File f(libFileName(filename), "r");
   if(!(bool)f) { setError("%s", errStr); goto error; }

   // FIXME there is a limit imposed here, and there probablyy shouldn't be, in fact, this '|' business is very nasty
   char s[256];
   if(f.scanf("%255[^\n]\n", s) == 1 && s[0] == '|')
    {
     char *p = s + 1, *name = strsep(&p, "|"), *descrip = strsep(&p, "|"), *picname = strsep(&p, "|");

     VPRINTF("<hatman> game scanned: <%s> <%s> <%s>\n", name, descrip, picname); (void)descrip;

     menuCtrl->add(itemId, name ?: "??");
     items->add(new GamesMenuItem(itemId, name, descrip, filename));
     itemId++;
    }

   f.close();
  }
 if(items->length())
  descripCtrl->text(items->at(menuCtrl->sel())->descrip());
 return true;

error:
 return false;
}

void GamesMenuCtrl::evCommand(int id)
{
 if(id == idOk) id = menuCtrl->itemId();
 switch(id)
  {

  case idBack:
   events->quit(gotoMainMenu);
   return;

  default:
   foreach(*items, i)
    {
     GamesMenuItem* item = items->at(i);
     if(item->id() == id)
      {
       VPRINTF("<hatman> selected game %d\n", i);
       if(game->loadGame(item->filename()))
	events->quit(gotoPlay);
       else
	fatal("%s\n", errStr);
       return;
      }
    }
   fatal("GamesMenuCtrl::evCommand(%d): don't know what to do\n", id);

  }
}

void GamesMenuCtrl::evContext(int id)
{
 if(id >= idRuntime)
  {
   descripCtrl->text(items->at(menuCtrl->sel())->descrip());
   events->redraw(descripCtrl->rect());
  }
}

//--------------------------------------------------------------------------------------------------------------------------------

void titleScreen()
{
 Sprite* s;
 if((s = (Sprite*) load(libFileName("title.jpg"), (objectCreateFunc) Sprite::createJpeg)))
  {
   s->draw(screen, origin);
   screen->wipeUpdate(16, 50000);
  }
 else nonFatal("cannot show screen: %s\n", errStr);
}

//--------------------------------------------------------------------------------------------------------------------------------

/* This is of course, the main function. The idea is that this contains no
   real code, just calls to functions which do a) initialisation, b) main loop
   stuff c) shutting down. */

int main(int argc, char **argv)
{
 // output startup banner
 printf("\n");
 printf("%s", startupMessage);

 // check parameters
 if(!parseOptions(argc, argv))
  {
   nonFatal("\n");
   return EXIT_FAILURE;
  }

 initError();

 // initialise sound
#ifdef SOUND
 if(options->mode == Options::modePlay && options->sound) Effects = new Sound;
#else
 printf("Sound support disabled at compilation.\n");
#endif

 // init console library
 bpp = options->bitdepth;
 if(!consoleInit()) fatal("%s\n", errStr);
 screen->clipping = true;

 if(bpp == 8)
  nonFatal("Running in 8bpp graphics - shame, this looks crap :(\n");

 // inititialise background object
 bk = new Background;

 // initialise screen and stuff
 if(screen->graphicsMode())
  {
   titleScreen();

   if(!loadFonts()) fatal("%s\n", errStr);
   if(!loadData()) return EXIT_FAILURE;
   if(!bk->load(libFileName(fileBackground))) fatal("%s\n", errStr); // FIXME default should be black
   game = new Game;
   keyboard->open();
   mouse->sprite = (Sprite*) load(libFileName("mouse/default.spr"), (objectCreateFunc) Sprite::create);
   if(!mouse->sprite) fatal("Cannot load mouse pointer: %s\n", errStr);

   // this is the top-level main loop / distributor code

   int next = gotoMainMenu;
   while(next) switch(next)
    {

    case gotoMainMenu:
     {
      ControlApp app;
      app.controls->add(new MainMenuCtrl(app.events));
      next = app();
     }
     break;

    case gotoGamesMenu:
     {
      ControlApp app;
      GamesMenuCtrl* gamesMenuCtrl;
      app.controls->add(gamesMenuCtrl = new GamesMenuCtrl(app.events));
      if(!gamesMenuCtrl->addGames())
       fatal("%s\n", errStr);
      next = app();
     }
     break;

    case gotoPlay:
     game->playGame();
     next = gotoMainMenu;
     break;

    case gotoLevelEdit:
     levelEditor();
     next = gotoMainMenu;
     break;

    default:
     fatal("main loop got invalid next=%d\n", next);

    }

   // shut down
   delete game;
   killData();
   killFonts();
  }

 consoleExit();

#ifdef SOUND
 if(Options->mode == OptionsType::modePlay && Options->sound) delete Effects;
#endif

 // delete global objects
 delete bk;
 
 // leave a line, this looks nice :)
 printf("\n");

 return EXIT_SUCCESS;
}

//--------------------------------------------------------------------------------------------------------------------------------
