/*
 * File: sl_io.c
 * Author: Brent Hendricks
 * Project: NetSpades
 * Date: 7/26/97
 *
 * This file contains functions for displaying cards, the current
 * score, etc.
 *
 * Copyright (C) 1998 Brent Hendricks.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */


#include <config.h>   /* Site config data */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>   /* For atoi */
#include <stdarg.h> 
#include <string.h>
#include <slang.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <readline/readline.h>
#include <netinet/in.h>
#include <netdb.h>

#ifdef HAVE_LIBHISTORY
#include <readline/history.h>
#endif

#include <card.h>
#include <client.h>
#include <display.h>
#include <sl_io.h>
#include <scroll.h>
#include <socketfunc.h>
#include <client.h>
#include <options.h>

#define TIMEOUT 120

/* Needed to launch offline game engine */
#include <engine.h>
gameInfo_t gameInfo;
int log = 0;


/* Global state of game variable */
extern gameState_t gameState;

/* Game options */
extern option_t options;


void DisplayInit( int* argc, char*** argv ) {

  /* Setup readline and history library */
  rl_instream = stdin;

#ifdef HAVE_LIBHISTORY
  using_history();
#endif
  
  /* Setup slang library */
  SLang_TT_Read_FD = fileno(stdin);
  SLtt_get_terminfo ();
  SLang_init_tty (-1, 0, 0);
  SLsmg_init_smg ();
  SLtt_set_color (REDCARD,   NULL, "red",       "lightgray");
  SLtt_set_color (BLACKCARD, NULL, "black",     "lightgray");
  SLtt_set_color (STDCOL,    NULL, "lightgray", "black");
  SLtt_set_color (TAUNTCOL,  NULL, "lightgray", "blue");
  SLtt_set_cursor_visibility(0);

  /* Setup scrolling status "window" */
  ScrollInit();

}


void InputServerInfo( void ) {

  char* user;
  char* server;
  char* opt;
  int code;

  if ( strcmp( gameState.userName,"") == 0 ) {
    
    SLsmg_set_color(STDCOL);
    SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
    SLsmg_write_string("Please enter your name (8 chars max): ");
    SLsmg_refresh();
    SLang_reset_tty ();
  
    user=readline("");
    gameState.userName = strdup(user);
    
    SLang_init_tty (-1, 0, 0);

    DisplayStatusLine("");
    
    SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
    SLsmg_erase_eol();
    SLsmg_touch_lines(STATTOP+STATHEIGHT,1);
    SLtt_set_cursor_visibility(0);
    SLsmg_refresh();

  }
  
  if( !(options.bitOpt&MSK_OFFLINE) && strcmp( gameState.serverName,"")== 0) {

    SLsmg_set_color(STDCOL);
    SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
    SLsmg_write_string("Please enter the server name: ");
    SLsmg_refresh();
    SLang_reset_tty ();
    
    server=readline("");
    gameState.serverName = strdup( server );
    
    SLang_init_tty (-1, 0, 0);

    DisplayStatusLine("");
    
    SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
    SLsmg_erase_eol();
    SLsmg_touch_lines(STATTOP+STATHEIGHT,1);
    SLtt_set_cursor_visibility(0);
    SLsmg_refresh();
  }

  /* Ask user to set options */
  SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
  SLsmg_write_string("Do you wish to set game options? [y/n] ");
  SLsmg_refresh();  
  code = 0;
  while( ( code != 110 ) && ( code != 121 ) ) {
    code=rl_read_key();
  }  
  SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
  SLsmg_erase_eol();
  SLsmg_refresh();
  
  /* Use wishes to change options */
  if( code == 121 ) {
    
    DisplayStatusLine("Setting options");
    SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
    SLsmg_write_string("Please enter end game score: ");
    SLsmg_refresh();
    SLsmg_gotorc(STATTOP+STATHEIGHT, 29);
    SLtt_set_cursor_visibility(1);
    SLsmg_refresh();
    SLang_reset_tty ();
  
    opt=readline("");
    code = atoi(opt);
    if( code < 100 ) {
      code = 100;
    }
    options.endGame = code;
    
    SLang_init_tty (-1, 0, 0);
    
    DisplayStatusLine( "Game ends at %d",code );
    
    /* Clean up after question */
    SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
    SLsmg_erase_eol();
    SLsmg_touch_lines(STATTOP+STATHEIGHT,1);
    SLsmg_refresh();

    SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
    SLsmg_write_string("Please enter minimum bid: ");
    SLsmg_refresh();
    SLsmg_gotorc(STATTOP+STATHEIGHT, 26);
    SLsmg_refresh();
    SLang_reset_tty ();
  
    opt=readline("");
    code = atoi(opt);
    if( code < 0 ) {
      code = 0;
    }
    options.minBid = code;

    SLang_init_tty (-1, 0, 0);
    
    DisplayStatusLine( "Minumum bid set to %d",code );
  
    /* Clean up after question */
    SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
    SLsmg_erase_eol();
    SLsmg_touch_lines(STATTOP+STATHEIGHT,1);
    SLsmg_refresh();

    if( !(options.bitOpt & MSK_OFFLINE) ) {
      
      SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
      SLtt_set_cursor_visibility(0);
      SLsmg_write_string("Player one (left) will be computer or human [c/h]");
      SLsmg_refresh();
      
      code = 0;
      while( ( code != 99 ) && ( code != 104 ) ) {
	code=rl_read_key();
      }
      if( code == 99 ) {
	DisplayStatusLine("Player one is a computer player");
	options.bitOpt |= MSK_COMP_1;
      }
      else {
	DisplayStatusLine("Player one is a human player");
	options.bitOpt &= ~MSK_COMP_1;
      }
      
      SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
      SLsmg_write_string("Player two (across) will be computer or human: [c/h]");
      SLsmg_refresh();
      
      code = 0;
      while( ( code != 99 ) && ( code != 104 ) ) {
	code=rl_read_key();
      }
      if( code == 99 ) {
	DisplayStatusLine("Player two is a computer player");
	options.bitOpt |= MSK_COMP_2;
      }
      else {
	DisplayStatusLine("Player two is a human player");
	options.bitOpt &= ~MSK_COMP_2;
      }
      
      SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
      SLsmg_write_string("Player three (right) will be computer or human: [c/h]");
      SLsmg_refresh();
      
      code = 0;
      while( ( code != 99 ) && ( code != 104 ) ) {
	code=rl_read_key();
      }
      if( code == 99 ) {
	DisplayStatusLine("Player three is a computer player");
	options.bitOpt |= MSK_COMP_3;
      }
      else {
	DisplayStatusLine("Player three is a human player");
	options.bitOpt &= ~MSK_COMP_3;
      }
      
      SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
      SLsmg_erase_eol();
      SLsmg_refresh();
    }
  }
  
  if( options.bitOpt & MSK_OFFLINE) {
    DisplayStatusLine("Playing offline");
  }
}

void DisplayStatusLine( const char* format, ... ) {
  
  va_list ap;
  char message[80];
  
  va_start( ap, format);
  vsnprintf( message, 80, format, ap);
  va_end(ap);


  AddScrollLine( message );
  update_display();
  SLsmg_refresh();
  DisplayPrompt();

}


void DisplayPrompt( void ) {

  switch( gameState.gameSegment ) {
    
  case ST_GET_HAND:
  case ST_GET_BIDDER:
  case ST_GET_TALLY:    
  case ST_GET_SCORE:
  case ST_GET_LEAD:
  case ST_END_GAME:
    break;

  case ST_GET_BIDS:
    if (gameState.curPlayer == gameState.playerId ) {
      SLtt_set_cursor_visibility(1);
      SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
      SLsmg_write_string("Your bid: ");
      SLsmg_refresh();
    }
    break;

  case ST_GET_TRICK:
    if( gameState.curPlayer == gameState.playerId ) {
      SLtt_set_cursor_visibility(0);
      SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
      SLsmg_write_string("Please play a card");
      SLsmg_refresh();
    }
    break;

  /*case ST_END_GAME:
    SLtt_set_cursor_visibility(0);
    SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
    SLsmg_write_string("Play Again? [y/n]");
    SLsmg_refresh();
    break;*/

  }
}


void DisplayTable( void ) {

  SLsmg_set_color( STDCOL );
  SLsmg_fill_region( TABLETOP, TABLELEFT, TABLEHEIGHT, TABLEWIDTH, ' ');  

  /* Top Player */
  SLsmg_gotorc( TABLETOP, TABLELEFT+(TABLEWIDTH-strlen(gameState.players[(gameState.playerId+2)%4]))/2);
  SLsmg_write_string(gameState.players[(gameState.playerId+2)%4]); 

  /* Left Player */
  SLsmg_gotorc( TABLETOP+TABLEHEIGHT/2, TABLELEFT);
  SLsmg_write_string(gameState.players[(gameState.playerId+1)%4]);

  /* Bottom Player */
  SLsmg_gotorc( TABLETOP+TABLEHEIGHT-1,
		TABLELEFT+(TABLEWIDTH-strlen( gameState.players[gameState.playerId] ))/2 );
  SLsmg_write_string(gameState.players[gameState.playerId]);
  
  /* Right Player */
  SLsmg_gotorc( TABLETOP+TABLEHEIGHT/2,
		TABLELEFT+TABLEWIDTH-strlen(gameState.players[(gameState.playerId+3)%4]));
  SLsmg_write_string(gameState.players[(gameState.playerId+3)%4]);

  SLsmg_refresh();
}


void DisplayHand( void ) {

  int i;
  
  for(i=0; i<13; i++) {
    DisplayCard( gameState.hand[i], CARDTOP, i*(CARDWIDTH+1)+1);

    /* Display Labels */
    SLsmg_set_color(BLACKCARD);
    SLsmg_fill_region( CARDTOP+CARDHEIGHT+1, i*(CARDWIDTH+1)+1,
		       1, CARDWIDTH, ' ');
    SLsmg_gotorc( CARDTOP+CARDHEIGHT+1, i*(CARDWIDTH+1)+3 );
    SLsmg_write_char('a'+i );

  } 

  SLsmg_refresh();
}


/**
 * Display team bids/scores
 */
void DisplayScores( void ) {

  SLsmg_set_color( STDCOL );
  
  /* Blank out status boxes 
  SLsmg_fill_region( TALLYTOP, TALLYLEFT, TALLYHEIGHT, TALLYWIDTH, ' ');  
  SLsmg_fill_region( SCORETOP, SCORELEFT, SCOREHEIGHT, SCOREWIDTH, ' ');*/
  SLsmg_draw_box ( TALLYTOP, TALLYLEFT, TALLYHEIGHT, TALLYWIDTH);
  SLsmg_draw_box ( SCORETOP, SCORELEFT, SCOREHEIGHT, SCOREWIDTH);

  /* Display Team Scores */
  SLsmg_gotorc( TALLYTOP+1, TALLYLEFT+2);
  SLsmg_printf( "%s and %s", gameState.players[0], gameState.players[2] );
  SLsmg_gotorc( TALLYTOP+2, TALLYLEFT+2);
  SLsmg_printf( "Score: % 4d", gameState.scores[0] );
  
  SLsmg_gotorc( SCORETOP+1, SCORELEFT+2);
  SLsmg_printf( "%s and %s", gameState.players[1], gameState.players[3] );
  SLsmg_gotorc( SCORETOP+2, SCORELEFT+2);
  SLsmg_printf( "Score: % 4d", gameState.scores[1] );

  /* Display Team Bids */
  SLsmg_gotorc( TALLYTOP+3, TALLYLEFT+2);
  if( gameState.bids[0] != BID_BLANK && gameState.bids[2] != BID_BLANK ) {      
    if ( gameState.bids[0] == BID_KNEEL || gameState.bids[2] == BID_KNEEL ) {
      SLsmg_printf( "  Bid: n%3d ", (gameState.bids[0]+gameState.bids[2] - BID_KNEEL));
    }  
    else {  
      SLsmg_printf( "  Bid:  %3d ", gameState.bids[0]+gameState.bids[2] );
    }
  }
  else {
      SLsmg_printf( "  Bid:     ", gameState.bids[0]+gameState.bids[2] );
  }

  SLsmg_gotorc( SCORETOP+3, SCORELEFT+2);
  if( gameState.bids[1] != BID_BLANK && gameState.bids[3] != BID_BLANK ) {      
    if ( gameState.bids[1] == BID_KNEEL || gameState.bids[3] == BID_KNEEL ) {
      SLsmg_printf( "  Bid: n%3d ", (gameState.bids[1]+gameState.bids[3] - BID_KNEEL));
    }  
    else {  
      SLsmg_printf( "  Bid:  %3d ", gameState.bids[1]+gameState.bids[3] );
    }
  }
  else {
      SLsmg_printf( "  Bid:     ", gameState.bids[1]+gameState.bids[3] );
  }


  SLsmg_refresh();
  
}


/**
 * Display Bids/Tallys
 */
void DisplayTallys( void ) {

  SLsmg_set_color( STDCOL );

  /* Player 0 */
  SLsmg_gotorc( TALLYTOP+5, TALLYLEFT+2);
  switch( gameState.bids[0] ) {
  case BID_BLANK:
    SLsmg_printf( "%-8s        ", gameState.players[0]);
    break;
  case BID_KNEEL:
    SLsmg_printf( "%-8s : %2d/ n", gameState.players[0], gameState.tallys[0] );
    break;
  default:
    SLsmg_printf( "%-8s : %2d/%2d", gameState.players[0], gameState.tallys[0], gameState.bids[0]);
  }

  /* Player 2 */
  SLsmg_gotorc( TALLYTOP+6, TALLYLEFT+2);
  switch( gameState.bids[2] ) {
  case BID_BLANK:
    SLsmg_printf( "%-8s        ", gameState.players[2]);
    break;
  case BID_KNEEL:
    SLsmg_printf( "%-8s : %2d/ n", gameState.players[2], gameState.tallys[2] );
    break;
  default:
    SLsmg_printf( "%-8s : %2d/%2d", gameState.players[2], gameState.tallys[2], gameState.bids[2]);
  }
  
  /* Player 1 */
  SLsmg_gotorc( SCORETOP+5, SCORELEFT+2);
  switch( gameState.bids[1] ) {
  case BID_BLANK:
    SLsmg_printf( "%-8s        ", gameState.players[1]);
    break;
  case BID_KNEEL:
    SLsmg_printf( "%-8s : %2d/ n", gameState.players[1], gameState.tallys[1] );
    break;
  default:
    SLsmg_printf( "%-8s : %2d/%2d", gameState.players[1], gameState.tallys[1], gameState.bids[1]);
  }
  
  /* Player 3 */
  SLsmg_gotorc( SCORETOP+6, SCORELEFT+2);
  switch( gameState.bids[3] ) {
  case BID_BLANK:
    SLsmg_printf( "%-8s        ", gameState.players[3]);
    break;
  case BID_KNEEL:
    SLsmg_printf( "%-8s : %2d/ n", gameState.players[3], gameState.tallys[3] );
    break;
  default:
    SLsmg_printf( "%-8s : %2d/%2d", gameState.players[3], gameState.tallys[3], gameState.bids[3]);
  }

  SLsmg_refresh();

}


void DisplayCard( Card card, int y, int x) {

  if ( card_suit_num(card) == HEARTS || card_suit_num(card) == DIAMONDS )
      SLsmg_set_color(REDCARD);
  else
      SLsmg_set_color(BLACKCARD);
  
  SLsmg_fill_region (y, x, CARDHEIGHT, CARDWIDTH, ' ');

  SLsmg_gotorc( y, x );
  SLsmg_write_string( (char*)card_name(card, SHORT_NAME ));
  
  if ( card_face( card ) == 10 ) {
    SLsmg_gotorc( y+CARDHEIGHT-1, x+CARDWIDTH-3 );
    SLsmg_write_string( (char*)card_name(card,SHORT_NAME) );
  }
  else {
    SLsmg_gotorc( y+CARDHEIGHT-1, x+CARDWIDTH-2 );
    SLsmg_write_string( (char*)card_name(card,SHORT_NAME) );
  }

}


void DisplayPlayedCard( Card card, int player, int id ) {

  int x, y, tablePos = (player - id + 4 ) % 4;
  
  switch( tablePos ) {
  case 0:
    x = TABLELEFT+(TABLEWIDTH-CARDWIDTH)/2;
    y = TABLETOP+TABLEHEIGHT-CARDHEIGHT-1;
    break;
  case 1:
    x = TABLELEFT+8;
    y = TABLETOP+(TABLEHEIGHT-CARDHEIGHT)/2;
    break;
  case 2:
    x = TABLELEFT+(TABLEWIDTH-CARDWIDTH)/2;
    y = TABLETOP+1;
    break;
  case 3:
    x = TABLELEFT+TABLEWIDTH-8-CARDWIDTH;
    y = TABLETOP+(TABLEHEIGHT-CARDHEIGHT)/2;
    break;
  }
  
  DisplayCard( card, y, x);
  SLsmg_refresh();
  SLsmg_set_color(STDCOL);
    
}


int InputBid( void ) {
  int code=14;
  char* bidString;

  SLsmg_set_color(STDCOL);
  SLsmg_gotorc(STATTOP+STATHEIGHT, 10);
  SLtt_set_cursor_visibility(1);
  SLsmg_refresh();
  SLang_reset_tty ();
  
  bidString=readline("");
    
  if ( (strncmp( bidString,"k",1) == 0) || (strncmp( bidString,"K",1) == 0) ||
      (strncmp( bidString,"N",1) == 0) || (strncmp( bidString,"n",1) == 0) )
      code = -1;
  else 
      code=atoi(bidString);

  SLang_init_tty (-1, 0, 0);

  if ( code == -1)
    DisplayStatusLine( "You bid nil" );
  else
    DisplayStatusLine( "You bid %d", code );
  
  SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
  SLsmg_erase_eol();
  SLsmg_touch_lines(STATTOP+STATHEIGHT,1);
  SLtt_set_cursor_visibility(0);
  SLsmg_refresh();

  return code;
}


int InputCard( void ) {
  int code=0;

  while( ( code < 97 ) || ( code > 109 ) ) {
    code=rl_read_key();
  }

  SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
  SLsmg_erase_eol();
  SLsmg_refresh();
  
  return code-97;
}


int InputQuery( void ) {
  int code=0;

  while( ( code != 110 ) && ( code != 121 ) ) {
    code=rl_read_key();
  }

  SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
  SLsmg_erase_eol();
  SLsmg_refresh();
  
  return (code == 121); /*User entered 'y', play again*/
}


int CheckForTaunt( void ) {

  int in = rl_read_key();

  if ( in == 27 && !(options.bitOpt & MSK_OFFLINE) )
      SendTaunt( InputTaunt() );
  else if( in == 12 ) {
    SLsmg_touch_lines(0,SLtt_Screen_Rows);
    SLsmg_refresh();
  }
  else
    rl_stuff_char( in );

  return( in == 27 && !(options.bitOpt & MSK_OFFLINE) );

}


char* InputTaunt( void ) {

  char* buf;

  SLsmg_set_color(TAUNTCOL);
  SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
  SLsmg_erase_eol();
  SLtt_set_cursor_visibility(1);
  SLsmg_refresh();
  SLang_reset_tty ();
  
  buf=readline("");

#ifdef HAVE_LIBHISTORY
  add_history( buf );
#endif
  
  SLang_init_tty (-1, 0, 0);

  update_display();
  
  SLsmg_gotorc(STATTOP+STATHEIGHT, 0);
  SLsmg_erase_eol();
  SLsmg_touch_lines(STATTOP+STATHEIGHT,1);
  SLtt_set_cursor_visibility(0);
  SLsmg_refresh();

  DisplayPrompt();  

  return buf;
}


void HideCard( int i ) {

  SLsmg_set_color( STDCOL );
  SLsmg_fill_region( CARDTOP, i*(CARDWIDTH+1)+1, CARDHEIGHT, CARDWIDTH, ' ');  
  SLsmg_fill_region( CARDTOP+CARDHEIGHT+1, i*(CARDWIDTH+1)+1,
		       1, CARDWIDTH, ' ');
  
  SLsmg_refresh();
}
  

void DisplayCleanup( void ) {

  free_lines();
  SLtt_set_cursor_visibility(1);
  SLsmg_reset_smg ();
  SLang_reset_tty ();

}

int MainIOLoop( void ) {
  
  Card playedCard;
  
  int l, status, handIndex;
  
  char *buf;
  fd_set read_fd_set, active_fd_set;
  struct timeval timeOutCount;
  struct timeval* timer;

  if( options.bitOpt & MSK_OFFLINE )
      PlayOffline();
  else
      Connect();
    
  /* Setup list of File descriptor to "listen" to */
  FD_ZERO( &active_fd_set );
  FD_SET( gameState.spadesSock, &active_fd_set );
  
  /* Setup timeout pointer for server while registering*/
  timer = &timeOutCount;
  
  while(!gameState.sessionOver) {

    /* Re-initialize at start of every game */
    GameInit();
    
    while(!gameState.gameOver) {

      read_fd_set = active_fd_set;

      /* Only time out on the server before game starts*/
      if( gameState.gameSegment < ST_REG_TAUNT ) {
	timeOutCount.tv_sec = 15;
	timeOutCount.tv_usec = 0;
      }
      
      /* Wait for message from server or user */
      status = select (FD_SETSIZE, &read_fd_set, NULL, NULL, timer);

      if( status < 0 ) {
	DisplayError("Selecting data input failed");
      }

      if( status == 0 ) {
	DisplayError("Server timed out");
      }

      /* Message from Server */
      if ( FD_ISSET( gameState.spadesSock, &read_fd_set ) ) {

	switch( gameState.gameSegment ) {
	  
	case ST_GET_ID:
#ifdef DEBUG
	  DisplayStatusLine("Getting Player ID...");
#endif 
	  if( GetPlayerId() == NET_FAIL ) {
	    DisplayError("Network error");
	  }
	  else if( gameState.playerId < 0 ) {
	    DisplayError( "Sorry, server full.  Try again later" );
	  }
	  else {
	    if( gameState.playerId != 0 ) {
	      DisplayWarning("Another player has already registered options for this game.");
	      DisplayWarning("Your options will be ignored." );
	    }
	    UpdateGame();
	  }
	  break;
	  
	case ST_GET_GAME:
#ifdef DEBUG
	  DisplayStatusLine("Getting Game Info...");
#endif 
	  GetGame();
	  UpdateGame();
	  break;
	  
	case ST_GET_NAMES:
#ifdef DEBUG
	  DisplayStatusLine("Getting Player Names...");
#endif
	  GetPlayers();
	  UpdateGame();
#ifdef DEBUG
	  DisplayStatusLine("Registering with TauntServer(tm)...");
#endif
	  if( RegisterTaunt() < 0 ) {
	    DisplayWarning( "Unable to register with TauntServer" );
	  }
	  else {
	    /* Start listening to taunt server*/
	    FD_SET( gameState.tauntSock, &active_fd_set );
	  }
	  /* Start listening to user*/
	  FD_SET( STDIN_FILENO, &active_fd_set );
	  UpdateGame();
	  /* Don't time out anymore */
	  timer = NULL;

	  break;

	case ST_GET_HAND:
	  GetHand();
	  DisplayHand();
	  DisplayTable();
	  DisplayScores();
	  UpdateGame();
	  break;

	case ST_GET_BIDDER:
	  GetLead();
	  DisplayStatusLine( "%s bids first.", gameState.players[gameState.lead]);
	  UpdateGame();
	  break;

	case ST_GET_BIDS:
	  if( gameState.curPlayer == gameState.playerId ) {
	    DisplayError("Network error");
	  }

	  GetBid();
	  if ( gameState.bids[ gameState.curPlayer] == BID_KNEEL )
	      DisplayStatusLine( "%s bid nil", gameState.players[ gameState.curPlayer] );
	  else 
	      DisplayStatusLine( "%s bid %d", gameState.players[ gameState.curPlayer],
		       gameState.bids[ gameState.curPlayer] );
	  
	  DisplayTallys();
	  UpdateGame();
	  if ( gameState.gameSegment == ST_GET_LEAD ) {
	    DisplayScores();
	  }
	  break;
      
	case ST_GET_LEAD:
	  GetLead();
	  DisplayStatusLine( "%s leads", gameState.players[ gameState.lead]);
	  UpdateGame();
	  break;
	  
	case ST_GET_TRICK:
	  if( gameState.curPlayer == gameState.playerId ) {
	    DisplayError("Network error");
	  }
	  
	  playedCard = GetPlayedCard();
	  DisplayPlayedCard( playedCard, gameState.curPlayer, 
			    gameState.playerId );
	  UpdateGame();
	  break;
	  
	case ST_GET_TALLY:
	  GetTallys();
	  DisplayTallys();
	  sleep(3);
	  DisplayTable();
	  UpdateGame();
	  break;
	  
	case ST_GET_SCORE:
	  GetScores();
	  /* Clear tallys and bids before next hand/game */
	  gameState.bids[0] = BID_BLANK;
	  gameState.bids[1] = BID_BLANK;
	  gameState.bids[2] = BID_BLANK;
	  gameState.bids[3] = BID_BLANK; 
	  gameState.tallys[0] = 0;
	  gameState.tallys[1] = 0; 
	  gameState.tallys[2] = 0;
	  gameState.tallys[3] = 0;
	  DisplayScores();
	  DisplayTallys();
	  if( gameState.gameOver & GAME_OVER ) {
	    DisplayStatusLine( "Game Over" );
	    if( gameState.gameOver & (1<<2) ^ ((gameState.playerId%2)<<2) ) {
	      DisplayStatusLine( "We lost :(" );
	    }
	    else {
	      DisplayStatusLine( "We win!" );
	    }
	    DisplayRecord();
	  }
	  UpdateGame();
	  break;
	  
	case ST_END_GAME:
	  GetNewGame();
	  UpdateGame();
	  switch( gameState.sessionOver ) {
    
	  case 0: /* Let's play another round, eh? */
	    GameInit();
	    DisplayTable();
	    DisplayTallys();
	    DisplayScores();
	    break;
	  case 1: /* Session is over, goodnight!*/
	    /*Disconnect( NULL, NULL);*/
	    break;
	  }
	}  /* switch( gameState.gameSegment )*/
      } /* if( FD_ISSET( gameState.spadesSock, &read_fd_set ) ) */
	
      /* Message from TauntServer */
      else if ( !(options.bitOpt & MSK_OFFLINE) && 
	       FD_ISSET(gameState.tauntSock, &read_fd_set) ) {
	DisplayStatusLine( "Message from TauntServer");
	if( CheckReadInt( gameState.tauntSock, &l ) == NET_OK ) {
	  if( CheckReadString( gameState.tauntSock, &buf ) == NET_OK ) {
	    if( l == -1 )
		DisplayStatusLine( "[ %s ] %s ", "TauntServer", buf );
	    else
		DisplayStatusLine( "[ %s ] %s ", gameState.players[ l ], buf );
	    
	    free(buf);
	  }
	}
      }
      
      /* Message from user */
      else {

	if( !CheckForTaunt() ) {
	  
	  switch( gameState.gameSegment ) {
	    
	  case ST_GET_HAND:
	  case ST_GET_BIDDER:
	  case ST_GET_LEAD:
	  case ST_GET_TALLY:
	  case ST_GET_SCORE:
	  case ST_END_GAME:
	    break;
	    
	  case ST_GET_BIDS:
	    if ( gameState.curPlayer == gameState.playerId ) {

	      gameState.bids[gameState.playerId] = InputBid();
	      status = SendMyBid();

	      switch( status ) {
		  
	      case -1: /* Invalid Bid */
		DisplayStatusLine( "Invalid Bid" );
		/*DisplayPrompt();*/
		break;
		  
	      case -2: /* Bid less than minimum */
		DisplayStatusLine( "Your team must bid at least the minimum" );
		/*DisplayPrompt();*/
		break;

	      default: /* Bid accepted */
		DisplayTallys();
		UpdateGame();
		if ( gameState.gameSegment == ST_GET_LEAD ) {
		  DisplayScores();
		}
  	      }
	    }

	    break;
	    
	  case ST_GET_TRICK:
	    if ( gameState.curPlayer == gameState.playerId ) {

	      status = SendCard( handIndex=InputCard() );
	      
	      switch( status ) {

	      case -1: /* Card already played */
		DisplayStatusLine( "You already played that card" );
		/*DisplayPrompt();*/
		break;
	      case -2: /* Renig */
		DisplayStatusLine( "Hey, no cheating, you're not out yet" );
		/*DisplayPrompt();*/
		break;
	      case 0: /* Card ok */
		DisplayPlayedCard( gameState.hand[handIndex], 0, 0);
		HideCard( handIndex );
		UpdateGame();
		break;
	      }
	    }
	    
	    break;
	    

	  } /* switch( gameState.gameSegment ) */
	} /* if !CheckForTaunt */
      } /* else message is from user*/
    } /* while !gameOver */
    /*DisplayStatusLine( "Game Over" );*/
  } /* while !sessionOver */
  
  
  return 0;
  
}


void DisplayRecord( void ) {
  
  int i, sc0, sc1;
  
  DisplayStatusLine("Session stats:");
  
  for( i=0; i<gameState.gameCount; i++ ) {
    sc0 = gameState.record[2*i+(gameState.playerId%2)];
    sc1 = gameState.record[2*i+((gameState.playerId+1)%2)];
    DisplayStatusLine( "Game %d:  %4d  %4d  (%c)", (i+1), sc0, sc1,
		      ( sc0 > sc1 ? 'W' : 'L' ) );
  }
  
  if( gameState.gameOver & GAME_QUERY ) {
    DisplayStatusLine("Play Again?");
    SendQuery( InputQuery() );
    gameState.gameOver = GAME_OVER;
  }
      
}

void DisplayError( char* message ) {
  
  DisplayStatusLine( "[%d]: Error: %s", getpid(), message );
  NetClose();
  DisplayCleanup();
  exit(-1);

}


void DisplayWarning( char* message ) {

  DisplayStatusLine( "*Warning* %s", message );

}

/* 
 * Create connection to server 
 */
void Connect( void ) {
  
  int status;
  
  DisplayStatusLine("Connecting to server -- please wait.");
  
  status = Register();
  
  switch( status ) {
  case ERR_SOCK_OPEN:
  case ERR_HOST: 
  case ERR_SOCK_CONN:
    DisplayError( errorMsg[ status + ERR_OFFSET ] );
    break;

  default:
    UpdateGame();
    break;
  }
  

}


/**
 * Begin offline game 
 */
void PlayOffline( void ) {

  int fd[2];
  
  DisplayStatusLine("Launching offline game");
  
  /* Create socket pair */
  socketpair( PF_UNIX, SOCK_STREAM, 0, fd );
  gameState.spadesSock = fd[0];

  gameInfo.clientPids[0] = getpid();

  /* Engine process */
  if ( ( gameInfo.gamePid = fork() ) == 0) {
    gameInfo.gamePid = getpid();

    /* Fill gameInfo struct */
    gameInfo.gameNum = 0;
    gameInfo.players[0] = strdup( gameState.userName );
    gameInfo.players[1] = "Yakko";
    gameInfo.players[2] = "Wakko";
    gameInfo.players[3] = "Dot";
    gameInfo.playerSock[0] = fd[1];
    gameInfo.playerSock[1] = SOCK_COMP;
    gameInfo.playerSock[2] = SOCK_COMP;
    gameInfo.playerSock[3] = SOCK_COMP;  
    gameInfo.opt.bitOpt = options.bitOpt;
    gameInfo.opt.endGame = options.endGame;
    gameInfo.opt.minBid = options.minBid;
    
    engine();
    _exit(0);
  }
  else { /* We're still the player process */
    gameState.gameSegment = ST_GET_GAME;
  }
}


