## vim:ts=4:et:nowrap
##
##---------------------------------------------------------------------------##
##
## PySol -- a Python Solitaire game
##
## Copyright (C) 1999 Markus Franz Xaver Johannes Oberhumer
## Copyright (C) 1998 Markus Franz Xaver Johannes Oberhumer
##
## This program is free software; you can redistribute it and/or modify
## it under the terms of the GNU General Public License as published by
## the Free Software Foundation; either version 2 of the License, or
## (at your option) any later version.
##
## This program is distributed in the hope that it will be useful,
## but WITHOUT ANY WARRANTY; without even the implied warranty of
## MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
## GNU General Public License for more details.
##
## You should have received a copy of the GNU General Public License
## along with this program; see the file COPYING.
## If not, write to the Free Software Foundation, Inc.,
## 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
##
## Markus F.X.J. Oberhumer
## <markus.oberhumer@jk.uni-linz.ac.at>
## http://wildsau.idv.uni-linz.ac.at/mfx/pysol.html
##
##---------------------------------------------------------------------------##


# imports
import sys

# PySol imports
if sys.modules.has_key("pysoltk"):                                  #bundle#
    from gamedb import registerGame, GameInfo, GI                   #bundle#
    from util import *                                              #bundle#
    from stack import *                                             #bundle#
    from game import Game                                           #bundle#
    from layout import Layout                                       #bundle#
    from hint import AbstractHint, DefaultHint, CautiousDefaultHint #bundle#


# /***********************************************************************
# //
# ************************************************************************/

class Montana_Hint(DefaultHint):
    def computeHints(self):
        game = self.game
        RLEN, RSTEP, RBASE = game.RLEN, game.RSTEP, game.RBASE
        freerows = filter(lambda s: not s.cards, game.s.rows)
        # for each stack
        for r in game.s.rows:
            if not r.cards:
                continue
            assert len(r.cards) == 1 and r.cards[-1].face_up
            c, pile, rpile = r.cards[0], r.cards, []
            if r.id % RSTEP > 0:
                left = game.s.rows[r.id - 1]
            else:
                left = None
                if c.rank == RBASE:
                    # do not move the leftmost card if corrcet rank
                    continue
            for t in freerows:
                if self.shallMovePile(r, t, pile, rpile):
                    # FIXME: this scoring is completely simple
                    if left and left.cards:
                        # prefer low-rank left neighbours
                        score = 40000 + (self.K - left.cards[-1].rank)
                    else:
                        score = 50000
                    self.addHint(score, 1, r, t)


# /***********************************************************************
# // Montana
# ************************************************************************/

class Montana_Talon(TalonStack):
    def canDealCards(self):
        return self.round != self.max_rounds

    def dealCards(self):
        RLEN, RSTEP, RBASE = self.game.RLEN, self.game.RSTEP, self.game.RBASE
        num_cards = 0
        # move cards to the Talon, shuffle and redeal
        # note: this is not a good shuffling, but otherwise I'd have to
        #       add even more atomic moves just for this game...
        assert len(self.cards) == 0
        rows = self.game.s.rows
        # move out-of-sequence cards from the Tableau to the Talon
        g, gaps = 0, [-1, -1, -1, -1]
        for i in range(0, RLEN, RSTEP):
            r = rows[i]
            if r.cards and r.cards[-1].rank == RBASE:
                in_sequence, suit = 1, r.cards[-1].suit
            else:
                in_sequence, suit = 0, NO_SUIT
            for j in range(RSTEP):
                r = rows[i + j]
                if in_sequence:
                    if not r.cards or r.cards[-1].suit != suit or r.cards[-1].rank != RBASE + j:
                        in_sequence = 0
                if not in_sequence:
                    if gaps[g] < 0:
                        gaps[g] = j
                    if r.cards:
                        self.game.moveMove(1, r, self, frames=0)
                        num_cards = num_cards + 1
            g = g + 1
        assert len(self.cards) == num_cards
        if num_cards == 0:          # game already finished
            return 0
        # shuffle
        self.game.shuffleStackMove(self)
        # redeal
        self.game.nextRoundMove(self)
        g = 0
        for i in range(0, RLEN, RSTEP):
            assert 0 <= gaps[g] < RSTEP
            to_stacks = rows[i+gaps[g]+1 : i+RSTEP]
            for r in to_stacks:
                frames = (0, 4)[i >= RLEN - RSTEP]
                self.game.moveMove(1, self, r, frames=frames)
            g = g + 1
        # done
        assert len(self.cards) == 0
        return num_cards


class Montana_RowStack(BasicRowStack):
    def acceptsPile(self, from_stack, cards):
        if not BasicRowStack.acceptsPile(self, from_stack, cards):
            return 0
        if self.id % self.game.RSTEP == 0:
            return cards[0].rank == self.game.RBASE
        left = self.game.s.rows[self.id - 1]
        return left.cards and left.cards[-1].suit == cards[0].suit and left.cards[-1].rank + 1 == cards[0].rank


class Montana(Game):
    RowStack_Class = Montana_RowStack
    Hint_Class = Montana_Hint

    RLEN, RSTEP, RBASE = 52, 13, 1

    def createGame(self):
        # create layout
        l, s = Layout(self, XM=4), self.s

        # set window
        self.setSize(l.XM + self.RSTEP*l.XS, l.YM + 5*l.YS)

        # create stacks
        for i in range(4):
            x, y, = l.XM, l.YM + i*l.YS
            for j in range(self.RSTEP):
                s.rows.append(self.RowStack_Class(x, y, self, max_accept=1, max_cards=1))
                x = x + l.XS
        x = l.XM + (self.RSTEP-1)*l.XS/2
        s.talon = Montana_Talon(x, self.height-l.YS, self, max_rounds=3)
        if self.RBASE:
            # create an invisble stack to hold the four Aces
            s.internals.append(InvisibleStack(self))

        # define stack-groups
        l.defaultStackGroups()


    #
    # game overrides
    #

    def startGame(self):
        for i in range(52):
            c = self.s.talon.cards[-1]
            if c.rank == ACE:
                self.s.talon.dealRow(rows=self.s.internals, frames=0)
            else:
                frames = (0, 4)[i >= 39]
                self.s.talon.dealRow(rows=(self.s.rows[i],), frames=frames)
        assert len(self.s.talon.cards) == 0

    def isGameWon(self):
        rows = self.s.rows
        for i in range(0, self.RLEN, self.RSTEP):
            for j in range(self.RSTEP - 1):
                r = rows[i + j]
                if not r.cards or r.cards[-1].rank != self.RBASE + j:
                    return 0
        return 1

    def getHighlightPilesStacks(self):
        return ()

    def getAutoStacks(self, event=None):
        return (self.sg.dropstacks, (), self.sg.dropstacks)


# /***********************************************************************
# // Blue Moon
# ************************************************************************/

class BlueMoon(Montana):
    RLEN, RSTEP, RBASE = 56, 14, 0

    def startGame(self):
        j = 0
        for i in range(52):
            if j % 14 == 0:
                j = j + 1
            frames = (0, 4)[i >= 39]
            self.s.talon.dealRow(rows=(self.s.rows[j],), frames=frames)
            j = j + 1
        assert len(self.s.talon.cards) == 0
        ace_rows = filter(lambda r: r.cards and r.cards[-1].rank == ACE, self.s.rows)
        j = 0
        for r in ace_rows:
            self.moveMove(1, r, self.s.rows[j])
            j = j + 14


# register the game
registerGame(GameInfo(53, Montana, "Montana",
                      GI.GT_1DECK_TYPE, 1, 2))
registerGame(GameInfo(63, BlueMoon, "Blue Moon",
                      GI.GT_1DECK_TYPE, 1, 2))

