#include <config.h>
#ifdef __GNUG__
#pragma implementation "board.hh"
#endif
#include "board.hh"
#include "panel.hh"
#include "tileset.hh"
#include "tile.hh"
#include "hint.hh"


Board::Board(Panel *panel, Game *game, Tileset *tileset)
  : SwWidget(panel),
    _panel(panel), _game(game), _tileset(0),
    _layout_x(0), _layout_y(0),
    _buffering(0), _buffer(None),
    _selected(0)
{
  set_tileset(tileset);
  _game->add_hook(this);
  
  _copygc = XCreateGC(display(), window(), 0, NULL);
  
  XGCValues gcv;
  gcv.function = GXor;
  _orgc = XCreateGC(display(), window(), GCFunction, &gcv);
  
  gcv.foreground = 97;
  _erasegc = XCreateGC(display(), window(), GCForeground, &gcv);
  
  gcv.foreground = 0UL;
  gcv.background = ~0UL;
  gcv.function = GXand;
  _maskgc = XCreateGC(display(), window(),
		      GCForeground | GCBackground | GCFunction, &gcv);
  
  _hint = new Hint(this);
}


Board::~Board()
{
  delete _hint;
}


void
Board::set_tileset(Tileset *ts)
{
  _tileset = ts;
  
  _tile_border = ts->border();
  _tile_halfwidth = ts->halfwidth();
  _tile_halfheight = ts->halfheight();
  
  /* When drawing a neighborhood, we will need to draw some other tiles as
     well -- the tiles that overlap this one. But what tiles?
     
     We have a tile (r,c,0). What are the biggest m and n so that the full
     tile at (r,c,0) and a halftile at (r+m,c-n,l) overlap?
     (r+m,c-n,l) == [hw*(c-n) + b*l, hh*(r+m) - b*l]
     (r,c,0)     == [hw*c, hh*r]
     Any intersection will happen between the points at
     [hw*(c-n+1) + b*(l+1), hh*(r+m) - b*l]  and  [hw*c, hh*(r+2) + b].
     
     So, we want greatest m and n so that:
     hw*n < hw + b*(l+1)    -->  n = 1 + floor(b*(l+1)/hw)
     hh*m < b*(l+1) + hh*2  -->  m = 2 + floor(b*(l+1)/hh)
  */
  int uppermost_level = TILE_LEVS - 2;
  _tile_surround_x = 1 + _tile_border*(uppermost_level+1) / _tile_halfwidth;
  _tile_surround_y = 2 + _tile_border*(uppermost_level+1) / _tile_halfheight;
  
  assert(!_buffering);
  _buffer_w = ts->width();
  _buffer_h = ts->height();
  if (_buffer) XFreePixmap(display(), _buffer);
  _buffer = XCreatePixmap(display(), window(), _buffer_w, _buffer_h,
			  _panel->depth());
}


void
Board::set_background(Pixmap background)
{
  XSetTile(display(), _erasegc, background);
  XSetFillStyle(display(), _erasegc, FillTiled);
}


#define CHECK(dr, dc, dl)	do { u = g->grid(r+dr, c+dc, l+dl); \
			if (u->marked()) display_order_dfs(g, u); } while(0)

void
Board::display_order_dfs(Game *g, Tile *t)
{
  t->unmark();
  
  int r = t->row();
  int c = t->col();
  int l = t->lev();
  
  Tile *u;

  // This loop is to be careful for hypothetical strange boards, where there
  // might be a bridge over a valley, for example: we have to check ALL
  // levels above.
  for (int dl = TILE_LEVS - 1 - l; dl >= 1; dl--)
    for (int dr = 0; dr <= 2; dr++)
      for (int dc = -1; dc <= 1; dc++)
	CHECK(dr, dc, dl);
  
  CHECK(0, -1, 0);
  CHECK(1, -1, 0);
  CHECK(2, -1, 0);
  CHECK(2, 0, 0);
  CHECK(2, 1, 0);
  
  _display_order.append(t);
}

#undef CHECK


void
Board::layout_hook(Game *g)
{
  assert(g == _game);
  const Vector<Tile *> &tiles = g->tiles();
  int ntiles = tiles.count();
  
  _display_order.clear();
  if (ntiles == 0) return;
  
  // Create the display order.
  for (int i = 0; i < ntiles; i++)
    tiles[i]->mark();
  
  for (int i = 0; i < ntiles; i++)
    if (tiles[i]->marked())
      display_order_dfs(g, tiles[i]);

  assert(_display_order.count() == ntiles);
  Vector<Tile *> d;
  for (int j = ntiles - 1; j >= 0; j--)
    d.append(_display_order[j]);
  _display_order = d;
  
  // Center the layout on the board.
  center_layout();
}


void
Board::center_layout()
{
  if (_game->ntiles() == 0) return;
  
  int tile_width = _game->rightmost() - _game->leftmost() + 2;
  int tile_height = _game->bottommost() - _game->topmost() + 2;
  int right_level = _game->right_highest();
  int top_level = _game->top_highest();
  
  int layout_width = tile_width * _tile_halfwidth
    + (right_level + 1) * _tile_border;
  int layout_height = tile_height * _tile_halfheight
    + (top_level + 1) * _tile_border;
  
  _layout_x = x() + (width() - layout_width) / 2
    - _game->leftmost() * _tile_halfwidth;
  _layout_y = y() + (height() - layout_height) / 2
    - _game->topmost() * _tile_halfheight
    + top_level * _tile_border;
}


void
Board::start_hook(Game *g)
{
  _tile_flags.assign(g->ntiles(), 0);
  draw_area(2, 2, TILE_ROWS - 2, TILE_COLS - 2);
}


inline void
Board::position(Tile *t, short *x, short *y) const
{
  *x = _layout_x
    + _tile_halfwidth * t->col()
    + _tile_border * t->lev();
  *y = _layout_y 
    + _tile_halfheight * t->row() 
    - _tile_border * t->lev();
}

void
Board::unposition(int x, int y, short *r, short *c) const
{
  *c = (x - _layout_x) / _tile_halfwidth;
  *r = (y - _layout_y) / _tile_halfheight;
}


Tile *
Board::find_tile(short x, short y) const
{
  for (int l = TILE_LEVS - 1; l >= 0; l--) {
    int r = (y - _layout_y + _tile_border * l) / _tile_halfheight;
    int c = (x - _layout_x - _tile_border * (l+1)) / _tile_halfwidth;
    if (r < 0 || r >= TILE_ROWS) continue;
    if (c < 0 || c >= TILE_COLS) continue;
    Tile *t = _game->grid(r, c, l);
    if (t->real()) {
      short xx, yy;
      position(t, &xx, &yy);
      if (x >= xx && y >= yy)
	return t;
    }
  }
  return 0;
}


void
Board::move(int x, int y)
{
  invalidate(_layout_x, _layout_y, TILE_COLS * _tile_halfwidth,
	     TILE_ROWS * _tile_halfheight);
  _layout_x = x;
  _layout_y = y;
}


void
Board::copy_buffer()
{
  if (_buffering)
    _panel->draw_image(_buffer, _buffer_w, _buffer_h, _buffer_x, _buffer_y);
}

void
Board::buffer_on(int x, int y, int w, int h, bool erase)
{
  copy_buffer();
  
  _buffer_x = x;
  _buffer_y = y;
  _buffer_w = w;
  _buffer_h = h;
  XSetTSOrigin(display(), _erasegc, -_buffer_x, -_buffer_y);
  _buffering = true;
  
  if (erase)
    XFillRectangle(display(), _buffer, _erasegc, 0, 0, w, h);
  else
    XCopyArea(display(), window(), _buffer, _copygc, x, y, w, h, 0, 0);
}

void
Board::buffer_off()
{
  copy_buffer();
  _buffer_x = 0;
  _buffer_y = 0;
  XSetTSOrigin(display(), _erasegc, 0, 0);
  _buffering = false;
}

void
Board::draw_subimage(Pixmap image, Pixmap mask, int src_x, int src_y,
		     int w, int h, int x, int y)
{
  if (_buffering) {
    x -= _buffer_x;
    y -= _buffer_y;
    XCopyPlane(display(), mask, _buffer, _maskgc, src_x, src_y, w, h, x, y, 1);
    XCopyArea(display(), image, _buffer, _orgc, src_x, src_y, w, h, x, y);
  } else
    _panel->draw_subimage(image, mask, src_x, src_y, w, h, x, y);
}


void
Board::draw(Tile *t)
{
  short x, y;
  position(t, &x, &y);
  
  if (!t->real())
    ;
  else if (t->obscured())
    _tileset->draw_obscured(t, this, x, y);
  else if (lit(t))
    _tileset->draw_lit(t, this, x, y);
  else
    _tileset->draw_normal(t, this, x, y);
}

void
Board::draw_marked()
{
  for (int i = 0; i < _display_order.count(); i++) {
    Tile *t = _display_order[i];
    if (t->marked() && t->real()) {
      draw(t);
      t->unmark();
    }
  }
}

void
Board::draw_neighborhood(Tile *t, bool erase)
{
  short x, y;
  position(t, &x, &y);
  buffer_on(x, y, _buffer_w, _buffer_h, erase);
  draw_area(t->row() - 1, t->col() - _tile_surround_x,
	    t->row() + _tile_surround_y, t->col() + 2);
  buffer_off();
}

void
Board::draw_area(short rowtop, short colleft, short rowbot, short colright)
{
  for (int lev = 0; lev < TILE_LEVS; lev++)
    for (int col = colleft; col <= colright; col++)
      for (int row = rowtop; row <= rowbot; row++)
	_game->grid(row, col, lev)->mark();
  draw_marked();
}


void
Board::set_lit(Tile *t, bool on)
{
  bool was_on = lit(t);
  set_tile_flag(t, fLit, on);
  if (on != was_on && t->real())
    draw_neighborhood(t, 0);
}


void
Board::select(Tile *t)
{
  if (_selected) deselect();
  light(t);
  set_tile_flag(t, fKeepLit, true);
  _selected = t;
}

void
Board::deselect()
{
  if (_selected) {
    unlight(_selected);
    set_tile_flag(_selected, fKeepLit, false);
    _selected = 0;
  }
}


void
Board::add_tile_hook(Game *g, Tile *t)
{
  assert(g == _game);
  draw_neighborhood(t, 0);
}

void
Board::remove_tile_hook(Game *g, Tile *t)
{
  assert(g == _game);
  if (_selected == t) _selected = 0;
  _tile_flags[t->number()] = 0;
  draw_neighborhood(t, 1);
}
