/*
                             Main Menu Management

	Functions:

	int GETLABELNUMFROMCOORD(
		xsw_main_menu_struct *mm,
		win_t w, shared_image_t *image,
		int x, int y
	)

        int XSWMainMenuIsLabelAllocated( 
                xsw_main_menu_struct *mm,
                int n
        )
	int XSWMainMenuCreateLabel(xsw_main_menu_struct *mm)

	void XSWMainMenuResize(
		xsw_main_menu_struct *mm,
		win_t w, shared_image_t *image
	)
	void XSWMainMenuDrawLabel(
		xsw_main_menu_struct *mm,
                win_t w, shared_image_t *image, int mm_label_num,
		bool_t put_to_window
	)
        void XSWMainMenuDraw(
                xsw_main_menu_struct *mm,
                win_t w, shared_image_t *image, int amount,
                bool_t put_to_window
        )
	int XSWMainMenuManage(
		xsw_main_menu_struct *mm,
		win_t w, shared_image_t *image, event_t *event
	)
	void XSWMainMenuMap(
		xsw_main_menu_struct *mm,
		win_t w, shared_image_t *image
	)
	void XSWMainMenuUnmap(
		xsw_main_menu_struct *mm,
		win_t w, shared_image_t *image
	)
        void XSWMainMenuDestroy(
		xsw_main_menu_struct *mm,
		win_t w, shared_image_t *image
        )

        int XSWMainMenuDoAction(int op_code)


	---

	The main menu is drawn on the viewscreen.

	Typically the passed arguments should be the window ID
	of the viewscreen and the image buffer of the viewscreen.



 */


#include "blitting.h"
#include "xsw.h"
#include "keymap.h"
#include "univlist.h"
#include "mainmenu.h"


#define MIN(a,b)        ((a) < (b) ? (a) : (b))
#define MAX(a,b)        ((a) > (b) ? (a) : (b))



/*
 *	Macro to get the main menu label number from
 *	given coordinates.
 *
 *	If labels overlap, older entry gets priority.
 *
 *	Returns -1 on error or no match.
 */
int GETLABELNUMFROMCOORD(
	xsw_main_menu_struct *mm,
	win_t w, shared_image_t *image,
	int x, int y 
)
{
	int i, n;
	int tmp_x, tmp_y;
	xsw_imglabel_struct *imglabel_ptr;


	if((mm == NULL) ||
           (w == 0) ||
           (image == 0)
	)
	    return(-1);


	for(i = 0; i < mm->total_labels; i++)
	{
	    if(mm->label[i] == NULL) continue;

	    /* Go through main menu label i's set of image labels. */
	    for(n = 0; n < MAIN_MENU_MAX_LABELS; n++)
	    {
		/* Get pointer to image label. */
		imglabel_ptr = (xsw_imglabel_struct *)
		    &mm->label[i]->imglabel[n];
		if(imglabel_ptr == NULL) continue;
		/* Image label not loaded? */
		if(imglabel_ptr->image == NULL) continue;

		/* Check coordinates. */
		if(imglabel_ptr->pos_by_percent)
		{
		    tmp_x = imglabel_ptr->x *
                        (int)image->width / 100;
                    tmp_y = imglabel_ptr->y *
                        (int)image->height / 100; 
		}
		else
		{
                    tmp_x = imglabel_ptr->x;
                    tmp_y = imglabel_ptr->y;
                }
		if((x >= tmp_x) &&
                   (y >= tmp_y) &&
                   (x < (tmp_x + (int)imglabel_ptr->image->width)) &&
                   (y < (tmp_y + (int)imglabel_ptr->image->height))
		)
		    return(i);

	    }
	}

	return(-1);
}



/*
 *	Checks if main menu label n is allocated on mm.
 */
int XSWMainMenuIsLabelAllocated(
	xsw_main_menu_struct *mm,
	int n
)
{
	if(mm == NULL)
	{
	    return(0);
	}
	else if((mm->label == NULL) ||
                (n < 0) ||
                (n >= mm->total_labels)
	)
	{
	    return(0);
	}
	else if(mm->label[n] == NULL)
	{
	    return(0);
	}
	else
	{
	    return(1);
	}
}


/*
 *	Allocates a new label on the mm structure.
 *
 *	The new label is reset and not loaded.
 *	Returns the label's number or -1 on error.
 */
int XSWMainMenuCreateLabel(
	xsw_main_menu_struct *mm
)
{
	int i, new;
	xsw_main_menu_label_struct *label_ptr;


	if(mm == NULL)
	    return(-1);;

	/* Sanitize total_labels. */
	if(mm->total_labels < 0)
	    mm->total_labels = 0;


	/* Look for already allocated label pointer. */
	for(i = 0; i < mm->total_labels; i++)
	{
	    if(mm->label[i] == NULL)
		break;
	}
	if(i < mm->total_labels)
	{
	    new = i;
	}
	else
	{
	    new = mm->total_labels;
	    mm->total_labels++;

	    mm->label = (xsw_main_menu_label_struct **)realloc(
		mm->label,
		mm->total_labels * sizeof(xsw_main_menu_label_struct *)
	    );
	    if(mm->label == NULL)
	    {
		mm->total_labels = 0;
		return(-1);
	    }
	}

	/* Allocate new main menu label structure. */
	mm->label[new] = (xsw_main_menu_label_struct *)calloc(1,
	    sizeof(xsw_main_menu_label_struct)
	);
	if(mm->label[new] == NULL)
	{
	    return(-1);
	}

	/* ********************************************************** */
	/* Reset values. */
	label_ptr = mm->label[new];

	label_ptr->map_state = 0;
        label_ptr->allow_transparency = 0;
        label_ptr->op_code = XSW_ACTION_NONE;

	/* Reset each xsw image label in the main menu label struct. */
	for(i = 0; i < MAIN_MENU_MAX_LABELS; i++)
            memset(
		&(label_ptr->imglabel[i]),
		0x00,
		sizeof(xsw_imglabel_struct)
	    );


	return(new);
}



/*
 *	Resizes the main menu.
 */
void XSWMainMenuResize(
	xsw_main_menu_struct *mm,
        win_t w, shared_image_t *image
)
{
	unsigned int width, height;
        image_t *ori_img = NULL;
        image_t *new_img = NULL;


	/* Error checks. */
	if(!IDC() ||
           (mm == NULL) ||
           (w == 0) ||
           (image == NULL)
	)
	    return;

	/* Do not resize if not mapped. */
	if(mm->map_state == 0)
	    return;


	/* ********************************************************** */

	/* Get size of image buffer. */
	width = image->width;
	height = image->height;


        /* Load background as needed. */
	if(mm->bg_filename != NULL)
	{
            if(mm->bg_image == NULL)
            {
	        mm->bg_image = MM_LOAD_IMAGE(mm->bg_filename);
		if(mm->bg_image == NULL)
		    return;
            }
	    ori_img = mm->bg_image;

	    /* Resize only as needed. */
	    if((width != ori_img->width) ||
               (height != ori_img->height)
	    )
	    {
	        /* Create new sized background. */
	        if(OSWCreateImage(&new_img, width, height))
	        {
	            return;
	        }

	        /* Resize background. */
	        WidgetResizeImageBuffer(
	            osw_gui[0].depth,
	            new_img->data,		/* Target. */
	            ori_img->data,		/* Source. */
	            new_img->width, new_img->height,
	            ori_img->width, ori_img->height
	        );

	        /* Destroy original image and set new image. */
	        OSWDestroyImage(&ori_img);

		/* Set new resized image for bg image. */
		mm->bg_image = new_img;
	    }
	}


	return;
}



/*
 *	Draws label mm_label_num in mm to image.
 */
void XSWMainMenuDrawLabel(
	xsw_main_menu_struct *mm,
	win_t w, shared_image_t *image, int mm_label_num,
	bool_t put_to_window
)
{
	static int x, y;
	static unsigned int width, height;
	static image_t *label_image;
        static xsw_main_menu_label_struct *mm_label_ptr;


        /* Error checks. */
        if(!IDC() ||
           (mm == NULL) ||
           (w == 0) ||
           (image == NULL)
        )
            return;

	/* Do not draw if not mapped. */
	if(mm->map_state == 0)
	    return;

	/* Do not draw if no background is allocated. */
	if(mm->bg_image == NULL)
	    return;


	/* Label must be allocated. */
	if(XSWMainMenuIsLabelAllocated(mm, mm_label_num))
	    mm_label_ptr = (xsw_main_menu_label_struct *)
		mm->label[mm_label_num];
	else
	    return;


	/* ******************************************************* */
	/* Get position and sizes. */

        x = 0;
        y = 0;
	width = 0;
	height = 0;

        switch(mm->sel_label_state)
        {
	  case MM_LABEL_STATE_ARMED:
	    if(mm_label_ptr->imglabel[MM_LABEL_STATE_ARMED].pos_by_percent)
            {
                x = mm_label_ptr->imglabel[MM_LABEL_STATE_ARMED].x
                    * (int)image->width / 100;
                y = mm_label_ptr->imglabel[MM_LABEL_STATE_ARMED].y
                    * (int)image->height / 100;
            }
            else
            {
                x = mm_label_ptr->imglabel[MM_LABEL_STATE_ARMED].x;
                y = mm_label_ptr->imglabel[MM_LABEL_STATE_ARMED].y;
            }
            label_image = mm_label_ptr->imglabel[
                MM_LABEL_STATE_ARMED].image;
	    if(label_image != NULL)
	    {
		width = label_image->width;
		height = label_image->height;
	    }
            break;

          case MM_LABEL_STATE_HIGHLIGHTED:
	    if(mm_label_ptr->imglabel[MM_LABEL_STATE_HIGHLIGHTED].pos_by_percent)
            {
                x = mm_label_ptr->imglabel[MM_LABEL_STATE_HIGHLIGHTED].x
                    * (int)image->width / 100;
                y = mm_label_ptr->imglabel[MM_LABEL_STATE_HIGHLIGHTED].y
                    * (int)image->height / 100;
            }
            else
            {
                x = mm_label_ptr->imglabel[MM_LABEL_STATE_HIGHLIGHTED].x;
                y = mm_label_ptr->imglabel[MM_LABEL_STATE_HIGHLIGHTED].y;
            }
            label_image = mm_label_ptr->imglabel[
                MM_LABEL_STATE_HIGHLIGHTED].image;
            if(label_image != NULL)
            {
                width = label_image->width;
                height = label_image->height;
            }
            break;

          default:	/* Default to unarmed. */
	    if(mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].pos_by_percent)
            {
                x = mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].x
                    * (int)image->width / 100;
                y = mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].y
                    * (int)image->height / 100;
            }
            else
            {
                x = mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].x;
                y = mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].y;
            }   
            label_image = mm_label_ptr->imglabel[
                MM_LABEL_STATE_UNARMED].image;
            if(label_image != NULL)
            {
                width = label_image->width;
                height = label_image->height;
            }
            break;
	}


	/* Blit portion of background to image. */
        BlitBufAbsolute(
            osw_gui[0].depth,
            image->data,		/* Target. */
            mm->bg_image->data,		/* Source. */
            x, y,			/* Target coordinates. */
            image->width, image->height,
            x, y,			/* Source coordinates. */
            mm->bg_image->width, mm->bg_image->height,
            width, height,		/* Copy width and height. */
            1.0,			/* Zoom. */
	    1.0				/* Magnification. */
        );


	/* Blit label to image. */
        if(label_image != NULL)
        {
	    if(mm_label_ptr->allow_transparency)
	    {
                BlitBufNormal(
                    osw_gui[0].depth,
                    image->data,		/* Target. */
                    label_image->data,		/* Source. */
                    x, y,			/* Target coordinates. */ 
                    image->width, image->height,
                    0, 0,			/* Source coordinates. */ 
                    label_image->width, label_image->height,
                    label_image->width, label_image->height,
                    1.0,	/* Zoom. */
                    1.0,	/* Visibility. */
		    1.0		/* Magnificaition. */
                );
	    }
	    else
	    {
                BlitBufAbsolute(
                    osw_gui[0].depth,
                    image->data,                /* Target. */
                    label_image->data,          /* Source. */
                    x, y,                       /* Target coordinates. */
                    image->width, image->height,
                    0, 0,                       /* Source coordinates. */
                    label_image->width, label_image->height,
                    label_image->width, label_image->height,
                    1.0,	/* Zoom. */
		    1.0		/* Magnification. */
                );
	    }
        }           

	/* Put to window? */
	if(put_to_window)
	{
	    OSWPutSharedImageToDrawableSect(
		image, w,
		x, y,		/* Target. */
		x, y,		/* Source. */
		width, height
	    );
	}




	return;
}



/*
 *	Redraws the main menu mm on window w.
 *
 *	If put_to_window is False, then only the image is updated,
 *	else both the image is updated and put to the window w.
 */
void XSWMainMenuDraw(
	xsw_main_menu_struct *mm,
	win_t w, shared_image_t *image,
	int amount,
	bool_t put_to_window
)
{
	int i, len;
	int mm_label_num;
	image_t *bkg_img;
	image_t *label_image;
	int x, y;
	xsw_main_menu_label_struct *mm_label_ptr;


	/* Error checks. */
	if(!IDC() ||
           (mm == NULL) ||
           (w == 0) ||
           (image == NULL)
	)
	    return;


	/* Map as needed. */
	if(mm->map_state == 0)
	{
            mm->map_state = 1;

	    /* Map all labels. */
	    for(i = 0; i < mm->total_labels; i++)
	    {
		if(mm->label[i] == NULL)
		    continue;
		mm->label[i]->map_state = 1;
	    }

	    /* Need to resize main menu on mapping. */
	    XSWMainMenuResize(mm, w, image);
	}


	/* ******************************************************** */
	/* Draw background. */

        /* Load background as needed. */
	if(mm->bg_filename != NULL)
	{
            if(mm->bg_image == NULL)
            {  
                mm->bg_image = MM_LOAD_IMAGE(mm->bg_filename);
                if(mm->bg_image == NULL)
		    return;
            }

            bkg_img = mm->bg_image;

	    /* Copy background image to image buffer. */
	    /* 8 bits. */
	    if(osw_gui[0].depth == 8)
	    {
	        len = MIN(
		    bkg_img->width * bkg_img->height * BYTES_PER_PIXEL8,
		    image->width * image->height * BYTES_PER_PIXEL8
	        );
	    }
	    /* 15 or 16 bits. */
	    else if((osw_gui[0].depth == 15) ||
                    (osw_gui[0].depth == 16)
            )
            {
                len = MIN(
                    bkg_img->width * bkg_img->height * BYTES_PER_PIXEL16,
                    image->width * image->height * BYTES_PER_PIXEL16
                );
            }
	    /* 24 or 32 bits. */
            else if((osw_gui[0].depth == 24) ||
                    (osw_gui[0].depth == 32)
	    )
            {
                len = MIN(
                    bkg_img->width * bkg_img->height * BYTES_PER_PIXEL32,
                    image->width * image->height * BYTES_PER_PIXEL32
                );
            }
	    else
	    {
	        /* Unsupported depth. */
	        return;
	    }
	    /* Copy it. */
	    for(i = 0; i < len; i++)
	        image->data[i] = bkg_img->data[i];
	}


        /* ******************************************************** */
        /* Draw labels. */

	for(mm_label_num = 0;
	    mm_label_num < mm->total_labels;
	    mm_label_num++
	)
	{
	    /* Get main menu label pointer. */
	    mm_label_ptr = (xsw_main_menu_label_struct *)
		mm->label[mm_label_num];
	    /* Cannot be NULL. */
	    if(mm_label_ptr == NULL)
		continue;

	    /* Skip if label is not mapped. */
	    if(mm_label_ptr->map_state == 0)
		continue;


	    /* Is this label selected? */
	    if(mm_label_num == mm->sel_label)
	    {
		x = 0;
		y = 0;
		/* Image label index corresponds with state. */
		switch(mm->sel_label_state)
		{
		  case MM_LABEL_STATE_ARMED:
		    if(mm_label_ptr->imglabel[MM_LABEL_STATE_ARMED].pos_by_percent)
		    {
                        x = mm_label_ptr->imglabel[MM_LABEL_STATE_ARMED].x
                            * (int)image->width / 100;
                        y = mm_label_ptr->imglabel[MM_LABEL_STATE_ARMED].y
                            * (int)image->height / 100;
	            }
		    else
		    {
		        x = mm_label_ptr->imglabel[MM_LABEL_STATE_ARMED].x;
                        y = mm_label_ptr->imglabel[MM_LABEL_STATE_ARMED].y;
		    }
		    label_image = mm_label_ptr->imglabel[
			MM_LABEL_STATE_ARMED].image;
		    break;

		  case MM_LABEL_STATE_HIGHLIGHTED:
                    if(mm_label_ptr->imglabel[MM_LABEL_STATE_HIGHLIGHTED].pos_by_percent)
                    {
                        x = mm_label_ptr->imglabel[MM_LABEL_STATE_HIGHLIGHTED].x
                            * (int)image->width / 100;      
                        y = mm_label_ptr->imglabel[MM_LABEL_STATE_HIGHLIGHTED].y
                            * (int)image->height / 100;
                    }
                    else
                    {
                        x = mm_label_ptr->imglabel[MM_LABEL_STATE_HIGHLIGHTED].x;
                        y = mm_label_ptr->imglabel[MM_LABEL_STATE_HIGHLIGHTED].y;
                    }
                    label_image = mm_label_ptr->imglabel[
			MM_LABEL_STATE_HIGHLIGHTED].image;
                    break;

		  default:	/* Default to unarmed. */
                    if(mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].pos_by_percent)
                    {
                        x = mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].x
                            * (int)image->width / 100;
                        y = mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].y
                            * (int)image->height / 100;
                    }
                    else
                    {
                        x = mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].x;
                        y = mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].y;
		    }
		    label_image = mm_label_ptr->imglabel[
			MM_LABEL_STATE_UNARMED].image;
                    break;
		}
		if(label_image == NULL)
		{
                    if(mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].pos_by_percent)
                    {
                        x = mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].x
                            * (int)image->width / 100;
                        y = mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].y
                            * (int)image->height / 100; 
                    }
                    else
                    {
                        x = mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].x;
                        y = mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].y;
                    }
		    label_image = mm_label_ptr->imglabel[
			MM_LABEL_STATE_UNARMED].image;
		}
	    }
	    else
	    {
		/* Not selected, use unarmed image. */
                if(mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].pos_by_percent)
                {
                    x = mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].x
                        * (int)image->width / 100;
                    y = mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].y
                        * (int)image->height / 100;
                }
		else
		{
                    x = mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].x;
                    y = mm_label_ptr->imglabel[MM_LABEL_STATE_UNARMED].y;
		}
		label_image = mm_label_ptr->imglabel[
		    MM_LABEL_STATE_UNARMED].image;
	    }
	    if(label_image == NULL)
		continue;

	    /* Blit label_image to image. */
	    if(mm_label_ptr->allow_transparency)
	    {
		BlitBufNormal(
		    osw_gui[0].depth,
		    image->data,	/* Target. */
		    label_image->data,	/* Source. */
		    x, y, 		/* Target coordinates. */
		    image->width, image->height,
		    0, 0,		/* Source coordinates. */
		    label_image->width, label_image->height,
                    label_image->width, label_image->height,
		    1.0,	/* Zoom. */
		    1.0,	/* Visibility. */
		    1.0		/* Magnification. */
		);
	    }
	    else
	    {
		BlitBufAbsolute(
		    osw_gui[0].depth,
		    image->data,        /* Target. */
                    label_image->data,  /* Source. */
                    x, y,               /* Target coordinates. */
                    image->width, image->height,
                    0, 0,               /* Source coordinates. */
                    label_image->width, label_image->height,
                    label_image->width, label_image->height,
                    1.0,	/* Zoom. */
		    1.0		/* Magnification. */
		);
	    }
	}


	/* Put image to window? */
	if(put_to_window)
	{
            OSWPutSharedImageToDrawable(image, w);
	}



	return;
}



int XSWMainMenuManage(
	xsw_main_menu_struct *mm,
	win_t w, shared_image_t *image, event_t *event
)
{
	int mm_label_num;
	int events_handled = 0;


	/* Error checks */
	if((mm == NULL) ||
           (w == 0) ||
           (image == NULL) ||
           (event == NULL)
	)
	    return(events_handled);

	if(!mm->map_state &&
           (event->type != MapNotify)
	)
	    return(events_handled);


	switch(event->type)
	{
	  /* ****************************************************** */
	  case KeyPress:
	    if(!mm->is_in_focus)
		break;


	    break;

          /* ****************************************************** */
          case KeyRelease:
            if(!mm->is_in_focus)
                break;

            break;

          /* ****************************************************** */
          case ButtonPress:
            /* Event must be on main menu's window. */
            if(event->xany.window != w)
	    {
		/* Unfocus as needed. */
		if(mm->is_in_focus)
		    mm->is_in_focus = 0;

		break;
	    }

	    /* Set into focus. */
	    mm->is_in_focus = 1;

	    /* Button press coordinates on a label? */
            mm_label_num = GETLABELNUMFROMCOORD(
		mm, w, image,
		event->xbutton.x,
		event->xbutton.y
	    );
	    if(XSWMainMenuIsLabelAllocated(mm, mm_label_num))
	    {
		/* Set newly selected label. */
		mm->sel_label = mm_label_num;
		mm->sel_label_state = MM_LABEL_STATE_ARMED;

		XSWMainMenuDrawLabel(mm, w, image, mm->sel_label, True);

		events_handled++;
		return(events_handled);
	    }
            break;

          /* ****************************************************** */
          case ButtonRelease:
            /* Event must be on main menu's window. */
            if(event->xany.window != w) break;

            mm_label_num = GETLABELNUMFROMCOORD(
                mm, w, image,
                event->xbutton.x,
                event->xbutton.y
            );
            if(XSWMainMenuIsLabelAllocated(mm, mm_label_num))
            {
		/* Was label previously armed? */
		if((mm_label_num == mm->sel_label) &&
                   (mm->sel_label_state == MM_LABEL_STATE_ARMED)
		)
		{
		    /* Unmap hint window. */
		    HintWinUnmap();

                    /* Set label back to highlighted. */
                    mm->sel_label_state = MM_LABEL_STATE_HIGHLIGHTED;

		    /* Redraw. */
                    XSWMainMenuDrawLabel(mm, w, image, mm->sel_label, True);
 
                    /* Play select sound. */
                    if((option.sounds > XSW_SOUNDS_NONE) &&
                       (mm->label[mm_label_num]->op_code != XSW_ACTION_NONE)
		    )
                        SoundPlay(
                            SOUND_CODE_MENU_SELECT,
                            1.00,
                            1.00,
                            0,
                            0
                        );

		    /* Perform action defined by label. */
                    XSWMainMenuDoAction(mm->label[mm_label_num]->op_code);

		    events_handled++;
		    return(events_handled);
		}
		else
		{
                    /*   The button was released on some other label,
                     *   in which case select that label and set it to
                     *   be highlighted, do not perform its action.
		     */
                    mm->sel_label = mm_label_num;
                    mm->sel_label_state = MM_LABEL_STATE_HIGHLIGHTED;

                    /* Redraw. */
                    XSWMainMenuDrawLabel(mm, w, image, mm->sel_label, True);

		    events_handled++;
                    return(events_handled);
		}
            }
            break;

          /* ****************************************************** */
          case MotionNotify:
	    /* Event must be on main menu's window. */
	    if(event->xany.window == w)
	    {
		/* Get label number that the pointer is over. */
                mm_label_num = GETLABELNUMFROMCOORD(
                    mm, w, image,
                    event->xmotion.x,
                    event->xmotion.y
                );
	        /* Got valid label? */
                if(XSWMainMenuIsLabelAllocated(mm, mm_label_num))
                {
		    if((mm->sel_label == mm_label_num) &&
                       (mm->sel_label_state == MM_LABEL_STATE_ARMED)
	            )
		    {
		        /* Label already selected, do nothing. */
		        events_handled++;
		        return(events_handled);
		    }
		    else if(mm->sel_label != mm_label_num)
		    {
			/* Pointer has moved over new label. */

                        HintWinUnmap();

                        /* Set new label highlighted. */
                        mm->sel_label = mm_label_num;
                        mm->sel_label_state = MM_LABEL_STATE_HIGHLIGHTED;

			/* Schedual hint message to be shown. */
			if(mm->label[mm_label_num]->op_code != XSW_ACTION_NONE)
			{
			    if(mm->label[mm_label_num]->hint_mesg != NULL)
	                        HintWinSetSchedualMessage(
			            widget_global.hintwin_map_delay,
			            w,
			            mm->label[mm_label_num]->hint_mesg
		                );
			}

			/* Play highlighted sound. */
                        if((option.sounds > XSW_SOUNDS_NONE) &&
                           (mm->label[mm_label_num]->op_code != XSW_ACTION_NONE)
                        )
                            SoundPlay(
                                SOUND_CODE_MENU_HIGHLIGHT,
                                1.00,
                                1.00,
                                0, 
                                0
                            );

		        events_handled++;
		    }
                }
	        else
	        {
		    /* Pointer is not over any label. */

		    /* Was a label previously selected? */
		    if(XSWMainMenuIsLabelAllocated(mm, mm->sel_label))
		    {
			HintWinUnmap();

		        events_handled++;
		    }

                    mm->sel_label = -1;
		    mm->sel_label_state = MM_LABEL_STATE_UNARMED;
		}
	    }
	    else
	    {
		/*   MotionNotify occured on another window,
		 *   check if mm->sel_label is valid, if so unmap hint
		 *   window.
		 */
                if(XSWMainMenuIsLabelAllocated(mm, mm->sel_label))
                {
		    mm->sel_label = -1;
                    HintWinUnmap();
                }
	    }
            break;

          /* ****************************************************** */
          case Expose:
            /* Event must be on main menu's window. */
            if(event->xany.window == w)
            {
		events_handled++;
	    }
	    break;
	}

	/* Redraw as needed. */
	if(events_handled > 0)
	{
	    XSWMainMenuDraw(
		mm,
		w,
		image,
		DRAW_AMOUNT_COMPLETE,
		True			/* Put to window. */
	    );
	}


	return(events_handled);
}


/*
 *	Maps and draws mm.
 */
void XSWMainMenuMap(
	xsw_main_menu_struct *mm,
	win_t w, shared_image_t *image
)
{
        /* Error checks. */
        if(!IDC() ||  
           (mm == NULL) ||
           (w == 0) ||
           (image == NULL)
        )
            return;


	/* Map by drawing it. */
	mm->map_state = 0;
	XSWMainMenuDraw(mm, w, image, DRAW_AMOUNT_COMPLETE, True);


	return;
}


/*
 *	Unmaps mm.
 */
void XSWMainMenuUnmap(
	xsw_main_menu_struct *mm,
	win_t w, shared_image_t *image
)
{
	/* Error checks. */
	if(!IDC() ||
           (mm == NULL) ||
           (w == 0) ||
           (image == NULL)
	)
	    return;


	/* Unmap as needed. */
	if(mm->map_state)
	{
	    /* Destroy the main menu background image. */
#ifdef DEBUG_MEM_FREE
if(mm->bg_image != NULL)
    printf("Main menu: Free'ed background image.\n");
#endif
            OSWDestroyImage(&mm->bg_image);

	    /* Set unmapped map state. */
            mm->map_state = 0;
	}


	/* Unmap hint window. */
        HintWinUnmap();


	return;
}


/*
 *	Deallocates all resources in mm but not the structure itself.
 */
void XSWMainMenuDestroy(
	xsw_main_menu_struct *mm,
	win_t w, shared_image_t *image
)
{
	int i, n;


        /* Error checks. */
        if(mm == NULL)
	    return;


	/* ******************************************************* */

	/* Free background image filename. */
#ifdef DEBUG_MEM_FREE
if(mm->bg_filename != NULL)
    printf("Main menu: Free'ed background image filename.\n");
#endif
	free(mm->bg_filename);
	mm->bg_filename = NULL;

	/* Destroy background image. */
#ifdef DEBUG_MEM_FREE
if(mm->bg_image != NULL)
    printf("Main menu: Free'ed background image.\n");
#endif
	OSWDestroyImage(&mm->bg_image);


        /* ******************************************************* */

	/* Deallocate all labels. */
	for(i = 0; i < mm->total_labels; i++)
	{
	    if(mm->label[i] == NULL)
		continue;

	    /* Free each image label's substructures. */
	    for(n = 0; n < MAIN_MENU_MAX_LABELS; n++)
	    {
#ifdef DEBUG_MEM_FREE
if(mm->label != NULL)
    printf("Main menu label %i: img label %i: Free'ed resources.\n", i, n);
#endif
	        ImgLabelReset(&mm->label[i]->imglabel[n]);
		/* Do not free imglabel[n] itself. */
	    }

            /* Free hint message. */
#ifdef DEBUG_MEM_FREE   
if(mm->label[i]->hint_mesg != NULL)
    printf("Main menu label %i: Free'ed hint message.\n", i);
#endif
            free(mm->label[i]->hint_mesg); 
            mm->label[i]->hint_mesg = NULL;

	    /* Free label structure itself. */
#ifdef DEBUG_MEM_FREE
if(mm->label[i] != NULL)
    printf("Main menu label %i: Free'ed.\n", i);
#endif
	    free(mm->label[i]);
	    mm->label[i] = NULL;
	}
#ifdef DEBUG_MEM_FREE
if(mm->label != NULL)
    printf("Main menu label pointers: Free'ed.\n");
#endif
	free(mm->label);
	mm->label = NULL;

	mm->total_labels = 0;


        /* ******************************************************* */
	/* Reset values. */

        mm->map_state = 0;




	return;
}


/*
 *	Performs main menu action.
 */
int XSWMainMenuDoAction(int op_code)
{
	char stringa[512];
        int status = 0;

            
        switch(op_code)
        {
	  /* ********************************************************** */
	  /* Do nothing. */
          case XSW_ACTION_NONE:
	    break;

          /* ********************************************************** */
          /* Run server. */
          case XSW_ACTION_RUN_SERVER:
            XSWMapFB(NULL, PRI_FB_LOADOP_RUN_SERVER);
            break;
                      
          /* Load OCSN file. */   
          case XSW_ACTION_LOAD_OCSN:
            XSWMapFB(NULL, PRI_FB_LOADOP_OCSN);
            break;
          
          /* Load Image Set Referance file. */
          case XSW_ACTION_LOAD_ISREF:
            XSWMapFB(NULL, PRI_FB_LOADOP_ISREF);
            break;
                
          /* Load Sound Scheme file */
          case XSW_ACTION_LOAD_SS:
            XSWMapFB(NULL, PRI_FB_LOADOP_SS);
            break;
        
          /* ********************************************************** */
          /* Network actions. */
          case XSW_ACTION_CONNECT:
            UnivListMap();
            status = 0;
            break;
        
          case XSW_ACTION_CONNECT_LAST:
            /* Reconnect using the last address and port. */
            status = NetOpenConnection(net_parms.address, net_parms.port);
            status = (status < 0) ? -1 : 0;
            break;
 
          case XSW_ACTION_DISCONNECT:
            XSWDoDisconnect();
            status = 0;
            break;
                      
          case XSW_ACTION_REFRESH:
            XSWDoRefresh();
            status = 0;
            break;
          
          case XSW_ACTION_AINT:
            if(auto_interval_tune.state)
                status = CmdAutoInterval("off");
            else  
                status = CmdAutoInterval("on");
            break;
          
          /* ********************************************************* */
          /* Client system. */
          case XSW_ACTION_SYNCTIME:
            status = CmdSynctime("");
            break;

          case XSW_ACTION_DISPLABELS:
            option.show_viewscreen_labels++;
            if(option.show_viewscreen_labels > 3)
                option.show_viewscreen_labels = 0;

            /* Redraw viewscreen. */
            VSDrawViewScreen(
                net_parms.player_obj_num,
		bridge_win.viewscreen,
                bridge_win.viewscreen_image,
		bridge_win.viewscreen_zoom
            );

            status = 0;
            break;

          case XSW_ACTION_MEMORY:
            status = CmdMemory("");
            break;
 
          case XSW_ACTION_OPTIONS:
            OptWinDoMapValues();
	    status = 0;
            break;      

	  case XSW_ACTION_EDIT_KEYMAP:
	    KeymapWinDoMapValues();
	    status = 0;
	    break;

          case XSW_ACTION_COMFERM_EXIT:
	    prompt_mode = PROMPT_CODE_EXIT;
            PromptChangeName(&bridge_win.prompt, "Exit?:");
	    PromptMap(&bridge_win.prompt);
            bridge_win.prompt.is_in_focus = 1;
	    status = 0;
            break;

	  case XSW_ACTION_EXIT:
	    status = 0;
	    runlevel = 1;
	    break;

          /* ********************************************************* */
          default:
            status = -2;
	    sprintf(stringa,
"Internal non-critical error:\n\n\
    Unsupported action operation code %i.\n\n",
		op_code
	    );
	    printdw(&err_dw, stringa);

            break;
        }


        return(status);
}
