/*
 * GTK See -- a image viewer based on GTK+
 * Copyright (C) 1998 Hotaru Lee <jkhotaru@mail.sti.com.cn> <hotaru@163.net>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#define HEADER_SIZE 128

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <gtk/gtk.h>

#include "gtypes.h"

#ifdef HAVE_LIBJPEG
#include <jpeglib.h>
#include "im_jpeg.h"
#endif

#ifdef HAVE_LIBTIFF
#include <tiffio.h>
#include "im_tiff.h"
#endif

#ifdef HAVE_LIBPNG
#include <png.h>
#include "im_png.h"
#endif

#include "im_gif.h"
#include "im_xpm.h"
#include "im_bmp.h"
#include "im_ico.h"
#include "im_pcx.h"
#include "im_pnm.h"
#include "im_psd.h"
#include "im_xbm.h"
#include "im_xcf.h"

#include "detect.h"

typedef gboolean (*DetectFunc) (guchar *header, guchar *filename, ImageInfo *info);

static gboolean
detect_jpeg(guchar *header, guchar *filename, ImageInfo *info)
{
#ifdef HAVE_LIBJPEG
	jpeg_info jinfo;
	
	if (strncmp(&header[6], "JFIF", 4) != 0) return FALSE;

	if (jpeg_get_header(filename, &jinfo))
	{
		info->type = JPG;
		info->width = jinfo.image_width;
		info->height = jinfo.image_height;
		info->ncolors = jinfo.num_components * 8;
		info->real_ncolors = FALSE;
		info->alpha = 0;
		return TRUE;
	} else
	{
		return FALSE;
	}
#else
	return FALSE;
#endif
}

static gboolean
detect_tiff(guchar *header, guchar *filename, ImageInfo *info)
{
#ifdef HAVE_LIBTIFF
	tiff_info tinfo;

	if (strncmp(header, "II*\0", 4) != 0) return FALSE;
	
	if (tiff_get_header(filename, &tinfo))
	{
		info->type = TIF;
		info->width = tinfo.width;
		info->height = tinfo.height;
		info->ncolors = tinfo.ncolors;
		info->real_ncolors = FALSE;
		info->alpha = tinfo.alpha;
		return TRUE;
	} else
	{
		return FALSE;
	}
#else
	return FALSE;
#endif
}

static gboolean
detect_png(guchar *header, guchar *filename, ImageInfo *info)
{
#ifdef HAVE_LIBPNG
	png_info pnginfo;

	if (header[0] != 0x89 || strncmp(&header[1], "PNG", 3) != 0)
		return FALSE;
	
	if (png_get_header(filename, &pnginfo))
	{
		info->type = PNG;
		info->width = pnginfo.width;
		info->height = pnginfo.height;
		switch (pnginfo.color_type)
		{
			case PNG_COLOR_TYPE_RGB :
				info->ncolors = 24;
				info->alpha = 0;
				info->bpp = 3;
				break;
			case PNG_COLOR_TYPE_RGB_ALPHA :
				info->ncolors = 24;
				info->alpha = 1;
				info->bpp = 4;
				break;
			case PNG_COLOR_TYPE_GRAY :
				info->ncolors = 8;
				info->alpha = 0;
				info->bpp = 1;
				break;
			case PNG_COLOR_TYPE_GRAY_ALPHA :
				info->ncolors = 8;
				info->alpha = 1;
				info->bpp = 2;
				break;
			case PNG_COLOR_TYPE_PALETTE :
				info->ncolors = 8;
				info->alpha = 0;
				info->bpp = 3;
				break;
		};
		info->real_ncolors = FALSE;
		return TRUE;
	} else
	{
		return FALSE;
	}
#else
	return FALSE;
#endif
}

static gboolean
detect_gif(guchar *header, guchar *filename, ImageInfo *info)
{
	gif_screen ginfo;
	
	if (strncmp(header, "GIF", 3) != 0) return FALSE;
	
	if (gif_get_header(filename, &ginfo))
	{
		info->type = GIF;
		info->width = ginfo.Width;
		info->height = ginfo.Height;
		info->ncolors = ginfo.BitPixel;
		info->real_ncolors = TRUE;
		info->alpha = 0;
		return TRUE;
	} else
	{
		return FALSE;
	}
}

static gboolean
detect_xpm(guchar *header, guchar *filename, ImageInfo *info)
{
	xpm_info xinfo;
	
	if (strstr(header, "XPM") == NULL) return FALSE;
	
	if (xpm_get_header(filename, &xinfo))
	{
		info->type = XPM;
		info->width = xinfo.width;
		info->height = xinfo.height;
		info->ncolors = xinfo.ncolors;
		info->real_ncolors = TRUE;
		info->alpha = 0;
		return TRUE;
	} else
	{
		return FALSE;
	}
}

static gboolean
detect_bmp(guchar *header, guchar *filename, ImageInfo *info)
{
	bmp_info binfo;
	
	if (strncmp(header, "BM", 2) != 0) return FALSE;
	
	if (bmp_get_header(filename, &binfo))
	{
		info->type = BMP;
		info->width = binfo.width;
		info->height = binfo.height;
		info->ncolors = binfo.bitCnt;
		info->real_ncolors = FALSE;
		info->alpha = 0;
		return TRUE;
	} else
	{
		return FALSE;
	}
}

/* We don't do a quick detect for ico format. */
static gboolean
detect_ico(guchar *header, guchar *filename, ImageInfo *info)
{
	ico_info icinfo;
	
	if (ico_get_header(filename, &icinfo))
	{
		info->type = ICO;
		info->width = icinfo.width;
		info->height = icinfo.height;
		info->ncolors = icinfo.ncolors;
		info->real_ncolors = FALSE;
		info->alpha = 0;
		return TRUE;
	} else
	{
		return FALSE;
	}
}

static gboolean
detect_pcx(guchar *header, guchar *filename, ImageInfo *info)
{
	pcx_info pinfo;
	
	if(header[0] != 10) return FALSE;
	
	if (pcx_get_header(filename, &pinfo))
	{
		info->type = PCX;
		info->width = pinfo.width;
		info->height = pinfo.height;
		info->ncolors = pinfo.ncolors;
		info->real_ncolors = FALSE;
		info->alpha = 0;
		return TRUE;
	} else
	{
		return FALSE;
	}
}

static gboolean
detect_pnm(guchar *header, guchar *filename, ImageInfo *info)
{
	pnm_info pnminfo;
	
	if (header[0] != 'P' || (header[1] < '1' && header[1] > '6'))
		return FALSE;
	
	if (pnm_get_header(filename, &pnminfo))
	{
		info->type = PNM;
		info->width = pnminfo.width;
		info->height = pnminfo.height;
		info->ncolors = pnminfo.ncolors;
		info->real_ncolors = FALSE;
		info->alpha = 0;
		return TRUE;
	} else {
		return FALSE;
	}
}

static gboolean
detect_psd(guchar *header, guchar *filename, ImageInfo *info)
{
	psd_info psdinfo;
	
	if (strncmp(header, "8BPS", 4) != 0) return FALSE;
	
	if (psd_get_header(filename, &psdinfo))
	{
		info->type = PSD;
		info->width = psdinfo.width;
		info->height = psdinfo.height;
		info->ncolors = psdinfo.ncolors;
		info->real_ncolors = FALSE;
		info->alpha = psdinfo.alpha;
		return TRUE;
	} else {
		return FALSE;
	}
}

static gboolean
detect_xbm(guchar *header, guchar *filename, ImageInfo *info)
{
	xbm_info xbminfo;
	
	if (strstr(header, "define") == NULL) return FALSE;
	if (strstr(header, "width") == NULL) return FALSE;
	
	if (xbm_get_header(filename, &xbminfo))
	{
		info->type = XBM;
		info->width = xbminfo.width;
		info->height = xbminfo.height;
		info->ncolors = 2;
		info->real_ncolors = TRUE;
		info->alpha = 0;
		return TRUE;
	} else {
		return FALSE;
	}
}

static gboolean
detect_xcf(guchar *header, guchar *filename, ImageInfo *info)
{
	xcf_info xcfinfo;
	
	if (strncmp(header, "gimp xcf ", 9) != 0) return FALSE;
	
	if (xcf_get_header(filename, &xcfinfo))
	{
		info->type = XCF;
		info->width = xcfinfo.width;
		info->height = xcfinfo.height;
		info->ncolors = xcfinfo.ncolors;
		info->real_ncolors = FALSE;
		info->alpha = xcfinfo.alpha;
		return TRUE;
	} else {
		return FALSE;
	}
}

static DetectFunc detect_funcs[MAX_IMAGE_TYPES - 1] =
{
	(DetectFunc)detect_xpm,
	(DetectFunc)detect_gif,
	(DetectFunc)detect_jpeg,
	(DetectFunc)detect_bmp,
	(DetectFunc)detect_ico,
	(DetectFunc)detect_pcx,
	(DetectFunc)detect_tiff,
	(DetectFunc)detect_png,
	(DetectFunc)detect_pnm,
	(DetectFunc)detect_psd,
	(DetectFunc)detect_xbm,
	(DetectFunc)detect_xcf,
};

gboolean
detect_image_type(guchar *filename, ImageInfo *info)
{
	guchar header[HEADER_SIZE+1];
	gint i;
	FILE *fp;
	
	if ((fp = fopen(filename, "rb")) == NULL) return FALSE;
	memset(header, 0, HEADER_SIZE+1);
	fread(header, HEADER_SIZE, 1, fp);
	fclose(fp);
	
	if (info->type < UNKNOWN)
	{
		if ((*detect_funcs[info->type])(header, filename, info))
			return TRUE;
	}
	
	for (i = 0; i < UNKNOWN; i++)
	{
		if (i != info->type && i != ICO)
		{
			if ((*detect_funcs[i])(header, filename, info))
				return TRUE;
		}
	}
	
	return FALSE;
}
