/***************************************************************************
                          relement.cpp  -  description
                             -------------------
    begin                : Mon Sep 27 1999
    copyright            : (C) 1999 by Andreas Mustun
    email                : andrew@ribbonsoft.com
 ***************************************************************************/


/****************************************************************************
** relement.cpp 1998/08/24 A. Mustun RibbonSoft
**
** Copyright (C) 1998 RibbonSoft.  All rights reserved.
**
*****************************************************************************/

#include <values.h>
#include <stdlib.h>

#include <qcolor.h>
#include <qnamespace.h>
#include <qpen.h>
#include <qstring.h>

#include "relement.h"

#include "rpainter.h"
#include "rfiledxf.h"
#include "rgraphic.h"
#include "rconfig.h"
#include "rlog.h"
#include "rmath.h"

// Constructor:
//
RElement::RElement(RGraphic* _graphic, 
                   unsigned int _flags)
{
  subElement=0;
  reset();
  graphic=_graphic;
  setFlags(_flags);
}



// Destructor:
//
RElement::~RElement()
{
  deleteSubElements();
}


// Reset all data but not the graphic pointer:
//
void
RElement::reset()
{
  setFlags(0);
  elementTyp = T_NO;
  x1 = y1 = x2 = y2 = cx = cy = cr = a1 = a2 = 0.0;
  color = Qt::black;
  width = 0;
  style = Qt::SolidLine;
  layer = 0;
  contour = 0;
  unumD = 0;
  unumC = 0;
  text = "";
  leftBorder=topBorder=rightBorder=bottomBorder=0.0;

  deleteSubElements();
  subElement=0;
  subElementNum=0;
}



// Get the number of text lines of this element or -1 if it's no text at all
//
int   
RElement::getNumberOfTextLines()
const
{
  if(elementTyp==T_TEXT) {
    int ret=1;
    
    for(int c=0; c<(int)text.length(); ++c) {
      if(text.at(c)==10) {
        ++ret;
      }
    }
  
    return ret;
  }
  else {
    return -1;
  }
  
}



// Get a line from this text element:
//
//   _num: number of line
//
void
RElement::getTextLine(int _num, char* _line, int _max)
const
{
  if(!_line) return;

  _line[0]='\0';

  if(elementTyp!=T_TEXT) {
    _line[0]='\0';
    return;
  }

  int lc=0;       // Line counter
  int i, c, end;  // counters

  // First line:
  //
  if(_num==0) {
    end=0;
    for(c=0; c<_max && c<(int)text.length(); ++c) {
      _line[c]=text.at(c);
      ++end;
      if(_line[c]==13 || _line[c]==10) {
        _line[c]='\0';
        break;
      }
    }
  
    _line[end]='\0';
  
    return;
  }

  // Get the line number num:
  //
  for(i=0; i<(int)text.length()-1; ++i) {
    if(text.at(i)==10 || text.at(i)==13) {
      ++i;
      if(text.at(i)==10 || text.at(i)==13) ++i;
      ++lc;
      if(lc==_num) {
        if(text.at(i)!='\0') {
          // Extract this line:
          //
          end=0;
          for(c=0; c<_max && i+c<(int)text.length(); ++c) {
            _line[c]=text.at(i+c);
            ++end;
            if(_line[c]==13 || _line[c]==10) {
              _line[c]='\0';
              break;
            }
          }
         
          _line[end]='\0';
        
        }
        else {
          // Line is empty:
          //
          _line[0]='\0';
        }
        break;
      }
    }
  }
}


/**
 * Gets the dimension text which is the measure or a user set text.
 */
void
RElement::getDimText(char* _text, int _len)
const
{
  // User defined:
  if(!text.isEmpty()) {
    strncpy(_text, text.data(), _len-1);
  }

  // Auto measure:
  else {
    float result=0.0;

    switch(getTextMode()) {
      case E_STRAIGHT:
        // Horizontal:
        if(getFlag(E_HORIZONTAL)) {
          result = mtGetDistance(x1, getDimDist(), x2, getDimDist());
          sprintf(_text, "%.2f", result);
        }
        // Vertical:
        else if(getFlag(E_VERTICAL)) {
          result = mtGetDistance(getDimDist(), y1, getDimDist(), y2);
          sprintf(_text, "%.2f", result);
        }
        // Diameter:
        else if(getFlag(E_DIAMETER)) {
          result = mtGetDistance(x1, y1, x2, y2);
          sprintf(_text, "%.2f", result);
        }
        // Radius:
        else if(getFlag(E_RADIUS)) {
          result = cr;
          sprintf(_text, "R%.2f", result);
        }
        // Parallel:
        else {
          result = mtGetDistance(x1, y1, x2, y2);
          sprintf(_text, "%.2f", result);
        }
        break;

      case E_ROUNDOUT:
        if(!getFlag(E_REVERSED)) result = mtCorrAngle(a2-a1);
        else                     result = mtCorrAngle(a1-a2);
        sprintf(_text, "%.2f", result);
        break;

      default:
        break;
    }
  }

  if(QString(_text).contains('.')) {
    // Remove zeros at the end:
    //
    while(_text[strlen(_text)-1]=='0') {
      _text[strlen(_text)-1] = '\0';
    }

    if(_text[strlen(_text)-1]=='.') {
      _text[strlen(_text)-1] = '\0';
    }
  }

  if(getTextMode()==E_ROUNDOUT) {
    char degr[2];
    degr[0]=176;
    degr[1]='\0';
    strcat(_text, degr);
  }
}



// Has this element an endpoint?
//
//   (Texts / Points / Rectangles have no endpoint)
//
bool  
RElement::hasEndpoint()
const
{
  if(elementTyp==T_TEXT  || 
     elementTyp==T_POINT ||
     elementTyp==T_RECT  ||
     elementTyp==T_DIMENSION ||
     elementTyp==T_HATCHING) {
    return false;
  }
  else {
    return true;
  }
}



// Is this element circular (arc or circle)?
//
bool  
RElement::isCircular()
const
{
  if(elementTyp==T_ARC    || 
     elementTyp==T_CIRCLE    ) {
    return true;
  }
  else {
    return false;
  }
}



// Check if element is in group _typGroup:
//
//   Groups are: T_NO, T_ALL, T_STRAIGHT, T_CIRCULAR, T_LONG
//
bool 
RElement::isInGroup(int _typGroup)
const
{
  bool ret=false;

  if(_typGroup==elementTyp) ret=true;
  
  switch(_typGroup) {
    case T_NO:
      ret=false;
      break;
    case T_ALL:
      ret=true;
      break;
    case T_STRAIGHT:
      if(elementTyp==T_LINE) ret=true;
      break;
    case T_LINE_ARC:
      if(elementTyp==T_LINE ||
         elementTyp==T_ARC    ) ret=true;
      break;
    case T_CIRCULAR:
      if(elementTyp==T_ARC   ||
         elementTyp==T_CIRCLE  ) ret=true;
      break;
    case T_LONG:
      if(elementTyp==T_LINE   ||
         elementTyp==T_ARC    ||
         elementTyp==T_CIRCLE   ) ret=true;
      break;
    default:
      break;
  }

  return ret;
}



// Compare this element with element _element
//
//   _element:   the element to compare with this
//   _tolerance: tolerance in mm
//   _reverse:   are reversed elements the same (=true)?
//
//   returns: true if they are identically otherwise false
bool
RElement::compareWith(const RElement* _element, float _tolerance, bool _reverse) 
{
  bool ret=true;
  
  if(_element) {

    // Test if they're NOT identically (unreversed):
    //

    // Common:
    //
    if(elementTyp!=_element->elementTyp ||
       getFlags()!=_element->getFlags()    ) {

      ret=false;
      goto testreversed;
    }

    // Lines / Arcs / Texts:
    //
    if(!mtCompFloat(x1, _element->x1, _tolerance) ||
       !mtCompFloat(y1, _element->y1, _tolerance) || 
       !mtCompFloat(x2, _element->x2, _tolerance) || 
       !mtCompFloat(y2, _element->y2, _tolerance)    ) {

      ret=false;
      goto testreversed;
    }
    
    // Arcs / Texts:
    //
    if((isCircular() || elementTyp==T_TEXT)        &&
       (!mtCompFloat(cx, _element->cx, _tolerance) || 
        !mtCompFloat(cy, _element->cy, _tolerance) || 
        !mtCompFloat(cr, _element->cr, _tolerance) || 
        !mtCompFloat(a1, _element->a1, _tolerance) || 
        !mtCompFloat(a2, _element->a2, _tolerance)    )    ) {

      ret=false;
      goto testreversed;
    }

    // Texts:
    //
    if(elementTyp==T_TEXT) {
      if(text!=_element->text) {
        ret=false;
        goto testreversed;
      }
    }

    // Dimensions:
    //
    if(elementTyp==T_DIMENSION) {
      if(!mtCompFloat(cx, _element->cx, _tolerance) ||
         !mtCompFloat(cy, _element->cy, _tolerance) ||
         !mtCompFloat(cr, _element->cr, _tolerance) ||
         !mtCompFloat(a1, _element->a1, _tolerance) ||
         !mtCompFloat(a2, _element->a2, _tolerance)    ) {

        ret=false;
        goto testreversed;
      }
    }

    // Hatchings:
    //
    if(elementTyp==T_HATCHING) {
      if(text!=_element->text ||
         !mtCompFloat(cr, _element->cr, _tolerance)) {

        ret=false;
        goto testreversed;
      }
      else {
        for(int i=0; i<subElementNum; ++i) {
          if(!subElement[i].getFlag(E_VISIBLE)) {
            if(!subElement[i].compareWith(&_element->subElement[i], _tolerance, _reverse)) {
              ret=false;
              break;
            }
          }
        }
        goto testreversed;
      }
    }
  }

  testreversed:

  // Test if they're identically (reversed):
  //
  if(_reverse && ret==false &&
     elementTyp!=T_TEXT &&
     elementTyp!=T_DIMENSION &&
     elementTyp!=T_HATCHING) {

    RElement revEl;      // The reversed element

    revEl.getMeasuresFrom(_element);
    revEl.turn();
    
    if(compareWith(&revEl, _tolerance, false)) {
      ret=true;
    }
  }
  
  return ret;
}



// Get ALL data from _element object:
//
//   copies also: graphic, undo/redo information
//
void 
RElement::getDataFrom(const RElement* _element) 
{
  getMeasuresFrom(_element);
  graphic   =_element->graphic;
  unumD     =_element->unumD;
  unumC     =_element->unumC;
}



// Get only measures data from _element object:
//
//   copies only measures and layer information
//
void 
RElement::getMeasuresFrom(const RElement* _element)
{
  if(_element) {
    elementTyp=_element->elementTyp;
    x1        =_element->x1;
    y1        =_element->y1;
    x2        =_element->x2;
    y2        =_element->y2;
    cx        =_element->cx;
    cy        =_element->cy;
    cr        =_element->cr;
    a1        =_element->a1;
    a2        =_element->a2;
    layer     =_element->layer;
    color     =_element->color;
    width     =_element->width;
    style     =_element->style;
    contour   =_element->contour;
    leftBorder=_element->leftBorder;
    topBorder =_element->topBorder;
    rightBorder=_element->rightBorder;
    bottomBorder=_element->bottomBorder;
    text      =_element->text.data();
    setFlags(_element->getFlags());

    if(_element->subElementNum!=0) {
      createSubElements(_element->subElementNum);
      for(int i=0; i<subElementNum; ++i) {
        setSubElement( i, &_element->subElement[i] );
      }
    }
  }
}



// Get attributes from _element object:
//
//   copies color, width, style
//
void
RElement::getAttribFrom(const RElement* _element)
{
  color = _element->color;
  width = _element->width;
  style = _element->style;
}



// Get angle amount (length of angle in degree)
//   (positive)
//
float 
RElement::getAngleAmount()
const
{
  float ret=0.0;

  switch(elementTyp) {
    case T_ARC:
      if(!getFlag(E_REVERSED)) {
        float dumAng=a2;
        if(a2<a1) dumAng+=360.0;
        ret=mtCorrAngle(dumAng-a1);
      }
      else {
        float dumAng=a1;
        if(a2>a1) dumAng+=360.0;
        ret=mtCorrAngle(dumAng-a2);
      }
      break;

    case T_CIRCLE:
      ret=360.0;
      break;

    default:
      break;
  }
  
  return ret;
}



// Get angle to add to a1 to reach a2:
//   (sign gives direction)
//
float 
RElement::getAngleFrom1To2()
const
{
  float ret=0.0;

  switch(elementTyp) {
    case T_ARC:
      ret=a2-a1;
      if(!getFlag(E_REVERSED) && ret<0.0) ret+=360.0;
      if( getFlag(E_REVERSED) && ret>0.0) ret-=360.0;
      break;
    case T_CIRCLE:
      if(!getFlag(E_REVERSED)) ret= 360.0;
      else                     ret=-360.0;
      break;
    default:
      break;
  }

  return ret;
}



// Get length of element:
//
float 
RElement::getLength()
const
{
  float ret=0.0;

  switch(elementTyp) {
    case T_RAPID:
    case T_LINE:
      ret=mtGetDistance(x1, y1, x2, y2);
      break;
      
    case T_ARC:
      ret=2.0*cr*M_PI/360.0*getAngleAmount();
      break;

    case T_CIRCLE:
      ret=2.0*cr*M_PI;
      break;

    case T_STARTWAY:
      if(getFlag(E_TANGENTIAL)) {
        /*RElement tmpEl;
        getPart(1, tmpEl);
        ret=tmpEl.getLength();*/
      }
      else {
        ret=mtGetDistance(x1, y1, x2, y2);
      }
      break;
      
    case T_ENDWAY:
      if(getFlag(E_TANGENTIAL)) {
        /*RElement tmpEl;
        getPart(2, tmpEl);
        ret=tmpEl.getLength();*/
      }
      else {
        ret=mtGetDistance(x1, y1, x2, y2);
      }
      break;


    default:
      break;
  }
  
  return ret;
}


// Get Center point of element (X):
//
float 
RElement::getCenterX()
const
{
  float ret=0.0;

  switch(elementTyp) {
    case T_POINT:
      ret=x1;
      break;
    case T_LINE:
    case T_RECT:
    case T_RAPID:
    case T_DIMENSION:
      ret=(x1+x2)/2.0;
      break;
    case T_ARC:
    case T_CIRCLE:
      ret=cx;
      break;
    case T_TEXT:
      if(getFlag(E_ROUNDOUT) || 
         getFlag(E_ROUNDIN)  ||
         getFlag(E_CENTER)      ) {
        ret=x1;
      }
      else {
        ret=(leftBorder+rightBorder)/2.0;
      }
      break;
    default:
      break;
  }
  return ret;
}



// Get Center point of element (Y):
//
float 
RElement::getCenterY()
const
{
  float ret=0.0;

  switch(elementTyp) {
    case T_POINT:
      ret=y1;
      break;
    case T_LINE:
    case T_RECT:
    case T_RAPID:
    case T_DIMENSION:
      ret=(y1+y2)/2.0;
      break;
    case T_ARC:
    case T_CIRCLE:
      ret=cy;
      break;
    case T_TEXT:
      if(getFlag(E_ROUNDOUT) || 
         getFlag(E_ROUNDIN)  ||
         getFlag(E_CENTER)      ) {
        ret=y1;
      }
      else {
        ret=(topBorder+bottomBorder)/2.0;
      }
      break;
    default:
      break;
  }
  return ret;
}



// Get Middle point of element (X):
//
float 
RElement::getMiddleX()
const
{
  float ret=0.0;

  switch(elementTyp) {
    case T_POINT:
      ret=x1;
      break;
    case T_LINE:
    case T_RECT:
    case T_RAPID:
      ret=(x1+x2)/2.0;
      break;
    case T_ARC:
    case T_CIRCLE:
      if(!getFlag(E_REVERSED)) {
        ret=cx + cos((a1+getAngleAmount()/2.0)/ARAD) * cr;
      }
      else {
        ret=cx + cos((a1-getAngleAmount()/2.0)/ARAD) * cr;
      }
      break;
    case T_TEXT:
    case T_DIMENSION:
      ret=getCenterX();
      break;
    default:
      break;
  }
  return ret;
}



// Get Middle point of element (Y):
//
float 
RElement::getMiddleY()
const
{
  float ret=0.0;

  switch(elementTyp) {
    case T_POINT:
      ret=y1;
      break;
    case T_LINE:
    case T_RECT:
    case T_RAPID:
      ret=(y1+y2)/2.0;
      break;
    case T_ARC:
    case T_CIRCLE:
      if(!getFlag(E_REVERSED)) {
        ret=cy + sin((a1+getAngleAmount()/2.0)/ARAD) * cr;
      }
      else {
        ret=cy + sin((a1-getAngleAmount()/2.0)/ARAD) * cr;
      }
      break;
    case T_TEXT:
    case T_DIMENSION:
      ret=getCenterY();
      break;
    default:
      break;
  }
  return ret;
}



// Get Direction of Middle point of element (degrees):
//
float 
RElement::getMiddleDirection()
const
{
  float ret=0.0;

  switch(elementTyp) {
    case T_POINT:
      ret=0.0;
      break;
    case T_LINE:
    case T_RAPID:
      ret=getDirection1();
      break;
    case T_ARC:
    case T_CIRCLE:
      if(!getFlag(E_REVERSED)) ret=getDirection1()+getAngleAmount()/2.0;
      else                     ret=getDirection1()-getAngleAmount()/2.0;
      break;
    default:
      break;
  }
  return ret;
}



// Get Direction of Start point of element (degrees):
//
float 
RElement::getDirection1()
const
{
  float ret=0.0;

  switch(elementTyp) {
    case T_POINT:
      ret=0.0;
      break;
    case T_LINE:
    case T_RAPID:
      ret = mtGetAngle(x1, y1, x2, y2);
      break;
    case T_ARC:
    case T_CIRCLE:
      if(!getFlag(E_REVERSED)) ret = mtCorrAngle(a1+90.0);
      else                     ret = mtCorrAngle(a1-90.0);
      break;
    default:
      break;
  }
  return ret;
}



// Get Direction of End point of element (degrees):
//
float 
RElement::getDirection2()
const
{
  float ret=0.0;

  switch(elementTyp) {
    case T_POINT:
      ret=0.0;
      break;
    case T_LINE:
    case T_RAPID:
      ret = mtGetAngle(x2, y2, x1, y1);
      break;
    case T_ARC:
    case T_CIRCLE:
      if(!getFlag(E_REVERSED)) ret = mtCorrAngle(a2-90.0);
      else                     ret = mtCorrAngle(a2+90.0);
      break;
    default:
      break;
  }
  return ret;
}



// Get point with distance from start- or endpoint (_startPoint):
//
//   _resx/_resy: result coordinate
//   _dist:       distance to start-/endpoint
//   _startPoint: true: measure distance from startpoint
//                false: measure distance from endpoint
//
void  
RElement::getDistPoint(float* _resx, 
                       float* _resy, 
                       float _dist, 
                       bool _startPoint)
const
{
  float ang;              // Angle
  *_resx = *_resy = DEF_AREAMAX;  // default value

  // Distances from startpoint:
  //
  if(_startPoint) {
    // Distance on a line:
    //
    if(elementTyp==T_LINE) {
      ang = getDirection1();
      *_resx = x1 + cos(ang/ARAD) * _dist;
      *_resy = y1 + sin(ang/ARAD) * _dist;
    }
    else {
      // Distance on an arc:
      //
      if(isCircular()) {
        if(!getFlag(E_REVERSED)) {
          ang = a1 + _dist/(cr*2.0*M_PI)*360.0;
          *_resx = cx + cos(ang/ARAD) * cr;
          *_resy = cy + sin(ang/ARAD) * cr;
        }
        else {
          ang = a1 - _dist/(cr*2.0*M_PI)*360.0;
          *_resx = cx + cos(ang/ARAD) * cr;
          *_resy = cy + sin(ang/ARAD) * cr;
        }
      }
    }
  }
  else {
    // Distances from the endpoint:
    //
    // Distance on a line:
    //
    if(elementTyp==T_LINE) {
      ang = getDirection2();
      *_resx = x2 + cos(ang/ARAD) * _dist;
      *_resy = y2 + sin(ang/ARAD) * _dist;
    }
    else {
      // Distance on an arc:
      //
      if(isCircular()) {
        if(!getFlag(E_REVERSED)) {
          ang = a2 - _dist/(cr*2.0*M_PI)*360.0;
          *_resx = cx + cos(ang/ARAD) * cr;
          *_resy = cy + sin(ang/ARAD) * cr;
        }
        else {
          ang = a2 + _dist/(cr*2.0*M_PI)*360.0;
          *_resx = cx + cos(ang/ARAD) * cr;
          *_resy = cy + sin(ang/ARAD) * cr;
        }
      }
    }
  }

}



// Get point on element which is next to given coordinate
//
// _resx :
// _resy : result coordinates
// _mx   :
// _my   : mouse coordinate (real in mm)
//
void
RElement::getTouchPoint(float* _resx, float* _resy,
                        float _mx, float _my)
const
{
  float ang;                      // Angle
  *_resx = *_resy = DEF_AREAMAX;  // default value

  //mt.SetElement1Data(this);

  switch(elementTyp) {
    case T_POINT:
      *_resx=x1;
      *_resy=y1;
      break;

    case T_LINE:
      // Touchpoint on a line:
      //
      if(true) {
        RElement tmpEl;    // Temporary element

        ang = mtCorrAngle(getDirection1()+90.0);
        tmpEl.createLine(_mx, _my,
                         _mx+cos(ang/ARAD)*100.0, _my+sin(ang/ARAD)*100.0);

        getIntersection(&tmpEl, 0, _resx, _resy, 0, 0, 0, 0, false);
      }
      break;

    case T_ARC:
    case T_CIRCLE:
      // Touchpoint on an arc:
      //
      ang = mtGetAngle(cx, cy, _mx, _my);
      *_resx = cx+cos(ang/ARAD)*cr;
      *_resy = cy+sin(ang/ARAD)*cr;
      break;

    default:
      break;
  }
}



// Get distance from this element to a point:
//
float 
RElement::getDistanceToPoint(float _px, float _py, bool _noLimit)
const
{
  float ret=DEF_AREAMAX;

  switch(elementTyp) {
    case T_POINT:
      ret = getDistanceFromPointToPoint(_px, _py);
      break;
    case T_LINE:
      ret = getDistanceFromLineToPoint(_px, _py, _noLimit);
      break;
    case T_ARC:
      ret = getDistanceFromArcToPoint(_px, _py, _noLimit);
      break;
    case T_CIRCLE:
      ret = getDistanceFromCircleToPoint(_px, _py);
      break;
    case T_TEXT:
    case T_DIMENSION:
    case T_HATCHING:
      ret = getDistanceFromTextToPoint(_px, _py);
      break;
    default:
      break;
  }
  return ret;
}



// Get distance from this point a point
//
float 
RElement::getDistanceFromPointToPoint(float _px, float _py)
const
{
  float ret=DEF_AREAMAX;

  if(elementTyp==T_POINT) {
    ret = mtGetDistance(x1, y1, _px, _py);
  }

  return ret;
}



// Get distance from this line a point
//
float 
RElement::getDistanceFromLineToPoint(float _px, float _py, bool _noLimit)
const
{
  float ret=DEF_AREAMAX;

  if(elementTyp==T_LINE) {

    // Point is on an endpoint of the line:
    //
    if( (mtCompFloat(_px, x1) &&
         mtCompFloat(_py, y1)    ) ||
        (mtCompFloat(_px, x2) &&
         mtCompFloat(_py, y2)    )    ) {
      ret = 0.0;
      goto bye;
    }

    // point has no distance to line:
    //
    if(!_noLimit) {
      float bet1 = getRelAngleToPoint(_px, _py, true);
      float bet2 = getRelAngleToPoint(_px, _py, false);

      if(bet1>90.0 || bet1<-90.0 ||
         bet2>90.0 || bet2<-90.0    ) {
        ret=DEF_AREAMAX;
        goto bye;
      }
    }

    // Calculate distance:
    //
    float alpha,         // angle
          alph1, alph2,  // help-angle
          c;             // distance l1 -> p

    c = mtGetDistance(x1, y1, _px, _py);

    alph1 = mtGetAngle(x1, y1, _px, _py);
    alph2 = getDirection1();

    //if(alph1 == alph2) return(0.0);

    alpha = alph2-alph1;
    if(alpha<0.0) {
      if(alpha<-90.0) alpha+=360.0;
      else            alpha*=-1.0;
    }

    ret = sin(alpha/ARAD)*c;
    if(ret<0.0) ret*=-1.0;
  }

  bye:
  return ret;
}



// Get distance from this arc to a point
//
float 
RElement::getDistanceFromArcToPoint(float _px, float _py, bool _noLimit)
const
{
  float ret=DEF_AREAMAX;

  if(elementTyp==T_ARC) {
    // Unlimited:
    //
    if(_noLimit) {
      ret = mtGetDistance(cx, cy, _px, _py) - cr;
    }
    
    // Limited:
    //
    else {
      float alpha;       // angle center->point
      
      alpha = mtGetAngle(cx, cy, _px, _py);

      // Wannabe circles:
      //
      if(mtCompFloat(a1, a2, 0.00001)) {
        ret = mtGetDistance(cx, cy, _px, _py) - cr;
      }

      // Arcs:
      //
      else if(!getFlag(E_REVERSED)) {
        if(mtAngleIsBetween(alpha, a1, a2)) {
          ret = mtGetDistance(cx, cy, _px, _py) - cr;
        }
      }
      else {
        if(mtAngleIsBetween(alpha, a2, a1)) {
          ret = mtGetDistance(cx, cy, _px, _py) - cr;
        }
      }
    }
    if(ret<0.0) ret*=-1.0;
  }
  
  return ret;
}



// Get distance from this circle to a point
//
float 
RElement::getDistanceFromCircleToPoint(float _px, float _py)
const
{
  float ret=DEF_AREAMAX;

  if(elementTyp==T_CIRCLE) {
    ret=mtGetDistance(cx, cy, _px, _py) - cr;
    if(ret<0.0) ret*=-1.0;
  }
  
  return ret;
}



// Get distance from this text to a point
//
float 
RElement::getDistanceFromTextToPoint(float _px, float _py)
const
{
  float ret=DEF_AREAMAX;

  if((elementTyp==T_TEXT || elementTyp==T_DIMENSION || elementTyp==T_HATCHING) && graphic) {
    if(_px>leftBorder   && _px<rightBorder &&
       _py>bottomBorder && _py<topBorder      ) {
      /*ret=((float)RIni::getRIni()->getSnapRange()/3.0) /
          graphic->getZoom();*/
      ret=15.0 / graphic->getZoom();;
    }
  }
  
  return ret;
}



// Get relative angle from this line to a point:
//
float 
RElement::getRelAngleToPoint(float _px, float _py, bool _startPoint)
const
{
  float ret = 0.0;
  
  if(elementTyp==T_LINE) {
    float basAngle;
    float relAngle;

    if(_startPoint) {
      basAngle = getDirection1();
      relAngle = mtGetAngle(x1, y1, _px, _py);
    }
    else {
      basAngle = getDirection2();
      relAngle = mtGetAngle(x2, y2, _px, _py);
    }

    ret = relAngle-basAngle;
    if(ret> 180.0) ret-=360.0;
    if(ret<-180.0) ret+=360.0;
  }

  return ret;
}



// Get intersection(s) with an other element:
//
//   _itan: contains true after call if the elements don't intersect but touch
//   _onElements: true: intersection must be on the elements
//
void  
RElement::getIntersection(const RElement* _el, 
                          bool* _ires1, float* _ix1, float* _iy1, 
                          bool* _ires2, float* _ix2, float* _iy2,
                          bool* _itan,
                          bool _onElements ) 
const
{
  if(_el) {
    if(_ix1)   *_ix1=DEF_AREAMAX;
    if(_iy1)   *_iy1=DEF_AREAMAX;
    if(_ix2)   *_ix2=DEF_AREAMAX;
    if(_iy2)   *_iy2=DEF_AREAMAX;
    if(_ires1) *_ires1=false;
    if(_ires2) *_ires2=false;
    if(_itan)  *_itan=false;

    // No intersections with texts, dimensions, hatchings:
    //
    if(elementTyp==T_TEXT || _el->getElementTyp()==T_TEXT ||
       elementTyp==T_DIMENSION || _el->getElementTyp()==T_DIMENSION ||
       elementTyp==T_HATCHING || _el->getElementTyp()==T_HATCHING) return;

    // Contact points:
    //
    if(!isCircular() && !_el->isCircular()) {
      if( (mtCompFloat(x1, _el->x1) &&
           mtCompFloat(y1, _el->y1)    ) ||
          (mtCompFloat(x1, _el->x2) &&
           mtCompFloat(y1, _el->y2)    )    ) {
           
        if(_ires1) *_ires1=true;
        if(_ix1)   *_ix1 = x1;
        if(_iy1)   *_iy1 = y1;
        return;
      }
  
      if( (mtCompFloat(x2, _el->x1) &&
           mtCompFloat(y2, _el->y1)    ) ||
          (mtCompFloat(x2, _el->x2) &&
           mtCompFloat(y2, _el->y2)    )    ) {
  
        if(_ires1) *_ires1=true;
        if(_ix1)   *_ix1 = x2;
        if(_iy1)   *_iy1 = y2;
        return;
      }
    }

    // no intersection is possible:
    //
    if(_onElements) {
      float e1xb1, e1yb1, e1xb2, e1yb2;   // Border of 1st element
      float e2xb1, e2yb1, e2xb2, e2yb2;   // Border of 2nd element
      getBorders(&e1xb1, &e1yb1, &e1xb2, &e1yb2);
      _el->getBorders(&e2xb1, &e2yb1, &e2xb2, &e2yb2);

      if(e1xb1 > e2xb2+DEF_MMTOL ||
         e1yb1 > e2yb2+DEF_MMTOL ||
         e1xb2 < e2xb1-DEF_MMTOL ||
         e1yb2 < e2yb1-DEF_MMTOL    )  {
        return;
      }
    }

    if(elementTyp==T_LINE) {
      if(_el->getElementTyp()==T_LINE) {
        getIntersectionLineWithLine(_el, _ires1, _ix1, _iy1);
      }
      else if(_el->isCircular()) {
        getIntersectionLineWithArc(_el, 
                                   _ires1, _ix1, _iy1,
                                   _ires2, _ix2, _iy2,
                                   _itan );
      }
    }

    else if(isCircular()) {
      if(_el->getElementTyp()==T_LINE) {
        _el->getIntersectionLineWithArc(this, 
                                        _ires1, _ix1, _iy1,
                                        _ires2, _ix2, _iy2,
                                        _itan );
      }
      
      else if(_el->isCircular()) {
        getIntersectionArcWithArc(_el, 
                                  _ires1, _ix1, _iy1,
                                  _ires2, _ix2, _iy2,
                                  _itan );
      }
    }

    if(_onElements) {
    
      // Is the 1st point on both elements?
      //
      if(_ix1 && _iy1) {
        if(!isPointOnElement(*_ix1, *_iy1) ||
           !_el->isPointOnElement(*_ix1, *_iy1)) {
          *_ires1=false;
        }
      }

      // Is the 2nd point on both elements?
      //
      if(_ix2 && _iy2) {
        if(!isPointOnElement(*_ix2, *_iy2) ||
           !_el->isPointOnElement(*_ix2, *_iy2)) {
          *_ires2=false;
        }
      }
    }
  }
}



// Get intersection of this line with line _el:
//
void  
RElement::getIntersectionLineWithLine(const RElement* _el, 
                                      bool* _ires1, float* _ix1, float* _iy1) 
const
{
  if(_el) {
    if(_ix1)   *_ix1=DEF_AREAMAX;
    if(_iy1)   *_iy1=DEF_AREAMAX;
    if(_ires1) *_ires1=false;

    // Lines are parallel:
    // 
    if(isParallelWith(_el)) {
      return;
    }

    float t2_divisor,
          t2_dividend,
          t2;

    t2_dividend = (_el->getX2()-_el->getX1())*(y2-y1)
                - (_el->getY2()-_el->getY1())*(x2-x1);

    if(t2_dividend > DEF_MMTOL || t2_dividend < -DEF_MMTOL) {
       t2_divisor = x1*y2
                  - y1*x2
                  - _el->getX1()*(y2-y1)
                  + _el->getY1()*(x2-x1);
      t2 = t2_divisor / t2_dividend;
    }
    else {
      t2 = 0.0;
    }
  
    if(_ires1) *_ires1=true;
    if(_ix1)   *_ix1 = _el->getX1() + t2*(_el->getX2() - _el->getX1());
    if(_iy1)   *_iy1 = _el->getY1() + t2*(_el->getY2() - _el->getY1());
  }

  return;
}



// Get intersection from this line with arc/circle _el:
//
//   _itan: contains true after call if the elements don't intersect but touch
//
void  
RElement::getIntersectionLineWithArc(const RElement* _el, 
                                     bool* _ires1, float* _ix1, float* _iy1,
                                     bool* _ires2, float* _ix2, float* _iy2,
                                     bool* _itan)
const
{
  if(_el) {
    RElement tmpEl;             // Copy of element
    tmpEl.getMeasuresFrom(_el);
  
    if(_ix1)   *_ix1=DEF_AREAMAX;
    if(_iy1)   *_iy1=DEF_AREAMAX;
    if(_ires1) *_ires1=true;
    if(_ix2)   *_ix2=DEF_AREAMAX;
    if(_iy2)   *_iy2=DEF_AREAMAX;
    if(_ires2) *_ires2=true;
    if(_itan)  *_itan=false;

    float disLC;                        // Distance Line <-> arc center

    tmpEl.forceDirection(true);

    disLC = getDistanceToPoint(tmpEl.getCx(), tmpEl.getCy(), true);
    
    // Distance Arccenter <-> Line > Radius: no intersection:
    //
    if(disLC > _el->getCr()+DEF_MMTOL) {
       if(_ires1) *_ires1 = false;
       if(_ires2) *_ires2 = false;
       return;
    }

    // Distance Arc center <-> Line == Radius: tangent:
    //
    if(mtCompFloat(disLC, tmpEl.getCr())) {
      float vx, vy;
      float an = getDirection1();
      vx = tmpEl.getCx() + cos((an+90.0)/ARAD) * 100.0;
      vy = tmpEl.getCy() + sin((an+90.0)/ARAD) * 100.0;
      tmpEl.createLine(tmpEl.getCx(), tmpEl.getCy(), vx, vy);
      getIntersectionLineWithLine(&tmpEl, _ires1, _ix1, _iy1);
      if(_itan)  *_itan = true;
      if(_ires2) *_ires2 = false;
      return;
    }


    float axs, ays, exs, eys;
    float gradient, lineOffsY;
    float d, p, q;
    float fdum;                 // Dummy float
    bool  bdum;                 // Dummy bool
    RElement tmpEl1;
    RElement tmpEl2;
  
    axs = x1 - _el->getCx();            // ax'
    ays = y1 - _el->getCy();            // ay'
    exs = x2 - _el->getCx();            // ex'
    eys = y2 - _el->getCy();            // ey'
  
    if(exs-axs!=0.0) {
       gradient = (eys-ays) / (exs-axs);      // Gradient
    }
    else {
       gradient = 0.0;                        // Line is vertical
    }

    // Y-offset of the line:
    //
    tmpEl1.createLine(axs, ays, exs, eys);
    tmpEl2.createLine(0.0, -1000.0, 0.0, 1000.0);
    tmpEl1.getIntersectionLineWithLine(&tmpEl2, &bdum, &fdum, &lineOffsY);

    d = 1.0 + gradient*gradient;
    p = (gradient*lineOffsY*2.0) / d;
    q = (lineOffsY*lineOffsY - _el->getCr()*_el->getCr()) / d;
  
    // Line is vertical:
    //
    if(mtCompFloat(x1, x2)) {
      if(_ix1) *_ix1 = x1;
      if(_ix2) *_ix2 = x1;
    }
    else {
       // calculate x-intersections:
       //
       if(_ix1) *_ix1 = _el->getCx() + (-(p/2.0) + sqrt(fabs((p/2.0)*(p/2.0) - q )));
       if(_ix2) *_ix2 = _el->getCx() + (-(p/2.0) - sqrt(fabs((p/2.0)*(p/2.0) - q )));
    }
  
    // Line is horizontal:
    //
    if(mtCompFloat(y1, y2)) {
      if(_iy1) *_iy1 = y1;
      if(_iy2) *_iy2 = y1;
    }
    else {
      // calculate y-intersections:
      //
      if(!mtCompFloat(x1, x2)) {
        if(_iy1 && _ix1) *_iy1 = gradient*(*_ix1 - _el->getCx()) + lineOffsY + _el->getCy();
        if(_iy2 && _ix2) *_iy2 = gradient*(*_ix2 - _el->getCx()) + lineOffsY + _el->getCy();
      }
      else {
        if(_iy1 && _ix1) *_iy1 = _el->getCy() + sqrt(fabs(_el->getCr()*_el->getCr() - (*_ix1 - _el->getCx())*(*_ix1 - _el->getCx())));
        if(_iy2 && _ix1) *_iy2 = _el->getCy() - sqrt(fabs(_el->getCr()*_el->getCr() - (*_ix1 - _el->getCx())*(*_ix1 - _el->getCx())));
      }
    }
  }
  
  return;
}



// Get intersection from this arc/circle with arc/circle _el:
//
//   _itan: contains true after call if the elements don't intersect but touch
//
void  
RElement::getIntersectionArcWithArc(const RElement* _el, 
                                    bool* _ires1, float* _ix1, float* _iy1,
                                    bool* _ires2, float* _ix2, float* _iy2,
                                    bool* _itan)
const
{
  RLOG("\n001");

  if(_el) {
    if(_ires1) *_ires1=true;
    if(_ires2) *_ires2=true;
    if(_itan)  *_itan=false;

    RLOG("\n002");

    float dist = mtGetDistance(cx, cy, _el->getCx(), _el->getCy());
    
    // No Intersection possible:
    //
    if(dist > cr+_el->getCr()+DEF_MMTOL ||
       cr-dist > _el->getCr()+DEF_MMTOL ||
       _el->getCr()-dist > cr+DEF_MMTOL ||
       dist<DEF_MMTOL                      ) {
       if(_ires1) *_ires1=false;
       if(_ires2) *_ires2=false;

      RLOG("\n003");

      return;
    }

    RLOG("\n004");
    

    // One touching point (tangent) first possibility:
    //
    if(mtCompFloat(dist, cr + _el->getCr(), 0.001) ||
       mtCompFloat(dist, cr - _el->getCr(), 0.001)    ) {

      float an = mtGetAngle(cx, cy, _el->getCx(), _el->getCy());
      if(_ix1)   *_ix1 = cx + cos(an/ARAD) * cr;
      if(_iy1)   *_iy1 = cy + sin(an/ARAD) * cr;
      if(_ires2) *_ires2 = false;
      if(_itan)  *_itan = true;

      RLOG("\n005");

      return;
    }

    RLOG("\n006");


    // One touching point (tangent) second possibility:
    //
    if(mtCompFloat(dist, _el->getCr() - _el->getCr(), 0.001)) {
       float an = mtGetAngle(cx, cy, _el->getCx(), _el->getCy());
       if(_ix1)   *_ix1 = cx - cos(an/ARAD) * cr;
       if(_iy1)   *_iy1 = cy - sin(an/ARAD) * cr;
       if(_ires2) *_ires2 = false;
       if(_itan)  *_itan = true;

       RLOG("\n007");

       return;
    }

    RLOG("\n008");

    // Centers have the same y-value:
    //
    //if(cy==_el->getCy()) {
    if(mtCompFloat(cy, _el->getCy(), 0.0001)) {
      /*if(_ix1)   *_ix1 = (cx+_el->getCx())/2.0;
      if(_ix2)   *_ix2 = (cx+_el->getCx())/2.0;
      if(_iy1)   *_iy1 = cy + sqrt(fabs(cr*cr - (dist/2.0)*(dist/2.0)));
      if(_iy2)   *_iy2 = cy - sqrt(fabs(cr*cr - (dist/2.0)*(dist/2.0)));

      RLOG("\n009");
      return;*/

      float dis = cx-_el->getCx();
      if(!mtCompFloat(dis, 0.0001) && !mtCompFloat(_el->getCr(), 0.0001)) {
        float ang = acos( -( (cr*cr-_el->getCr()*_el->getCr()-dis*dis) / (2*_el->getCr()*dis) ) );
        if(_ix1)   *_ix1 = _el->getCx()+cos(ang)*_el->getCr();
        if(_ix2)   *_ix2 = _el->getCx()+cos(ang)*_el->getCr();
        if(_iy1)   *_iy1 = _el->getCy()+sin(ang)*_el->getCr();
        if(_iy2)   *_iy2 = _el->getCy()-sin(ang)*_el->getCr();
      }
      else {
        if(_ires1) *_ires1=false;
        if(_ires2) *_ires2=false;
      }
      return;
    }

    RLOG("\n010");

    float ca, da;          // Quadrats of el1->cr and el2->cr
    float p, q;
    float fact_x2;         // Factor of x Quadrats
    float fact_x;          // Factor of x
    float fact;            // Factor of itself :)
    float vz, vz2, vn;     // Divident & Divisor
   
    ca = cr*cr;
    da = _el->getCr()*_el->getCr();
   
    vz  = cx*cx - _el->getCx()*_el->getCx() +
          cy*cy - _el->getCy()*_el->getCy() - ca + da;
    vz2 = cx-_el->getCx();
    vn  = 2.0 * (cy-_el->getCy());

    if(vn==0.0) vn=0.000001;
   
    fact_x2 = 1.0 + (4.0*vz2*vz2)/(vn*vn);
    fact_x  = -2.0*_el->getCx() - (4.0*vz2*vz)/(vn*vn) +
                                  (4.0*_el->getCy()*vz2)/(vn);
    fact    = _el->getCx()*_el->getCx() + (vz*vz)/(vn*vn) -
              (2.0*_el->getCy()*vz)/(vn) + _el->getCy()*_el->getCy() - da;
   
    p = fact_x / fact_x2;
    q = fact   / fact_x2;
   
    if(_ix1) *_ix1 = (-(p/2.0) + sqrt(fabs( (p/2.0)*(p/2.0) - q )));
    if(_ix2) *_ix2 = (-(p/2.0) - sqrt(fabs( (p/2.0)*(p/2.0) - q )));
    if(_iy1 && _ix1) *_iy1 = (-2.0* *_ix1*(cx-_el->getCx()) +
                             cx*cx - _el->getCx()*_el->getCx() +
                             cy*cy - _el->getCy()*_el->getCy() - ca + da) /
                             (2.0*(cy-_el->getCy()));
    if(_iy2 && _ix2) *_iy2 = (-2.0* *_ix2*(cx-_el->getCx()) +
                             cx*cx - _el->getCx()*_el->getCx() +
                             cy*cy - _el->getCy()*_el->getCy() - ca + da) /
                             (2.0*(cy-_el->getCy()));

    RLOG("\n011");
  }
}


// Get Gliding ("The point somewhere between the elements") between this element and an other:
//  _el   The other element
//  _gres true is in this variable returned if there is a gliding
//  _gx   X-Coordinate of the 1st gliding is retruned in this variable
//  _gy   Y-Coordinate of the 1st gliding is retruned in this variable
//
void  
RElement::getGliding(const RElement* _el, 
                     bool* _gres, float* _gx, float* _gy) 
const
{
  if(_el) {
    if(_gx)   *_gx=DEF_AREAMAX;
    if(_gy)   *_gy=DEF_AREAMAX;
    if(_gres) *_gres=false;

    // No glidings with texts:
    //
    if(elementTyp==T_TEXT || _el->getElementTyp()==T_TEXT ||
       elementTyp==T_DIMENSION || _el->getElementTyp()==T_DIMENSION ||
       elementTyp==T_HATCHING || _el->getElementTyp()==T_HATCHING) return;

    
    if(elementTyp==T_LINE) {
      if(_el->getElementTyp()==T_LINE) {
        // No gliding is possible between lines:
        //
        return;
      }
      else if(_el->isCircular()) {
        getGlidingLineWithArc(_el, 
                              _gres, _gx, _gy);
      }
    }

    else if(isCircular()) {
      if(_el->getElementTyp()==T_LINE) {
        _el->getGlidingLineWithArc(this, 
                                   _gres, _gx, _gy);
      }
      
      else if(_el->isCircular()) {
        getGlidingArcWithArc(_el, 
                             _gres, _gx, _gy);
      }
    }

  }

}



// Get Gliding between this line and an arc:
//   _el   The arc
//   _gres true is in this variable returned if there is a gliding
//   _gx   X-Coordinate of the 1st gliding is retruned in this variable
//   _gy   Y-Coordinate of the 1st gliding is retruned in this variable
//
void  
RElement::getGlidingLineWithArc(const RElement* _el, 
                                bool* _gres, float* _gx, float* _gy) 
const
{
  if(_el) {
    if(_gres) *_gres=true;
    float disLC;                        // Distance Line <-> arc center
  
    //el2->ForceDirection();
  
    disLC = getDistanceToPoint(_el->getCx(), _el->getCy(), true);
  
    // Distance Arccenter <-> Line > Radius: no intersection:
    //
    if(disLC > _el->getCr()) {
      float ang1c;        // angle from 1st line point to arc center
      float angEl1;       // angle from 1st line point to 2nd line point
      float angN;         // angle from normale from cx/cy to line
      float disGC;        // distance from center to gliding point
      
      ang1c = mtGetAngle(x1, y1, _el->getCx(), _el->getCy());
      angEl1 = mtGetAngle(x1, y1, x2, y2);
  
      if(ang1c<angEl1) {
        ang1c+=360.0;
      }
      if(ang1c-angEl1<180.0) {
        angN = mtCorrAngle(angEl1-90.0);
      }
      else {
        angN = mtCorrAngle(angEl1+90.0);
      }
  
      disGC = (_el->getCr()+(disLC-_el->getCr())/2); 
      if(_gx) *_gx = _el->getCx() + cos(angN/ARAD) * disGC;
      if(_gy) *_gy = _el->getCy() + sin(angN/ARAD) * disGC;
    }
    else {
      if(_gres) *_gres=true;
    }
  }
}



// Get Gliding between this arc and an other one:
//   _el   The other arc
//   _gres true is in this variable returned if there is a gliding
//   _gx   X-Coordinate of the 1st gliding is retruned in this variable
//   _gy   Y-Coordinate of the 1st gliding is retruned in this variable
//
void  
RElement::getGlidingArcWithArc(const RElement* _el, 
                               bool* _gres, float* _gx, float* _gy) 
const
{
  if(_el) {
    if(_gres) *_gres=true;

    float dist = mtGetDistance(cx, cy, _el->getCx(), _el->getCy());
    float ang = mtGetAngle(cx, cy, _el->getCx(), _el->getCy());
    float disGC;

    // No intersection but gliding:
    //
    if( dist > cr+_el->getCr()+DEF_MMTOL ||
        cr-dist > _el->getCr()+DEF_MMTOL ||
        _el->getCr()-dist > cr+DEF_MMTOL    ) {

      disGC = cr + (dist - cr - _el->getCr()) / 2.0;
      if(_gx) *_gx = cx + cos(ang/ARAD) * disGC;
      if(_gy) *_gy = cy + sin(ang/ARAD) * disGC;
    }
    else {
      if(_gres) *_gres=true;
    }
  }
}



// Get intersections of the parallels of this element with the parallels of another element:
//
// _el The other element
// _dist Distance to parallels
// _ires Pointer to Array of eight bools (for returning if intersection n is valid)
// _ipx  Pointer to Array of eight floats (for returning the x-values of the intersection points)
// _ipy  Pointer to Array of eight floats (for returning the y-values of the intersection points)
//
void  
RElement::getParallelIntersections(const RElement* _el, 
                                   float _dist,
                                   bool* _ires, float* _ipx, float* _ipy) 
const
{
  if(_el) {
    
    int i;                  // counter
    int i1, i2;             // counter
    int pc1, pc2;           // parallel counter
    RElement parallel[4];   // the parallels
             
    // Ini values:
    //
    for(i=0; i<8; ++i) {
      _ipx[i]=DEF_AREAMAX;
      _ipy[i]=DEF_AREAMAX;
      _ires[i]=false;
    }

    // No intersections with texts:
    //
    if(elementTyp==T_TEXT || _el->getElementTyp()==T_TEXT ||
       elementTyp==T_DIMENSION || _el->getElementTyp()==T_DIMENSION ||
       elementTyp==T_HATCHING || _el->getElementTyp()==T_HATCHING) return;
    
    // Get parallels:
    //
    getParallels(&parallel[0], &parallel[1], _dist);
    _el->getParallels(&parallel[2], &parallel[3], _dist);
    
    i1=i2=0;
    for(pc1=0; pc1<=1; ++pc1) {
      for(pc2=2; pc2<=3; ++pc2) {
        parallel[pc1].getIntersection(&parallel[pc2], 
                                      &_ires[i1  ], &_ipx[i1  ], &_ipy[i1  ],
                                      &_ires[i1+1], &_ipx[i1+1], &_ipy[i1+1],
                                      0, false);
        
        if(_ires[i1]) {
          _ires[i2] = _ires[i1];
          _ipx[i2]  = _ipx[i1];
          _ipy[i2]  = _ipy[i1];
          if(i2<8) ++i2;
        }
        if(_ires[i1+1]) {
          _ires[i2] = _ires[i1+1];
          _ipx[i2]  = _ipx[i1+1];
          _ipy[i2]  = _ipy[i1+1];
          if(i2<8) ++i2;
        }
        
        if(i1<6) i1+=2;
      }
    }
    
    while(i2<8) {
      _ires[i2]=false;
      _ipx[i2] =DEF_AREAMAX;
      _ipy[i2] =DEF_AREAMAX;
      ++i2;
    }
    
  }
}



// Get the point which we must trim if the user 
//   clicked an element for trimming
//
// return: true: trim endpoint
//         false: trim startpoint
//
bool  
RElement::getTrimPoint(float _mx, float _my, float _ix, float _iy)
{
  bool ret=false;         // returned value

  float angEl;            // angle of element
  float angM;             // angle from intersection to mouse
  float angDif;           // angle difference

  switch(elementTyp) {
    case T_POINT:
      ret = false;
      break;

    case T_LINE:
      angEl  = mtGetAngle(x1, y1, x2, y2);
      angM   = mtGetAngle(_ix, _iy, _mx, _my);
      angDif = angEl-angM;
      if(angDif<  0.0) angDif*=-1.0;
      if(angDif>180.0) angDif=360.0-angDif;
      
      if(angDif< 90.0) ret = false;
      else ret = true;
      break;

    case T_ARC:
      angEl = mtGetAngle(cx, cy, _ix, _iy);
      angM  = mtGetAngle(cx, cy, _mx, _my);

      if(mtGetAngleDiff(angM, angEl)>180.0) ret = false;
      else ret = true;

      if(getFlag(E_REVERSED)) {
        ret = !ret;
      }
      break;

    case T_CIRCLE:
      ret=false;
      /*angM  = mtGetAngle(cx, cy, _mx, _my);

      if(mtGetAngleDiff(angM, angEl)>180.0) ret = false;
      else ret = true;

      if(getFlag(E_REVERSED)) {
        ret = !ret;
      }*/
      break;

    default:
      break;
  }

  return ret;
}



// Test if the element gets trimmed to zero:
//
// _ep      : trim endpoint
// _ix/_iy  : trim to this point
//
bool 
RElement::isTrimmedToZero(bool _ep, float _ix1, float _iy1)
{
  bool ret;            // return value

  if(elementTyp==T_CIRCLE) return false;

  if( ( !_ep && mtCompFloat(_ix1, x2)
             && mtCompFloat(_iy1, y2) ) ||
      (  _ep && mtCompFloat(_ix1, x1)
             && mtCompFloat(_iy1, y1) )    ) {
    ret = true;
  }
  else {
    ret = false;
  }

  return ret;
}



// Trim this element:
//
// _ep        : trim endpoint
// _ix1/_iy1  : trim to this point
// _ix2/_iy2  : trim to this point too for circles
//
void 
RElement::trim(bool _ep,
               float _ix1, float _iy1,
               float _ix2, float _iy2,
               float _mang)
{
  switch(elementTyp) {

    case T_POINT:
      // we usually don't trim points, do we?
      break;

    case T_LINE:
      if(!_ep) {         // trim start point:
        x1 = _ix1;
        y1 = _iy1;
      }
      else {             // trim end point:
        x2 = _ix1;
        y2 = _iy1;
      }
      break;

    case T_ARC:
      if(!_ep) {         // trim start point:
        x1 = _ix1;
        y1 = _iy1;
      }
      else {             // trim end point:
        x2 = _ix1;
        y2 = _iy1;
      }
      recalculateAngles();
      break;

    case T_CIRCLE:
      elementTyp = T_ARC;
      x1 = _ix1;
      y1 = _iy1;
      if(!mtCompFloat(_ix2, DEF_AREAMAX) &&
         !mtCompFloat(_iy2, DEF_AREAMAX)    ) {
        x2 = _ix2;
        y2 = _iy2;
      }
      else {
        x2 = cx + (cx-_ix1);
        y2 = cy + (cy-_iy1);
      }
      recalculateAngles();

      RLOG("\nAngles: ");
      RLOG("\na1: ");
      RLOG(a1);
      RLOG("\na2: ");
      RLOG(a2);
      RLOG("\nmang: ");
      RLOG(_mang);
      RLOG("\nbetween: ");
      RLOG(mtAngleIsBetween(_mang, a1, a2));

      if(mtAngleIsBetween(_mang, a1, a2)) {
        delFlag(E_REVERSED);
      }
      else {
        setFlag(E_REVERSED);
      }
      break;

    default:
      break;
  }

  recalculateBorders();
}




// Test if point _x/_y is on the element:
//
bool  
RElement::isPointOnElement(float _x, float _y, float _tol)
const
{
  if(getDistanceToPoint(_x, _y)    < _tol ||
     mtGetDistance(x1, y1, _x, _y) < _tol ||
     mtGetDistance(x2, y2, _x, _y) < _tol    ) {

    return true;
  }
  else {
    return false;
  }
}



/**
 * Checks, if the given point is inside the "contour" defined
 *   by the sub elements.
 */
bool
RElement::isPointInsideSubElements(float _x, float _y)
const
{
  RElement testLine;
  bool ires1, ires2;

  int is;      // Number of normal intersections
  int isUp;    // Number of touching points upwards
  int isDown;  // Number of touching points downwards

  int trues=0,   // How many trues/falses have we found?
      falses=0;

  // Try different directions until it's clear -> then: break
  for(int dir=0; dir<3; ++dir) {
    is=0;
    isUp=0;
    isDown=0;

    switch(dir) {
      case 0: testLine.createLine(_x,_y, leftBorder-1.0,_y); break;
      case 1: testLine.createLine(_x,_y, rightBorder+1.0,_y); break;
      case 2: testLine.createLine(_x,_y, _x,topBorder+1.0); break;
      default: break;
    }

    for(int i=0; i<subElementNum; ++i) {

      // Count intersections:
      ires1=ires2=false;
      float ix1, iy1, ix2, iy2;
      bool itan;
      testLine.getIntersection(&subElement[i],
                               &ires1, &ix1,&iy1,
                               &ires2, &ix2,&iy2,
                               &itan, true);

      if(!itan) {
        if(subElement[i].getElementTyp()==T_CIRCLE) {
          if(ires1 && (!mtCompFloat(ix1, testLine.x1) || !mtCompFloat(iy1, testLine.y1))) {
            is++;
          }
          if(ires2) is++;
        }
        else {
          if(ires1) {
            // No real intersection but touching points:
            if(mtCompFloat(ix1, subElement[i].x1) && mtCompFloat(iy1, subElement[i].y1)) {
              if(dir<2) {
                if(subElement[i].getDirection1()<180.0) isUp++;
                else                                    isDown++;
              }
              else {
                if(subElement[i].getDirection1()>90.0 &&
                   subElement[i].getDirection1()<270.0   ) isUp++;
                else                                       isDown++;
              }
            }
            else if(mtCompFloat(ix1, subElement[i].x2) && mtCompFloat(iy1, subElement[i].y2)) {
              if(dir<2) {
                if(subElement[i].getDirection2()<180.0) isUp++;
                else                                    isDown++;
              }
              else {
                if(subElement[i].getDirection2()>90.0 &&
                   subElement[i].getDirection2()<270.0   ) isUp++;
                else                                       isDown++;
              }
            }
            else {
              is++;
            }
          }

          if(ires2) {
            // No real intersection but touching points:
            if(mtCompFloat(ix2, subElement[i].x1) && mtCompFloat(iy2, subElement[i].y1)) {
              if(dir<2) {
                if(subElement[i].getDirection1()<180.0) isUp++;
                else                                    isDown++;
              }
              else {
                if(subElement[i].getDirection1()>90.0 &&
                   subElement[i].getDirection1()<270.0   ) isUp++;
                else                                       isDown++;
              }
            }
            else if(mtCompFloat(ix2, subElement[i].x2) && mtCompFloat(iy2, subElement[i].y2)) {
              if(dir<2) {
                if(subElement[i].getDirection2()<180.0) isUp++;
                else                                    isDown++;
              }
              else {
                if(subElement[i].getDirection2()>90.0 &&
                   subElement[i].getDirection2()<270.0   ) isUp++;
                else                                       isDown++;
              }
            }
            else {
              is++;
            }
          }
        }
      }
    }

    if(isUp>isDown) is+=isDown;
    else            is+=isUp;

    if(is%2==1) trues++;
    else        falses++;
  }

  if(trues>falses) return true;
  else             return false;
}



// Test if this element is parallel with element _el:
//
bool
RElement::isParallelWith(const RElement* _el)
const
{
  bool ret=false;
  
  if(_el) {
    // Two lines:
    //
    if(elementTyp==T_LINE && _el->getElementTyp()==T_LINE) {
      float ang1, ang2;
      ang1 = getDirection1();
      ang2 = _el->getDirection1();
      if(mtCompFloat(ang1, ang2) ||
         mtCompFloat(ang1, mtCorrAngle(ang2+180.0)) ) {
        ret=true;
      }
    }

    // Two arcs/circles:
    //
    if(isCircular() && _el->isCircular()) {
      if(mtCompFloat(cx, _el->getCx()) && mtCompFloat(cy, _el->getCy())) {
        ret = true;
      }
    }
  }

  return ret;
}



// Get parallel to this element:
//
bool
RElement::getParallel(RElement* _parallel, 
                      float _px, float _py, 
                      float _dist)
const
{
  if(_parallel) {
    RElement p1, p2;
    float dist1, dist2;

    _parallel->reset();
  
    getParallels(&p1, &p2, _dist);
    dist1 = p1.getDistanceToPoint(_px, _py);
    dist2 = p2.getDistanceToPoint(_px, _py);
  
    if(dist1<dist2 || !p2.getFlag(E_VISIBLE)) _parallel->getMeasuresFrom(&p1);
    else                                      _parallel->getMeasuresFrom(&p2);
  }
  
  if(_parallel && _parallel->getFlag(E_VISIBLE)) return true;
  else                                           return false;
}



// Get both parallels for this elemennt
//   (used from getParallel())
//
void  
RElement::getParallels(RElement* _parallel1, 
                       RElement* _parallel2, 
                       float _dist)
const
{
  float angle;

  if(_parallel1 && _parallel2) {
  
    _parallel1->reset();
    _parallel2->reset();
    
    switch(elementTyp) {
      case T_LINE:
        // 1. Parallel of a line:
        //
        angle = mtCorrAngle(getDirection1()+90.0);
        _parallel1->createLine( x1 + cos(angle/ARAD) * _dist,
                                y1 + sin(angle/ARAD) * _dist,
                                x2 + cos(angle/ARAD) * _dist,
                                y2 + sin(angle/ARAD) * _dist );
      
        // 2. Parallel of a line:
        //
        angle = mtCorrAngle(getDirection1()-90.0);
        _parallel2->createLine( x1 + cos(angle/ARAD) * _dist,
                                y1 + sin(angle/ARAD) * _dist,
                                x2 + cos(angle/ARAD) * _dist,
                                y2 + sin(angle/ARAD) * _dist );

        break;

      case T_ARC:
        // 1. Parallel of an arc:
        //
        _parallel1->createArc( cx, cy, cr+_dist,
                               a1, a2, getFlag(E_REVERSED) );

        // 2. Parallel of an arc:
        //
        if(cr-_dist>0.0) {
          _parallel2->createArc( cx, cy, cr-_dist,
                                 a1, a2, getFlag(E_REVERSED) );
        }
        break;

      case T_CIRCLE:
        // 1. Parallel of an arc:
        //
        _parallel1->createCircle( cx, cy, cr+_dist,
                                  a1, a2, getFlag(E_REVERSED) );

        // 2. Parallel of an arc:
        //
        if(cr-_dist>0.0) {
          _parallel2->createCircle( cx, cy, cr-_dist,
                                    a1, a2, getFlag(E_REVERSED) );
        }
        break;

      default:
        break;
    }
  }
}



// Get the bisector from this line with line _leg which is
//   next to _px/_py with length _length:
//
bool
RElement::getBisector(RElement* _bisector,
                      const RElement* _leg,
                      float _px, float _py,
                      float _length ) 
const
{
  if(_leg && _bisector) {
    RElement b1, b2, b3, b4;    // The four bisectors
    float mAngle,               // Angle from inters. to mousepointer
          angEl1,               // Angles from inters. to endp. of elements
          angEl2,               //
          angEl1r,              // Angles from inters. to startp. of elements
          angEl2r;              //
  
    // Get intersection between the two elements:
    //
    bool  ires;
    float ix, iy;
    getIntersection(_leg, 
                    &ires, &ix, &iy, 
                    0,     0,   0, 
                    0, false);

    if(ires) {
  
      angEl1  = getDirection1();
      angEl2  = _leg->getDirection1();
      angEl1r = getDirection2();
      angEl2r = _leg->getDirection2();
    
      getBisectors(&b1, &b2, &b3, &b4, _leg, ix, iy, _length);
    
      // Find line which is in the same sector like the mouse cursor:
      //
      mAngle = mtGetAngle(ix, iy, _px, _py);
    
      // 1st element is the solution:
      //
      if(mtGetAngleDiff(angEl1, angEl2)<180.0) {
        if(mtAngleIsBetween(mAngle, angEl1, angEl2)) {
          _bisector->getMeasuresFrom(&b1);
          return true;
        }
      }
      else {
        if(mtAngleIsBetween(mAngle, angEl2, angEl1)) {
          _bisector->getMeasuresFrom(&b1);
          return true;
        }
      }
    
      // 2nd element is the solution:
      //
      if(mtGetAngleDiff(angEl2, angEl1r)<180.0) {
        if(mtAngleIsBetween(mAngle, angEl2, angEl1r)) {
          _bisector->getMeasuresFrom(&b2);
          return true;
        }
      }
      else {
        if(mtAngleIsBetween(mAngle, angEl1r, angEl2)) {
          _bisector->getMeasuresFrom(&b2);
          return true;
        }
      }
    
      // 3rd element is the solution:
      //
      if(mtGetAngleDiff(angEl1r, angEl2r)<180.0) {
        if(mtAngleIsBetween(mAngle, angEl1r, angEl2r)) {
          _bisector->getMeasuresFrom(&b3);
          return true;
        }
      }
      else {
        if(mtAngleIsBetween(mAngle, angEl2r, angEl1r)) {
          _bisector->getMeasuresFrom(&b3);
          return true;
        }
      }
    
      // 4th element is the solution:
      //
      if(mtGetAngleDiff(angEl2r, angEl1)<180.0) {
        if(mtAngleIsBetween(mAngle, angEl2r, angEl1)) {
          _bisector->getMeasuresFrom(&b4);
          return true;
        }
      }
      else {
        if(mtAngleIsBetween(mAngle, angEl1, angEl2r)) {
          _bisector->getMeasuresFrom(&b4);
          return true;
        }
      }
    }
  }
  
  return false;
  
}



// Get all four bisectors from this line with line _leg 
//   and length _length:
//
//  _ix/_iy: intersection of the elements 
//    (calculated in function getBisector())
//
void
RElement::getBisectors(RElement* _bisector1,
                       RElement* _bisector2,
                       RElement* _bisector3,
                       RElement* _bisector4,
                       const RElement* _leg,
                       float _ix, float _iy,
                       float _length ) 
const
{

  if(_leg && _bisector1 && _bisector2 && _bisector3 && _bisector4) {
    float angEl1,               // Angle from inters. to endp.   el1
          angEl2,               // "                             el2
          angEl1r,              // "                     startp. el1
          angEl2r,              // "                             el2
          angMid;               // Angle in the middle of angEl1 and angEl2
    int   secCount;             // Security-Counter
  
    // Works only for lines:
    //
    if(elementTyp==T_LINE && _leg->getElementTyp()==T_LINE) {

      angEl1  = getDirection1();
      angEl2  = _leg->getDirection1();
      angEl1r = getDirection2();
      angEl2r = _leg->getDirection2();
  
      // Get the mid angle between 'angEl1' and 'angEl2':
      //
      angMid = mtCorrAngle((angEl1+angEl2)/2.0);
  
      // 1st element is between 'angEl1' and 'angEl2':
      //
      if(mtGetAngleDiff(angEl1, angEl2)<180.0) {
        secCount = 0;
        while(!mtAngleIsBetween(angMid, angEl1, angEl2) && secCount<4) {
          angMid = mtCorrAngle(angMid+90.0);
          ++secCount;
        }
      }
      else {
        secCount = 0;
        while(!mtAngleIsBetween(angMid, angEl2, angEl1) && secCount<4) {
          angMid = mtCorrAngle(angMid+90.0);
          ++secCount;
        }
      }
      _bisector1->createLine(_ix, _iy,
                             _ix + cos(angMid/ARAD) * _length,
                             _iy + sin(angMid/ARAD) * _length);
  
      // 2nd element is between 'angEl2' and 'angEl1r':
      //
      if(mtGetAngleDiff(angEl2, angEl1r)<180.0) {
        secCount = 0;
        while(!mtAngleIsBetween(angMid, angEl2, angEl1r) && secCount<4) {
          angMid = mtCorrAngle(angMid+90.0);
          ++secCount;
        }
      }
      else {
        secCount = 0;
        while(!mtAngleIsBetween(angMid, angEl1r, angEl2) && secCount<4) {
          angMid = mtCorrAngle(angMid+90.0);
          ++secCount;
        }
      }
      _bisector2->createLine(_ix, _iy,
                             _ix + cos(angMid/ARAD) * _length,
                             _iy + sin(angMid/ARAD) * _length);
  
      // 3rd element is between 'angEl1r' and 'angEl2r':
      //
      if(mtGetAngleDiff(angEl1r, angEl2r)<180.0) {
        secCount = 0;
        while(!mtAngleIsBetween(angMid, angEl1r, angEl2r) && secCount<4) {
          angMid = mtCorrAngle(angMid+90.0);
          ++secCount;
        }
      }
      else {
        secCount = 0;
        while(!mtAngleIsBetween(angMid, angEl2r, angEl1r) && secCount<4) {
          angMid = mtCorrAngle(angMid+90.0);
          ++secCount;
        }
      }
      _bisector3->createLine(_ix, _iy,
                             _ix + cos(angMid/ARAD) * _length,
                             _iy + sin(angMid/ARAD) * _length);
  
  
      // 4th element is between 'angEl2r' and 'angEl1':
      //
      if(mtGetAngleDiff(angEl2r, angEl1)<180.0) {
        secCount = 0;
        while(!mtAngleIsBetween(angMid, angEl2r, angEl1) && secCount<4) {
          angMid = mtCorrAngle(angMid+90.0);
          ++secCount;
        }
      }
      else {
        secCount = 0;
        while(!mtAngleIsBetween(angMid, angEl1, angEl2r) && secCount<4) {
          angMid = mtCorrAngle(angMid+90.0);
          ++secCount;
        }
      }
      _bisector4->createLine(_ix, _iy,
                             _ix + cos(angMid/ARAD) * _length,
                             _iy + sin(angMid/ARAD) * _length);
    }
  }
}



// Get the tangent from this circle to point _x/_y
//   which is next to _px/_py:
//
bool
RElement::getTangent1(RElement* _tangent,
                      float _x, float _y,
                      float _px, float _py)
const
{
  bool ret=false;

  if(_tangent && isInGroup(T_CIRCULAR)) {
    RElement t1, t2;            // The two tangents
    float dist1, dist2;         // Distances to two tangents

    if(mtGetDistance(cx, cy, _x, _y) > cr) {
      if(getTangents1(&t1, &t2, _x, _y)) {

        // Get distances to points (find element next to mouse cursor):
        //
        dist1 = t1.getDistanceToPoint(_px, _py, true);
        dist2 = t2.getDistanceToPoint(_px, _py, true);

        if(dist1<dist2) _tangent->getMeasuresFrom(&t1);
        else            _tangent->getMeasuresFrom(&t2);

        ret=true;
      }
    }
  }

  return ret;
}


// Get the tangents from this line to circle _circle which is
//   next to _px/_py:
//
bool
RElement::getTangents1(RElement* _tangent1,
                       RElement* _tangent2,
                       float _x, float _y)
const
{
  bool ret=false;
  
  if(_tangent1 && _tangent2 && isInGroup(T_CIRCULAR)) {
    RElement t1, t2;            // The two tangents
    float dist1  = mtGetDistance(_x, _y, cx, cy);

    if(dist1>cr && cr>0.0001) {
      float angle1 = mtGetAngle(_x, _y, cx, cy);
      float angle2 = asin(cr/dist1)*ARAD;
      float dist2  = cos(angle2/ARAD)*dist1;
      float angt1  = mtCorrAngle(angle1+angle2);
      float angt2  = mtCorrAngle(angle1-angle2);

      // 1. tangent point-circle:
      //
      _tangent1->createLine( _x, _y,
                             _x + cos(angt1/ARAD) * dist2,
                             _y + sin(angt1/ARAD) * dist2);

      // 2. tangent point-circle:
      //
      _tangent2->createLine( _x, _y,
                             _x + cos(angt2/ARAD) * dist2,
                             _y + sin(angt2/ARAD) * dist2);

      ret=true;
    }
  }

  return ret;
}



// Get tangent from this circle to circle _secondCircle
//   which is next to point _px, _py
//
bool
RElement::getTangent2(RElement* _tangent,
                      const RElement* _secondCircle,
                      float _px, float _py)
const
{
  bool ret=false;
  RElement p1, p2, p3, p4;
  float dist1, dist2, dist3, dist4;

  if(_tangent && _secondCircle) {

    if(mtGetDistance(cx, cy, _secondCircle->getCx(), _secondCircle->getCy())+cr                     > _secondCircle->getCr() &&
       mtGetDistance(cx, cy, _secondCircle->getCx(), _secondCircle->getCy())+_secondCircle->getCr() > cr                        ) {

      // Get all possible tangents
      //
      getTangents2(&p1, &p2, &p3, &p4, _secondCircle);

      // Get the distances from mousecursor to the tangents
      //
      dist1 = p1.getDistanceToPoint(_px, _py, true);
      dist2 = p2.getDistanceToPoint(_px, _py, true);
      dist3 = p3.getDistanceToPoint(_px, _py, true);
      dist4 = p4.getDistanceToPoint(_px, _py, true);

      // Get the tangent next to the mousecursor
      //
      if(dist1<dist2 && dist1<dist3 && dist1<dist4) {
        _tangent->getMeasuresFrom(&p1);
      }
      else {
        if(dist2<dist3 && dist2<dist4) {
          _tangent->getMeasuresFrom(&p2);
        }
        else {
          if(dist3<dist4) {
            _tangent->getMeasuresFrom(&p3);
          }
          else {
            _tangent->getMeasuresFrom(&p4);
          }
        }
      }
      ret=true;
    }
  }
  return ret;
}



// Get all possible tangents from this circle to ^circle _secondCircle:
//
bool
RElement::getTangents2(RElement* _tangent1,
                       RElement* _tangent2,
                       RElement* _tangent3,
                       RElement* _tangent4,
                       const RElement* _secondCircle)
const
{
  bool ret=false;

  if(_tangent1 && _tangent2 && _tangent3 && _tangent4 &&
      getFlag(E_VISIBLE) && isInGroup(T_CIRCULAR) &&
      _secondCircle->getFlag(E_VISIBLE) && _secondCircle->isInGroup(T_CIRCULAR) &&
      cr>0.0 && _secondCircle->getCr()>0.0) {

    // (el1->cx / el1->cy) <-> (el2->cx / el2->cy):
    //
    float angle1 = mtGetAngle(cx, cy, _secondCircle->getCx(), _secondCircle->getCy());
    float dist1  = mtGetDistance(cx, cy, _secondCircle->getCx(), _secondCircle->getCy());

    if(dist1>0.0) {
      // Outer Tangents:
      //
      float dist2  = _secondCircle->getCr() - cr;
      if(dist1>dist2) {
        float angle2 = asin(dist2/dist1)*ARAD;
        float angt1  = mtCorrAngle(angle1+angle2+90.0);      // Angle normal to 1st tan
        float angt2  = mtCorrAngle(angle1-angle2-90.0);      // Angle normal to 2st tan

        // 1. tangent circle-circle:
        //
        _tangent1->createLine(cx + cos(angt1/ARAD) * cr,
                              cy + sin(angt1/ARAD) * cr,
                              _secondCircle->getCx() + cos(angt1/ARAD) * _secondCircle->getCr(),
                              _secondCircle->getCy() + sin(angt1/ARAD) * _secondCircle->getCr() );

        // 2. tangent circle-circle:
        //
        _tangent2->createLine(cx + cos(angt2/ARAD) * cr,
                              cy + sin(angt2/ARAD) * cr,
                              _secondCircle->cx + cos(angt2/ARAD) * _secondCircle->getCr(),
                              _secondCircle->cy + sin(angt2/ARAD) * _secondCircle->getCr() );
        ret=true;
      }

      // Inner Tangents:
      //
      float dist3  = _secondCircle->cr + cr;
      if(dist1>dist3) {
        float angle3 = asin(dist3/dist1)*ARAD;
        float angt3  = mtCorrAngle(angle1+angle3+90.0);      // Angle normal to 3rd tan
        float angt4  = mtCorrAngle(angle1-angle3-90.0);      // Angle normal to 4th tan

        // 3. tangent circle-circle:
        //
        _tangent3->createLine(cx - cos(angt3/ARAD) * cr,
                              cy - sin(angt3/ARAD) * cr,
                              _secondCircle->getCx() + cos(angt3/ARAD) * _secondCircle->getCr(),
                              _secondCircle->getCy() + sin(angt3/ARAD) * _secondCircle->getCr() );

        // 4. tangent circle-circle:
        //
        _tangent4->createLine(cx - cos(angt4/ARAD) * cr,
                              cy - sin(angt4/ARAD) * cr,
                              _secondCircle->getCx() + cos(angt4/ARAD) * _secondCircle->getCr(),
                              _secondCircle->getCy() + sin(angt4/ARAD) * _secondCircle->getCr() );
        ret=true;
      }
    }
  }
  return ret;
}



// Get orthogonal line to this element:
//
bool
RElement::getOrtho(RElement* _ortho,
                   float _px, float _py,
                   float _ang, float _length)
const
{
  if(_ortho) {
    float angle = 0.0;
    float halfWidth = 100.0;
    float posX=0.0, posY=0.0;

    getTouchPoint(&posX, &posY, _px, _py);

    halfWidth = _length/2;

    _ortho->reset();

    switch(elementTyp) {
      case T_LINE:
        angle = mtCorrAngle( getDirection1()+_ang );
        _ortho->createLine( _px+cos(angle/ARAD)*halfWidth,
                            _py+sin(angle/ARAD)*halfWidth,
                            _px-cos(angle/ARAD)*halfWidth,
                            _py-sin(angle/ARAD)*halfWidth );
        break;

      case T_ARC:
      case T_CIRCLE:
        angle = mtGetAngle( cx, cy, _px, _py );
        _ortho->createLine( cx, cy,
                            cx+cos(angle/ARAD)*halfWidth,
                            cy+sin(angle/ARAD)*halfWidth );
        break;

      default:
        break;
    }

  }

  if(_ortho && _ortho->getFlag(E_VISIBLE)) return true;
  else                                     return false;
}



// Force direction of element:
//
//   _pos: true: force positive direction
//         false: force negative direction
//
void
RElement::forceDirection(bool _pos)
{
  if(isCircular()) {
    if((_pos  &&  getFlag(E_REVERSED)) ||
       (!_pos && !getFlag(E_REVERSED))    ) {
      turn();
    }
  }
}




// Recalculate the borders of the element:
//
void  
RElement::recalculateBorders()
{
  switch(elementTyp) {
    
    case T_POINT:
      leftBorder=rightBorder =x1;
      topBorder =bottomBorder=y1;
      break;

    case T_LINE:
    case T_RECT:
    case T_ARC:
    case T_RAPID:
      if(x1<x2) {
        leftBorder =x1;
        rightBorder=x2;
      }
      else {
        leftBorder =x2;
        rightBorder=x1;
      }

      if(y1<y2) {
        topBorder   =y2;
        bottomBorder=y1;
      }
      else {
        topBorder   =y1;
        bottomBorder=y2;
      }

      if(elementTyp==T_ARC) {
        if(!getFlag(E_REVERSED)) {
          if((a1<180.0 && a2>180.0) ||
             (a1>a2    && a2>180.0) ||
             (a1>a2    && a1<180.0)   ) {

            if(cx-cr<leftBorder) leftBorder=cx-cr;
          }

          if((a1>a2)) {
            if(cx+cr>rightBorder) rightBorder=cx+cr;
          }

          if((a1<270.0 && a2>270.0) ||
             (a1>a2    && a2>270.0) ||
             (a1>a2    && a1<270.0) ) {

            if(cy-cr<bottomBorder) bottomBorder=cy-cr;
          }

          if((a1<90.0  && a2>90.0) ||
             (a1>a2    && a2>90.0) ||
             (a1>a2    && a1<90.0) ) {

            if(cy+cr>topBorder) topBorder=cy+cr;
          }
        }
        else {
          if((a1>180.0 && a2<180.0) ||
             (a1<a2    && a2<180.0) ||
             (a1<a2    && a1>180.0)   ) {

            if(cx-cr<leftBorder) leftBorder=cx-cr;
          }

          if((a1<a2)) {
            if(cx+cr>rightBorder) rightBorder=cx+cr;
          }

          if((a1>270.0 && a2<270.0) ||
             (a1<a2    && a2<270.0) ||
             (a1<a2    && a1>270.0) ) {

            if(cy-cr<bottomBorder) bottomBorder=cy-cr;
          }

          if((a1>90.0 && a2<90.0) ||
             (a1<a2   && a2<90.0) ||
             (a1<a2   && a1>90.0) ) {

            if(cy+cr>topBorder) topBorder=cy+cr;
          }
        }
      }
      break;

    case T_CIRCLE:
      leftBorder=cx-cr;
      rightBorder=cx+cr;
      bottomBorder=cy-cr;
      topBorder=cy+cr;
      break;

    case T_TEXT:
    case T_DIMENSION:
      recalculateTextBorders();
      break;

    case T_HATCHING:
      if(subElementNum!=0) {
        // The border of a hatching is defined by the (invisible)
        //   border elements:
        float l, b, r, t;
        leftBorder=DEF_AREAMAX;
        topBorder=DEF_AREAMIN;
        rightBorder=DEF_AREAMIN;
        bottomBorder=DEF_AREAMAX;
        for(int i=0; i<subElementNum; ++i) {
          if(!subElement[i].getFlag(E_VISIBLE)) {
            subElement[i].getBorders( &l, &b, &r, &t );
            if(l<leftBorder  ) leftBorder  =l;
            if(b<bottomBorder) bottomBorder=b;
            if(r>rightBorder ) rightBorder =r;
            if(t>topBorder   ) topBorder   =t;
          }
        }
      }
      break;
      
    default:
      break;
  }

}



// Recalculate borders of text:
//
void
RElement::recalculateTextBorders()
{
  // No complex element:
  if(elementTyp!=T_TEXT &&
     elementTyp!=T_DIMENSION &&
     elementTyp!=T_HATCHING) return;

  if(graphic) {
    graphic->singleElements(this, 
                            false,
                            false, 0,
                            true, 
                            &leftBorder,
                            &topBorder,
                            &rightBorder,
                            &bottomBorder);

  }

}



// Recalculate the endpoints of the element:
//
void  
RElement::recalculateEndpoints()
{
  switch(elementTyp) {
    
    case T_ARC:
    case T_CIRCLE:
      x1=cx + cos(a1/ARAD) * cr;
      y1=cy + sin(a1/ARAD) * cr;
      x2=cx + cos(a2/ARAD) * cr;
      y2=cy + sin(a2/ARAD) * cr;
      break;

    default:
      break;
  }
}



// Recalculate the angles of the element:
//
void  
RElement::recalculateAngles()
{
  switch(elementTyp) {
    
    case T_ARC:
    case T_CIRCLE:
      a1 = mtGetAngle(cx, cy, x1, y1);
      a2 = mtGetAngle(cx, cy, x2, y2);
      break;

    default:
      break;
  }
}



// Get all borders of this element:
//
void  
RElement::getBorders(float* _left, float* _bottom, float* _right, float* _top) const
{ 
  if(_left)   *_left=leftBorder; 
  if(_bottom) *_bottom=bottomBorder; 
  if(_right)  *_right=rightBorder; 
  if(_top)    *_top=topBorder; 
}



// Is the element on screen?
//
bool 
RElement::isOnScreen()
const
{
  if(graphic->realToScreenX(rightBorder)<0 ||
     graphic->realToScreenY(bottomBorder)<0 ||
     graphic->realToScreenX(leftBorder)>graphic->width() ||
     graphic->realToScreenY(topBorder)>graphic->height()) {
    return false;
  }
  else { 
    return true;
  }
}



// Create a point in this elements:
//
// _x1, _y1: coordinates
//
void 
RElement::createPoint(float _x1, float _y1, 
                      byte _layer)
{
  elementTyp=T_POINT;
  setFlags(E_VISIBLE|E_ACTIVE);
  x1=_x1;
  y1=_y1;
  x2=_x1;
  y2=_y1;
  layer=_layer;
  recalculateBorders();
}



// Create a line in this elements:
//
// _x1, _y1, _x2, _y2: coordinates
//
void 
RElement::createLine(float _x1, float _y1, 
                     float _x2, float _y2, 
                     byte _layer)
{
  elementTyp=T_LINE;
  setFlags(E_VISIBLE|E_ACTIVE);
  x1=_x1;
  y1=_y1;
  x2=_x2;
  y2=_y2;
  layer=_layer;
  recalculateBorders();
}



// Create an arc in this elements:
//
// _cx, _cy, _cr, _a1, _a2: coordinates
// _reversed:               clockwise?
//
void 
RElement::createArc(float _cx, float _cy, 
                    float _cr, 
                    float _a1, float _a2, 
                    bool  _reversed, 
                    byte _layer)
{
  elementTyp=T_ARC;
  setFlags(E_VISIBLE|E_ACTIVE);
  cx=_cx;
  cy=_cy;
  cr=_cr;
  a1=mtCorrAngle(_a1);
  a2=mtCorrAngle(_a2);
  layer=_layer;
  if(_reversed) setFlag(E_REVERSED);
  recalculateEndpoints();
  recalculateBorders();
}


// Create an arc in this elements:
//
// _x1, _y1, _x2, _y2, _x3, _y3: coordinates of the three points on arc
// _reversed:                    clockwise?
//
void 
RElement::createArcPPP(float _x1, float _y1,
                       float _x2, float _y2,
                       float _x3, float _y3,
                       bool  /*_reversed*/,
                       byte _layer)
{
  RElement hLine1(graphic);    // Help lines
  RElement hLine2(graphic);    //
  float xm1, ym1,     // Middle between 1st and 2nd point
        xm2, ym2;     // Middle between 2nd and 3rd point
  float am1, am2;     // Angles from middles to center (or away)
  bool  is;           // Was the intersecion function succesful?
  float ix1, iy1;     // Intersection point

  // 'Construct' helplines to find the center:
  //
  xm1 = (_x1+_x2)/2;
  ym1 = (_y1+_y2)/2;
  xm2 = (_x2+_x3)/2;
  ym2 = (_y2+_y3)/2;
  am1 = mtCorrAngle(mtGetAngle(_x1, _y1, _x2, _y2)+90.0);
  am2 = mtCorrAngle(mtGetAngle(_x2, _y2, _x3, _y3)+90.0);

  hLine1.createLine(xm1, ym1,
                    xm1+cos(am1/ARAD)*100.0, ym1+sin(am1/ARAD)*100.0);
  hLine2.createLine(xm2, ym2,
                    xm2+cos(am2/ARAD)*100.0, ym2+sin(am2/ARAD)*100.0);

  hLine1.getIntersection(&hLine2,
                         &is, &ix1, &iy1,
                         0, 0, 0,
                         0,
                         false);

  hLine1.outputDebug();
  hLine2.outputDebug();

  if(is) {
    elementTyp=T_ARC;
    setFlags(E_VISIBLE|E_ACTIVE);

    cx=ix1;
    cy=iy1;
    cr=mtGetDistance(cx, cy, _x1, _y1);
    a1=mtGetAngle(cx, cy, _x1, _y1);
    a2=mtGetAngle(cx, cy, _x3, _y3);

    if(!mtAngleIsBetween(mtGetAngle(cx, cy, _x2, _y2), a1, a2)) {
      setFlag(E_REVERSED);
    }

    layer=_layer;
    //if(_reversed) setFlag(E_REVERSED);
    recalculateEndpoints();
    recalculateBorders();
  }
}



// Create a circle in this elements:
//
// _cx, _cy, _cr, _a1, _a2: coordinates
// _reversed:               clockwise?
//
void 
RElement::createCircle(float _cx, float _cy, 
                       float _cr, 
                       float _a1, float _a2, 
                       bool  _reversed, 
                       byte _layer)
{
  elementTyp=T_CIRCLE;
  setFlags(E_VISIBLE|E_ACTIVE);
  cx=_cx;
  cy=_cy;
  cr=_cr;
  a1=mtCorrAngle(_a1);
  a2=mtCorrAngle(_a2);
  layer=_layer;
  if(_reversed) setFlag(E_REVERSED);
  recalculateEndpoints();
  recalculateBorders();
}



// Create a polyline element (arc or line) in this elements:
//
// _x1, _y1, _x2, _y2, _ab: coordinates
//
void 
RElement::createPolylineElement(float _x1, float _y1,
                                float _x2, float _y2,
                                float _ab, 
                                byte _layer)
{
  setFlags(E_VISIBLE|E_ACTIVE);
  x1=_x1;
  y1=_y1;
  x2=_x2;
  y2=_y2;
  layer=_layer;

  if(_ab!=0.0) {
    elementTyp=T_ARC;
    
    // 'Ausbuchtung':
    //
    if(_ab<0) {
      setFlag(E_REVERSED);
    }

    float alpha =atan(_ab)*4.0*ARAD;

    float xm =(x1+x2)/2.0;
    float ym =(y1+y2)/2.0;

    if(mtCompFloat(alpha, 0.0, 0.0001)) {
      cr = 1.0;
    }
    else {
      cr = ( sqrt( (x1-x2)*(x1-x2)+(y2-y1)*(y2-y1) )/2.0 )
                   / sin( alpha/ARAD/2.0 );
    }

    if(cr<0.0) cr=-cr;

    float dist = mtGetDistance(x1, y1, x2, y2)/2.0;
    float wu = cr*cr - dist*dist;
    if(wu<0.0) wu*=-1.0;
    float h = sqrt(wu);

    float a;
    if( _ab>0.0 ) a = mtGetAngle(x1, y1, x2, y2)+90.0;
    else          a = mtGetAngle(x1, y1, x2, y2)-90.0;

    if(alpha>180.0 || alpha<-180.0) h*=-1.0;

    if(a==0.0) {
       cx = xm+h;
       cy = ym;
    }
    else {
       cx = xm+cos(a/ARAD)*h;
       cy = ym+sin(a/ARAD)*h;
    }

    recalculateAngles();
 }
 else {
   elementTyp=T_LINE;
 }
 
 recalculateBorders();
}



// Create a rectangle in this elements:
//
// _x1, _y1, _x2, _y2: coordinates of corners
//
void 
RElement::createRect(float _x1, float _y1, 
                     float _x2, float _y2, 
                     byte _layer)
{
  elementTyp=T_RECT;
  setFlags(E_VISIBLE|E_ACTIVE);
  x1=_x1;
  y1=_y1;
  x2=_x2;
  y2=_y2;
  layer=_layer;
  recalculateBorders();
}



// Create Text:
//
//   _x/_y        : reference point
//   _font        : font number
//   _text        : the text (can include '\n')
//   _flags       : OPTIONAL flags for text (align, round)
//   _texthigh    : texthigh in mm
//   _angle       : ref. angle
//   _radius      : radius of first textline
//   _letterspace : distance between letters in mm
//   _wordspace   : distance between words in mm
//   _linedistance: distance between lines in mm
//   _layer       : layer number
//
void  
RElement::createText(float _x, float _y,
                     int   _font,
                     const unsigned char* _text,
                     unsigned int _flags,
                     float _textheight,
                     float _angle,
                     float _radius,
                     float _letterspace,
                     float _wordspace,
                     float _linedistance,
                     byte _layer)
{
  if(_text && strlen((char*)_text)>0) {
    elementTyp=T_TEXT;
    setFlags(E_VISIBLE|E_ACTIVE);
    x1=_x;
    y1=_y;
    text=(char*)_text;
    a2 = (float)_font;
    y2 = _textheight;
    a1 = _angle;
    cr = _radius;
    x2 = _letterspace;
    cx = _wordspace;
    cy = _linedistance;
    setFlag(_flags);
    layer=_layer;
    recalculateBorders();
  }
}



// Create a rapid move in this elements:
//   (only ONE point / last point was last 
//    point in drawing)
//
// _x1, _y1: coordinates
//
void 
RElement::createRapid(float _x1, float _y1, 
                      byte _layer)
{
  elementTyp=T_RAPID;
  setFlags(E_VISIBLE|E_ACTIVE);
  x1=_x1;
  y1=_y1;
  x2=_x1;
  y2=_y1;
  layer=_layer;
  recalculateBorders();
}



// Create a rapid move in this elements:
//   (TWO points)
//
// _x1, _y1: coordinates
//
void 
RElement::createRapid(float _x1, float _y1, 
                      float _x2, float _y2, 
                      byte _layer)
{
  elementTyp=T_RAPID;
  setFlags(E_VISIBLE|E_ACTIVE);
  x1=_x1;
  y1=_y1;
  x2=_x2;
  y2=_y2;
  layer=_layer;
  recalculateBorders();
}


/**
 * Creates a dimension.
 *
 * @param _v1 Value 1: pos1 x for straight / center x for angle dimension
 * @param _v2 Value 2: pos1 y for straight / center y for angle dimension
 * @param _v3 Value 3: pos2 x for straight / angle1 for angle dimension
 * @param _v4 Value 4: pos2 y for straight / angle2 for angle dimension
 * @param _v5 Value 5: start radius of 1st extension line for angle dimension
 * @param _v6 Value 6: start radius of 2nd extension line for angle dimension
 * @param _pos Position (distance, x, y, or radius)
 * @param _flags Special Flags (E_STRAIGHT, E_ROUNDOUT, E_HORIZONTAL, E_VERTICAL)
 */
void
RElement::createDimension(float _v1, float _v2, float _v3, float _v4,
                          float _v5, float _v6,
                          float _pos,
                          uint _flags,
                          byte _layer)
{
  elementTyp=T_DIMENSION;
  setFlags(E_VISIBLE|E_ACTIVE|_flags);

  if(getFlag(E_STRAIGHT)) {
    if(getFlag(E_RADIUS)) {
      cx=_v1;
      cy=_v2;
      x1=_v3;
      y1=_v4;
      cr=_pos;
    }
    else {
      x1=_v1;
      y1=_v2;
      x2=_v3;
      y2=_v4;
      cr=_pos;
    }
  }
  else if(getFlag(E_ROUNDOUT)) {
    cx=_v1;
    cy=_v2;
    a1=_v3;
    a2=_v4;
    cr=_pos;
    x1=_v5;
    x2=_v6;
  }
  layer=_layer;
  recalculateBorders();
}



/**
 * Creates a hatching.
 *   Note: you have to add the border elements afterwards.
 *
 * @param _factor  Factor for pattern.
 * @param _pattern Name of pattern (DXF file in hatch-directory)
 */
void
RElement::createHatching(float _factor, QString _pattern,
                         byte _layer)
{
  if(!graphic) return;

  // Check if pattern can be found:
  QString patternPath;
  patternPath = RCONFIG->searchFile( QString("hatches/")+_pattern+".dxf" );
  if(patternPath.isEmpty()) return;

  int c, i;                 // Counters

  elementTyp=T_HATCHING;
  setFlags(E_VISIBLE|E_ACTIVE);

  cr=_factor;
  text=_pattern;
  layer=_layer;

  // List of countour and hatching elements :
  //
  QList<RElement> elList;
  elList.setAutoDelete( true );

  // Add selected elements to the element buffer of the new hatching;
  //   and get min/max of border elements:
  //
  RElement *elT;            // pointer which walks through elements
  RElement* newEl;          // pointer for new elements
  float l, b, r, t;
  leftBorder=DEF_AREAMAX;
  topBorder=DEF_AREAMIN;
  rightBorder=DEF_AREAMIN;
  bottomBorder=DEF_AREAMAX;
  for(elT=graphic->elementFirst(); elT!=0; elT=graphic->elementNext()) {
    if(elT->getFlag(E_ACTIVE) && elT->getFlag(E_VISIBLE) && elT->getFlag(E_TAGGED)) {
      newEl=new RElement( graphic );
      newEl->getDataFrom( elT );
      newEl->delFlag( E_VISIBLE );
      elList.append( newEl );

      newEl->getBorders( &l, &b, &r, &t );
      if(l<leftBorder  ) leftBorder  =l;
      if(b<bottomBorder) bottomBorder=b;
      if(r>rightBorder ) rightBorder =r;
      if(t>topBorder   ) topBorder   =t;
    }
  }

  // Add sub elements (the contour):
  //   This is needed only temporary for the method isPointInsideSubElements()
  //
  int sec=0;
  createSubElements(elList.count());
  for(elT=elList.first(); elT!=0; elT=elList.next()) {
    setSubElement( sec, elT );
    ++sec;
  }

  // Create temp elements for the NON-TRIMMED pattern:
  //
  RGraphic pattern(0);


  /*
  // Search pattern in program dir (/usr/local/qcad/hatches/xxxx.dxf):
  patternPath = RCONFIG->getPrgDir().path()+"/hatches/"+_pattern+".dxf";

#ifndef DEF_WINDOWS
  // Search pattern in users dir (~/.<proj>/hatches/xxxx.dxf):
  if(!QFileInfo(patternPath).exists()) {
    patternPath = RCONFIG->getConfigDir().path()+"/hatches/"+_pattern+".dxf";
  }

  // Search pattern in /usr/share/<proj>/hatches dir:
  if(!QFileInfo(patternPath).exists()) {
    patternPath = QString("/usr/share/")+DEF_APPNAME_L+"/hatches/"+_pattern+".dxf";
  }
#endif
  */

  RFileDxf patternFile(patternPath.latin1(), &pattern);
  patternFile.load(false);

  // Move pattern around and do whatever there is to do:
  //
  float offsetX, offsetY;
  QValueList<float> ixList;    // Intersection point list
  QValueList<float> iyList;
  float ix1, iy1, ix2, iy2;    // One intersection
  bool  ires1, ires2, itan;
  RElement* tmpEl = new RElement[pattern.count()];
  int patternNum=pattern.count();

  for(offsetY=bottomBorder; offsetY<topBorder; offsetY+=100.0*cr) {
    for(offsetX=leftBorder; offsetX<rightBorder; offsetX+=100.0*cr) {
      for(c=0; c<patternNum; ++c) {
        tmpEl[c].getDataFrom( pattern.elementAt(c) );
        tmpEl[c].scale( cr, 0.0, 0.0, false);
        tmpEl[c].move( offsetX, offsetY );
      }

      // Look at every element of the pattern and
      //   calc intersections
      //
      for(c=0; c<patternNum; ++c) {
        ixList.clear();
        iyList.clear();

        ixList.append(tmpEl[c].getX1());
        iyList.append(tmpEl[c].getY1());

        // Collect intersections on this pattern line:
        //
        RElement* elP;
        for(elP=elList.first(); elP!=0 && !elP->getFlag(E_VISIBLE); elP=elList.next()) {
          ires1=ires2=false;
          tmpEl[c].getIntersection( elP,
                                    &ires1, &ix1, &iy1,
                                    &ires2, &ix2, &iy2,
                                    &itan, true );
          if(!itan) {
            if(ires1) {
              ixList.append(ix1);
              iyList.append(iy1);
            }

            if(ires2) {
              ixList.append(ix2);
              iyList.append(iy2);
            }
          }
        }

        ixList.append(tmpEl[c].getX2());
        iyList.append(tmpEl[c].getY2());

        // Step through patteren element parts:
        //
        RElement partLine;
        partLine.setGraphic( graphic );
        partLine.getAttribFrom( this );

        float hx1=0.0, hy1=0.0, hx2=0.0, hy2=0.0;

        hx1 = *ixList.at(0);
        hy1 = *iyList.at(0);
        ixList.remove(ixList.at(0));
        iyList.remove(iyList.at(0));

        float dist;
        float minDist;
        int found;

        do {
          found=-1;
          minDist=DEF_AREAMAX;

          // Find next point:
          for(i=0; i<(int)ixList.count(); ++i) {
            dist = mtGetDistance(hx1,hy1, *ixList.at(i),*iyList.at(i));
            if(dist<minDist) {
              found=i;
              hx2 = *ixList.at(i);
              hy2 = *iyList.at(i);
              minDist = dist;
            }
          }

          if(found!=-1) {
            ixList.remove(ixList.at(found));
            iyList.remove(iyList.at(found));
          }

          partLine.createLine( hx1,hy1, hx2,hy2 );

          if( isPointInsideSubElements((hx1+hx2)/2, (hy1+hy2)/2) ) {

            // Add elements:
            //
            newEl = new RElement( graphic );
            if(newEl) {
              newEl->getMeasuresFrom(&partLine);
              newEl->setLayer(getLayer());
              elList.append( newEl );
            }
          }

          hx1=hx2;
          hy1=hy2;

        }while(found!=-1);
      }

    }
  }
  delete[] tmpEl;

  // Add sub elements (definitely):
  //
  sec=0;
  createSubElements(elList.count());
  for(elT=elList.first(); elT!=0; elT=elList.next()) {
    setSubElement( sec, elT );
    ++sec;
  }
}



// Turn element:
//
void
RElement::turn()
{
  // We can't turn texts, dimensions:
  //
  if(elementTyp==T_TEXT ||
     elementTyp==T_DIMENSION ||
     elementTyp==T_HATCHING) return;
  
  float tmp;
  
  tmp=x1;
  x1=x2;
  x2=tmp;
 
  tmp=y1;
  y1=y2;
  y2=tmp;

  // Contour position:
  //
  if(!getFlag(E_CONTBEG) || !getFlag(E_CONTEND)) {
    if(getFlag(E_CONTBEG)) {
      delFlag(E_CONTBEG);
      setFlag(E_CONTEND);
    }
    else {
      if(getFlag(E_CONTEND)) {
        delFlag(E_CONTEND);
        setFlag(E_CONTBEG);
      }
    }
  }

  // Tool offset:
  //
  if(getFlag(E_TOOLOFFL)) {
    delFlag(E_TOOLOFFL);
    setFlag(E_TOOLOFFR);
  }
  else {
    if(getFlag(E_TOOLOFFR)) {
      delFlag(E_TOOLOFFR);
      setFlag(E_TOOLOFFL);
    }
  }

  if(isCircular()) {
    tmp=a1;
    a1=a2;
    a2=tmp;
    turnFlag(E_REVERSED);
  }

}



// Set color for drawing this element and 
//   draw startpoint if highlighted:
//
void  
RElement::prepareDrawing(RPainter* _paint, int _x, int _y, bool _curr)
{
  static QColor oldColor=Qt::black;
  static byte oldWidth=0;
  static Qt::PenStyle oldStyle=Qt::SolidLine;

  QPen pen;
  if(graphic && graphic->isPrinting()) pen.setWidth(width*2);
  else                                 pen.setWidth(width==1 ? 0 : width);
  pen.setStyle(style);
  pen.setColor(_paint->pen().color());
  if(_paint->rasterOp() != Qt::XorROP) {
    if(!_curr && pen.color()!=Qt::white) {
      pen.setColor(color);
    }
  }

  if(!_curr && pen.color()!=Qt::white) {
    // Highlight (next to mouse):
    //
    if(this==graphic->getHighlightElement()) {
      if(getFlag(E_TAGGED)) {
        pen.setColor(QColor(220, 140, 140));
      }
      else {
        pen.setColor(Qt::lightGray);
      }
    }

    // Highlight broken contours:
    //
    else if(graphic->getFlag(G_BROKEN) && getFlag(E_BROKEN)) {

      // Highlight startpoint:
      //
      if(getFlag(E_BROKEN1)) {
        _paint->setPen(Qt::darkGreen);
        _paint->drawRect(_x-2, _y-2, 4, 4);
      }

      // Highlighted:
      //
      pen.setColor(Qt::green);
    }

    // Only tagged:
    //
    else if(getFlag(E_TAGGED)) {
      pen.setColor(Qt::red);
    }
  }

  // Normal element color:
  //
  //else if(oldColor!=color || oldWidth!=width || oldStyle!=style) {
  _paint->setPen(pen);

  oldWidth=pen.width();
  oldStyle=pen.style();
  oldColor=pen.color();
}



// Finish drawing
//
void  
RElement::finishDrawing(RPainter* _paint, int _x, int _y)
{
  // Highlight broken contours:
  //
  if(graphic->getFlag(G_BROKEN) && getFlag(E_BROKEN)) {
    if(getFlag(E_BROKEN2)) {
      _paint->setPen(Qt::darkGreen);
      _paint->drawRect(_x-2, _y-2, 4, 4);
    }
    _paint->setPen(Qt::black);
  }
    
  // Was highlighted (next to mouse):
  //
  else if(this==graphic->getHighlightElement() ||
          getFlag(E_TAGGED)) {
    _paint->setPen(Qt::black);
  }
}




// Draw element:
//
void
RElement::draw(RPainter* _paint, 
               float _ox, float _oy, 
               bool _prev,
               bool _curr)
{

  if(_paint && graphic && getFlag(E_VISIBLE) && getFlag(E_ACTIVE)) {
    int ix1, iy1, ix2, iy2;

    ix1=graphic->realToScreenX(x1+_ox);
    iy1=graphic->realToScreenY(y1+_oy);
    ix2=graphic->realToScreenX(x2+_ox);
    iy2=graphic->realToScreenY(y2+_oy);

    if(!_prev) prepareDrawing(_paint, ix1, iy1, _curr);

    switch(elementTyp) {
      case T_POINT:
        _paint->drawLine(ix1, iy1-width/2+2, ix1, iy1+width/2+2);
        _paint->drawLine(ix1-width/2+2, iy1, ix1+width/2+2, iy1);
        break;
        
      case T_LINE:
        _paint->drawLine(ix1, iy1, ix2, iy2);
        break;

      case T_ARC:
        _paint->drawExactArc(graphic->realToExactScreenX(cx+_ox),
                             graphic->realToExactScreenY(cy+_oy),
                             graphic->realToExactScreenDist(cr),
                             a1, a2,
                             ix1, iy1, ix2, iy2,
                             getFlag(E_REVERSED));
        break;

      case T_CIRCLE:
        _paint->drawExactArc(graphic->realToExactScreenX(cx+_ox),
                             graphic->realToExactScreenY(cy+_oy),
                             graphic->realToExactScreenDist(cr),
                             a1, a2,
                             ix1, iy1, ix2, iy2,
                             getFlag(E_REVERSED));
        break;

      case T_RECT:
        _paint->drawRect(ix1, iy1, ix2-ix1, iy2-iy1);
        break;

      case T_TEXT:
        if(!_prev) {
          if(!text.isEmpty()) {
            graphic->singleElements(this,
                                    false,
                                    true, _paint);
          }
        }
        break;

      case T_DIMENSION:
        if(!_prev) {
          graphic->singleElements(this,
                                  false,
                                  true, _paint);
        }
        break;

      case T_HATCHING:
        if(true) {
          for(int i=0; i<subElementNum; ++i) {
            if(subElement[i].getFlag(E_VISIBLE)) {
              _paint->drawLine(graphic->realToScreenX(subElement[i].x1+_ox),
                               graphic->realToScreenY(subElement[i].y1+_oy),
                               graphic->realToScreenX(subElement[i].x2+_ox),
                               graphic->realToScreenY(subElement[i].y2+_oy));
            }
          }
        }

        /*
        for(float hi=leftBorder; hi<=rightBorder; hi+=1.0) {
          for(float hj=bottomBorder; hj<=topBorder; hj+=1.0) {
            if(isPointInsideSubElements(hi, hj)) {
              _paint->setPen(Qt::blue);
            }
            else {
              _paint->setPen(Qt::yellow);
            }

            _paint->drawPoint(graphic->realToScreenX(hi),
                              graphic->realToScreenY(hj));
          }
        }
        */
        break;

      case T_RAPID:
        if(!_prev) {
          _paint->setPen(Qt::blue);
          _paint->drawLine(ix1, iy1, ix2, iy2);
          _paint->setPen(Qt::black);
        }
        break;

      default:
        break;
    }
    
    
    if(!_prev && !_curr) finishDrawing(_paint, ix2, iy2);
  }
}



// Draw Element fast (only lines/arcs of texts/dimensions):
//
// Draw element:
//
void
RElement::fastDraw(RPainter* _paint)
{
  if(_paint && graphic) {
    int ix1, iy1, ix2, iy2;
    ix1=graphic->realToScreenX(x1);
    iy1=graphic->realToScreenY(y1);
    ix2=graphic->realToScreenX(x2);
    iy2=graphic->realToScreenY(y2);

    if(elementTyp==T_LINE) {
      _paint->drawLine(ix1, iy1, ix2, iy2);
    }
    else if(elementTyp==T_ARC) {
      _paint->drawExactArc(graphic->realToExactScreenX(cx),
                           graphic->realToExactScreenY(cy),
                           graphic->realToExactScreenDist(cr),
                           a1, a2,
                           ix1, iy1, ix2, iy2,
                           getFlag(E_REVERSED));
    }
    else if(elementTyp==T_TEXT) {
      if(!text.isEmpty()) {
        graphic->singleElements(this,
                                false,
                                true, _paint);
      }
    }
  }
}



// Draw element smooth:
//
void
RElement::drawSmooth(RPainter* _paint)
{ 
  if(graphic && getFlag(E_VISIBLE) && getFlag(E_ACTIVE)) {

    int ix1, iy1, ix2, iy2;
    
    ix1=graphic->realToScreenX(x1);
    iy1=graphic->realToScreenY(y1);
    ix2=graphic->realToScreenX(x2);
    iy2=graphic->realToScreenY(y2);

    prepareDrawing(_paint, ix1, iy1);

    switch(elementTyp) {
      case T_POINT:
        ix1=graphic->realToScreenX(x1);
        iy1=graphic->realToScreenY(y1);
        _paint->drawLine(ix1, iy1-2, ix1, iy1+2);
        _paint->drawLine(ix1-2, iy1, ix1+2, iy1);
        graphic->simulationDelay();
        break;
        
      case T_LINE:
        drawSmoothLine(_paint);
        break;

      case T_ARC:
        drawSmoothArc(_paint);
        break;

      case T_CIRCLE:
        drawSmoothArc(_paint);
        break;

      case T_RECT:
        //_paint->drawRect(ix1, iy1, ix2, iy2);
        break;

      case T_TEXT:
        if(!text.isEmpty()) {
          graphic->singleElements(this,
                                  false,
                                  true, _paint);
        }
        break;

      case T_DIMENSION:
      case T_HATCHING:
        graphic->singleElements(this,
                                false,
                                true, _paint);
        break;

      case T_RAPID:
        _paint->setPen(Qt::blue);
        drawSmoothLine(_paint);
        _paint->setPen(Qt::black);
        break;

      default:
        break;
    }
    
    finishDrawing(_paint, ix2, iy2);

  }
}



// Draw Smooth Line:
//
void
RElement::drawSmoothLine(RPainter* _paint)
{
  float alpha = getDirection1();
  float xStep, yStep;
  bool  xIsOne;

  int ix1, iy1, ix2, iy2;

  ix1=graphic->realToScreenX(x1);
  iy1=graphic->realToScreenY(y1);
  ix2=graphic->realToScreenX(x2);
  iy2=graphic->realToScreenY(y2);

  if(mtCompFloat(alpha, 0.0)) { 
    xStep = 1.0;
    yStep = 0.0;
    xIsOne=true;
  }
  else if(mtCompFloat(alpha, 90.0)) {
    xStep = 0.0;
    yStep =-1.0;
    xIsOne=false;
  }
  else if(mtCompFloat(alpha, 180.0)) {
    xStep =-1.0;
    yStep = 0.0;
    xIsOne=true;
  }
  else if(mtCompFloat(alpha, 270.0)) {
    xStep = 0.0;
    yStep = 1.0;
    xIsOne=false;
  }
  else if(abs(ix2-ix1)>abs(iy2-iy1)) {
    if(ix2>ix1) xStep=1.0;
    else        xStep=-1.0;
    yStep=-tan(alpha/ARAD)*xStep;
    xIsOne=true;
  }
  else {
    if(iy2>iy1) yStep=1.0;
    else        yStep=-1.0;
    xStep=-yStep/tan(alpha/ARAD);
    xIsOne=false;
  }

  float lx=(float)ix1, ly=(float)iy1;

  do {
    if(graphic->isPointOnScreen((int)lx, (int)ly)) {
      _paint->drawPoint((int)lx, (int)ly);
      graphic->simulationDelay();
    }

    lx+=xStep;
    ly+=yStep;

  } while( ( xIsOne && (((int)lx>=ix1 && (int)lx<=ix2) || ((int)lx>=ix2 && (int)lx<=ix1))) ||
           (!xIsOne && (((int)ly>=iy1 && (int)ly<=iy2) || ((int)ly>=iy2 && (int)ly<=iy1)))   );
}



// Draw a smooth arc:
//
void
RElement::drawSmoothArc(RPainter* _paint)
{
  int icx = graphic->realToScreenX(cx);
  int icy = graphic->realToScreenY(cy);

  if(cr*graphic->getZoom()<=1.4) {
    _paint->drawPoint(icx, icy);
  }
  else {
    int ix1 = graphic->realToScreenX(x1);
    int iy1 = graphic->realToScreenY(y1);
    int ix2 = graphic->realToScreenX(x2);
    int iy2 = graphic->realToScreenY(y2);
    int k2x;              // Next point on circle
    int k2y;              //
    int k1x=0;            // Prev point on circle
    int k1y=0;            //
    float aStep;          // Angle Step (rad)
    float a;              // Actual Angle (rad)
    float a2cp = a2;      // Copy of a2

    //if(!graphic->isPrinting()) {
      if(1.0/(cr*graphic->getZoom())<=1.0)
        aStep=asin(1.0/(cr*graphic->getZoom()));
      else
        aStep = 0.01;

      if(aStep<0.01) aStep = 0.01;
    /*}
    else {
      if(1.0/(cr*graphic->getZoom())<=1.0)
        aStep=asin(1.0/(cr*graphic->getZoom()))*10.0;
      else
        aStep = 0.1;

      if(aStep<0.1) aStep = 0.1;
    }*/

    if(!getFlag(E_REVERSED)) {
    
      // Arc Counterclockwise:
      //
      if(a1>a2cp-0.01) a2cp+=360.0;
      _paint->moveTo(ix1, iy1);
      for(a=(a1/ARAD)+aStep; a<=(a2cp/ARAD); a+=aStep) {
        k2x = graphic->realToScreenX(cx+cos(a)*cr);
        k2y = graphic->realToScreenY(cy+sin(a)*cr);
        if(graphic->isPointOnScreen(k2x, k2y) ||
           graphic->isPointOnScreen(k1x, k1y)    ) {
          _paint->lineTo(k2x, k2y);
          graphic->simulationDelay();
        }
        _paint->moveTo(k2x, k2y);
        k1x=k2x;
        k1y=k2y;
      }
      _paint->lineTo(ix2, iy2);
    }
    else {
    
      // Arc Clockwise:
      //
      if(a1<a2cp+0.01) a2cp-=360.0;
      _paint->moveTo(ix1, iy1);
      for(a=(a1/ARAD)-aStep; a>=(a2cp/ARAD); a-=aStep) {
        k2x = graphic->realToScreenX(cx+cos(a)*cr);
        k2y = graphic->realToScreenY(cy+sin(a)*cr);
        if(graphic->isPointOnScreen(k2x, k2y) ||
           graphic->isPointOnScreen(k1x, k1y)    ) {
          _paint->lineTo(k2x, k2y);
          graphic->simulationDelay();
        }
        _paint->moveTo(k2x, k2y);
        k1x=k2x;
        k1y=k2y;
      }
      _paint->lineTo(ix2, iy2);
    }
  }
}



// Move element by _x, _y:
//
void  
RElement::move(float _x, float _y, 
               bool _updBorder)
{
  switch(elementTyp) {

    case T_ARC:
    case T_CIRCLE:
      cx+=_x;
      cy+=_y;
      // fall through!
    case T_POINT:
    case T_LINE:
    case T_RECT:
    case T_RAPID:
    case T_STARTWAY:
    case T_ENDWAY:
      x1+=_x;
      y1+=_y;
      x2+=_x;
      y2+=_y;
      break;

    case T_TEXT:
      x1+=_x;
      y1+=_y; 
      break;

    case T_DIMENSION:
      cx+=_x;
      cy+=_y;
      if(!getFlag(E_ROUNDOUT)) {
        x1+=_x;
        y1+=_y;
        x2+=_x;
        y2+=_y;
      }
      if(getFlag(E_HORIZONTAL)) {
        cr+=_y;
      }
      if(getFlag(E_VERTICAL)) {
        cr+=_x;
      }
      break;

    case T_HATCHING:
      if(subElement) {
        for(int i=0; i<subElementNum; ++i) {
          subElement[i].move( _x, _y, _updBorder );
        }
      }
      break;

    default:
      break;
  }

  if(_updBorder) recalculateBorders();
}



// Rotate element. Center: _cx, _cy / Angle: _angle:
//
void  
RElement::rotate(float _angle, 
                 float _cx, float _cy, 
                 bool _updBorder)
{
  _angle = mtCorrAngle(_angle);

  if(mtCompFloat(_angle, 0.0)) return;
  
  switch(elementTyp) {
    case T_POINT:
    case T_LINE:
    case T_RAPID:
    case T_STARTWAY:
    case T_ENDWAY:
      mtRotatePoint(_cx, _cy, &x1, &y1, _angle);
      mtRotatePoint(_cx, _cy, &x2, &y2, _angle);
      break;

    case T_ARC:
    case T_CIRCLE:
      mtRotatePoint(_cx, _cy, &cx, &cy, _angle);
      a1 = mtCorrAngle(a1+_angle);
      a2 = mtCorrAngle(a2+_angle);
      recalculateEndpoints();
      break;

    case T_RECT:
      mtRotatePoint(_cx, _cy, &x1, &y1, _angle);
      a1 = mtCorrAngle(a1+_angle);
      break;

    case T_TEXT:
      mtRotatePoint(_cx, _cy, &x1, &y1, _angle);
      a1 = mtCorrAngle(a1+_angle);
      break;

    case T_DIMENSION:
      if(!getFlag(E_ROUNDOUT)) {
        mtRotatePoint(_cx, _cy, &x1, &y1, _angle);
        mtRotatePoint(_cx, _cy, &x2, &y2, _angle);
      }
      mtRotatePoint(_cx, _cy, &cx, &cy, _angle);
      a1 = mtCorrAngle(a1+_angle);
      a2 = mtCorrAngle(a2+_angle);
      break;

    case T_HATCHING:
      if(subElement) {
        for(int i=0; i<subElementNum; ++i) {
          subElement[i].rotate( _angle, _cx, _cy, _updBorder );
        }
      }
      break;

    default:
      break;
  }

  if(_updBorder) recalculateBorders();
}



// Scale:
//
void  
RElement::scale(float _fact, 
                float _cx, float _cy, 
                bool _updBorder)
{
  if(elementTyp!=T_HATCHING) move(-_cx, -_cy, false);

  switch(elementTyp) {
    case T_ARC:
    case T_CIRCLE:
      cx*=_fact;
      cy*=_fact;
      cr*=_fact;
      // fall through!
    case T_POINT:
    case T_LINE:
    case T_RECT:
      x1*=_fact;
      y1*=_fact;
      x2*=_fact;
      y2*=_fact;
      break;

    case T_TEXT:
      x1*=_fact;
      y1*=_fact;
      x2*=_fact;
      y2*=_fact;
      cx*=_fact;
      cy*=_fact;
      cr*=_fact;
      break;

    case T_DIMENSION:
      if(!getFlag(E_ROUNDOUT)) {
        x1*=_fact;
        y1*=_fact;
        x2*=_fact;
        y2*=_fact;
      }
      cx*=_fact;
      cy*=_fact;
      cr*=_fact;
      break;

    case T_HATCHING:
      if(subElement) {
        for(int i=0; i<subElementNum; ++i) {
          //move(-_cx, -_cy, false);
          subElement[i].scale( _fact, _cx, _cy, _updBorder );
          //move(_cx, _cy, false);
        }
      }
      break;

    default:
      break;
  }
  
  if(elementTyp!=T_HATCHING) move(_cx, _cy, false);

  if(_updBorder) recalculateBorders();
}



// Scale:
//
void  
RElement::mirror(float _x1, float _y1, 
                 float _x2, float _y2, 
                 bool _updBorder)
{
  
  switch(elementTyp) {

    case T_ARC:
    case T_CIRCLE:
      turnFlag(E_REVERSED);
      mtMirrorPoint(_x1, _y1, _x2, _y2, &x1, &y1);
      mtMirrorPoint(_x1, _y1, _x2, _y2, &x2, &y2);
      mtMirrorPoint(_x1, _y1, _x2, _y2, &cx, &cy);
      recalculateAngles();
      break;

    case T_POINT:
    case T_LINE:
      mtMirrorPoint(_x1, _y1, _x2, _y2, &x1, &y1);
      mtMirrorPoint(_x1, _y1, _x2, _y2, &x2, &y2);
      break;

    case T_TEXT:
      turnFlag(E_REVERSED);
      mtMirrorPoint(_x1, _y1, _x2, _y2, &x1, &y1);
      break;

    case T_DIMENSION:
      if(!getFlag(E_ROUNDOUT)) {
        mtMirrorPoint(_x1, _y1, _x2, _y2, &x1, &y1);
        mtMirrorPoint(_x1, _y1, _x2, _y2, &x2, &y2);
      }
      mtMirrorPoint(_x1, _y1, _x2, _y2, &cx, &cy);
      break;

    case T_HATCHING:
      if(subElement) {
        for(int i=0; i<subElementNum; ++i) {
          subElement[i].mirror( _x1, _y1, _x2, _y2, _updBorder );
        }
      }
      break;

    default:
      break;
  }
  
  if(_updBorder) recalculateBorders();
}



// Stretch element (move start point):
//
void  
RElement::moveStartPoint(float _x, float _y, bool _updBorder)
{
  switch(elementTyp) {
    case T_POINT:
    case T_LINE:
    case T_RECT:
    case T_RAPID:
    case T_STARTWAY:
    case T_ENDWAY:
      x1+=_x;
      y1+=_y;
      break;

    case T_DIMENSION:
      if(!getFlag(E_ROUNDOUT)) {
        x1+=_x;
        y1+=_y;
      }
      break;

    default:
      break;
  }

  if(_updBorder) recalculateBorders();
}



// Stretch element (move end point):
//
void  
RElement::moveEndPoint(float _x, float _y, bool _updBorder)
{
  switch(elementTyp) {
    case T_POINT:
    case T_LINE:
    case T_RECT:
    case T_RAPID:
    case T_STARTWAY:
    case T_ENDWAY:
      x2+=_x;
      y2+=_y;
      break;

    case T_DIMENSION:
      if(!getFlag(E_ROUNDOUT)) {
        x2+=_x;
        y2+=_y;
      }
      break;

    default:
      break;
  }

  if(_updBorder) recalculateBorders();
}



/**
 * Creates _num sub elements in this one.
 */
void
RElement::createSubElements(int _num)
{
  deleteSubElements();
  subElementNum=_num;
  subElement = new RElement[subElementNum];
}



/**
 * Deletes all sub elments in this one.
 */
void
RElement::deleteSubElements()
{
  if(subElement) delete[] subElement;
  subElement=0;
  subElementNum=0;
}



/**
 * Sets a sub element.
 */
void
RElement::setSubElement(int _i, RElement* _el)
{
  if(_el && _i>=0 && _i<=subElementNum) {
    subElement[_i].setGraphic( graphic );
    subElement[_i].getMeasuresFrom( _el );
  }
}



/**
 * Sets a sub element.
 */
RElement*
RElement::getSubElement(int _i)
{
  if(_i>=0 && _i<=subElementNum) {
    return &subElement[_i];
  }
  else {
    return 0;
  }
}



// Output some debug info in "debug.txt":
//
void  
RElement::outputDebug()
{
  RLOG("\n---------------\n ");
  RLOG("Typ: ");
  switch(elementTyp) {
    case T_POINT:  RLOG("POINT "); break;
    case T_LINE:   RLOG("LINE "); break;
    case T_ARC:    RLOG("ARC "); break;
    case T_CIRCLE: RLOG("CIRCLE "); break;
    case T_TEXT:   RLOG("TEXT "); break;
    case T_DIMENSION: RLOG("DIMENSION "); break;
    case T_HATCHING: RLOG("HATCHING "); break;
    case T_RECT:   RLOG("RECT "); break;
    case T_RAPID:  RLOG("RAPID "); break;
    default:       break;
  }
  
  RLOG("Layer: ");
  RLOG(layer);
  RLOG(" ");
  RLOG("Contour: ");
  RLOG(contour);
  
  RLOG(" ");
  RLOG("x1: ");
  RLOG(x1);
  RLOG(" ");
  RLOG("y1: ");
  RLOG(y1);
  RLOG(" ");
  if(elementTyp==T_TEXT) RLOG("Letter spacing: "); 
  else                   RLOG("x2: ");
  RLOG(x2);
  RLOG(" ");
  if(elementTyp==T_TEXT) RLOG("Textheight: "); 
  else                   RLOG("y2: ");
  RLOG(y2);
  RLOG(" ");
  if(elementTyp==T_TEXT) RLOG("Word spacing: "); 
  else                   RLOG("cx: ");
  RLOG(cx);
  RLOG(" ");
  if(elementTyp==T_TEXT) RLOG("Line distance: "); 
  else                   RLOG("cy: ");
  RLOG(cy);
  RLOG(" ");
  RLOG("cr: ");
  RLOG(cr);
  RLOG(" ");
  RLOG("a1: ");
  RLOG(a1);
  RLOG(" ");
  if(elementTyp==T_TEXT) RLOG("Font: "); 
  else                   RLOG("a2: ");
  RLOG(a2);
  RLOG(" ");
  
  RLOG("\n\tFlags: ");
  if(getFlag(E_VISIBLE))  RLOG("VISIBLE ");
  if(getFlag(E_ACTIVE))   RLOG("ACTIVE ");
  if(getFlag(E_REVERSED)) RLOG("REVERSED ");
  if(getFlag(E_TOOLOFFL)) RLOG("TOOLOFFL ");
  if(getFlag(E_TOOLOFFR)) RLOG("TOOLOFFR ");
  if(getFlag(E_TOOLOFFO)) RLOG("TOOLOFFO ");
  if(getFlag(E_CONV))     RLOG("CONV ");
  if(getFlag(E_CONTBEG))  RLOG("CONTBEG ");
  if(getFlag(E_CONTEND))  RLOG("CONTEND ");
  if(getFlag(E_TAGGED))   RLOG("TAGGED ");
  if(getFlag(E_TAGSTP))   RLOG("TAGSTP ");
  if(getFlag(E_TAGEND))   RLOG("TAGEND ");

  RLOG("\n\tText: ");
  RLOG(text.data());
}


// EOF













































































