#ifndef lint
static char SccsId[] = "%W%  %G%";
#endif

/* Module:	readiraf.c (Read IRAF)
 * Purpose:	Read "old style" IRAF OIF (.imh) files directly
 * Copyright:	1998 Smithsonian Astrophysical Observatory
 *		You may do anything you like with this file except remove
 *		this copyright.  The Smithsonian Astrophysical Observatory
 *		makes no representations about the suitability of this
 *		software for any purpose.  It is provided "as is" without
 *		express or implied warranty.
 * Modified:	{see end of file}
 */

#include <stdio.h>		/* define stderr, FD, and NULL */
#include <unistd.h>		/* define lseek arguments */
#include <stdlib.h>
#include <fcntl.h>
#include <sys/stat.h>
#include "wcs.h"
#include "fitsfile.h"

#ifndef VMS
#ifdef SYSV
#include <string.h>		/* strlen, strcat, strcpy, strrchr */
#else
#include <strings.h>		/* strlen, strcat, strcpy, rindex */
#define strchr index
#define strrchr rindex
#endif
#else
#include <string.h>		/* strlen, strcat, strcpy, strrchr */
#endif

#include <X11/Xlib.h>           /* X window stuff */
#include <X11/Xutil.h>          /* X window manager stuff */
#include "hfiles/constant.h"
#include "hfiles/struct.h"      /* declare structure types */
#include "hfiles/extern.h"      /* extern main parameter structure */

/* Local subroutines used to decode the IRAF header */
extern char *iraf2str();
extern int head_version ();
extern int pix_version ();
extern int irafncmp ();

/*
 * Subroutine:	init_irafimh
 * Purpose:	Open and read the iraf .imh file and set up the pixfile
 		for reading
 * Returns:	-1 if failure, else FD to image file ready for reading data
 * Notes:	The imhdr format is defined in iraf/lib/imhdr.h, some of
 *		which defines or mimiced, above.
 */
int init_irafimh ( img )
     struct imageRec *img;
{
  int fd;
  int nbfhead;
  char *irafheader;
  char *fitsheader;
  int open_disk(), lseek_disk(), read_disk();
  char *calloc_errchk();
  void close_disk();
  struct WorldCoor *wcsinit();
  int nbhead, pixoff, imhver;
  int bitpix, naxis;
  char pixpath[256];
  char *pixfile;

  /* open the image header file */
  if( (fd = open_disk(img->filename, IOP_Read, 0)) <= 0 )
    return( -1 );

  /* Find size of image header file */
  nbhead = (int) lseek (fd, 0, SEEK_END);
  (void) lseek (fd, 0, SEEK_SET);

  /* allocate initial sized buffer */
  irafheader = calloc_errchk(nbhead+2, 1, "IRAF header");

  /* read in all of IRAF image header */
  if (read_disk(fd, irafheader, nbhead, 0,
		img->filename, "IRAF header") < nbhead ) {
    free(irafheader);
    close_disk(fd, img->filename);
    return( -1 );
  }
  close_disk (fd, img->filename);

  /* check header magic word */
  imhver = head_version (irafheader);
  if (imhver < 1) {
    free (irafheader);
    (void)fprintf(stderr, "File %s not valid IRAF imhdr.\n", img->filename);
    return( -1 );
  }

  /* Translate IRAF header of either version 1 or 2 to FITS header */
  fitsheader = iraf2fits (img->filename, irafheader, nbhead, &nbfhead);

  /* check number of image dimensions */
  naxis = 0;
  hgeti4 (fitsheader,"NAXIS", &naxis);
  if (naxis < 2) {
    free (irafheader);
    free (fitsheader);
    (void)fprintf(stderr, "File %s does not contain 2d image\n", img->filename);
    return(-1);
  }

  /* get file parameters */
  hgeti4 (fitsheader, "BITPIX", &bitpix);
  switch (bitpix) {
  case 8:
    img->storage_type = ARR_U1;
    img->bytepix = 1;
    break;
  case -8:
    img->storage_type = ARR_U1;
    img->bytepix = 1;
    break;
  case -16:
    img->storage_type = ARR_U2;
    img->bytepix = 2;
    break;
  case 16:
    img->storage_type = ARR_I2;
    img->bytepix = 2;
    break;
  case 32:
    img->storage_type = ARR_I4;
    img->bytepix = 4;
    break;
  case -32:
    img->storage_type = ARR_R4;
    img->bytepix = 4;
    break;
  case -64:
    img->storage_type = ARR_R8;
    img->bytepix = 8;
    break;
  default:
    (void)fprintf(stderr,"Unsupported data type: %d\n", bitpix);
    free (irafheader);
    free (fitsheader);
    return( -1 );
  }
  hgeti4 (fitsheader, "NAXIS1", &img->filecols);
  hgeti4 (fitsheader, "NAXIS2", &img->filerows);
  hgeti4 (fitsheader, "NAXIS3", &img->filenimg);

  /* Check for existence and open pix file */
  hgets (fitsheader, "PIXFILE", 256, pixpath); 
  if ((fd = open_disk(pixpath, IOP_Read, 0)) <= 2 ) {
    pixfile = strrchr (pixpath,'/');
    if (pixfile == NULL || (fd = open_disk(pixfile+1, IOP_Read, 0)) <= 2 ) {
      free (irafheader);
      free (fitsheader);
      return( -1 );
      }
    }

  /* Check for data in pixel file */
  hgeti4 (fitsheader, "PIXOFF", &pixoff);
  if( lseek_disk(fd, pixoff, img->filename)
     < 0 ) {
    free(irafheader);
    free(fitsheader);
    close_disk(fd, img->filename);
    return( -1 );
    }

  /* Set byte-swapping for input */
  hgetl (fitsheader, "PIXSWAP", &img->byte_swap);

  /* Initialize World Coordinate System */
  if (nbfhead > 0) {
    wcs = wcsinit (fitsheader);
    (void)setwcslin (wcs, 1);
    if (control.verbose) {
      if (wcs)
        wcscent (wcs);
      else
        wcserr ();
      }   
    wcsoutinit (wcs, getwcscoor());
    }

  free (irafheader);
  img->header = fitsheader;
  return (fd);
}

/* Jan 23 1989	Michael VanHilst initial version
 * Feb 19 1990	MVH BSDonly strings.h compatability
 * Dec 12 1995	Doug Mink IRAF spatial WCS implemented
 * Dec 14 1995	Doug Mink No message if header too small
 * Jan  3 1996	Doug Mink IRAF2FITS arguments revised
 * Feb  9 1996  Doug Mink Allocate actual header size
 * Feb  9 1996	Doug Mink Change IRAF2FITS arguments
 * Dec 18 1997	Doug Mink Implement IRAF version 2 format
 * Jan 14 1998	Doug Mink Fix byte swapping so it works
 * Apr 17 1998	Doug Mink Add IRAF 2.11 unsigned data types
 * Jul  7 1998	Change setlinmode() to setwcslin()
 * Jul 17 1998	Use getwcscom() and getwcscoor() instead of external variables
 * Aug 12 1998	Include wcs.h instead of fitshead.h
 * Aug 12 1998	Include fitsfile.h instead of fitsio.h
 * Aug 14 1998	Use setwcscom() instead of wcscominit()
 * Sep 29 1998	Drop setwcscom() as it is in wcsinit()
 * Nov 16 1998	If pixfile not found, try smae file in current directory
 * Nov 16 1998	Use utility subroutines in imhfile.c
 * Nov 17 1998	Use separate flags for header and pixfile byte order
 * Nov 18 1998	Read correct number of header bytes
 */
