/* Zgv v3.1 - GIF, JPEG and PBM/PGM/PPM viewer, for VGA PCs running Linux.
 * Copyright (C) 1993-1998 Russell Marks. See README for license details.
 *
 * 3deffects.c - provides the `3d' style boxes, text etc.
 *                used by zgv.c (for file selector) and vgadisp.c
 *                (for help screen)
 */


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <vga.h>
#include <vgagl.h>
#include <vgamouse.h>
#include "3deffects.h"
#include "font.h"
#include "readnbkey.h"
#include "zgv.h"
#include "rc_config.h"
#include "rcfile.h"
#include "mousecur.h"


int msgbox_draw_ok=0;	/* 1 if ok to draw msgbox in current mode */

static unsigned char *save_mem=NULL;
static int save_x,save_y,save_width,save_height;



/* produce tacky 3d text */
void drawtext3d(int x,int y,int s,char *str,
                int isout,int light,int dark,int txt)
{
vga_setcolor(isout?light:dark);
vgadrawtext(x-1,y-1,s,str);
vga_setcolor(isout?dark:light);
vgadrawtext(x+1,y+1,s,str);
vga_setcolor(txt);
vgadrawtext(x,y,s,str);
}


/* restore sanity */
void undrawtext3d(int x,int y,int s,char *str)
{
vga_setcolor(idx_medium);
vgadrawtext(x-1,y-1,s,str);
vgadrawtext(x+1,y+1,s,str);
vgadrawtext(x,y,s,str);
}


/* render each bock in 3d */
void draw3dbox(int x1,int y1,int x2,int y2,int depth,
               int isout,int light,int dark)
{
int f;

for(f=0;f<depth;f++)
  {
  vga_setcolor(isout?light:dark);
  vga_drawline(x1+f,y2-f,x1+f,y1+f);
  vga_drawline(x1+f,y1+f,x2-f,y1+f);
  vga_setcolor(isout?dark:light);
  vga_drawline(x2-f,y1+f,x2-f,y2-f);
  vga_drawline(x2-f,y2-f,x1+f,y2-f);
  }
}


/* Undraw each relevant bock */
void undraw3dbox(int x1,int y1,int x2,int y2,int depth)
{
int f;

vga_setcolor(idx_medium);
for(f=0;f<depth;f++)
  {
  vga_drawline(x1+f,y2-f,x1+f,y1+f);
  vga_drawline(x1+f,y1+f,x2-f,y1+f);
  vga_drawline(x2-f,y1+f,x2-f,y2-f);
  vga_drawline(x2-f,y2-f,x1+f,y2-f);
  }
}


/* save old contents of area to put right-button menu on, and draw it.
 * we know it's either a 16-col or 256-col mode.
 */
static void save_area(int x1,int y1,int x2,int y2)
{
int fs256=(vga_getcurrentmode()!=G640x480x16);

save_x=x1;
save_y=y1;
save_width=x2-x1+1;
save_height=y2-y1+1;

if((save_mem=malloc(save_width*save_height))==NULL) return;

if(fs256)
  {
  /* use vgagl. mouse cursor does this too, but since we're doing the
   * same thing it won't hurt. :-)
   * (besides which, mouse might not be enabled...)
   */
  gl_setcontextvga(vga_getcurrentmode());
  gl_getbox(save_x,save_y,save_width,save_height,save_mem);
  }
else
  {
  /* 16-colour, use vga_getscansegment */
  unsigned char *ptr=save_mem;
  int y;
  
  for(y=0;y<save_height;y++,ptr+=save_width)
    vga_getscansegment(ptr,save_x,save_y+y,save_width);
  }
}


/* restore old contents of area */
static void restore_area()
{
if(save_mem==NULL) return;	/* ran out of memory, can't do much! */

if(vga_getcurrentmode()!=G640x480x16)
  {
  /* use vgagl again */
  gl_putbox(save_x,save_y,save_width,save_height,save_mem);
  }
else
  {
  /* 16-colour */
  unsigned char *ptr=save_mem;
  int y;
  
  for(y=0;y<save_height;y++,ptr+=save_width)
    vga_drawscansegment(ptr,save_x,save_y+y,save_width);
  }

free(save_mem);
}


static void blank_area(int x1,int y1,int x2,int y2,int col)
{
static unsigned char scanbit[640];	/* only used for 640x480 16-col */
int f;

/* must be 640x480 16-col, or unknown-size 256-col */
if(vga_getcurrentmode()!=G640x480x16)
  {
  gl_setcontextvga(vga_getcurrentmode());
  gl_fillbox(x1,y1,x2-x1+1,y2-y1+1,col);
  return;
  }

memset(scanbit,col,sizeof(scanbit));
for(f=y1;f<y2;f++)
  vga_drawscansegment(scanbit,x1,f,x2-x1+1);
}


/* if msgbox_draw_ok is zero, it changes to 640x480x{256,16} mode before
 * drawing the box, and obviously this blasts the screen. :-)
 * Otherwise, it saves/restores the area used as needed. This only
 * works for 16 and 256-col modes, but we know we must be on file selector
 * if msgbox_draw_ok is non-zero, so that's ok.
 *
 * NB: for type MSGBOXTYPE_TWOLINE_OK, `message' is corrupted by the
 * routine (the first `\n' is overwritten with a NUL).
 */
int msgbox(int ttyfd,char *message,int replytype,int light,int dark,int txt)
{
int x1,y1,x2,y2,wide,high,key;
struct { int x1,y1,x2,y2; } but1,but2;
char *ptr,*secondline=NULL;

if(!msgbox_draw_ok)
  {
  /* fairly nasty, but it'll be followed by a mode change so it
   * doesn't matter.
   */
  vga_setmode(fs_vgamode);
  vga_setpalette(idx_medium=1,cfg.medium.r,cfg.medium.g,cfg.medium.b);
  vga_setpalette(dark=2,cfg.dark.r,cfg.dark.g,cfg.dark.b);
  vga_setpalette(light=3,cfg.light.r,cfg.light.g,cfg.light.b);
  vga_setpalette(txt =4,cfg.black.r,cfg.black.g,cfg.black.b);
  /* needed for mouse cursor to look sane */
  vga_setpalette(5,0,0,0);	/* a non-zero-index black */
  mousecur_init(5,light);
  
  /* this isn't necessarily a nice thing to do, but it does make
   * sure the mouse is definitely in a sane position onscreen,
   * which might not be the case currently.
   * (Thinking about this, I'm not sure that it can *ever*
   * be anything other than a sane position when we're called,
   * but it doesn't hurt much to make sure in this
   * not-called-on-file-selector-screen case.)
   *
   * We don't need to save it after, as we're running in the fs's vga
   * mode and there'll be no need to rescale the mouse location.
   */
  restore_mouse_pos();
  }

set_max_text_width(vga_getxdim()-70);

high=90;
if(replytype!=MSGBOXTYPE_TWOLINE_OK)
  wide=vgatextsize(3,message);
else
  {
  int w1,w2;
  
  /* make the box a bit taller */
  high+=30;
  
  /* split the message string into two lines, and set width
   * to widest of the two. String length doesn't mean anything;
   * it's the onscreen width we care about here.
   */
  
  if((ptr=strchr(message,'\n'))==NULL)
    secondline=message+strlen(message);	/* point to a NUL */
  else
    {
    *ptr=0;
    secondline=ptr+1;
    }
  
  w1=vgatextsize(3,message);
  w2=vgatextsize(3,secondline);
  wide=w1;
  if(w2>w1) wide=w2;
  }

wide+=60;

x1=((vga_getxdim()-wide)>>1);
y1=((vga_getydim()-high)>>1);
x2=((vga_getxdim()+wide)>>1);
y2=((vga_getydim()+high)>>1);

if(msgbox_draw_ok)
  save_area(x1,y1,x2,y2);

blank_area(x1,y1,x2,y2,idx_medium);

/* draw outer box */
draw3dbox(x1 ,y1  ,x2  ,y2  ,2,1, light,dark);
draw3dbox(x1+9,y1+9,x2-9,y2-9,1,0, light,dark);

/* finally, I've got around to doing different types of msgbox! */
switch(replytype)
  {
  /* a box with a single 'OK' button, for warnings, errors, etc. */
  case MSGBOXTYPE_OK:
    /* draw 'button' */
    draw3dbox(but1.x1=((vga_getxdim()-40)>>1),but1.y1=y2-45,
              but1.x2=((vga_getxdim()+40)>>1),but1.y2=y2-20,
              1,1, light,dark);

    vga_setcolor(txt);
    vgadrawtext(x1+30,y1+20,3,message);
    drawtext3d(((vga_getxdim()-15)>>1)-3*(cfg.linetext==0),y2-39,3,
    	"OK",0,light,dark,txt);
    set_max_text_width(NO_CLIP_FONT);

    do
      {
      key=mousecur_wait_for_keys_or_mouse(ttyfd);
      if(has_mouse && is_end_click_left())
        {
        int mx=mouse_getx(),my=mouse_gety();
        if(mx>=but1.x1 && mx<=but1.x2 && my>=but1.y1 && my<=but1.y2)
          key=RK_ENTER;
        }
      }
    while(key!=RK_ESC && key!=RK_ENTER);
    
    if(msgbox_draw_ok) restore_area();
    return(1);

  /* a variant of the `OK' one, with two lines of text. The line is
   * split with a `\n'. Basically a special-case hack for `:'.
   */
  case MSGBOXTYPE_TWOLINE_OK:
    /* draw 'button' */
    draw3dbox(but1.x1=((vga_getxdim()-40)>>1),but1.y1=y2-45,
              but1.x2=((vga_getxdim()+40)>>1),but1.y2=y2-20,
              1,1, light,dark);
    
    vga_setcolor(txt);
    vgadrawtext(x1+(wide-vgatextsize(3,message))/2,y1+20,3,message);
    vgadrawtext(x1+(wide-vgatextsize(3,secondline))/2,y1+45,3,secondline);
    drawtext3d(((vga_getxdim()-15)>>1)-3*(cfg.linetext==0),y2-39,3,
    	"OK",0,light,dark,txt);
    set_max_text_width(NO_CLIP_FONT);

    do
      {
      key=mousecur_wait_for_keys_or_mouse(ttyfd);
      if(has_mouse && is_end_click_left())
        {
        int mx=mouse_getx(),my=mouse_gety();
        if(mx>=but1.x1 && mx<=but1.x2 && my>=but1.y1 && my<=but1.y2)
          key=RK_ENTER;
        }
      }
    while(key!=RK_ESC && key!=RK_ENTER);
    
    if(msgbox_draw_ok) restore_area();
    return(1);

  /* a box with two buttons, 'Yes' and 'No'. Enter or 'y' selects yes,
   * Esc or 'n' selects no.
   */    
  case MSGBOXTYPE_YESNO:
    /* draw 'yes' button */
    draw3dbox(but1.x1=(vga_getxdim()>>1)-50,but1.y1=y2-45,
              but1.x2=(vga_getxdim()>>1)-10,but1.y2=y2-20,
              1,1, light,dark);
    vga_setcolor(txt);
    vgadrawtext(x1+30,y1+20,3,message);		/* draw message */
    drawtext3d((vga_getxdim()>>1)-43-1*(cfg.linetext==0),y2-39,3,
    	"Yes",0,light,dark,txt);

    /* draw 'no' button */
    draw3dbox(but2.x1=(vga_getxdim()>>1)+10,but2.y1=y2-45,
              but2.x2=(vga_getxdim()>>1)+50,but2.y2=y2-20,
              1,1, light,dark);
    vga_setcolor(txt);
    drawtext3d((vga_getxdim()>>1)+23-2*(cfg.linetext==0),y2-39,3,
    	"No",0,light,dark,txt);

    set_max_text_width(NO_CLIP_FONT);
    
    do
      {
      key=mousecur_wait_for_keys_or_mouse(ttyfd);
      if(has_mouse && is_end_click_left())
        {
        int mx=mouse_getx(),my=mouse_gety();
        if(mx>=but1.x1 && mx<=but1.x2 && my>=but1.y1 && my<=but1.y2)
          key='y';
        if(mx>=but2.x1 && mx<=but2.x2 && my>=but2.y1 && my<=but2.y2)
          key='n';
        }
      }
    while(key!=RK_ESC && key!=RK_ENTER && key!='y' && key!='n');

    if(msgbox_draw_ok) restore_area();
    return(key==RK_ENTER || key=='y');
  }

if(msgbox_draw_ok) restore_area();

return(0);
}


/* get line of input (presumed to be a directory, though this is
 * not checked). Expands `~' to $HOME.
 */
char *cm_getline(int ttyfd,char *prompt,int light,int dark,int txt,int med)
{
static char buf[256],*retptr=NULL;
char *homeptr;
int pos,c,tmp;
int x1,y1,x2,y2,boxwidth,boxheight;
int insert=1;
int maxlen=sizeof(buf)-1;
int firsttime=1;
int curofs=0;

if(retptr!=NULL && retptr!=buf)
  {
  free(retptr);
  retptr=NULL;
  }

pos=0; *buf=0;

boxwidth=620;
boxheight=90;
x1=((vga_getxdim()-boxwidth)>>1);
y1=((vga_getydim()-boxheight)>>1);
x2=((vga_getxdim()+boxwidth)>>1);
y2=((vga_getydim()+boxheight)>>1);

save_area(x1,y1,x2,y2);

blank_area(x1,y1,x2,y2,med);

/* draw outer box */
draw3dbox(x1 ,y1  ,x2  ,y2  ,2,1, light,dark);
draw3dbox(x1+9,y2-46,x2-9,y2-9,1,0, light,dark);

/* draw prompt */
drawtext3d(x1+30,y1+15,3,prompt,0,light,dark,txt);

set_max_text_width(560);

do
  {
  if(firsttime)
    c=RK_NO_KEY;
  else
    do
      {
      c=mousecur_wait_for_keys_or_mouse(ttyfd);
      }
    while(c==RK_NO_KEY);
  
  vga_setcolor(med);
  tmp=vgadrawtext(x1+30,y2-35,3,buf);
  vgadrawtext(x1+30+curofs,y2-35,3,"_");
  
  if(firsttime)
    firsttime=0;
  else
    switch(c)
      {
      case 'b'-0x60: case RK_CURSOR_LEFT:  if(pos>0) pos--; break;
      case 'f'-0x60: case RK_CURSOR_RIGHT: if(pos<strlen(buf)) pos++; break;
      case 'a'-0x60: case RK_HOME:	pos=0; break;
      case 'e'-0x60: case RK_END:	pos=strlen(buf); break;
      case 'i'-0x60: case RK_INSERT:	insert=!insert; break;
      case 'd'-0x60: case RK_DELETE:
        /* delete forwards */
        if(pos<strlen(buf)) memmove(buf+pos,buf+pos+1,strlen(buf)-pos);
        break;
      case 8: case 127:
        /* delete backwards */
        if(pos>0)
          {
          memmove(buf+pos-1,buf+pos,strlen(buf+pos)+1);
          pos--;
          }
        break;
      
      default:
        /* if printable, insert it (or overwrite) */
        if(c>=32 && c<127)
          {
          if(insert)
            {
            if(strlen(buf)<maxlen)
              {
              if(pos<=strlen(buf))
                memmove(buf+pos+1,buf+pos,strlen(buf+pos)+1);
              buf[pos++]=c;
              }
            }
          else
            {
            if(strlen(buf)<maxlen || pos<strlen(buf))
              {
              if(pos>=strlen(buf))
                buf[pos+1]=0;
              
              buf[pos++]=c;
              }
            }
          }
      }
  
  /* draw text and cursor */
  vga_setcolor(txt);
  tmp=buf[pos];
  buf[pos]=0;
  curofs=vgatextsize(3,buf);
  buf[pos]=tmp;
  vgadrawtext(x1+30,y2-35,3,buf);
  vgadrawtext(x1+30+curofs,y2-35,3,"_");
  }
while(c!=RK_ENTER && c!=RK_ESC);

set_max_text_width(NO_CLIP_FONT);

restore_area();

if(c==RK_ESC) return(NULL);

/* expand tilde if needed; only dealt with if it's first char,
 * and the ~user form isn't supported.
 */
if(*buf!='~') return(buf);

homeptr=getenv("HOME");
if(homeptr==NULL) homeptr=buf+strlen(buf);	/* point at a NUL */

/* we want length of homeptr string, plus length of existing string
 * minus the first char (the tilde), plus space for the NUL.
 */

if((retptr=malloc(strlen(homeptr)+strlen(buf)))==NULL)
  return(NULL);

/* the memory will be freed on any subsequent call to this routine */

strcpy(retptr,homeptr);
strcat(retptr,buf+1);

return(retptr);
}
