// ct8file.cpp
//
// Copyright (C) 1999  Robert Barron, KA5WSS
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

#include "ct8file.h"
#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#define MAX_CT8_CALL_SIZE        16

/* Used to convert from CT file based band numbers to qso_data type band numbers. */
unsigned char RCt8File::bands[] = {0,BAND_160M,BAND_80M,BAND_40M,BAND_20M,BAND_15M,
                                   BAND_10M, 0, 0, 0, BAND_NOVICE, BAND_SATELLITE,
                                   BAND_PACKET, 0};
unsigned char RCt8File::vhfbands[] = {0, BAND_50MHZ, BAND_144MHZ, BAND_222MHZ, BAND_432MHZ,
                                      BAND_902MHZ, BAND_1GHZ, BAND_2GHZ, BAND_3GHZ,
                                      BAND_5GHZ, BAND_10GHZ, BAND_24GHZ, BAND_LIGHT, 0};
unsigned char RCt8File::dxpedbands[] = {0,BAND_160M,BAND_80M,BAND_40M,BAND_30M,BAND_20M,BAND_18M,
                                       BAND_15M,BAND_12M,BAND_10M,BAND_SATELLITE,0};
unsigned char RCt8File::contests[] = {CQ_WW_CONTEST,CQ_160M_CONTEST,CQ_WPX_CONTEST,
                                      ARRL_DX_CONTEST,ARRL_DX_CONTEST_DX,ARRL_10M_CONTEST,
                                      ARRL_160M_CONTEST,ARRL_SS_CONTEST,ARRL_VHF_QSO_PARTY,
                                      FD_CONTEST,WAE_CONTEST,DXPEDITION_MODE,CAL_QP_CONTEST,
                                      IARU_CONTEST,ALL_ASIAN_CONTEST,0};
unsigned char RCt8File::CTBands[] = {0, 1, 2, 3, 0, 4, 0, 5, 0, 6, 1, 0, 2, 3, 4, 5, 6, 7, 8, 9,
                                     10, 11, 0, 0, 0, 12, 13, 11, 0};
unsigned char RCt8File::CTContests[] = {0, 7, 3, 0, 2, 13, 5, 6, 1, 9, 12, 4, 10, 8, 11, 14, 0};
char *RCt8File::radioNames[] = {"NONE", "TS440", "TS850", "TS940", "TS950", "IC725", "IC735", "IC751",
                              "IC761", "IC765", "IC781", "FT890", "FT990", "FT1000", NULL};
                              
RCt8File::RCt8File(char *name)
{
    RCt8File::open(name);
}

boolean RCt8File::open(char *name, int mode)
{
#ifdef LINUX
    if (!RFile::open(name, mode))
        return FALSE;
#else
    // State ios::binary again just in case user does not.
    if (!RFile::open(name, mode | ios::binary))
        return FALSE;
#endif

    if (!(mode & ios::out))
    {
        RFile::read((void *)&ct8header, sizeof(Ct8Header));
        strncpy(header.callsign, ct8header.call, CT8_CALL_SIZE);
        strncpy(header.name, ct8header.name, CT8_NAME_SIZE);
        strncpy(header.address, ct8header.address, CT8_ADDRESS_SIZE);
        strncpy(header.city, ct8header.town, CT8_TOWN_SIZE);
        strncpy(header.state, ct8header.state, CT8_STATE_SIZE);
        strncpy(header.zip, ct8header.zip, CT8_ZIP_SIZE);

        header.stationNumber = (unsigned char)(ct8header.station + 1);
        strcpy(header.lastOperator, ct8header.last_op);
        strcpy(header.radio, RCt8File::radioNames[ct8header.radio]);
        header.tnc = ct8header.tnc;
        header.cwPort = ct8header.cw_port;
        header.dvkPort = ct8header.dvk_port;

        header.zone = (unsigned char)atoi(ct8header.zone);
        strncpy(header.grid, ct8header.my_grid, CT8_GRID_SIZE);
        strncpy(header.club, ct8header.club, CT8_CLUB_SIZE);

        if ((header.contest = contests[ct8header.contest]) == 0)
        {
            /* This contest is not supported. */
            RFile::close();
            return FALSE;
        }
        strncpy(header.contestName, ct8header.contest_title, 32);
        header.category = ct8header.category;
        header.mode = ct8header.mode + 1;
        
        strncpy(header.CQ_CW_Message, ct8header.cq_msg, 64);
        strncpy(header.Exchange_CW_Message, ct8header.ex_msg, 64);
        strncpy(header.QRZ_CW_Message, ct8header.qrz_msg, 64);
        strncpy(header.F6_CW_Message, ct8header.F6_text, 64);
        strncpy(header.F7_CW_Message, ct8header.F7_text, 64);
        strncpy(header.Dupe_CW_Message, ct8header.dupe_msg, 32);

        // Remember the COM: port information.
        for (int comm_count = 0; comm_count < 4; comm_count++)
        {
            header.comm[comm_count].device = ct8header.comm[comm_count].device;
            header.comm[comm_count].baud = ct8header.comm[comm_count].baud;
        }
        
        header.dvp.ingain = ct8header.dvp.ingain;
        header.dvp.outgain = ct8header.dvp.outgain;
        header.dvp.onair = ct8header.dvp.onair;
        header.dvp.mon = ct8header.dvp.mon;
        header.dvp.ram_disk = ct8header.dvp.ram_disk;
        header.dvp.ptt = ct8header.dvp.ptt;
        header.dvp.backcopy = ct8header.dvp.backcopy;
        header.dvp.spare = ct8header.dvp.spare;
        header.dvp.clipping_pt = ct8header.dvp.clipping_pt;
        header.dvp.rpt_delay = ct8header.dvp.rpt_delay;
        header.dvp.auto_space = ct8header.dvp.auto_space;
        
        header.flags.no_work_dupe = ct8header.flags.no_work_dupe;
        header.flags.m_stn = ct8header.flags.m_stn;
        header.flags.post = ct8header.flags.post;
        header.flags.beep = ct8header.flags.beep;
        header.flags.band_rate = ct8header.flags.band_rate;
        header.flags.sound = ct8header.flags.sound;
        header.flags.correct = ct8header.flags.correct;
        header.flags.cw_abbrev = ct8header.flags.cw_abbrev;
        header.flags.autosave = ct8header.flags.autosave;
        header.flags.nocompress = ct8header.flags.nocompress;
        header.flags.see_warc = ct8header.flags.see_warc;
        header.flags.print = ct8header.flags.print;
        header.flags.rpt = ct8header.flags.rpt;
        header.flags.kw_right = ct8header.flags.kw_right;
        
        // Don't bother with these fields unless they are specifically needed.
        if (header.contest == ARRL_SS_CONTEST)
        {
            header.ss.precedence = ct8header.ss.prec[0];
            header.ss.check = (unsigned char)atoi(ct8header.ss.chk);
            strncpy(header.ss.section, ct8header.sec, CT8_SECTION_SIZE);
        }
        else if (header.contest == FD_CONTEST)
        {
            header.fd.category = ct8header.fd.category;
            header.fd.transmitters = ct8header.fd.tx_count;
            strncpy(header.fd.section, ct8header.sec, CT8_SECTION_SIZE);

            header.fd.ep_bonus = ct8header.fd.ep_bonus;
            header.fd.pr_bonus = ct8header.fd.pr_bonus;
            header.fd.loc_bonus = ct8header.fd.loc_bonus;
            header.fd.info_bonus = ct8header.fd.info_bonus;
            header.fd.msg_bonus = ct8header.fd.msg_bonus;
            header.fd.sat_bonus = ct8header.fd.sat_bonus;
            header.fd.nat_bonus = ct8header.fd.nat_bonus;
            header.fd.w1aw_bonus = ct8header.fd.w1aw_bonus;
            header.fd.pkt_bonus = ct8header.fd.pkt_bonus;
            header.fd.tfc_bonus = ct8header.fd.tfc_bonus;
        }
    }
    
    location = 0;

    return TRUE;
}

boolean RCt8File::nextQso(qso_data *qso)
{
    ct8_log_data CTqso;
    unsigned short sentRST;
             
    qso->init();
    if (RFile::read((void *)&CTqso, sizeof(ct8_log_data)))
    {
        int code = CTqso.status & 0x44;
        
        /* Some CT files somehow got spaces in their info field.  Remove them. */
        /*for (char count = 0; count < strlen(CTqso.info); count++)
        if (!isalnum(CTqso.info[count]))
            CTqso.info[count] = NULL;*/
    
        qso->setCallsign(CTqso.call, strlen(CTqso.call));
        /* WARNING!  The following kluge "fixes" a 7 hour time difference between */
        /*  v7 and v8 generated time stamps.  Not 100% sure this is correct! */
        qso->setTime(CTqso.time - (7 * 60 * 60));
        
        if (header.contest == ARRL_VHF_QSO_PARTY)
            qso->setBand(vhfbands[CTqso.band]);
        else if (header.contest == DXPEDITION_MODE)
            qso->setBand(dxpedbands[CTqso.band]);
        else if (header.contest == ARRL_10M_CONTEST)
            qso->setBand(BAND_10M);        
        else
            qso->setBand(bands[CTqso.band]);
            
        qso->setMode(CTqso.mode);

        if (header.contest != ARRL_SS_CONTEST)
        {
            // Error check.  CT can allow an incorrect RST for a mode!
            if ((qso->getMode() != CW_MODE) && (CTqso.rst > 59))
                CTqso.rst = (short)(CTqso.rst / 10);
            else if ((qso->getMode() == CW_MODE) && (CTqso.rst < 100))
                CTqso.rst = (short)((CTqso.rst * 10) + 9);
            if ((qso->getMode() == CW_MODE) || (qso->getMode() == RTTY_MODE)
                || (qso->getMode() == PACKET_MODE))
                sentRST = 599;
            else
                sentRST = 59;
        }
        qso->setStationNumber(CTqso.stn_num - '0');
        qso->setQslSent(CTqso.qsl_sent);

        switch (header.contest) {
            case ARRL_SS_CONTEST:
                /* SS serial number should be in serial.  However many */
                /* CT files seem to have that value in rst instead. */
                if (CTqso.serial)
                    qso->setSerialNumber(CTqso.serial);
                else
                    qso->setSerialNumber(CTqso.rst);
                if (code == 0)
                    qso->setSSPrecedence('Q');
                else if (code == 0x40)
                    qso->setSSPrecedence('B');
                else
                    qso->setSSPrecedence('A');
                qso->setSSCheck(CTqso.check);
                qso->setSection(CTqso.info, 8);
                break;
            case CQ_WW_CONTEST:
                qso->setReceivedRST(CTqso.rst);
                qso->setSentRST(sentRST);
                qso->setCqZone(CTqso.info);
                break;
            case CQ_WPX_CONTEST:
                qso->setReceivedRST(CTqso.rst);
                qso->setSentRST(sentRST);
                qso->setSerialNumber(CTqso.info);
                break;
            case IARU_CONTEST:
                qso->setReceivedRST(CTqso.rst);
                qso->setSentRST(sentRST);
                if (isdigit(CTqso.info[0]))
                    qso->setItuZone(CTqso.info);
                else
                    qso->setInfo(CTqso.info, 8);
                break;
            case FD_CONTEST:
                /* Another kluge.  FD information SOMETIMES appears in RST for some reason. */
                if (CTqso.fd.tx_count != 0)
                {
                    qso->setFdTransmitters(CTqso.fd.tx_count);        // Transmitter count.
                    qso->setFdCategory(CTqso.fd.category);            // A-E.
                }
                else
                {
                    qso->setFdTransmitters((char)(CTqso.rst & 0xf));
                    qso->setFdCategory((char)(CTqso.rst >> 8));
                }
                qso->setSection(CTqso.info, 5);             // ARRL section.
                break;
            case WAE_CONTEST:
                qso->setReceivedRST(CTqso.rst);
                qso->setSentRST(sentRST);
                qso->setInfo(CTqso.info, 8);
                qso->setQtcNum(CTqso.qtc_num);
                break;
            case CAL_QP_CONTEST:
                if (CTqso.serial)
                    qso->setSerialNumber(CTqso.serial);
                else
                    qso->setSerialNumber(CTqso.rst);
                qso->setInfo(CTqso.info, 8);
                break;
            default:
                qso->setReceivedRST(CTqso.rst);
                qso->setSentRST(sentRST);
                qso->setInfo(CTqso.info, 8);
        }
        location++;
        return TRUE;
    }
    else
        return FALSE;
}

boolean RCt8File::writeQso(qso_data *qso)
{
    ct8_log_data ct_qso;
    char szTempString[16];

    memset(&ct_qso, '\0', sizeof(ct8_log_data));

    strcpy(ct_qso.call, qso->getCallsign());
    /* WARNING!  The following kluge "fixes" a 7 hour time difference between */
    /*  v7 and v8 generated time stamps.  Not 100% sure this is correct! */
    ct_qso.time = qso->getTime() + (7 * 60 * 60);
    ct_qso.mode = qso->getMode();
    ct_qso.band = CTBands[qso->getBand()];
    ct_qso.stn_num = qso->getStationNumber() + '0';
    ct_qso.qsl_sent = qso->getQslSent();

    switch (header.contest) {
        case ARRL_SS_CONTEST:    
            ct_qso.rst = (short)qso->getSerialNumber();
            switch (qso->getSSPrecedence()) {
                case 'A':
                    /* What to do about this one????? */
                    break;
                case 'B':
                    ct_qso.status |= 0x40;
                    break;
                case 'Q':
                    ct_qso.status |= 0x04;
                    /* The value in code needs to be 0, it already is. */
                    break;
            }
            ct_qso.check = qso->getSSCheck();
            strcpy(szTempString, qso->getSection());
            strupr(szTempString);
            strncpy(ct_qso.info, szTempString, 8);
            break;
        case CQ_WW_CONTEST:
            ct_qso.rst = (short)qso->getReceivedRST();
            sprintf(ct_qso.info, "%02d", qso->getCqZone());
            break;
        case CQ_WPX_CONTEST:
            ct_qso.rst = (short)qso->getReceivedRST();
            sprintf(ct_qso.info, "%d", qso->getSerialNumber());
            break;
        case IARU_CONTEST:
            ct_qso.rst = (short)qso->getReceivedRST();
            if (qso->getItuZone() > 0)
                sprintf(ct_qso.info, "%02d", qso->getItuZone());
            else
            {
                strcpy(szTempString, qso->getInfo());
                strupr(szTempString);
                strncpy(ct_qso.info, szTempString, 8);
            }
            break;
        case FD_CONTEST:
            ct_qso.fd.tx_count = qso->getFdTransmitters();        // Transmitter count.
            ct_qso.fd.category = qso->getFdCategory();            // A-E.
            
            /* FD information SOMETIMES appears in CT v8's RST field. So we had better */
            /* write it to both places to make sure that all versions of CT8 can read it. */
            ct_qso.rst = qso->getFdTransmitters();
            ct_qso.rst |= qso->getFdCategory() << 8;

            strcpy(szTempString, qso->getSection());
            strupr(szTempString);
            strncpy(ct_qso.info, szTempString, 5);           // ARRL section.
            break;
        case WAE_CONTEST:
            ct_qso.rst = (short)qso->getReceivedRST();
            strcpy(szTempString, qso->getInfo());
            strupr(szTempString);
            strncpy(ct_qso.info, szTempString, 8);
            ct_qso.qtc_num = qso->getQtcNum();
            break;
        case CAL_QP_CONTEST:
            /* CT v8 puts the CQP serial number in the RST field! */
            /* Put it in both rst and serial and hope for the best! */
            ct_qso.rst = (short)qso->getSerialNumber();
            ct_qso.serial = (short)qso->getSerialNumber();
            strcpy(szTempString, qso->getInfo());
            strupr(szTempString);
            strncpy(ct_qso.info, szTempString, 8);
            break;
        default:
            ct_qso.rst = (short)qso->getReceivedRST();
            strcpy(szTempString, qso->getInfo());
            strupr(szTempString);
            strncpy(ct_qso.info, szTempString, 8);
    }
    location++;

    RFile::write(&ct_qso, sizeof(ct8_log_data));
    
    return TRUE;
}

boolean RCt8File::writeHeader(header_data old_header)
{
    Ct8Header ct_header;

    RQsoFile::storeHeader(old_header);

    memset(&ct_header, '\0', sizeof(Ct8Header));

    strncpy(ct_header.version_str, "CT 8.19", 16);
    
    strncpy(ct_header.call, old_header.callsign, CT8_CALL_SIZE);
    strncpy(ct_header.name, old_header.name, CT8_NAME_SIZE);
    strncpy(ct_header.address, old_header.address, CT8_ADDRESS_SIZE);
    strncpy(ct_header.town, old_header.city, CT8_TOWN_SIZE);
    strncpy(ct_header.state, old_header.state, CT8_STATE_SIZE);
    strncpy(ct_header.zip, old_header.zip, CT8_ZIP_SIZE);

    ct_header.station = (unsigned char)(old_header.stationNumber - 1);
    strcpy(ct_header.last_op, old_header.lastOperator);
    for (unsigned char nameCount = 0; RCt8File::radioNames[nameCount] != NULL; nameCount++)
        if (strcmp(old_header.radio, RCt8File::radioNames[nameCount]) == 0)
            ct_header.radio = nameCount;
    ct_header.tnc = old_header.tnc;
    ct_header.cw_port = old_header.cwPort;
    ct_header.dvk_port = old_header.dvkPort;

    if (old_header.contest > ALL_ASIAN_CONTEST)
    {
        /* This contest is not supported. */
        RFile::close();
        return FALSE;
    }
    else
        ct_header.contest = CTContests[old_header.contest];
        
    strncpy(ct_header.contest_title, old_header.contestName, 32);
    ct_header.category = old_header.category;
    if ((old_header.mode > 0) && (old_header.mode < 4))
        ct_header.mode = old_header.mode - 1;       // CW, SSB and RTTY are supported.

    sprintf(ct_header.zone, "%d", old_header.zone);
    strncpy(ct_header.my_grid, old_header.grid, CT8_GRID_SIZE);
    strncpy(ct_header.club, old_header.club, CT8_CLUB_SIZE);

    strncpy(ct_header.cq_msg, old_header.CQ_CW_Message, 64);
    strncpy(ct_header.qrz_msg, old_header.QRZ_CW_Message, 64);
    strncpy(ct_header.ex_msg, old_header.Exchange_CW_Message, 64);
    strncpy(ct_header.F6_text, old_header.F6_CW_Message, 64);
    strncpy(ct_header.F7_text, old_header.F7_CW_Message, 64);
    strncpy(ct_header.dupe_msg, old_header.Dupe_CW_Message, 32);

    // Remember the COM: port information.
    for (int comm_count = 0; comm_count < 4; comm_count++)
    {
        ct_header.comm[comm_count].device = old_header.comm[comm_count].device;
        ct_header.comm[comm_count].baud = old_header.comm[comm_count].baud;
    }
        
    ct_header.dvp.ingain = old_header.dvp.ingain;
    ct_header.dvp.outgain = old_header.dvp.outgain;
    ct_header.dvp.onair = old_header.dvp.onair;
    ct_header.dvp.mon = old_header.dvp.mon;
    ct_header.dvp.ram_disk = old_header.dvp.ram_disk;
    ct_header.dvp.ptt = old_header.dvp.ptt;
    ct_header.dvp.backcopy = old_header.dvp.backcopy;
    ct_header.dvp.spare = old_header.dvp.spare;
    ct_header.dvp.clipping_pt = old_header.dvp.clipping_pt;
    ct_header.dvp.rpt_delay = old_header.dvp.rpt_delay;
    ct_header.dvp.auto_space = old_header.dvp.auto_space;

    ct_header.flags.no_work_dupe = old_header.flags.no_work_dupe;
    ct_header.flags.m_stn = old_header.flags.m_stn;
    ct_header.flags.post = old_header.flags.post;
    ct_header.flags.beep = old_header.flags.beep;
    ct_header.flags.band_rate = old_header.flags.band_rate;
    ct_header.flags.sound = old_header.flags.sound;
    ct_header.flags.correct = old_header.flags.correct;
    ct_header.flags.cw_abbrev = old_header.flags.cw_abbrev;
    ct_header.flags.autosave = old_header.flags.autosave;
    ct_header.flags.nocompress = old_header.flags.nocompress;
    ct_header.flags.see_warc = old_header.flags.see_warc;
    ct_header.flags.print = old_header.flags.print;
    ct_header.flags.rpt = old_header.flags.rpt;
    ct_header.flags.kw_right = old_header.flags.kw_right;
        
        ct_header.fd.power = 1;     /* Kluge to make CT8 output like   */
        ct_header.spare[0] = 32;    /* CT8 input for testing purposes! */
        ct_header.spare[1] = 244;
        ct_header.spare[2] = 1;
        ct_header.spare[3] = 4;
        
    // Don't bother with these fields unless they are specifically needed.
    if (old_header.contest == ARRL_SS_CONTEST)
    {
        ct_header.ss.prec[0] = old_header.ss.precedence;
        sprintf(ct_header.ss.chk, "%d", old_header.ss.check);
        strncpy(ct_header.sec, old_header.ss.section, CT8_SECTION_SIZE);
    }
    else if (old_header.contest == FD_CONTEST)
    {
        ct_header.fd.category = old_header.fd.category;
        ct_header.fd.tx_count = old_header.fd.transmitters;
        strncpy(ct_header.sec, old_header.fd.section, CT8_SECTION_SIZE);
        
        ct_header.fd.ep_bonus = old_header.fd.ep_bonus;
        ct_header.fd.pr_bonus = old_header.fd.pr_bonus;
        ct_header.fd.loc_bonus = old_header.fd.loc_bonus;
        ct_header.fd.info_bonus = old_header.fd.info_bonus;
        ct_header.fd.msg_bonus = old_header.fd.msg_bonus;
        ct_header.fd.sat_bonus = old_header.fd.sat_bonus;
        ct_header.fd.nat_bonus = old_header.fd.nat_bonus;
        ct_header.fd.w1aw_bonus = old_header.fd.w1aw_bonus;
        ct_header.fd.pkt_bonus = old_header.fd.pkt_bonus;
        ct_header.fd.tfc_bonus = old_header.fd.tfc_bonus;
    }
        
    RFile::write((void *)&ct_header, sizeof(Ct8Header));    
    return TRUE;
}
