// ---------------------------------------------------------------------------
// - Symbol.hpp                                                             -
// - aleph engine - symbol class definition                                 -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_SYMBOL_HPP
#define  ALEPH_SYMBOL_HPP

#ifndef  ALEPH_STRING_HPP
#include "String.hpp"
#endif

namespace aleph {

  /// The Symbol class is the fundamental class for all names in the aleph
  /// engine. A symbol binds a name to an object. A symbol can hold some
  /// flags as well. Typically one is the const flag. When a symbol is
  /// created, it is bound by default to the nil object. A symbol is by
  /// default mutable, that is, it can be changed.
  /// @author amaury darsch

  class Symbol : public Object {
  private:
    /// the symbol name
    String  d_name;
    /// the symbol object
    Object* p_object;
    /// the const flag
    bool d_const;

  public:
    /// create a new symbol by name
    /// @param name the symbol name
    Symbol (const String& name);

    /// create a new symbol with a name and an object
    /// @param name   the symbol name
    /// @param object the symbol object
    Symbol (const String& name, Object* object);

    /// destroy this symbol
    ~Symbol (void);

    /// @return the class name
    String repr (void) const;

    /// set the const flag for this symbol
    /// @param flag the flag to set
    void setconst (const bool flag);

    /// @return the const flag for this symbol
    inline bool getconst (void) const {
      return d_const;
    }

    /// set the symbol object
    /// @param object the object to set
    void setobj (Object* object);

    /// @return the symbol object
    inline Object* getobj (void) const {
      return p_object;
    }

    /// @return the symbol name
    inline String getname (void) const {
      return d_name;
    }

    /// set this symbol as a const object
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* cdef (Interp* interp, Nameset* nset, Object* object);

    /// set this symbol with an object
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Interp* interp, Nameset* nset, Object* object);

    /// evaluate an object in this symbol
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @return the evaluated object
    Object* eval (Interp* interp, Nameset* nset);

  private:
    // make the copy constructor private
    Symbol (const Symbol&);
    // make the assignment operator private
    Symbol& operator = (const Symbol&);
  };
}

#endif
