// ---------------------------------------------------------------------------
// - While.cpp                                                               -
// - aleph engine - builtin do/while loop functions implementation           -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Interp.hpp"
#include "Builtin.hpp"
#include "Boolean.hpp"
#include "Exception.hpp"

namespace aleph {

  // this procedure evaluate an object and check that we have a boolean. It
  // returns the boolean value
  static inline bool check_cond (Interp* interp, Nameset* nset, Object* obj) {
    bool result = false;
    Object*  object = (obj == nilp) ? nilp : obj->eval (interp, nset);
    Boolean* bval   = dynamic_cast <Boolean*> (object);
    if (bval == nilp) 
      throw Exception ("type-error", "illegal object in loop condition",
		       Object::repr (object));
    result = bval->toBoolean ();
    Object::cref (bval);
    return result;
  }

  // run the while loop

  Object* builtin_while (Interp* interp, Nameset* nset, Cons* args) {
    // trivial check first
    if ((args == nilp) || (args->length () < 2))
      throw Exception ("argument-error",
		       "missing or too many arguments with while loop");
    // extract condition and form
    Object* cond = args->getcar  ();
    Object* form = args->getcadr ();    
    // loop until false condition
    Object* result = nilp;
    while (check_cond (interp, nset, cond) == true) 
      result = (form == nilp) ? nilp : form->eval (interp, nset);
    return result;
  }

  // run the do loop

  Object* builtin_do (Interp* interp, Nameset* nset, Cons* args) {
    // trivial check first
    if ((args == nilp) || (args->length () < 2))
      throw Exception ("argument-error",
		       "missing or too many arguments with do loop");
    // extract condition and form
    Object* form = args->getcar  ();
    Object* cond = args->getcadr ();
    
    // loop until false condition
    Object* result = nilp;
    do {
      result = (form == nilp) ? nilp : form->eval (interp, nset);
    } while (check_cond (interp, nset, cond) == true);
    return result;
  }
}
