// ---------------------------------------------------------------------------
// - Buffer.hpp                                                              -
// - standard object library - character buffer class definition             -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_BUFFER_HPP
#define  ALEPH_BUFFER_HPP

#ifndef  ALEPH_STRING_HPP
#include "String.hpp"
#endif

namespace aleph {
  /// The Buffer class is a character buffer which can be used to accumulate
  /// character. The class is built as a circular buffer. The class maintains
  /// a set of indexes which can be used for editing purpose. The buffer class
  /// is also used by the terminal class to support editing operation.
  /// @author amaury darsch

  class Buffer {
  private:
    /// the character buffer
    char* p_buffer;
    /// the buffer allocated size
    long d_size;
    /// the insert flag
    bool d_insert;
    /// the start index
    long d_start;
    /// the end index
    long d_end;
    /// the cursor position in this buffer
    long d_cursor;

  public:
    /// Create a new buffer class. The buffer is created with a default
    /// size of 1024 characters.
    Buffer (void);

    /// Create a new buffer class with a default size.
    /// @param size the buffer default size
    Buffer (const long size);
    
    /// Destroy this buffer
    ~Buffer (void);

    /// add a character to this buffer
    /// @param value the character to add to this buffer
    void add (const char value);

    /// @return the character at the current index position
    char read (void);

    /// pushback a character in this buffer
    /// @param value the character to push back
    void pushback (const char value);

    /// pushback a string in this buffer
    void pushback (const String& value);

    /// remove a character before the cursor (backspace mode)
    /// @return true if the backspace succeded
    bool erase (void);

    /// remove a character at the cursor (delete mode)
    /// @return true if the delete succeded
    bool del (void);

    /// kill one or several characters
    void kill (const long num);

    /// @return true if the cursor was moved one character left
    bool movel (void);

    /// @return true if the cursor was moved one character right
    bool mover (void);

    /// move cursor to the begining of the buffer and return the amount
    /// of move done by the cursor
    /// @return the number of cursor move
    long setcb (void);

    /// move cursor to the end of the buffer and return the amount
    /// of move done by the cursor
    /// @return the number of cursor move
    long setce (void);

    /// set the buffer insert flag
    /// @param mode the insert mode
    void setimode (const bool mode);

    /// @return the length of this buffer
    long length (void) const;

    /// @return the buffer contents as a string
    String toString (void);

    /// @return the cursor position
    long getCursor (void) const;

    /// reset this buffer 
    void reset (void);

  private:
    // make the copy constructor private
    Buffer (const Buffer&);
    // make the assignment operator private
    Buffer& operator =  (const Buffer&);
  };
}

#endif
