// ---------------------------------------------------------------------------
// - Cons.cpp                                                                -
// - standard object library - cons cell class implementation                -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Cons.hpp"
#include "Real.hpp"
#include "Method.hpp"
#include "Boolean.hpp"
#include "Character.hpp"
#include "Exception.hpp"

namespace aleph {

  // -------------------------------------------------------------------------
  // - cons class section                                                    -
  // -------------------------------------------------------------------------

  // create a new cons cell initialized to nil

  Cons::Cons (void) {
    d_type = Cons::NORMAL;
    p_car  = nilp;
    p_cdr  = nilp;
  }

  // create a new cons cell with a type

  Cons::Cons (t_type type) {
    d_type = type;
    p_car  = nilp;
    p_cdr  = nilp;
  }

  // create a new cons cell with a car

  Cons::Cons (Object* car) {
    d_type = Cons::NORMAL;
    p_car  = Object::iref (car);
    p_cdr  = nilp;
  }

  // create a new cons cell with a type and a car

  Cons::Cons (t_type type, Object* car) {
    d_type = type;
    p_car  = Object::iref (car);
    p_cdr  = nilp;
  }

  // copy constructor for this cons cell

  Cons::Cons (const Cons& that) {
    d_type = that.d_type;
    p_car  = Object::iref (that.p_car);
    p_cdr  = that.p_cdr;
    Object::iref (that.p_cdr);
  }

  // destroy this cons cell

  Cons::~Cons (void) {
    Object::dref (p_car);
    Object::dref (p_cdr);
  }

  // return the class name
  String Cons::repr (void) const {
    return "Cons";
  }

  // assign a cons cell to this one

  Cons& Cons::operator = (const Cons& that) {
    // protect again same assignation
    Object::dref (p_car);
    Object::dref (p_cdr);
    // assign cell
    d_type = that.d_type;
    p_car  = Object::iref (that.p_car);
    p_cdr  = that.p_cdr; Object::iref (p_cdr);
    return *this;
  }

  // append an object to the last cdr of this cons cell

  void Cons::append (Object* object) {
    // create a new cons cell 
    Cons* cons = new Cons (object);
    // find the last cons cell
    Cons* last = this;
    while (last->p_cdr != nilp) last = last->p_cdr;
    // create attach this new cons cell
    last->p_cdr = cons;
    Object::iref (cons);
  }

  // set the car if the object is nil or append the object

  void Cons::lnkobj (Object* object) {
    if ((p_car == nilp) && (p_cdr == nilp)) {
      setcar (object);
      return;
    }
    append (object);
  }

  // set the car of this cons cell

  void Cons::setcar (Object* object) {
    Object::dref (p_car);
    p_car = Object::iref (object);
  }

  // set the cdr of this cons cell

  void Cons::setcdr (Cons* cdr) {
    Object::dref (p_cdr);
    p_cdr = cdr; Object::iref (cdr);
  }

  // return the car of the cdr of this cons cell

  Object* Cons::getcadr (void) const {
    if (p_cdr == nilp) return nilp;
    return p_cdr->p_car;
  }

  // return the car of the cdr of the cdr of this cons cell

  Object* Cons::getcaddr (void) const {
    if (p_cdr == nilp) return nilp;
    Cons* cdr = p_cdr->p_cdr;
    if (cdr == nilp) return nilp;
    return cdr->p_car;
  }
  
  // return the length of this cons cell

  long Cons::length (void) const {
    long result      = 0;
    const Cons* cons = this;
    do {
      result++;
    } while ((cons = cons->p_cdr) != nilp);
    return result;
  }

  // return an object by index

  Object* Cons::get (const long index) const {
    long count       = 0;
    const Cons* cons = this;
    if (index < 0) throw Exception ("index-error",
				    "invalid negative index in cons get");
    // loop in the cons cell list
    while (cons != nilp) {
      if (count == index) return cons->p_car;
      count++;
      cons = cons->p_cdr;
    }
    throw Exception ("index-error", "invalid index in cons get method");
  }

  // return a cons iterator

  Iterator* Cons::makeit (void) {
    return new ConsIterator (this);
  }

  // check for a certain number of argument

  void Cons::check (Cons* args, const long size) {
    long len = (args == nilp) ? 0 : args->length ();
    if (len != size) 
      throw Exception ("argument-error", "invalid number of arguments");
  }
  
  // get an integer value from a cons cell

  t_long Cons::getint (Interp* interp, Nameset* nset, Cons* args) {
    Object* cdr  = (args == nilp) ? nilp : args->getcdr ();
    if (cdr != nilp) throw Exception ("argument-error",
				      "too many arguments for one integer");
    Object* car = (args == nilp) ? nilp : args->getcar ();    
    Object* obj = (car == nilp) ? nilp : car->eval (interp, nset); 
    Integer* iobj = dynamic_cast <Integer*> (obj);
    if (iobj == nilp) 
      throw Exception ("type-error", "looking for integer but got",
		       Object::repr (obj));
    return iobj->toInteger ();
  }

  // get a real value from a cons cell

  t_real Cons::getreal (Interp* interp, Nameset* nset, Cons* args) {
    Object* cdr  = (args == nilp) ? nilp : args->getcdr ();
    if (cdr != nilp) throw Exception ("argument-error",
				      "too many arguments for one real");
    Object* car = (args == nilp) ? nilp : args->getcar ();
    Object* obj = (car == nilp) ? nilp : car->eval (interp, nset); 
    Real* robj = dynamic_cast <Real*> (obj);
    if (robj == nilp) 
      throw Exception ("type-error", "looking for real but got",
		       Object::repr (obj));
    return robj->toReal ();
  }

  // get a boolean value from a cons cell

  bool Cons::getbool (Interp* interp, Nameset* nset, Cons* args) {
    Object* cdr  = (args == nilp) ? nilp : args->getcdr ();
    if (cdr != nilp) throw Exception ("argument-error",
				      "too many arguments for one boolean");
    Object* car = (args == nilp) ? nilp : args->getcar ();
    Object* obj = (car == nilp) ? nilp : car->eval (interp, nset); 
    Boolean* bobj = dynamic_cast <Boolean*> (obj);
    if (bobj == nilp) 
      throw Exception ("type-error", "looking for boolean but got",
		       Object::repr (obj));
    return bobj->toBoolean ();
  }

  // get a character value from a cons cell

  char Cons::getchar (Interp* interp, Nameset* nset, Cons* args) {
    Object* cdr  = (args == nilp) ? nilp : args->getcdr ();
    if (cdr != nilp) throw Exception ("argument-error",
				      "too many arguments for one character");
    Object* car = (args == nilp) ? nilp : args->getcar ();
    Object* obj = (car == nilp) ? nilp : car->eval (interp, nset); 
    Character* cobj = dynamic_cast <Character*> (obj);
    if (cobj == nilp) 
      throw Exception ("type-error", "looking for character but got",
		       Object::repr (obj));
    return cobj->toCharacter ();
  }

  // get a string value from a cons cell

  String Cons::getstring (Interp* interp, Nameset* nset, Cons* args) {
    Object* cdr  = (args == nilp) ? nilp : args->getcdr ();
    if (cdr != nilp) throw Exception ("argument-error",
				      "too many arguments for one string");
    Object* car = (args == nilp) ? nilp : args->getcar ();
    Object* obj = (car == nilp) ? nilp : car->eval (interp, nset); 
    String* sobj = dynamic_cast <String*> (obj);
    if (sobj == nilp) 
      throw Exception ("type-error", "looking for string but got",
		       Object::repr (obj));
    return *sobj;
  }

  // create a new cons cell in a generic way

  Object* Cons::mknew (Vector* argv) {
    long len = 0;
    if ((argv == nilp) || ((len = argv->length ()) == 0)) return nilp;
    // build the cons cell
    Cons* result = nilp;
    for (long i = 0; i < len; i++) {
      if (result == nilp)
	result = new Cons (argv->get (i));
      else
	result->append (argv->get (i));
    }
    return result;
  }

  // set an object to the car of this cons cell

  Object* Cons::vdef (Interp* interp, Nameset* nset, Object* object) {
    setcar (object);
    return object;
  }

  // evaluate this cons cell in the current nameset

  Object* Cons::eval (Interp* interp, Nameset* nset) {
    Object* result = nilp;
    if (d_type == Cons::BLOCK) {
      Cons*   cons   = this;
      while (cons != nilp) {
	Object::cref (result);
	Object* car = cons->getcar ();
	result = (car == nilp) ? nilp : car->eval (interp,nset);
	cons   = cons->getcdr ();
      }
    } else {
      if (p_car == nilp) return nilp;
      Object* func = Object::iref (p_car->eval (interp, nset));
      if (func == nilp) return nilp;
      try {
	result = func->apply (interp, nset, p_cdr);
	Object::dref (func);
      } catch (...) {
	Object::dref (func);
	throw;
      }
    }
    return result;
  }

  // evaluate this cons with a member name

  Object* Cons::eval (Interp* interp, Nameset* nset, const String& name) {
    return new Method (name, this);
  }

  // apply a cons method by name

  Object* Cons::apply (Interp* interp, Nameset* nset, const String& name,
		       Cons* args) {
    // evaluate the arguments
    Vector* argv = Vector::eval (interp, nset, args);
    long    argc = (argv == nilp) ? 0 : argv->length ();

    // dispatch 0 argument
    if ((name == "get-car") && (argc == 0)) {
      delete argv;
      return getcar ();
    }
    if ((name == "get-cdr") && (argc == 0)) {
      delete argv;
      return getcdr ();
    }
    if ((name == "get-cadr") && (argc == 0)) {
      delete argv;
      return getcadr ();
    }
    if ((name == "get-caddr") && (argc == 0)) {
      delete argv;
      return getcaddr ();
    }
    if ((name == "length") && (argc == 0)) {
      delete argv;
      return new Integer (length ());
    }
    if ((name == "nil-p") && (argc == 0)) {
      delete argv;
      return new Boolean (isnil ());
    }
    if ((name == "block-p") && (argc == 0)) {
      delete argv;
      return new Boolean (isblock ());
    }
    if ((name == "get-iterator") && (argc == 0)) {
      delete argv;
      return makeit ();
    }

    // dispatch 1 argument
    if ((name == "set-car") && (argc == 1)) {
      Object* result = argv->get (0);
      setcar (result);
      delete argv;
      return result;
    }
    if ((name == "set-cdr") && (argc == 1)) {
      Object* result = argv->get (0);
      if (result == nilp) {
	setcdr ((Cons*) nilp);
	delete argv;
	return nilp;
      }
      Cons* cdr = dynamic_cast <Cons*> (result);
      if (cdr == nilp) 
	throw Exception ("type-error", "invalid object with set-cdr method",
			 Object::repr (result));
					
      setcdr (cdr);
      delete argv;
      return result;
    }
    if ((name == "append") && (argc == 1)) {
      Object* result = argv->get (0);
      append (result);
      delete argv;
      return result;
    }
    if ((name == "link") && (argc == 1)) {
      Object* result = argv->get (0);
      lnkobj (result);
      delete argv;
      return result;
    }

    if ((name == "get") && (argc == 1)) {
      long val = argv->getint (0);
      Object* result = get (val);
      delete argv;
      return result;
    }

    // call the object method
    Object* result = nilp;
    try {
      result =  Object::apply (interp, nset, name, argv);
    } catch (...) {
      delete argv;
      throw;
    }
    return result;
  }

  // -------------------------------------------------------------------------
  // - cons iterator class section                                           -
  // -------------------------------------------------------------------------

  // create a new cons iterator

  ConsIterator::ConsIterator (Cons* cons) {
    p_cons = cons;
    Object::iref (cons);
    p_cell = cons;
    Object::iref (cons);
    begin ();
  }

  // destroy this cons iterator

  ConsIterator::~ConsIterator (void) {
    Object::dref (p_cons);
    Object::dref (p_cell);
  }

  // return the class name

  String ConsIterator::repr (void) const {
    return "ConsIterator";
  }

  // reset the iterator to the begining

  void ConsIterator::begin (void) {
    Object::dref (p_cell);
    p_cell = p_cons;
    Object::iref (p_cell);
  }

  // reset the iterator to the end

  void ConsIterator::end (void) {
    throw Exception ("iterator-error", "cannot set a cons iterator to end");
  }

  // go to the next object

  void ConsIterator::next (void) {
    if (p_cell == nilp) return;
    Cons* cdr = p_cell->p_cdr;
    Object::iref (cdr);
    Object::dref (p_cell);
    p_cell = cdr;
  }

  // go to the previous object
  void ConsIterator::prev (void) {
    throw Exception ("iterator-error", "cannot move back a cons iterator");
  }

  // get the object at the current position

  Object* ConsIterator::getobj (void) {
    if (p_cell == nilp) return nilp;
    return p_cell->getcar ();
  }

  // return true if the iterator is at the end

  bool ConsIterator::isend (void) {
    if (p_cell == nilp) return true;
    return false;
  }
}
