// ---------------------------------------------------------------------------
// - Cons.hpp                                                                -
// - standard object library - cons cell class definition                    -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_CONS_HPP
#define  ALEPH_CONS_HPP

#ifndef  ALEPH_ITERABLE_HPP
#include "Iterable.hpp"
#endif

#ifndef  ALEPH_VECTOR_HPP
#include "Vector.hpp"
#endif

namespace aleph {

  /// The Cons class is the class used to create generic forms. A cons
  /// cell consists of a car and a cdr. The car holds an object while the cdr
  /// is the link to the next cell. When a cons cell is destroyed, the car and
  /// the cdr are destroyed as well. A cons cell is either a normal cell or
  /// a block cell. In the case of block cell, the car hold the cons cell 
  /// which is a list of forms.
  /// @author amaury darsch

  class Cons : public Iterable {
  public:
    /// the cons cell type
    enum t_type {NORMAL, BLOCK};

  private:
    /// the cons cell type
    t_type  d_type;
    /// the car holds the object
    Object* p_car;
    /// the cdr is always a cons
    Cons*   p_cdr;

  public:
    /// create a new cons cell with a null car and cdr
    Cons (void);

    /// create a new cons cell with a type
    /// @param type the cons cell type
    Cons (t_type type);

    /// create a cons cell with a car
    /// @param car the car of this cons cell
    Cons (Object* car);

    /// create a cons cell with a type and a car
    /// @param type the cons cell type
    /// @param car the car of this cons cell
    Cons (t_type type, Object* car);

    /// copy constructor for this cons cell
    /// @param that the cons to copy
    Cons (const Cons& that);

    /// destroy this cons cell
    ~Cons (void);

    /// @return the class name
    String repr (void) const;

    /// assign a cons cell to another one
    /// @param that the cons cell to assign
    Cons& operator = (const Cons& that);

    /// append an object to the last cdr of this cons. This method creates
    /// a new cons cell and link it to the last cdr.
    /// @param object the object to append
    void append (Object* object);

    /// set the car if nil or append an object to the last cdr of this cons.
    /// @param object the object to lnkobj
    void lnkobj (Object* object);

    /// set the car of this cons cell
    /// @param car the car to set
    void setcar (Object* object);

    /// set the cdr of this cons cell
    /// @param cdr the cdr to set
    void setcdr (Cons* cdr);
  
    /// @return the car of this cons cell or nil.
    inline Object* getcar (void) const {
      return p_car;
    }

    /// @return the cdr of this cons cell or nil
    inline Cons* getcdr (void) const {
      return p_cdr;
    }

    /// @return the car of the cdr or nil
    Object* getcadr (void) const;

    /// @return the car of the cdr of the cdr or nil
    Object* getcaddr (void) const;

    /// @return the length of this cons cell
    long length (void) const;

    /// @return true if the cdr of this cons cell is nil
    inline bool isnil (void) const {
      return (p_cdr == nilp);
    }

    /// @return true if the cons cell is of type block
    inline bool isblock (void) const {
      return (d_type == BLOCK);
    }

    /// @return an object by index
    Object* get (const long index) const;

    /// @return a new iterator for this cons
    Iterator* makeit (void);

    /// check for a certain number of arguments or throw an exception
    /// @param args the the arguments to check
    /// @param size the number of argument to check
    static void check (Cons* args, const long size);

    /// return a long integer value by object index
    static t_long getint (Interp* interp, Nameset* nset, Cons* args);

    /// return a real value by object index
    static t_real getreal (Interp* interp, Nameset* nset, Cons* args);

    /// return a boolean value by object index
    static bool getbool (Interp* interp, Nameset* nset, Cons* args);

    /// return a character value by object index
    static char getchar (Interp* interp, Nameset* nset, Cons* args);

    /// return a string value by object index
    static String getstring (Interp* interp, Nameset* nset, Cons* args);

    /// generate a new cons cell
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// set an object to the car of this cons cell
    /// @param interp the current interpreter
    /// @param nset   the current nameset
    /// @param object the object to set
    Object* vdef (Interp* interp, Nameset* nset, Object* object);

    /// evaluate this cons cell withing the calling nameset
    /// @param interp the current interpreter
    /// @param nset   the current nameset 
    Object* eval (Interp* interp, Nameset* nset);

    /// evaluate a cons member name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to evaluate
    Object* eval (Interp* interp, Nameset* nset, const String& name);

    /// apply this cons cell with a set of arguments and a method name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply this arguments
    /// @param args   the arguments to apply
    Object* apply (Interp* interp, Nameset* nset, const String& name,
		   Cons* args);
  private:
    // make the cons iterator a friend
    friend class ConsIterator;
  };

  /// The ConsIterator class is the iterator for the cons class. Such iterator
  /// is constructed with the "makeit" cons method. The iterator is reset to
  /// the beginning of the cons cell.
  /// @author amaury darsch

  class ConsIterator : public Iterator {
  private:
    /// the cons to iterate
    Cons* p_cons;
    /// the current cons cell
    Cons* p_cell;

  public:
    /// create a new iterator from a cons
    /// @param cons the list to iterate
    ConsIterator (Cons* cons);

    /// destroy this cons iterator
    ~ConsIterator (void);

    /// @return the class name
    String repr (void) const;

    /// reset the iterator to the begining
    void begin (void);

    /// reset the iterator to the end
    void end (void);

    /// move the list iterator to the next position
    void next (void);

    /// move the list iterator to the previous position
    void prev (void);

    /// @return the object at the current position
    Object* getobj (void);

    /// @return true if the iterator is at the end
    bool isend (void);

  private:
    // make the copy constructor private
    ConsIterator (const ConsIterator&);
    // make the assignment operator private
    ConsIterator& operator = (const ConsIterator&);
  };
}

#endif
