// ---------------------------------------------------------------------------
// - OutputFile.hpp                                                          -
// - standard object library - output file class definition                  -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2000 amaury darsch                                   -
// ---------------------------------------------------------------------------

#ifndef  ALEPH_OUTPUTFILE_HPP
#define  ALEPH_OUTPUTFILE_HPP

#ifndef  ALEPH_OUTPUT_HPP
#include "Output.hpp"
#endif

namespace aleph {

  /// The OutputFile class is a class to access file for writing. The
  /// class is constructed from the file name. If the file does not exist, it
  /// created. If the file exist, it is overwritten. All write method are
  /// available with this class, including the one defined in the base class.
  /// @author amaury darsch

  class OutputFile : public Output {
  private:
    /// the file name
    String d_name;
    /// the stream id
    int d_sid;

  public:
    /// create a new output stream by name. The file is open for writing and
    /// is created if it does not exist.
    /// @param name the file name
    OutputFile (const String& name);
    
    /// close and destroy this file output stream
    ~OutputFile (void);

    /// @return the class name
    String repr (void) const;

    /// close this output file
    void close (void);

    /// write one character on the output stream.
    /// @param value the character to write  
    void write (const char value);

    /// write a character string to the output stream
    /// @param value the character string to write
    void write (const char* value);

    /// write an integer to the output stream
    /// @param value the integer to write
    void write (const long value);

    /// @return the file name associated with this stream
    String getFileName (void) const;

    /// generate a new output file
    /// @param argv the argument vector
    static Object* mknew (Vector* argv);

    /// evaluate an output file member name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to evaluate
    Object* eval (Interp* interp, Nameset* nset, const String& name);

    /// apply this output file with a set of arguments and a method name
    /// @param interp interp the current interpreter
    /// @param nset   the current nameset    
    /// @param name   the name to apply this arguments
    /// @param args   the arguments to apply
    Object* apply (Interp* interp, Nameset* nset, const String& name,
		   Cons* args);

  private:
    // make the copy constructor private
    OutputFile (const OutputFile&);
    // make the assignment operator private
    OutputFile& operator = (const OutputFile&);
  };
}

#endif
