# Copyright (c) 1996, 1997, The Regents of the University of California.
# All rights reserved.  See Legal.htm for full text and disclaimer.

"""
PDB basic writer class PW.py by Paul Dubois, LLNL
Version 2.0 July, 1997
"""
import pypdb

class PW:
    "PDB file writer class."

    no_file_message = '(PW object not open on any file)'

    Error = 'PW error'

    def __del__ (self):
        "Close any file open when this object disappears."
        self.close()

    def __init__ (self, filename='', mode="w", verbose = 1):
        "PW(filename='', verbose=1) creates filename if given" 

        self.__dict__['_handle'] = None
        self.set_verbosity (verbose)
        if filename:
            self.open (filename, mode)

    def __setattr__ (self, name, value):
        self.write (name, value)

    def __repr__ (self):
        if self.is_open ():
            current_mode = 'opened for ' + \
                           self.inquire_handle ().mode()
            return 'PDB file %s %s.' % \
               (self.inquire_filename(), current_mode)
        else:
            return PW.no_file_message

    __str__ = __repr__

    def check_open (self):
        "check_open (): raise exception if not open for write."
        if not self.is_open ():
            raise PW.Error, 'PW object not open for write.'

    def close (self):
        "close(): close the file."
        h = self.inquire_handle ()
        if not (h is None):
            if self.inquire_verbosity():
                print "Closing PDB file being written:", \
                      self.inquire_filename()
            h.close()
        self.__dict__['_handle'] = None

    def inquire_filename (self):
        "inquire_filename() = name of this file."
        if self.is_open ():
            return self.inquire_handle ().filename()
        else:
            return ''

    def inquire_handle (self):
        "inquire_handle () = PWBfile object open on this file."
        return self._handle

    def inquire_mode (self):
        "inquire_mode () = mode ('w', or 'a') of this file."
        self.check_open()
        return self.inquire_handle ().mode()

    def inquire_pwd(self):
        "inquire_pwd () = present PDB directory"
        self.check_open ()
        return self.inquire_handle ().pwd()

    def inquire_verbosity (self):
        "inquire_verbosity () = current value of verbose flag."
        return self._verbose_flag

    def is_open (self):
        "is_open () = true if file is open"
        if self.inquire_handle () == None:
            return 0
        return 1

    def open (self, filename, mode = "w"):
        "open (filename, 'w' or 'a')"
        self.close ()
        if mode == "w":
            self.__dict__['_handle'] = pypdb.create (filename)
        elif mode == "a":
            self.__dict__['_handle'] = pypdb.append (filename)
        else:
            raise ValueError, "Improper mode: " + mode


    def make_directory(self, name):
        """make_directory(name) 
        -- create a new PDB directory, return status"""
        self.check_open()
        return self.inquire_handle().mkdir(name)

    def make_link(self, var, link):
        """make_directory(var, link) 
        -- make a link, return status"""
        self.check_open()
        return self.inquire_handle().ln(name)

    def set_directory (self, name):
        """set_directory(name) 
        -- change PDB directory to name, return status"""
        self.check_open;
        return self.inquire_handle ().cd(name)

    def set_verbosity (self, flag):
        """set_verbosity (flag) sets verbosity level to flag.
        0 for quiet operation, 
        1 to report closing only, 
        2 to report access to data."""
        if 0 <= flag <= 2:
            self.__dict__['_verbose_flag'] = flag
        else:
            self.__dict__['_verbose_flag'] = 2

    def write (self, name, quantity, record = 0):
        """Write quantity to file as 'name'"""
        self.check_open()
        if self.inquire_verbosity () > 1: 
            if record == 0:
                print "PW::write writing", name
            else:
                print "PW::write writing record", record, \
                      "of", name
        self.inquire_handle ().write (name, quantity, record)


if __name__ == "__main__":
    f=PW("foo.pdb")
    a = 1
    b = 2.0
    c = "Hello world"
    from multiarray import *
    d = array ([1.,2., 3.])
    e = array ([1,2,3])
    g = array (["hello", "world", "array"])
    h = array ([[1.,2.,3.], [4.,5.,6]])
    k = 3
    f.a = a
    f.b = b
    f.c = c
    f.d = d
    f.e = e
    f.g = g
    f.h = h
    f.close ()
    f.open ("foo.pdb", "a")
    f.k = k
    f.close ()
# read-back test	
    from PR import PR
    f = PR('foo.pdb')
    for x in f.inquire_names ():
        print x, "is", eval(x), ", in file it is", eval('f.'+x)
    f.close ()
# record-writing
    g = PW ('goo.pdb')
    g.set_verbosity (1)
    xh = array([0.]*4)
    for i in range (len(xh)):
        x = i / 10.
        g.write ('xh', x, i + 1)
        xh [i] = x
    g.close ()
    g = PR ('goo.pdb')
    print "xh is", xh, ", file it is ", g.xh
    g.close ()










