/********************************************************************************
*                                                                               *
*                      O p e n G L   V i e w e r   W i d g e t                  *
*                                                                               *
*********************************************************************************
* Copyright (C) 1997 by Jeroen van der Zijp.   All Rights Reserved.             *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Library General Public                   *
* License as published by the Free Software Foundation; either                  *
* version 2 of the License, or (at your option) any later version.              *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Library General Public License for more details.                              *
*                                                                               *
* You should have received a copy of the GNU Library General Public             *
* License along with this library; if not, write to the Free                    *
* Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.            *
*********************************************************************************
* $Id: FXGLViewer.h,v 1.9 1999/11/09 23:58:37 jeroen Exp $                      *
********************************************************************************/
#ifndef FXGLVIEWER_H
#define FXGLVIEWER_H


class FXDCPrint;


// Pick tolerance
#define PICK_TOL  3


// GL Viewer options
enum FXGLViewerOption {
  VIEWER_LOCKED   = 0x00008000,     // Mouse based view manipulation is locked
  VIEWER_LIGHTING = 0x00010000,     // Lighting is on
  VIEWER_FOG      = 0x00020000,     // Fog mode on
  VIEWER_DITHER   = 0x00040000      // Dithering
  };

  
/*******************************  Viewer  Structs  *****************************/
  

struct FXViewport {
  FXint      w,h;               // Viewport dimensions
  FXdouble   left,right;        // World box
  FXdouble   bottom,top;
  FXdouble   hither,yon;
  };

struct FXAPI FXLight {
  FXHVec     ambient;           // Ambient light color
  FXHVec     diffuse;           // Diffuse light color
  FXHVec     specular;          // Specular light color
  FXHVec     position;          // Light position
  FXVec      direction;         // Spot direction
  FXfloat    exponent;          // Spotlight exponent
  FXfloat    cutoff;            // Spotlight cutoff angle
  FXfloat    c_attn;            // Constant attenuation factor
  FXfloat    l_attn;            // Linear attenuation factor
  FXfloat    q_attn;            // Quadratic attenuation factor
  };

struct FXAPI FXMaterial {
  FXHVec     ambient;           // Ambient material color
  FXHVec     diffuse;           // Diffuse material color
  FXHVec     specular;          // Specular material color
  FXHVec     emission;          // Emissive material color
  FXfloat    shininess;         // Specular shininess
  };

  
// Feedback buffer sort routine
typedef FXbool (*FXZSortFunc)(FXfloat*& buffer,FXint& used,FXint& size);


/********************************  Viewer  Class  ******************************/

  
// Canvas, an area drawn by another object
class FXAPI FXGLViewer : public FXGLCanvas {
  FXDECLARE(FXGLViewer)
  friend class FXGLObject;
protected:
  FXViewport      wvt;              // Window viewport transform
  FXHMat          transform;        // Current transformation matrix
  FXHMat          itransform;       // Inverse of current transformation matrix
  FXdouble        screenmin;        // Minimum screen dimension
  FXdouble        screenmax;        // Maximum screen dimension
  FXuint          projection;       // Projection mode
  FXQuat          rotation;         // Viewer orientation
  FXdouble        aspect;           // Aspect ratio
  FXdouble        fov;              // Field of view
  FXdouble        zoom;             // Zoom factor
  FXdouble        offset;           // Offset for lines on surfaces
  FXVec           center;           // Model center
  FXVec           scale;            // Model scale
  FXdouble        worldpx;          // Pixel size in world
  FXdouble        modelpx;          // Pixel size in model
  FXuint          op;               // Operation being performed
  FXint           maxhits;          // Maximum number of hits
  FXdouble        ax,ay;            // Quick view->world coordinate mapping
  FXdouble        diameter;         // Size of model diameter ( always > 0)
  FXdouble        distance;         // Distance of PRP to target
  FXHVec          background;       // Background color
  FXHVec          ambient;          // Global ambient light
  FXLight         light;            // Light source
  FXMaterial      material;         // Base material properties
  FXint           dial[3];          // Dial positions
  FXTimer        *timer;            // Motion timer
  FXString        help;             // Status help
  FXString        tip;              // Tooltip for background
  FXbool          doesturbo;        // Doing turbo mode
  FXbool          turbomode;        // Turbo mode
  FXGLObject     *dropped;          // Object being dropped on
  FXGLObject     *selection;        // Current object
  FXZSortFunc     zsortfunc;        // Routine to sort feedback buffer
  FXGLObject     *scene;            // What we're looking at

protected:
  
  // Mouse actions when in viewing window
  enum {
    PICKING,                        // Pick mode
    TRANSLATING,                    // Translating camera
    ROTATING,                       // Rotating camera around target
    ZOOMING,                        // Zooming
    FOVING,                         // Change field-of-view
    DRAGGING,                       // Dragging objects
    TRUCKING,                       // Trucking camera
    LASSOING,                       // Lassoing rectangle
    GYRATING                        // Rotation of camera around eye
    };

protected:
  FXGLViewer(){}
  void glsetup();
  void updateProjection();
  void updateTransform();
  FXVec spherePoint(FXint px,FXint py);
  FXQuat turn(FXint fx,FXint fy,FXint tx,FXint ty);
  void drawWorld(FXViewport& wv);
  void drawAnti(FXViewport& wv);
  void drawLasso(FXint x0,FXint y0,FXint x1,FXint y1);
  FXint selectHits(FXuint *hits,FXint x,FXint y,FXint w,FXint h,FXint maxhits);
  FXint renderFeedback(FXfloat *buffer,FXint x,FXint y,FXint w,FXint h,FXint maxbuffer);
  void drawFeedback(FXDCPrint& pdc,const FXfloat* buffer,FXint used);
  virtual FXGLObject* processHits(FXuint *pickbuffer,FXint nhits);
  virtual void layout();
private:
  FXGLViewer(const FXGLViewer&);
  FXGLViewer &operator=(const FXGLViewer&);
public:
  long onPaint(FXObject*,FXSelector,void*);
  long onUpdate(FXObject*,FXSelector,void*);
  long onEnter(FXObject*,FXSelector,void*);
  long onLeave(FXObject*,FXSelector,void*);
  long onMotion(FXObject*,FXSelector,void*);
  long onChanged(FXObject*,FXSelector,void*);
  long onClicked(FXObject*,FXSelector,void*);
  long onDoubleClicked(FXObject*,FXSelector,void*);
  long onTripleClicked(FXObject*,FXSelector,void*);
  long onLassoed(FXObject*,FXSelector,void*);
  long onSelected(FXObject*,FXSelector,void*);
  long onDeselected(FXObject*,FXSelector,void*);
  long onInserted(FXObject*,FXSelector,void*);
  long onDeleted(FXObject*,FXSelector,void*);
  long onLeftBtnPress(FXObject*,FXSelector,void*);
  long onLeftBtnRelease(FXObject*,FXSelector,void*);
  long onMiddleBtnPress(FXObject*,FXSelector,void*);
  long onMiddleBtnRelease(FXObject*,FXSelector,void*);
  long onRightBtnPress(FXObject*,FXSelector,void*);
  long onRightBtnRelease(FXObject*,FXSelector,void*);
  long onUngrabbed(FXObject*,FXSelector,void*);
  long onKeyPress(FXObject*,FXSelector,void*);
  long onKeyRelease(FXObject*,FXSelector,void*);
  long onFocusIn(FXObject*,FXSelector,void*);
  long onFocusOut(FXObject*,FXSelector,void*);
  long onCmdPerspective(FXObject*,FXSelector,void*);
  long onUpdPerspective(FXObject*,FXSelector,void*);
  long onCmdParallel(FXObject*,FXSelector,void*);
  long onUpdParallel(FXObject*,FXSelector,void*);
  long onCmdFront(FXObject*,FXSelector,void*);
  long onUpdFront(FXObject*,FXSelector,void*);
  long onCmdBack(FXObject*,FXSelector,void*);
  long onUpdBack(FXObject*,FXSelector,void*);
  long onCmdLeft(FXObject*,FXSelector,void*);
  long onUpdLeft(FXObject*,FXSelector,void*);
  long onCmdRight(FXObject*,FXSelector,void*);
  long onUpdRight(FXObject*,FXSelector,void*);
  long onCmdTop(FXObject*,FXSelector,void*);
  long onUpdTop(FXObject*,FXSelector,void*);
  long onCmdBottom(FXObject*,FXSelector,void*);
  long onUpdBottom(FXObject*,FXSelector,void*);
  long onCmdResetView(FXObject*,FXSelector,void*);
  long onCmdFitView(FXObject*,FXSelector,void*);
  long onDNDEnter(FXObject*,FXSelector,void*);
  long onDNDLeave(FXObject*,FXSelector,void*);
  long onDNDMotion(FXObject*,FXSelector,void*);
  long onDNDDrop(FXObject*,FXSelector,void*);
  long onTipTimer(FXObject*,FXSelector,void*);
  long onQueryHelp(FXObject*,FXSelector,void*);
  long onQueryTip(FXObject*,FXSelector,void*);
  long onCmdXYZDial(FXObject*,FXSelector,void*);
  long onUpdXYZDial(FXObject*,FXSelector,void*);
  long onUpdCurrent(FXObject*,FXSelector,void*);
  long onCmdCutSel(FXObject*,FXSelector,void*);
  long onCmdCopySel(FXObject*,FXSelector,void*);
  long onCmdPasteSel(FXObject*,FXSelector,void*);
  long onCmdDeleteSel(FXObject*,FXSelector,void*);
  long onUpdDeleteSel(FXObject*,FXSelector,void*);
  long onCmdBackColor(FXObject*,FXSelector,void*);
  long onUpdBackColor(FXObject*,FXSelector,void*);
  long onCmdAmbientColor(FXObject*,FXSelector,void*);
  long onUpdAmbientColor(FXObject*,FXSelector,void*);
  long onCmdLock(FXObject*,FXSelector,void*);
  long onUpdLock(FXObject*,FXSelector,void*);
  long onCmdLighting(FXObject*,FXSelector,void*);
  long onUpdLighting(FXObject*,FXSelector,void*);
  long onCmdFog(FXObject*,FXSelector,void*);
  long onUpdFog(FXObject*,FXSelector,void*);
  long onCmdDither(FXObject*,FXSelector,void*);
  long onUpdDither(FXObject*,FXSelector,void*);
  long onCmdFov(FXObject*,FXSelector,void*);
  long onUpdFov(FXObject*,FXSelector,void*);
  long onCmdZoom(FXObject*,FXSelector,void*);
  long onUpdZoom(FXObject*,FXSelector,void*);
  long onCmdLightAmbient(FXObject*,FXSelector,void*);
  long onUpdLightAmbient(FXObject*,FXSelector,void*);
  long onCmdLightDiffuse(FXObject*,FXSelector,void*);
  long onUpdLightDiffuse(FXObject*,FXSelector,void*);
  long onCmdLightSpecular(FXObject*,FXSelector,void*);
  long onUpdLightSpecular(FXObject*,FXSelector,void*);
  long onCmdTurbo(FXObject*,FXSelector,void*);
  long onUpdTurbo(FXObject*,FXSelector,void*);
  long onCmdPrintImage(FXObject*,FXSelector,void*);
  long onCmdPrintVector(FXObject*,FXSelector,void*);

public:
  
  // Projection modes
  enum {
    PARALLEL,
    PERSPECTIVE
    };

  // Precompiled display lists
  enum {
    OFFSETPROJECTION  = 1,
    SURFACEPROJECTION = 2
    };
    
  // Messages
  enum {
    ID_PERSPECTIVE=FXGLCanvas::ID_LAST,
    ID_PARALLEL,
    ID_FRONT,
    ID_BACK,
    ID_LEFT,
    ID_RIGHT,
    ID_TOP,
    ID_BOTTOM,
    ID_RESETVIEW,
    ID_FITVIEW,
    ID_TIPTIMER,
    ID_BACK_COLOR,    
    ID_AMBIENT_COLOR,
    ID_LIGHT_AMBIENT,
    ID_LIGHT_DIFFUSE,
    ID_LIGHT_SPECULAR,
    ID_LIGHTING,
    ID_TURBO,
    ID_FOG,
    ID_DITHER,
    ID_DIAL_X,
    ID_DIAL_Y,
    ID_DIAL_Z,
    ID_FOV,
    ID_ZOOM,
    ID_LOCK,
    ID_CUT_SEL,
    ID_COPY_SEL,
    ID_PASTE_SEL,
    ID_DELETE_SEL,
    ID_PRINT_IMAGE,
    ID_PRINT_VECTOR,
    ID_LAST
    };

public:
  FXGLViewer(FXComposite* p,FXGLVisual *vis,FXObject* tgt=NULL,FXSelector sel=0,FXuint opts=0,FXint x=0,FXint y=0,FXint w=0,FXint h=0);
  virtual long handle(FXObject* sender,FXSelector key,void* data);
  virtual void create();
  virtual void resize(FXint w,FXint h);
  virtual void position(FXint x,FXint y,FXint w,FXint h);
  FXdouble worldPix() const { return worldpx; }
  FXdouble modelPix() const { return modelpx; }
  FXGLObject** lasso(FXint x1,FXint y1,FXint x2,FXint y2);
  virtual FXGLObject** select(FXint x,FXint y,FXint w,FXint h);
  virtual FXGLObject* pick(FXint x,FXint y);
  FXbool setBounds(const FXRange& box);
  FXbool fitToBounds(const FXRange& box);
  void getViewport(FXViewport& v) const;
  void eyeToScreen(FXint& sx,FXint& sy,const FXVec& e);
  FXVec screenToEye(FXint sx,FXint sy,FXfloat eyez=0.0);
  FXVec screenToTarget(FXint sx,FXint sy);
  FXVec worldToEye(const FXVec& w);
  FXfloat worldToEyeZ(const FXVec& w);
  FXVec eyeToWorld(const FXVec& e); 
  FXVec worldVector(FXint fx,FXint fy,FXint tx,FXint ty);
  void setMaterial(const FXMaterial &mtl);
  void getMaterial(FXMaterial &mtl) const;
  void setFieldOfView(FXdouble fv);
  FXdouble getFieldOfView() const { return fov; }
  void setZoom(FXdouble zm);
  FXdouble getZoom() const { return zoom; }
  void setDistance(FXdouble ed);
  FXdouble getDistance() const { return distance; }
  void setOrientation(const FXQuat& rot);
  const FXQuat& getOrientation() const { return rotation; }
  void setCenter(const FXVec& cntr);
  const FXVec& getCenter() const { return center; }
  void translate(const FXVec& vec);
  FXbool getBoreVector(FXint sx,FXint sy,FXVec& point,FXVec& dir);
  FXVec getEyeVector() const;
  FXVec getEyePosition() const;
  virtual FXbool canFocus() const;
  void setHelpText(const FXString& text);
  FXString getHelpText() const { return help; }
  void setTipText(const FXString&  text);
  FXString getTipText() const { return tip; }
  const FXHMat& getTransform() const { return transform; }
  const FXHMat& getInvTransform() const { return itransform; }
  void setScene(FXGLObject* sc);
  FXGLObject* getScene() const { return scene; }
  void setSelection(FXGLObject* sel);
  FXGLObject* getSelection() const { return selection; }
  void setProjection(FXuint proj);
  FXuint getProjection() const { return projection; }
  void setViewLock(FXbool lock=TRUE);
  FXbool getViewLock() const;
  void setBackgroundColor(const FXHVec& clr);
  FXHVec getBackgroundColor() const { return background; }
  void setAmbientColor(const FXHVec& clr);
  FXHVec getAmbientColor() const { return ambient; }
  FXbool readPixels(FXuchar*& buffer,FXint x,FXint y,FXint w,FXint h);
  FXbool readFeedback(FXfloat*& buffer,FXint& used,FXint& size,FXint x,FXint y,FXint w,FXint h);
  void setZSortFunc(FXZSortFunc func){ zsortfunc=func; }
  FXZSortFunc getZSortFunc() const { return zsortfunc; }
  void setOffset(FXdouble offs);
  FXdouble getOffset() const { return offset; }
  void setMaxHits(FXint maxh) { maxhits=maxh; }
  FXint getMaxHits() const { return maxhits; }
  FXbool doesTurbo() const { return doesturbo; }
  FXbool getTurboMode() const { return turbomode; }
  void setTurboMode(FXbool turbo=TRUE);
  void getLight(FXLight& lite) const;
  void setLight(const FXLight& lite);
  virtual void save(FXStream& store) const;
  virtual void load(FXStream& store);
  virtual ~FXGLViewer();
  };


#endif

