/* $Id: visual.c,v 1.62 1999/05/09 09:47:47 marcus Exp $
******************************************************************************

   Display-FBDEV: visual handling

   Copyright (C) 1998 Andrew Apted    [andrew@ggi-project.org]

   Permission is hereby granted, free of charge, to any person obtaining a
   copy of this software and associated documentation files (the "Software"),
   to deal in the Software without restriction, including without limitation
   the rights to use, copy, modify, merge, publish, distribute, sublicense,
   and/or sell copies of the Software, and to permit persons to whom the
   Software is furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL
   THE AUTHOR(S) BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
   IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
   CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

******************************************************************************
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include <linux/kd.h>
#include <linux/vt.h>
#include <linux/tty.h>

#include <ggi/internal/ggi-dl.h>
#include <ggi/display/fbdev.h>
#include <ggi/display/linvtsw.h>


static gg_option options[] =
{
	{ "nokbd",   "no" },
	{ "nomouse", "no" },
	{ "noinput", "no" },  /* shorthand for nokbd + nomouse */
	{ "novt",    "no" }
};

#define OPT_NOKBD	0
#define OPT_NOMOUSE	1
#define OPT_NOINPUT	2
#define OPT_NOVT	3

#define NUM_OPTIONS	(sizeof(options)/sizeof(gg_option))


#ifdef HAVE_NEW_FBDEV
static char accel_prefix[] = "tgt-fbdev-";
#define PREFIX_LEN	(sizeof(accel_prefix))

typedef struct {
	int   async;
	char *str;
} accel_info;


static accel_info accel_strings[] = {
	{ 0, "kgicon-generic",},	/* no accel - check for KGIcon	*/
	{ 0, NULL },			/* Atari Blitter		*/
	{ 0, NULL },			/* Amiga Blitter                */
	{ 0, NULL },			/* Cybervision64 (S3 Trio64)    */
	{ 0, NULL },			/* RetinaZ3 (NCR 77C32BLT)      */
	{ 0, NULL },			/* Cybervision64/3D (S3 ViRGE)	*/
	{ 0, NULL },			/* ATI Mach 64GX family		*/
	{ 0, NULL },			/* DEC 21030 TGA		*/
	{ 0, NULL },			/* ATI Mach 64CT family		*/
	{ 0, NULL },			/* ATI Mach 64CT family VT class */
	{ 0, NULL },			/* ATI Mach 64CT family GT class */
	{ 0, NULL },			/* Sun Creator/Creator3D	*/
	{ 0, NULL },			/* Sun cg6			*/
	{ 0, NULL },			/* Sun leo/zx			*/
	{ 0, NULL },			/* IMS Twin Turbo		*/
	{ 0, NULL },			/* 3Dlabs Permedia 2		*/
	{ 1, "mga-2164w" },		/* Matrox MGA2064W (Millenium)	*/
	{ 1, "mga-2164w" },		/* Matrox MGA1064SG (Mystique)	*/
	{ 1, "mga-2164w" },		/* Matrox MGA2164W (Millenium II) */
	{ 1, "mga-2164w" },		/* Matrox MGA2164W (Millenium II AGP)*/
	{ 1, "mga-2164w" },		/* Matrox G100 (Productiva G100) */
	{ 1, "mga-2164w" },		/* Matrox G200 (Myst, Mill, ...) */
	{ 0, NULL },			/* Sun cgfourteen		 */
	{ 0, NULL },			/* Sun bwtwo			 */
	{ 0, NULL },			/* Sun cgthree			 */
	{ 0, NULL }			/* Sun tcx			 */
};

#define NUM_ACCELS	(sizeof(accel_strings)/sizeof(accel_info))
#endif /* HAVE_NEW_FBDEV */

static void
switched_away(void *arg)
{
	ggi_visual *vis = arg;
	fbdev_hook *priv = LIBGGI_PRIVATE(vis);

	GGIDPRINT_MISC("display-fbdev: switched_away() called\n");
	if (!priv->autoswitch && !priv->switchpending) {
		gii_event ev;
		ggi_cmddata_switchrequest *data;
		
		GGIDPRINT_MISC("display-fbdev: queueing GGICMD_REQUEST_SWITCH"
			       " event\n");

		_giiEventBlank(&ev, sizeof(gii_cmd_event));

		data = (void *)ev.cmd.data;

		ev.size   = sizeof(gii_cmd_event);
		ev.cmd.type = evCommand;
		ev.cmd.code = GGICMD_REQUEST_SWITCH;
		data->request = GGI_REQSW_UNMAP;

		_giiSafeAdd(vis->input, &ev);
		
		priv->switchpending = 1;
	} else if (priv->autoswitch) {
		priv->ismapped = 0;
	}

}


static void
do_setpalette(ggi_visual *vis)
{
	ggi_graphtype gt = LIBGGI_MODE(vis)->graphtype;
	fbdev_hook *priv = LIBGGI_PRIVATE(vis);
	ggi_color *colormap;
	struct fb_cmap cmap;
	int len, i;

	if (GT_SCHEME(gt) != GT_PALETTE) return;

	len = 1 << GT_DEPTH(gt);
	cmap.start  = 0;
	cmap.len    = len;
	cmap.red    = priv->reds;
	cmap.green  = priv->greens;
	cmap.blue   = priv->blues;
	cmap.transp = NULL;
	colormap = vis->palette;

	for (i = 0; i < len; i++) {
		priv->reds[i]   = colormap[i].r;
		priv->greens[i] = colormap[i].g;
		priv->blues[i]  = colormap[i].b;
	}

	if (ioctl(LIBGGI_FD(vis), FBIOPUTCMAP, &cmap) < 0) {
		GGIDPRINT_COLOR("display-fbdev: PUTCMAP failed.");
	}
}



static void
switched_back(void *arg)
{
	ggi_visual *vis = arg;
	gii_event ev;

	GGIDPRINT_MISC("display-fbdev: switched_back() called\n");

	_giiEventBlank(&ev, sizeof(gii_expose_event));

	ev.any.size   = sizeof(gii_expose_event);
	ev.any.type   = evExpose;

	ev.expose.x = ev.expose.y = 0;
	ev.expose.w = LIBGGI_VIRTX(vis);
	ev.expose.h = LIBGGI_VIRTY(vis);

	_giiSafeAdd(vis->input, &ev);
	GGIDPRINT_MISC("fbdev: EXPOSE sent.\n");

	do_setpalette(vis);
	FBDEV_PRIV(vis)->ismapped = 1;
}


static int 
GGI_fbdev_sendevent(ggi_visual *vis, gii_event *ev)
{
	fbdev_hook *priv = LIBGGI_PRIVATE(vis);

	GGIDPRINT_MISC("GGI_fbdev_sendevent() called\n");

	if (ev->any.type != evCommand) {
		return GGI_EEVUNKNOWN;
	}
	switch (ev->cmd.code) {
	case GGICMD_ACKNOWLEDGE_SWITCH:
		GGIDPRINT_MISC("display-fbdev: switch acknowledge\n");
		if (priv->switchpending) {
			priv->switchpending = 0;
			priv->doswitch(vis);
			priv->ismapped = 0;
			return 0;
		} else {
			/* No switch pending */
			return GGI_EEVNOTARGET;
		}
		break;
	case GGICMD_NOHALT_ON_UNMAP:
		GGIDPRINT_MISC("display-fbdev: nohalt on\n");
		priv->dohalt = 0;
		priv->autoswitch = 0;
		break;
	case GGICMD_HALT_ON_UNMAP:
		GGIDPRINT_MISC("display-fbdev: halt on\n");
		priv->dohalt = 1;
		priv->autoswitch = 1;
		if (priv->switchpending) {
			/* Do switch and halt */
			priv->switchpending = 0;
			priv->doswitch(vis);
			priv->ismapped = 0;
			pause();
		}
		break;
	}
	
	return GGI_EEVUNKNOWN;
}


static int
GGI_fbdev_flush(ggi_visual *vis, int x, int y, int w, int h, int tryflag)
{
	fbdev_hook *priv = FBDEV_PRIV(vis);
	
	if (priv->flush) return priv->flush(vis, x, y, w, h, tryflag);

	return 0;
}

static int
GGI_fbdev_idleaccel(ggi_visual *vis)
{
	fbdev_hook *priv = FBDEV_PRIV(vis);

	GGIDPRINT_DRAW("GGI_fbdev_idleaccel(%p) called \n", vis);
	
	if (priv->idleaccel) return priv->idleaccel(vis);

	return 0;
}


static void 
save_palette(ggi_visual *vis)
{
	fbdev_hook *priv = LIBGGI_PRIVATE(vis);
	struct fb_cmap cmap;
	int len;

	if ((priv->orig_var.bits_per_pixel < 1) || 
	    (priv->orig_var.bits_per_pixel > 8)) {
		priv->orig_reds = NULL;
		return;
	}
	len = 1 << priv->orig_var.bits_per_pixel;
	
	priv->orig_reds = malloc(sizeof(uint16)*len*3);
	if (priv->orig_reds == NULL) {
		return;
	}

	cmap.start = 0;
	cmap.len   = len;
	cmap.red   = priv->orig_reds;
	cmap.green = priv->orig_greens = priv->orig_reds + len;
	cmap.blue  = priv->orig_blues  = priv->orig_greens + len;
	cmap.transp = NULL;

	if (ioctl(LIBGGI_FD(vis), FBIOGETCMAP, &cmap) < 0) {
		GGIDPRINT_COLOR("display-fbdev: GETCMAP failed.\n");
	} else {
		GGIDPRINT_COLOR("display-fbdev: Saved palette (len=%d).\n",
				cmap.len);
	}
}


static void 
restore_palette(ggi_visual *vis)
{
	fbdev_hook *priv = LIBGGI_PRIVATE(vis);
	struct fb_cmap cmap;

	if (priv->orig_reds == NULL) return;
	
	cmap.start  = 0;
	cmap.len    = 1 << priv->orig_var.bits_per_pixel;
	cmap.red    = priv->orig_reds;
	cmap.green  = priv->orig_greens;
	cmap.blue   = priv->orig_blues;
	cmap.transp = NULL;

	if (ioctl(LIBGGI_FD(vis), FBIOPUTCMAP, &cmap) < 0) {
		GGIDPRINT_COLOR("display-fbdev: PUTCMAP failed.\n");
	} else {
		GGIDPRINT_COLOR("display-fbdev: Restored palette (len=%d).\n", 
				cmap.len);
	}

	free(priv->orig_reds);
	priv->orig_reds = NULL;
}


static int do_cleanup(ggi_visual *vis)
{
	fbdev_hook *priv = LIBGGI_PRIVATE(vis);

	/* We may be called more than once due to the LibGG cleanup stuff */
	if (priv == NULL) return 0;

	GGIDPRINT("display-fbdev: GGIdlcleanup start.\n");

	if (LIBGGI_FD(vis) >= 0) {
		GGI_fbdev_resetmode(vis);
		restore_palette(vis);
		close(LIBGGI_FD(vis));
	}

	if (vis->input != NULL) {
		giiClose(vis->input);
		vis->input = NULL;
	}

	if (priv->normalgc) {
		free(priv->normalgc);
	}
	if (priv->accel) {
		free(priv->accel);
	}
	free(priv);

	ggUnregisterCleanup((ggcleanup_func *)do_cleanup, vis);

	GGIDPRINT("display-fbdev: GGIdlcleanup done.\n");

	return 0;
}

static int
get_fbdev(char *devname)
{
#ifdef FBIOGET_CON2FBMAP
	struct vt_stat qry_stat;  
	struct fb_con2fbmap map;
	int fb, fd;
	int devfs = 0;

	/* determine VT number */
	fd = open("/dev/tty", O_RDONLY);

	if (fd < 0) {
		perror("display-fbdev: failed to open tty");
		return -1;
	}

	if (ioctl(fd, VT_GETSTATE, &qry_stat) == 0) {
		map.console = qry_stat.v_active;
		GGIDPRINT_MISC("display-fbdev: Using VT %d.\n", map.console);
	} else {
		perror("display-fbdev: ioctl(VT_GETSTATE) failed");
		close(fd);
		return -1;
	}
	close(fd);

	/* find a framebuffer to open */
	for (fb=0; fb < 32; fb++) {
		sprintf(devname, "/dev/fb%d", fb);
		fd = open(devname, O_RDONLY);
		if (fd >= 0) {
			break;
		}
		sprintf(devname, "/dev/fb/%d", fb);
		fd = open(devname, O_RDONLY);
		if (fd >= 0) {
			devfs = 1;
			break;
		}
	}

	if (fb >= 32) {
		/* not found */
		GGIDPRINT_MISC("display-fbdev: Could not find a framebuffer "
			       "device with read permission.\n");
		return -1;
	}

	if (ioctl(fd, FBIOGET_CON2FBMAP, &map) != 0) {
		perror("display-fbdev: ioctl(FBIOGET_CON2FBMAP) failed");
		close(fd);
		return -1;
	}

	close(fd);

	if (devfs) {
		sprintf(devname, "/dev/fb/%d", map.framebuffer);
	} else {
		sprintf(devname, "/dev/fb%d", map.framebuffer);
	}

	GGIDPRINT_MISC("display-fbdev: Determined VT %d is on FB %d\n",
		       map.console, map.framebuffer);

	return 0;
#else
	return -1;
#endif
}


#define MAX_DEV_LEN	63

int GGIdlinit(ggi_visual *vis, const char *args, void *argptr)
{
	char  devicename[MAX_DEV_LEN+1];
	fbdev_hook *priv;

	GGIDPRINT("display-fbdev: GGIdlinit start.\n");

	if (getenv("GGI_FBDEV_OPTIONS") != 0) {
		if (ggParseOptions(getenv("GGI_FBDEV_OPTIONS"),
				   options, NUM_OPTIONS) == NULL) {
			fprintf(stderr, "display-fbdev: error in "
				"$GGI_FBDEV_OPTIONS\n");
			return GGI_DL_ERROR;
		}
	}
	
	if (args) {
		args = ggParseOptions(args, options, NUM_OPTIONS);
		if (args == NULL) {
			fprintf(stderr, "display-fbdev: error in "
				"arguments.\n");
			return GGI_DL_ERROR;
		}
	}

	LIBGGI_PRIVATE(vis) = priv = malloc(sizeof(fbdev_hook));
	if (priv == NULL) {
		return GGI_DL_ERROR;
	}
	
	priv->fb_ptr = NULL;
	priv->dohalt = 1;
	priv->autoswitch = 1;
	priv->switchpending = 0;
	priv->ismapped = 1;
	priv->accel = NULL;
	priv->have_accel = 0;
	priv->accelpriv = NULL;
	priv->mmioaddr = NULL;
	priv->flush = NULL;
	priv->idleaccel = NULL;
	priv->orig_reds = NULL;

	/* handle args */
	while (args && *args && isspace(*args)) args++;

	if (args && *args) {
		strncpy(devicename, args, MAX_DEV_LEN);
	} else if (getenv("FRAMEBUFFER") != NULL) {
		strncpy(devicename, getenv("FRAMEBUFFER"), MAX_DEV_LEN);
	} else {
		if (get_fbdev(devicename) != 0) {
			*devicename = '\0';
		}
	}
	/* Make sure string is terminated */
	devicename[MAX_DEV_LEN] = '\0';
	
	priv->inputs = FBDEV_INP_KBD | FBDEV_INP_MOUSE | FBDEV_INP_VT;

	if (toupper(options[OPT_NOKBD].result[0]) != 'N') {
		priv->inputs &= ~FBDEV_INP_KBD;
	}
	if (toupper(options[OPT_NOMOUSE].result[0]) != 'N') {
		priv->inputs &= ~FBDEV_INP_MOUSE;
	}
	if (toupper(options[OPT_NOINPUT].result[0]) != 'N') {
		priv->inputs &= ~(FBDEV_INP_KBD | FBDEV_INP_MOUSE);
	}
	if (toupper(options[OPT_NOVT].result[0]) != 'N') {
		priv->inputs &= ~FBDEV_INP_VT;
	}

	/* Open keyboard and mouse input */
	if (priv->inputs & FBDEV_INP_KBD) {
		ggi_linvtsw_arg vtswarg;
		int  vtnum = -1;
		char strbuf[64];
		char *inputstr = "input-linux-kbd";

		/* Fbcon handles the hw on console switch */
		vtswarg.switchaway = switched_away;
		vtswarg.switchback = switched_back;
		vtswarg.funcarg = vis;

		vtswarg.dohalt     = &priv->dohalt;
		vtswarg.autoswitch = &priv->autoswitch;
		vtswarg.onconsole = 1;
		if (getenv("GGI_NEWVT")) {
			vtswarg.forcenew = 1;
		} else {
			vtswarg.forcenew = 0;
		}

		if (_ggiAddDL(vis, "helper-linux-vtswitch", NULL, &vtswarg, 0)
		    != NULL) {
			vtnum = vtswarg.vtnum;
			priv->doswitch = vtswarg.doswitch;
		} else {
			priv->doswitch = NULL;
		}
		if (vtnum != -1) {
			sprintf(strbuf, "linux-kbd:/dev/tty%d", vtnum);
			inputstr = strbuf;
		}

		vis->input = giiOpen(inputstr, NULL);
		if (vis->input == NULL) {
			if (vtnum != -1) {
				sprintf(strbuf, "linux-kbd:/dev/vc/%d", vtnum);
				vis->input = giiOpen(inputstr, NULL);
			}
			if (vis->input == NULL) {
				fprintf(stderr,
"display-fbdev: Unable to open linux-kbd, trying stdin input.\n");
				/* We're on the Linux console so we want
				   ansikey. */
				vis->input = giiOpen("stdin:ansikey", NULL);
				if (vis->input == NULL) {
					fprintf(stderr,
"display-fbdev: Unable to open stdin input, try running with '-nokbd'.\n");
					do_cleanup(vis);
					return GGI_DL_ERROR;
				}
			}
		}
	}
	if (priv->inputs & FBDEV_INP_MOUSE) {
		gii_input *inp;
		if ((inp = giiOpen("linux-mouse:auto", &args, NULL)) != NULL) {
			vis->input = giiJoinInputs(vis->input, inp);
		}
	}

	/* Now open the framebuffer device */
	if (*devicename != '\0') {
		LIBGGI_FD(vis) = open(devicename, O_RDWR);
	} else {
		strcpy(devicename, "/dev/fb0");
		LIBGGI_FD(vis) = open(devicename, O_RDWR);
		if (LIBGGI_FD(vis) < 0) {
			strcpy(devicename, "/dev/fb/0");
			LIBGGI_FD(vis) = open(devicename, O_RDWR);
		}
	}

	if (LIBGGI_FD(vis) < 0) {
		fprintf(stderr, "display-fbdev: Couldn't open "
			"framebuffer device %s: %s\n", devicename,
			strerror(errno));
		do_cleanup(vis);
		return GGI_DL_ERROR;
	}

	/* Read original mode on framebuffer */
	if ((ioctl(LIBGGI_FD(vis),FBIOGET_FSCREENINFO,&priv->orig_fix) < 0) ||
	    (ioctl(LIBGGI_FD(vis),FBIOGET_VSCREENINFO,&priv->orig_var) < 0)) {
		perror("display-fbdev: GET_SCREENINFO");
		close(LIBGGI_FD(vis));
		LIBGGI_FD(vis) = -1;
		do_cleanup(vis);
		return GGI_DL_ERROR;
	}
	save_palette(vis);

	LIBGGI_GC(vis) = priv->normalgc = malloc(sizeof(ggi_gc));
	if (priv->normalgc == NULL) {
		do_cleanup(vis);
		return GGI_DL_ERROR;
	}

#ifdef HAVE_NEW_FBDEV
	GGIDPRINT_MISC("display-fbdev: accel: %d, supported: %d\n",
		       priv->orig_fix.accel, NUM_ACCELS);
	if (priv->orig_fix.accel >= 0 && priv->orig_fix.accel < NUM_ACCELS &&
	    accel_strings[priv->orig_fix.accel].str != NULL) {
		priv->accel
			= malloc(strlen(accel_strings[priv->orig_fix.accel].str)
				 + PREFIX_LEN);
		if (priv->accel == NULL) {
			do_cleanup(vis);
			return GGI_DL_ERROR;
		}
		sprintf(priv->accel, "%s%s", accel_prefix,
			accel_strings[priv->orig_fix.accel].str);
		vis->needidleaccel = accel_strings[priv->orig_fix.accel].async;
		GGIDPRINT_MISC("display-fbdev: Has accel: \"%s\"\n",
			       accel_strings[priv->orig_fix.accel].str);
	}
#endif /* HAVE_NEW_FBDEV */

	/* Mode management */
	vis->opdisplay->getmode   = GGI_fbdev_getmode;
	vis->opdisplay->setmode   = GGI_fbdev_setmode;
	vis->opdisplay->checkmode = GGI_fbdev_checkmode;
	vis->opdisplay->getapi    = GGI_fbdev_getapi;
	vis->opdisplay->flush     = GGI_fbdev_flush;
	vis->opdisplay->idleaccel = GGI_fbdev_idleaccel;
	vis->opdisplay->setflags  = GGI_fbdev_setflags;
	vis->opdisplay->sendevent = GGI_fbdev_sendevent;

	/* Register cleanup handler */
	ggRegisterCleanup((ggcleanup_func *)do_cleanup, vis);

	GGIDPRINT("display-fbdev: GGIdlinit success.\n");

	return GGI_DL_OPDISPLAY;
}

int GGIdlcleanup(ggi_visual *vis)
{
	return do_cleanup(vis);
}

#include <ggi/internal/ggidlinit.h>
