///-*-C++-*-//////////////////////////////////////////////////////////////////
//
// Hoard: A Fast, Scalable, and Memory-Efficient Allocator
//        for Shared-Memory Multiprocessors
// Contact author: Emery Berger, http://www.cs.utexas.edu/users/emery
//
// Copyright (c) 1998-2000, The University of Texas at Austin.
//
// This library is free software; you can redistribute it and/or modify
// it under the terms of the GNU Library General Public License as
// published by the Free Software Foundation, http://www.fsf.org.
//
// This library is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Library General Public License for more details.
//
//////////////////////////////////////////////////////////////////////////////

/*
  superblock.h
  ------------------------------------------------------------------------
  The superblock class controls a number of blocks (which are
  allocatable units of memory).
  ------------------------------------------------------------------------
  @(#) $Id: superblock.h,v 1.28 2000/02/11 13:48:02 emery Exp $
  ------------------------------------------------------------------------
  Emery Berger                    | <http://www.cs.utexas.edu/users/emery>
  Department of Computer Sciences |             <http://www.cs.utexas.edu>
  University of Texas at Austin   |                <http://www.utexas.edu>
  ========================================================================
*/

#ifndef _SUPERBLOCK_H_
#define _SUPERBLOCK_H_

#include "config.h"

#include <assert.h>
#include <new.h>
#include <stdio.h>
#include <stdlib.h>

#include "arch-specific.h"
#include "block.h"

class hoardHeap; // forward declaration


class superblock {

public:

  // Construct a superblock for a given size class,
  // and set the owner to heap o.
  superblock (int numblocks,
	      int sizeclass,
	      hoardHeap * owner);

  ~superblock (void)
    {}

  // Make (allocate or re-use) a superblock for a given size class.
  static superblock * makeSuperblock (int sizeclass);

  // Find out who allocated this superblock.
  inline hoardHeap * getOwner (void);

  // Set the superblock's owner.
  inline void setOwner (hoardHeap * o);

  // Get a block from the superblock.
  inline block * getBlock (void);

  // Put a block back in the superblock.
  inline void putBlock (block * b);

  // How many blocks are available?
  inline int getNumAvailable (void);

  // How many blocks are there, in total?
  inline int getNumBlocks (void);

  // What size class are blocks in this superblock?
  inline int getBlockSizeClass (void);

  // Insert this superblock before the next one.
  inline void insertBefore (superblock * nextSb);

  // Return the next pointer (to the next superblock in the list).
  inline superblock * const getNext (void);

  // Return the prev pointer (to the previous superblock in the list).
  inline superblock * const getPrev (void);

  // Return the 'fullness' of this superblock.
  inline int getFullness (void);
  
#if HEAP_FRAG_STATS
  // Return the amount of waste in every allocated block.
  int getMaxInternalFragmentation (void);
#endif

  // Remove this superblock from its linked list.
  inline void remove (void);

  // Is this superblock valid? (i.e.,
  // does it have the right magic number?)
  inline int isValid (void);

private:

  // Compute the 'fullness' of this superblock.
  inline int computeFullness (void);

  // Disable copying and assignment.

  superblock (const superblock&);
  const superblock& operator= (const superblock&);

  // Used for sanity checking.
  enum { SUPERBLOCK_MAGIC = 0xCAFEBABE };

#if HEAP_DEBUG
  unsigned long _magic;
#endif

  const int 	_sizeClass;	// The size class of blocks in the superblock.
  const int 	_numBlocks;	// The number of blocks in the superblock.
  int		_numAvailable;	// The number of blocks available.
  int		_fullness;	// How full is this superblock?
				// (which SUPERBLOCK_FULLNESS group is it in)
  block *	_freeList;	// A pointer to the first free block.
  hoardHeap * 	_owner;		// The heap who owns this superblock.
  superblock * 	_next;		// The next superblock in the list.
  superblock * 	_prev;		// The previous superblock in the list.

  // We insert a cache pad here to prevent false sharing with the
  // first block (which immediately follows the superblock).

  double _pad[CACHE_LINE / sizeof(double)];
};


hoardHeap * superblock::getOwner (void)
{
#if HEAP_DEBUG
  assert (_magic == SUPERBLOCK_MAGIC);
#endif
  hoardHeap * o = _owner;
  return o;
}


void superblock::setOwner (hoardHeap * o) 
{
#if HEAP_DEBUG
  assert (_magic == SUPERBLOCK_MAGIC);
#endif
  // Either the current owner is NULL or the new owner will be.
  assert ((_owner == NULL) || (o == NULL));
  _owner = o;
}


block * superblock::getBlock (void)
{
#if HEAP_DEBUG
  assert (_magic == SUPERBLOCK_MAGIC);
#endif
  // Pop off a block from this superblock's freelist,
  // if there is one available.
  if (_freeList == NULL) {
    // The freelist is empty.
    return NULL;
  }
  assert (getNumAvailable() > 0);
  block * b = _freeList;
  _freeList = _freeList->getNext();
  _numAvailable--;

  b->setNext(NULL);

  _fullness = computeFullness();

  return b;
}


void superblock::putBlock (block * b)
{
#if HEAP_DEBUG
  assert (_magic == SUPERBLOCK_MAGIC);
#endif
  // Push a block onto the superblock's freelist.
#if HEAP_DEBUG
  assert (b->isValid());
#endif
  assert (b->getSuperblock() == this);
  assert (getNumAvailable() < getNumBlocks());
  b->setNext (_freeList);
  _freeList = b;
  _numAvailable++;
  _fullness = computeFullness();
}

int superblock::getNumAvailable (void)
{
#if HEAP_DEBUG
  assert (_magic == SUPERBLOCK_MAGIC);
#endif
  return _numAvailable;
}


int superblock::getNumBlocks (void)
{
#if HEAP_DEBUG
  assert (_magic == SUPERBLOCK_MAGIC);
#endif
  return _numBlocks;
}


int superblock::getBlockSizeClass (void)
{
#if HEAP_DEBUG
  assert (_magic == SUPERBLOCK_MAGIC);
#endif
  return _sizeClass;
}


superblock * const superblock::getNext (void)
{
#if HEAP_DEBUG
  assert (_magic == SUPERBLOCK_MAGIC);
#endif
  return _next; 
}

superblock * const superblock::getPrev (void)
{
#if HEAP_DEBUG
  assert (_magic == SUPERBLOCK_MAGIC);
#endif
  return _prev; 
}


void superblock::insertBefore (superblock * nextSb) {
#if HEAP_DEBUG
  assert (_magic == SUPERBLOCK_MAGIC);
#endif
  // Insert this superblock before the next one (nextSb).
  assert (nextSb != this);
  _next = nextSb;
  if (nextSb) {
    _prev = nextSb->_prev;
    nextSb->_prev = this;
  }
}


void superblock::remove (void) {
#if HEAP_DEBUG
  assert (_magic == SUPERBLOCK_MAGIC);
#endif
  // Remove this superblock from a doubly-linked list.
  if (_next) {
    _next->_prev = _prev;
  }
  if (_prev) {
    _prev->_next = _next;
  }
  _prev = NULL;
  _next = NULL;
}


int superblock::isValid (void)
{
#if HEAP_DEBUG
  return (_magic == SUPERBLOCK_MAGIC)
    && (_numBlocks > 0)
    && (_numAvailable <= _numBlocks)
    && (_sizeClass >= 0);
#else
  return 1;
#endif
}


int superblock::computeFullness (void)
{
  return (((SUPERBLOCK_FULLNESS_GROUP - 1)
	   * (getNumBlocks() - getNumAvailable())) / getNumBlocks());
}

int superblock::getFullness (void)
{
  return _fullness;
}

#endif // _SUPERBLOCK_H_
