# File "tools.tcl":
# Tools for malaga TCL programs.

# This file is part of Malaga, a system for Left Associative Grammars.
# Copyright (C) 1995-1998 Bjoern Beutel
#
# Bjoern Beutel
# Universitaet Erlangen-Nuernberg
# Abteilung fuer Computerlinguistik
# Bismarckstrasse 12
# D-91054 Erlangen
# e-mail: malaga@linguistik.uni-erlangen.de
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#------------------------------------------------------------------------------

# Return the left border position of $widget_id in $canvas

proc widget_left {canvas widget_id} {
  return [lindex [$canvas bbox $widget_id] 0]
}

#------------------------------------------------------------------------------

# Return the right border position of $widget_id in $canvas.

proc widget_right {canvas widget_id} {
  return [lindex [$canvas bbox $widget_id] 2]
}

#------------------------------------------------------------------------------

# Return the top border position of $widget_id in $canvas.

proc widget_top {canvas widget_id} {
  return [lindex [$canvas bbox $widget_id] 1]
}

#------------------------------------------------------------------------------

# Return the bottom border position of $widget_id in $canvas.

proc widget_bottom {canvas widget_id} {
    return [lindex [$canvas bbox $widget_id] 3]
}

#------------------------------------------------------------------------------

# Read line from file $fp, ignore empty lines and comments starting with "#".

proc read_line {fp} {
  set line ""
  while {[gets $fp line] >= 0} {
    # ignore empty line and comment
    if {([string length $line] == 0) || ([string index $line 0] == "\#")} {
      continue
    }
    return $line
  }
  return $line
}

#------------------------------------------------------------------------------

# Compare $val_1 and $val_2 and return the maximum.

proc maximum {val_1 val_2} {
  if {$val_1 > $val_2} {return $val_1} else {return $val_2}
}

#------------------------------------------------------------------------------

# Export "canvas" as postscript file.
# Pop up a file requester and use "default_name".

proc export_postscript {canvas default_name} {
  set file_name [select_file "Save Postscript File as:" $default_name]
  if {$file_name == ""} {return}
    
  set canvas_bbox [lindex [$canvas configure -scrollregion] 4]
  set width [lindex $canvas_bbox 2]
  set height [lindex $canvas_bbox 3]
  
  set max_page_width 19
  set max_page_height 27
  set pixel_per_cm [expr 75 / 2.54]
  set page_width [expr $width / $pixel_per_cm]
  set page_height [expr $height / $pixel_per_cm]
  
  if {($page_width <= $max_page_width) && ($page_height <= $max_page_height)} {
    # Print original size in portrait mode
    $canvas postscript \
    -colormode mono \
    -file $file_name \
    -x 0 \
    -y 0 \
    -width $width \
    -height $height
  } elseif {($page_height <= $max_page_width) \
	    && ($page_width <= $max_page_height)} {
    # Print original size in landscape mode
    $canvas postscript \
    -colormode mono \
    -file $file_name \
    -x 0 \
    -y 0 \
    -width $width \
    -height $height \
    -rotate true
  } elseif {$page_width <= $page_height} {
    # Downscale in portrait mode
    $canvas postscript \
    -colormode mono \
    -file $file_name \
    -pageheight 27c \
    -x 0 \
    -y 0 \
    -width $width \
    -height $height
  } else {
    # Downscale in landscape mode
    $canvas postscript \
    -colormode mono \
    -file $file_name \
    -rotate true \
    -pagewidth 27c \
    -x 0 \
    -y 0 \
    -width $width \
    -height $height
  }
}

#------------------------------------------------------------------------------
