/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997, 1998, 1999  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_mutex.c,v 1.2 1999/11/23 19:01:41 hercules Exp $";
#endif

/* Semaphore functions using the Win32 API */

#include <stdio.h>
#include <stdlib.h>
#include <windows.h>

#include "SDL_error.h"
#include "SDL_mutex.h"


struct SDL_mutex {
	HANDLE id;
};

/* This is a binary semphore -- increase for a counting semaphore */
#define MAXSEMLEN	1

/* Create a semaphore */
SDL_mutex *SDL_CreateMutex(void)
{
	SECURITY_ATTRIBUTES security;
	SDL_mutex *mutex;

	/* Allocate mutex memory */
	mutex = (SDL_mutex *)malloc(sizeof(*mutex));
	if ( mutex == NULL ) {
		SDL_SetError("Out of memory");
		return(NULL);
	}

	/* Allow the semaphore to be inherited */
	security.nLength = sizeof(security);
	security.lpSecurityDescriptor = NULL;
	security.bInheritHandle = TRUE;

	/* Create the semaphore, with initial value signaled */
	mutex->id = CreateSemaphore(&security, 1, MAXSEMLEN, NULL);
	if ( mutex->id == NULL ) {
		SDL_SetError("Couldn't create semaphore");
		free(mutex);
		return(NULL);
	}
	return(mutex);
}

/* Lock the semaphore */
int SDL_mutexP(SDL_mutex *mutex)
{
	if ( WaitForSingleObject(mutex->id, INFINITE) == WAIT_FAILED ) {
		SDL_SetError("Couldn't wait on semaphore");
		return(-1);
	}
	return(0);
}

/* Unlock the semaphore */
int SDL_mutexV(SDL_mutex *mutex)
{
	if ( ReleaseSemaphore(mutex->id, 1, NULL) == FALSE ) {
		SDL_SetError("Couldn't release semaphore");
		return(-1);
	}
	return(0);
}

/* Free the semaphore */
void SDL_DestroyMutex(SDL_mutex *mutex)
{
	if ( mutex ) {
		CloseHandle(mutex->id);
		free(mutex);
	}
}
