/*
    SDL - Simple DirectMedia Layer
    Copyright (C) 1997  Sam Lantinga

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

    Sam Lantinga
    5635-34 Springhouse Dr.
    Pleasanton, CA 94588 (USA)
    slouken@devolution.com
*/

#ifdef SAVE_RCSID
static char rcsid =
 "@(#) $Id: SDL_sysaudio.cc,v 1.11 1999/07/26 13:06:20 slouken Exp $";
#endif

/* Allow access to the audio stream on BeOS */

#include <stdio.h>
#include <string.h>
#include <SoundPlayer.h>

#include "SDL_BeApp.h"

extern "C" {

#include "SDL_audio.h"
#include "SDL_audio_c.h"
#include "SDL_sysaudio.h"
#include "SDL_systhread_c.h"


static BSoundPlayer *audio = NULL;

/* The BeOS callback for handling the audio buffer */
static void FillSound(void *unused, void *stream, size_t len, 
					const media_raw_audio_format &format)
{
	/* Silence the buffer, since it's ours */
	memset(stream, audio_spec.silence, len);

	/* Only do soemthing if audio is enabled */
	if ( ! SDL_AudioEnabled )
		return;

	if ( ! SDL_AudioPaused ) {
		if ( convert.needed ) {
			SDL_LockAudio();
			(*audio_spec.callback)(audio_spec.userdata,
					(Uint8 *)convert.buf, convert.len);
			SDL_UnlockAudio();
			SDL_ConvertAudio(&convert);
			memcpy(stream, convert.buf, convert.len_cvt);
		} else {
			SDL_LockAudio();
			(*audio_spec.callback)(audio_spec.userdata,
						(Uint8 *)stream, len);
			SDL_UnlockAudio();
		}
	}
	return;
}

/* Dummy functions -- we don't use thread-based audio */
void SDL_SYS_WaitAudio(void)
{
	return;
}
void SDL_SYS_PlayAudio(void)
{
	return;
}
Uint8 *SDL_SYS_GetAudioBuf(void)
{
	return(NULL);
}

void SDL_SYS_CloseAudio(void)
{
	if ( audio ) {
		audio->Stop();
		delete audio;
		audio = NULL;
	}

	/* Quit the Be Application, if there's nothing left to do */
	SDL_QuitBeApp();
}

int SDL_SYS_OpenAudio(SDL_AudioSpec *spec)
{
	media_raw_audio_format format;

	/* Initialize the Be Application, if it's not already started */
	if ( SDL_InitBeApp() < 0 ) {
		return(-1);
	}

	/* Parse the audio format and fill the Be raw audio format */
	format.frame_rate = (float)spec->freq;
	format.channel_count = spec->channels;
	switch (spec->format&~0x1000) {
		case AUDIO_S8:
			/* Signed 8-bit audio unsupported, convert to U8 */
			spec->format = AUDIO_U8;
		case AUDIO_U8:
			format.format = media_raw_audio_format::B_AUDIO_UCHAR;
			format.byte_order = 0;
			break;
		case AUDIO_U16:
			/* Unsigned 16-bit audio unsupported, convert to S16 */
			spec->format ^= 0x8000;
		case AUDIO_S16:
			format.format = media_raw_audio_format::B_AUDIO_SHORT;
			if ( spec->format & 0x1000 ) {
				format.byte_order = 1; /* Big endian */
			} else {
				format.byte_order = 2; /* Little endian */
			}
			break;
	}
	format.buffer_size = spec->samples;
	
	/* Calculate the final parameters for this audio specification */
	SDL_CalculateAudioSpec(spec);

	/* Subscribe to the audio stream (creates a new thread) */
	{ sigset_t omask;
		SDL_MaskSignals(&omask);
		audio = new BSoundPlayer(&format, "SDL Audio", FillSound);
		SDL_UnmaskSignals(&omask);
	}
	audio->Start();
	audio->SetHasData(true);

	/* We're running! */
	return(1);
}

};	/* Extern C */
