# Arrow.w -- Arrow widget, usually part of a scrollbar
#
# Bert Bos <bert@let.rug.nl>
# Version 1.2
#
# $Id: Arrow.w,v 1.1 1996-09-25 09:23:35+02 mho Exp $

@CLASS XfwfArrow (XfwfBoard)  @file = Arrow

@ The Arrow widget is usually part of a composite scrollbar widget. It
draws a triangle pointing up, down, left or right, depending on the
|direction| resource. It has a single callback, that is repeatedly
called as long as a mouse button -- button 1 by default -- is pressed.

The triangle has a 3D shadow, the size of which can be controlled with
a resource. The shadow is either stippled or colored, depending on the
|shadowScheme| and associated resources (see the XfwfFrame widget).

@PUBLIC

@ The direction of the arrow (triangle) is given by the |direction|
resource, which is of type |Alignment|. Only |"top"| (|XfwfTop|),
|"bottom"| (|XfwfBottom|), |"left"| (|XfwfLeft|) and |"right"|
(|XfwfRight|) are valid directions. Other directions result in a
warning.

@var Alignment direction = XfwfTop

@ The color of the arrow also determines the color of the 3D shadow,
at least if |shadowScheme| is set to |XfwfAuto|, as it is by default.

@var Pixel foreground = <String> XtDefaultBackground

@ A place is needed to care for the computed shadow colors or shadow
pixmaps. 

        @var Pixel   arrowTopShadowColor = NULL
        @var Pixel   arrowBottomShadowColor = NULL
        @var Bitmap  arrowTopShadowStipple = NULL
        @var Bitmap  arrowBottomShadowStipple = NULL

@ The width of the arrow's shadow is by default 2 pixels.

@var Dimension arrowShadow = 2

@ The action should be usually repeated.

@var Boolean repeat = TRUE

@ When the user presses and then holds the mouse button, the action
function waits some milliseconds before it starts repeating the
callbacks.

@var Cardinal initialDelay = 500

@ Between repeated calls to the callback routines, the arrow widget
will wait a few milliseconds.

@var Cardinal repeatDelay = 200

@ The |callback| function is called by the |activate| action. It is
called repeatedly until the mouse button that triggered the action is
released again.

@var <Callback> XtCallbackList callback = NULL

@PRIVATE

@ The three GC's are used for drawing the arrow and its shadows.

@var GC arrowgc
@var GC arrowlightgc
@var GC arrowdarkgc

@ The repeating callback is implemented with a time out routine. The
timer is a private variable of the widget.

@var XtIntervalId timer

@ Is the arrow currently pushed?

@var Boolean pushed
  
@METHODS

@ The |initialize| method sets initial values for the three GC's and
checks the |direction| resource.

@proc initialize
{
    $pushed = FALSE;
    if ($direction != XfwfTop && $direction != XfwfLeft
	&& $direction != XfwfRight && $direction != XfwfBottom) {
	XtWarning("direction of Arrow widget incorrect; set to `top'");
	$direction = XfwfTop;
    }
    $timer = 0;
    $arrowgc = NULL; create_arrowgc($);
    /* free allocations done by superclass */
    XfwfFreeShadowGC($, $shadowScheme, $sunkengc,
		     &($sunkenShadowColor), &($sunkenShadowStipple));
    /* compute shadow colours */
    $arrowlightgc = XfwfGetShadowGC($, $shadowScheme, $beNiceToColormap,
				    1.20, XfwfLighter, &($arrowTopShadowColor),
				    &($foreground), &($arrowTopShadowStipple));
    $arrowdarkgc  = XfwfGetShadowGC($, $shadowScheme, $beNiceToColormap,
				    0.6, XfwfDarker, &($arrowBottomShadowColor),
				    &($foreground), &($arrowBottomShadowStipple));
    $sunkengc  = XfwfGetShadowGC($, $shadowScheme, $beNiceToColormap,
				 0.85, XfwfGray, &($sunkenShadowColor),
				 &($foreground), &($sunkenShadowStipple));
}

@ The |destroy| method has to free the local resources.

@proc destroy
{
    XfwfFreeShadowGC($, $shadowScheme, $arrowlightgc,
		     &($arrowTopShadowColor), &($arrowTopShadowStipple));
    XfwfFreeShadowGC($, $shadowScheme, $arrowdarkgc,
		     &($arrowBottomShadowColor), &($arrowBottomShadowStipple));
    /* sunkengc is freed by xfwfFrameWidgetClass */
    XtReleaseGC($, $arrowgc);
}

@ When the |foreground|, |arrowShadow| or |direction| resource changes,
the widget has to be redrawn. Like in the |initialize| method, the
|direction| resource needs to be checked for valid values.

If the inherited resource |shadowScheme| or one of its family changes, new
GC's need to be created.

@proc set_values
{
    Boolean need_redisplay = False;

    /* has the direction changed */
    if ($direction != XfwfTop && $direction != XfwfLeft
	&& $direction != XfwfRight && $direction != XfwfBottom) {
	XtWarning("direction of Arrow widget incorrect; set to `top'");
	$direction = XfwfTop;
    }
    if ($old$direction != $direction)
	need_redisplay = True;
    /* has something affected the current GCs? */
    if ($beNiceToColormap         != $old$beNiceToColormap
    ||  $shadowScheme             != $old$shadowScheme
    ||  $background_pixel         != $old$background_pixel
    ||  $arrowTopShadowColor      != $old$arrowTopShadowColor
    ||  $arrowTopShadowStipple    != $old$arrowTopShadowStipple
    ||  $arrowBottomShadowColor   != $old$arrowBottomShadowColor
    ||  $arrowBottomShadowStipple != $old$arrowBottomShadowStipple
    ||  $background_pixel         != $old$background_pixel
    ||  $foreground               != $old$foreground             ) {
	/* free old GCs */
	XfwfFreeShadowGC($, $old$shadowScheme, $old$arrowlightgc,
			 &($old$arrowTopShadowColor), &($old$arrowTopShadowStipple));
	XfwfFreeShadowGC($, $old$shadowScheme, $old$arrowdarkgc,
			 &($old$arrowBottomShadowColor), &($old$arrowBottomShadowStipple));
	XfwfFreeShadowGC($, $shadowScheme, $sunkengc,
			 &($sunkenShadowColor), &($sunkenShadowStipple));
	/* create new GCs */
	$arrowlightgc = XfwfGetShadowGC($, $shadowScheme, $beNiceToColormap,
					1.20, XfwfLighter, &($arrowTopShadowColor),
					&($foreground), &($arrowTopShadowStipple));
	$arrowdarkgc  = XfwfGetShadowGC($, $shadowScheme, $beNiceToColormap,
					0.6, XfwfDarker, &($arrowBottomShadowColor),
					&($foreground), &($arrowBottomShadowStipple));
	$sunkengc  = XfwfGetShadowGC($, $shadowScheme, $beNiceToColormap,
				     0.85, XfwfGray, &($sunkenShadowColor),
				     &($foreground), &($sunkenShadowStipple));
	/* change arrowGC */
	create_arrowgc($);
	need_redisplay = True;
    }
    /* shadowwidth has changed */
    if ($old$arrowShadow != $arrowShadow)
	need_redisplay = True;
    /* return if redisplay is desired */
    return need_redisplay;
}

@proc _expose
{
    Position      x, y;
    Dimension     width, height;
    XfwfArrowType type = XfwfArrowUp;

    if (! XtIsRealized($)) return;
    if (region != NULL) {
	XSetRegion(XtDisplay($), $arrowgc, region);
	XSetRegion(XtDisplay($), $arrowlightgc, region);
	XSetRegion(XtDisplay($), $arrowdarkgc, region);
    }
    $compute_inside($, &x, &y, &width, &height);
    switch ($direction) {
    case XfwfLeft:   type = XfwfArrowLeft; break;
    case XfwfBottom: type = XfwfArrowDown; break;
    case XfwfRight:  type = XfwfArrowRight; break;
    default:         break; /* default is up */
    }
    XfwfDrawArrow($, XtWindow($), $arrowlightgc, $darkgc, $sunkengc, $arrowgc,
		  x, y, width, height, $arrowShadow, type, $pushed);
    if (region != NULL) {
	XSetClipMask(XtDisplay($), $arrowgc, None);
	XSetClipMask(XtDisplay($), $arrowlightgc, None);
	XSetClipMask(XtDisplay($), $arrowdarkgc, None);
    }
}

@TRANSLATIONS

	@trans <Btn1Down>:	push_down() activate_and_start_timer()
	@trans <Btn1Up>:	push_up() stop_timer()

@ACTIONS

@ The |activate| action calls the |callback| routine once and installs
a timeout routine.

@proc activate_and_start_timer
{
    if (event->type != ButtonPress) {
        XtWarning("The Arrow activate action isn't bound to a BtnDown event");
	return;
    }
    XtCallCallbackList($, $callback, NULL);
    if ($repeat)
	$timer = XtAppAddTimeOut(XtWidgetToApplicationContext($),
				 $initialDelay, timer_callback, $);
    else
	push_up($, event, params, num_params);
}

@proc stop_timer
{
    if ($timer)
	XtRemoveTimeOut($timer);
    $timer = 0;
}

@proc push_up
{
    Position      x, y;
    Dimension     width, height;
    XfwfArrowType type = XfwfArrowUp;

    $pushed = False;

    $compute_inside($, &x, &y, &width, &height);
    switch ($direction) {
    case XfwfLeft:   type = XfwfArrowLeft; break;
    case XfwfBottom: type = XfwfArrowDown; break;
    case XfwfRight:  type = XfwfArrowRight; break;
    default:         break; /* default is up */
    }
    XfwfDrawArrow($, XtWindow($), $arrowlightgc, $darkgc, $sunkengc, $arrowgc,
		  x, y, width, height, $arrowShadow, type, $pushed);
}

@proc push_down
{
    Position      x, y;
    Dimension     width, height;
    XfwfArrowType type = XfwfArrowUp;

    $pushed = True;

    $compute_inside($, &x, &y, &width, &height);
    switch ($direction) {
    case XfwfLeft:   type = XfwfArrowLeft; break;
    case XfwfBottom: type = XfwfArrowDown; break;
    case XfwfRight:  type = XfwfArrowRight; break;
    default:         break; /* default is up */
    }
    XfwfDrawArrow($, XtWindow($), $arrowlightgc, $darkgc, $sunkengc, $arrowgc,
		  x, y, width, height, $arrowShadow, type, $pushed);
}

@UTILITIES

@ The time-out calls the |timer_callback| routine.  The routine
re-installs the time-out and calls the |callback| function (but in the
reverse order, because we do not want time-outs to overtake each
other). The delay is now |repeatDelay| instead of |initialDelay|.

@proc timer_callback(XtPointer client_data, XtIntervalId *timer)
{
    Widget $ = (Widget) client_data;

    XtCallCallbackList($, $callback, NULL);
    $timer = XtAppAddTimeOut(XtWidgetToApplicationContext($),
			     $repeatDelay, timer_callback, $);
}

@ The GC for the triangle is created by a utility function. It destroys the
old GC and then creates a new one, based on the |foreground| resource.

@proc create_arrowgc($)
{
    XtGCMask mask;
    XGCValues values;

    if ($arrowgc != NULL) XtReleaseGC($, $arrowgc);
    mask = GCForeground;
    values.foreground = $foreground;
    $arrowgc = XtGetGC($, mask, &values);
}

@IMPORTS

@ The stipple for the shadows are loaded from a bitmap file.

@incl <stdio.h>
@incl <assert.h>
