
#include "lispassert.h"
#include "infixparser.h"
#include "lispatom.h"
#include "standard.h"
#include "lisperror.h"


void InfixParser::Parse(LispPtr& aResult )
{
    ParsedObject object(*this);
    object.Parse();
    aResult = object.iResult;
    return ;
}

void LispOperators::SetOperator(LispInt aPrecedence, LispStringPtr aString)
{
    LispInFixOperator op(aPrecedence);
    SetAssociation(op, aString);
}


void LispOperators::SetRightAssociative(LispStringPtr aString)
{
    LispInFixOperator* op = LookUp(aString);
    Check(op != NULL,KLispErrNotAnInFixOperator);
    op->SetRightAssociative();
}

void LispOperators::SetLeftPrecedence(LispStringPtr aString,LispInt aPrecedence)
{
    LispInFixOperator* op = LookUp(aString);
    Check(op != NULL,KLispErrNotAnInFixOperator);
    op->SetLeftPrecedence(aPrecedence);
}

void LispOperators::SetRightPrecedence(LispStringPtr aString,LispInt aPrecedence)
{
    LispInFixOperator* op = LookUp(aString);
    Check(op != NULL,KLispErrNotAnInFixOperator);
    op->SetRightPrecedence(aPrecedence);
}


void ParsedObject::ReadToken()
{
//TODO weg    LispString token;
    // Get token.
    iLookAhead = iParser.iTokenizer.NextToken(iParser.iInput,
                                              iParser.iHashTable);
    if (iLookAhead->String()[0] == '\0')
        iEndOfFile=LispTrue;
//TODO weg    iLookAhead = iParser.iHashTable.LookUp(token.String());
}

void ParsedObject::MatchToken(LispStringPtr aToken)
{
    if (aToken != iLookAhead)
        iError=LispTrue;
    ReadToken();
}

void ParsedObject::Parse()
{
    ReadToken();
    if (iEndOfFile)
    {
        iResult.Set(LispAtom::New(iParser.iHashTable.LookUp("EndOfFile")));
        return;
    }
    ReadExpression(KMaxPrecedence);  // least precedence

    if (iLookAhead != iParser.iHashTable.LookUp(";"))
        iError = LispTrue;
    if (iError)
    {
        while ((*iLookAhead)[0] != '\0' && iLookAhead != iParser.iHashTable.LookUp(";"))
        {
            ReadToken();
        }
    }
    
    if (iError)
    {
        iResult.Set(NULL);
    }
    Check(!iError,KLispErrInvalidExpression);
}


void ParsedObject::Combine(LispInt aNrArgsToCombine)
{
    LispPtr subList;
    subList.Set(LispSubList::New(iResult.Get()));

    LispIterator iter(iResult);
    LispInt i;
    for (i=0;i<aNrArgsToCombine;i++)
    {
        if (iter() == NULL)
        {
            iError = LispTrue;
            return;
        }
        iter.GoNext();
    }
    if (iter() == NULL)
    {
        iError = LispTrue;
        return;
    }
    subList.Get()->Next().Set(iter()->Next().Get());
    iter()->Next().Set(NULL);

    InternalReverseList(subList.Get()->SubList()->Get()->Next(),
                     subList.Get()->SubList()->Get()->Next());
    iResult = subList;
}

void ParsedObject::GetOtherSide(LispInt aNrArgsToCombine, LispInt depth)
{
    LispStringPtr theOperator = iLookAhead;
    MatchToken(iLookAhead);
    ReadExpression(depth);
    InsertAtom(theOperator);
    Combine(aNrArgsToCombine);
}

void ParsedObject::InsertAtom(LispStringPtr aString)
{
    LispPtr ptr;
    ptr.Set(LispAtom::New(aString));
    ptr.Get()->Next().Set(iResult.Get());
    iResult.Set(ptr.Get());
}


void ParsedObject::ReadExpression(LispInt depth)
{
    ReadAtom();

    for(;;)
    {
        //Handle special case: a[[b]]. a is matched with lowest precedence!!
        if (iLookAhead == iParser.iHashTable.LookUp("[["))
        {
            // Match opening bracket
            MatchToken(iLookAhead);
            // Read "index" argument
            ReadExpression(KMaxPrecedence);
            // Match closing bracket
            if (iLookAhead != iParser.iHashTable.LookUp("]]"))
            {
                iError = LispTrue;
                return;
            }
            MatchToken(iLookAhead);
            // Build into Ntn(...)
            LispStringPtr theOperator = iParser.iHashTable.LookUp("Nth");
            InsertAtom(theOperator);
            Combine(2);
        }
        else
        {
            LispInFixOperator* op = iParser.iInfixOperators.LookUp(iLookAhead);
            if (op == NULL)
                return;
            if (depth < op->iPrecedence)
                return;
            LispInt upper=op->iPrecedence;
            if (!op->iRightAssociative)
                upper--;
            GetOtherSide(2,upper);
        }
    }
}



void ParsedObject::ReadAtom()
{
    LispInFixOperator* op;
    // Parse prefix operators
    op = iParser.iPrefixOperators.LookUp(iLookAhead);
    if (op != NULL)
    {
        LispStringPtr theOperator = iLookAhead;
        MatchToken(iLookAhead);
        ReadAtom();
        InsertAtom(theOperator);
        Combine(1);
    }
    // Else parse brackets
    else if (iLookAhead == iParser.iHashTable.LookUp("("))
    {
        MatchToken(iLookAhead);
        ReadExpression(KMaxPrecedence);  // least precedence
        MatchToken(iParser.iHashTable.LookUp(")"));
    }
    //Parse lists
    else if (iLookAhead == iParser.iHashTable.LookUp("{"))
    {
        LispInt nrargs=0;
        MatchToken(iLookAhead);
        while (iLookAhead != iParser.iHashTable.LookUp("}"))
        {
            ReadExpression(KMaxPrecedence);  // least precedence
            nrargs++;

            if (iLookAhead == iParser.iHashTable.LookUp(","))
            {
                MatchToken(iLookAhead);
            }
            else if (iLookAhead != iParser.iHashTable.LookUp("}"))
            {
                iError = LispTrue;
                return;
            }
        }
        MatchToken(iLookAhead);
        LispStringPtr theOperator = iParser.iHashTable.LookUp("List");
        InsertAtom(theOperator);
        Combine(nrargs);

    }
    // Parse prog bodies
    else if (iLookAhead == iParser.iHashTable.LookUp("["))
    {
        LispInt nrargs=0;
        MatchToken(iLookAhead);
        while (iLookAhead != iParser.iHashTable.LookUp("]"))
        {
            ReadExpression(KMaxPrecedence);  // least precedence
            nrargs++;

            if (iLookAhead == iParser.iHashTable.LookUp(";"))
            {
                MatchToken(iLookAhead);
            }
            else
            {
                iError = LispTrue;
                return;
            }
        }
        MatchToken(iLookAhead);
        LispStringPtr theOperator = iParser.iHashTable.LookUp("Prog");
        InsertAtom(theOperator);
        Combine(nrargs);
    }
    // Else we have an atom.
    else
    {
        LispStringPtr theOperator = iLookAhead;
        MatchToken(iLookAhead);

        LispInt nrargs=-1;
        if (iLookAhead == iParser.iHashTable.LookUp("("))
        {
            nrargs=0;
            MatchToken(iLookAhead);
            while (iLookAhead != iParser.iHashTable.LookUp(")"))
            {
                ReadExpression(KMaxPrecedence);  // least precedence
                nrargs++;

                if (iLookAhead == iParser.iHashTable.LookUp(","))
                {
                    MatchToken(iLookAhead);
                }
                else if (iLookAhead != iParser.iHashTable.LookUp(")"))
                {
                    iError = LispTrue;
                    return;
                }
            }
            MatchToken(iLookAhead);
            op = iParser.iBodiedOperators.LookUp(theOperator);
            if (op)
            {
                ReadExpression(KMaxPrecedence);  // least precedence
                nrargs++;
            }
        }
        InsertAtom(theOperator);
        if (nrargs>=0)
            Combine(nrargs);

    }

    // Parse postfix operators
    op = iParser.iPostfixOperators.LookUp(iLookAhead);
    if (op != NULL)
    {
        InsertAtom(iLookAhead);
        MatchToken(iLookAhead);
        Combine(1);
    }

}

void InfixPrinter::Print(LispPtr& aExpression, LispOutput& aOutput)
{
    Print(aExpression, aOutput, KMaxPrecedence);
}

void InfixPrinter::Print(LispPtr& aExpression, LispOutput& aOutput,
                         LispInt iPrecedence)
{
    LISPASSERT(aExpression.Get() != NULL);

    LispStringPtr string = aExpression.Get()->String();
    if (string != NULL)
    {
        aOutput.Write(string->String());
        return;
    }

    if (aExpression.Get()->Generic() != NULL)
    {
        //TODO display genericclass
        aOutput.Write(aExpression.Get()->Generic()->TypeName());
        return;
    }
    
    LispPtr* subList = aExpression.Get()->SubList();
    Check(subList!=NULL, KLispErrUnprintableToken);
    if (subList->Get() == NULL)
    {
        aOutput.Write("( )");
    }
    else
    {
        LispInt length = InternalListLength(*subList);
        string = subList->Get()->String();
        LispInFixOperator* prefix  = iPrefixOperators.LookUp(string);
        LispInFixOperator* infix   = iInfixOperators.LookUp(string);
        LispInFixOperator* postfix = iPostfixOperators.LookUp(string);
        LispInFixOperator* bodied  = iBodiedOperators.LookUp(string);
        LispInFixOperator* op = NULL;

        if (length!=2)
        {
            prefix=NULL;
            postfix=NULL;
        }
        if (length!=3)
        {
            infix=NULL;
        }
        if (prefix  )  op=prefix;
        if (postfix )  op=postfix;
        if (infix   )  op=infix;

        if (op)
        {
            LispPtr* left  = NULL;
            LispPtr* right = NULL;

            if (prefix)
            {
                right = &subList->Get()->Next();
            }
            else if (infix)
            {
                left  = &subList->Get()->Next();
                right = &subList->Get()->Next().Get()->Next();
            }
            else if (postfix)
            {
                left = &subList->Get()->Next();
            }

            if (iPrecedence < op->iPrecedence)  aOutput.Write("(");
            if (left)  Print(*left, aOutput,op->iLeftPrecedence);
            aOutput.Write(string->String());
            if (right) Print(*right, aOutput,op->iRightPrecedence);
            if (iPrecedence < op->iPrecedence)   aOutput.Write(")");
        }
        else
        {
            LispIterator iter(subList->Get()->Next());
            if (*string == "List")
            {
                aOutput.Write("{");
                while (iter())
                {
                    Print(*iter.Ptr(), aOutput, KMaxPrecedence);
                    iter.GoNext();
                    if (iter())
                        aOutput.Write(",");
                }
                aOutput.Write("}");
            }
            else if (*string == "Prog")
            {
                aOutput.Write("[");
                while (iter())
                {
                    Print(*iter.Ptr(), aOutput, KMaxPrecedence);
                    iter.GoNext();
                    aOutput.Write(";");
                }
                aOutput.Write("]");
            }
            else if (*string == "Nth")
            {
                Print(*iter.Ptr(), aOutput, 0);
                iter.GoNext();
                aOutput.Write("[[");
                Print(*iter.Ptr(), aOutput, KMaxPrecedence);
                aOutput.Write("]]");
            }
            else
            {
                aOutput.Write(string->String());
                aOutput.Write("(");

                LispIterator counter(*iter.Ptr());
                LispInt nr=0;

                while (counter())
                {
                    counter.GoNext();
                    nr++;
                }
                
                if (bodied)
                    nr--;
                while (nr--)
                {
                    Print(*iter.Ptr(), aOutput, KMaxPrecedence);
                    iter.GoNext();
                    if (nr)
                        aOutput.Write(",");
                }
                aOutput.Write(")");
                if (iter())
                    Print(*iter.Ptr(), aOutput, KMaxPrecedence);
            }
        }
    }
}


