/**************************************************************************\
 gatos (General ATI TV and Overlay Software)

  Project Coordinated By Insomnia (Steaphan Greene)
  (insomnia@core.binghamton.edu)

  Copyright (C) 1999 Steaphan Greene, yvind Aabling, Octavian Purdila, 
	Vladimir Dergachev and Christian Lupien.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111, USA.

\**************************************************************************/

#define GATOS_BT829_C 1

#include "gatos.h"
#include "bt829.h"
#include "i2c.h"

#include <stdio.h>
#include <errno.h>
#include <string.h>

/* Private global vars */
static int xtsel=0, htotal=0, vtotal=0, ldec=1, cbsense=0 ;
static u16 adelay=0, bdelay=0, hdelay=0, vdelay=22 ;
static u16 luma=0, sat_u=0, sat_v=0 ;

/* ------------------------------------------------------------------------ */
/* Initialization routines */

int bt829_init(void) {
  int has88=i2c_device(0x88), has8A=i2c_device(0x8A) ;
  gatos.bt829.addr = 0xFF ;
  switch (gatos.cardtype) {
    case CARD_STAND_ALONE:
    case CARD_NEC:
      if (has8A) bt829_register(0x8A,0,0) ;
      if (has88) bt829_register(0x88,1,0) ; break ;
    case CARD_INTEGRATED:  
      if (has88 && has8A) {
        if (has8A) bt829_register(0x8A,0,1) ;
        if (has88) bt829_register(0x88,1,0) ; }
      else {
        if (has8A) bt829_register(0x8A,1,1) ;
        if (has88) bt829_register(0x88,1,0) ; }
      break ;
    case CARD_EXTERNAL_POD:
    case CARD_ALL_IN_WONDER:
      if (has88 && has8A) {
        if (has88) bt829_register(0x88,0,0) ;
        if (has8A) bt829_register(0x8A,1,0) ; }
      else {
        if (has88) bt829_register(0x88,1,0) ;
        if (has8A) bt829_register(0x8A,1,0) ; }
      break ; 
    case CARD_INTEL_BOARD:
    case CARD_ALL_IN_WONDER_PRO:
    case CARD_ALL_IN_WONDER_128:
      if (has88) bt829_register(0x88,0,1) ;
      if (has8A) bt829_register(0x8A,1,1) ;
      break ; }
  if (gatos.bt829.addr == 0xFF) RETURN(ENODEV) ; RETURN0 ; }

int bt829_setformat(void) {

  /* NTSC, PAL or SECAM ? */
  switch (gatos.format) {
    case 2: case 4:						/* NTSC */
      if (gatos.bt829.deviceid <= BT819) RETURN(EINVAL) ;
    case 1:							/* NTSC */
      adelay = 104 ; bdelay = 93 ; xtsel = 1 ; vdelay = 22 ;
      htotal = 754 ; vtotal = 480 ; break ;
    case 5: case 7:						/* PAL */
      if (gatos.bt829.deviceid <= BT819) RETURN(EINVAL) ;
    case 3:							/* PAL */
      vdelay = (gatos.tunertype==5) ? 34 : 22 ;
      adelay = 127 ; bdelay = 114 ; xtsel = 2 ;
      htotal = 888 ; vtotal = 576 ; break ;
    case 6:							/* SECAM */
      if (gatos.bt829.deviceid <= BT819) RETURN(EINVAL) ;
      adelay = 127 ; bdelay = 160 ; xtsel = 2 ; vdelay = 34 ;
      htotal = 888 ; vtotal = 576 ; break ; }

  /* Program the Bt829 */
  bt829_iform() ;
  BTWRITE(TDEC,0x00) ;
  BTWRITE(VDELAY_LO,L(vdelay)) ;
  BTWRITE(OFORM,0x0A) ;
  BTWRITE(ADELAY,adelay) ;
  BTWRITE(BDELAY,bdelay) ;
  BTWRITE(ADC,0x82) ;
  if (gatos.bt829.deviceid >= BT827) {
    BTWRITE(SCLOOP,(gatos.format==6)?0x10:0x00) ;
    BTWRITE(WC_UP,0xCF) ;
    bt829_setCC() ;
    BTWRITE(WC_DN,0x7F) ;
    BTWRITE(P_IO,0x00) ; }
  bt829_setmux() ;
  bt829_setbrightness() ;
  bt829_setcontrast() ;
  bt829_setsaturation() ;
  bt829_sethue() ;
  RETURN0 ; }

/* ------------------------------------------------------------------------ */
/* Public routines */

int bt829_setmux(void) { bt829_ctrl() ; return bt829_iform() ; }

int bt829_setbrightness(void) {
  int b = gatos.brightness + gatos.channelinfo[gatos.channel].mod_brightness;
  b = 127*(b-100)/100 ;
  b = LIMIT(b,-128,127) ; BTWRITE(BRIGHT,b) ; RETURN0 ; }

int bt829_setcontrast(void) {
  int c = gatos.contrast + gatos.channelinfo[gatos.channel].mod_contrast;
  c = 216*c/100 ; luma = LIMIT(c,0,511) ;
  bt829_ctrl() ; BTWRITE(CONTRAST_LO,L(luma)) ; RETURN0 ; }

int bt829_setsaturation(void) {
  int s = gatos.saturation + gatos.channelinfo[gatos.channel].mod_saturation;
  s = LIMIT(s, -100, 100);
  sat_u = 254*s/100 ; sat_v = 180*s/100 ;
  bt829_ctrl() ; BTWRITE(SAT_U_LO,L(sat_u)) ; BTWRITE(SAT_V_LO,L(sat_v)) ;
  RETURN0 ; }

int bt829_sethue(void) {
  int h = gatos.hue + gatos.channelinfo[gatos.channel].mod_hue;
  h = 128*h/90 ; h = LIMIT(h,-128,127) ;
  BTWRITE(HUE,h) ; RETURN0 ; }

int bt829_setcaptsize(void) {

  u16 hscale, vscale ; u8 yci=0x60, vfilt=0 ;

  /* NTSC, PAL or SECAM ? */
  switch (gatos.format) {
    case 1: case 2: case 4:				/* NTSC */
      hdelay = gatos.xcapt*135/754 ;
      if (gatos.xcapt <= 240) vfilt = 1 ;
      if (gatos.xcapt <= 120) vfilt = 2 ;
      if (gatos.xcapt <=  60) vfilt = 3 ; break ;
    case 3: case 5: case 6: case 7:			/* PAL/SECAM */
      hdelay = gatos.xcapt*176/768 ;
      if (gatos.xcapt <= 384) vfilt = 1 ;
      if (gatos.xcapt <= 192) vfilt = 2 ;
      if (gatos.xcapt <=  96) vfilt = 3 ; break ;
    default: RETURN(EINVAL) ; }

  ldec = (gatos.xcapt > 384) ;
  cbsense = hdelay & 1 ;
  hscale = 4096*htotal/gatos.xcapt-4096 ;
  vscale = (0x10000 - (512*vtotal/gatos.ycapt-512)) & 0x1FFF ;

  bt829_crop() ; bt829_ctrl() ;
  BTWRITE(VACTIVE_LO,L(vtotal)) ;
  BTWRITE(HDELAY_LO,L(hdelay)) ;
  BTWRITE(HACTIVE_LO,L(gatos.xcapt)) ;
  BTWRITE(HSCALE_HI,H(hscale)) ;
  BTWRITE(HSCALE_LO,L(hscale)) ;
  BTWRITE(VSCALE_HI,H(vscale)|yci) ;
  BTWRITE(VSCALE_LO,L(vscale)) ;
  if (gatos.bt829.deviceid >= BT827) BTWRITE(VTC,0x18|vfilt) ;

  RETURN0 ; }

int bt829_setCC() {
  if (gatos.bt829.deviceid < BT827) {
    gatos.CCmode = 0;
    RETURN(EINVAL) ; }
  if (gatos.CCmode == 0) { /* Powering off circuitry */
    BTWRITE(CC_STATUS,0x00);
    RETURN0 ; }
  /* 0x40 is activate to set the CCVALID line. Not required yet */
  BTWRITE(CC_STATUS,(gatos.CCmode<<4)|0x40);
/* we write to STATUS to reset the CCVALID flag */
  BTWRITE(STATUS,0x00);
  RETURN0 ; }

int bt829_getCCdata(struct CCdata *data) {
  u8 status;
  data->num_valid=0;
  /* wait for buffer to be half full (means 8/16 bytes)
   * either 4 (one of CC/EDS) or 2 (both CC/EDS) frames */
  if(!(BTREAD(STATUS)&0x04)) RETURN0; /* could comment this line */
  for(;data->num_valid<CC_FIFO_SIZE;data->num_valid++) {
    status=BTREAD(CC_STATUS);
    if(!(status&0x04)) break;
    data->data[data->num_valid]= BTREAD(CC_DATA)&0x7f;
                         /* stripped high bit (parity) */
    data->status[data->num_valid]= (CCS_EDS*((status&0x02)>>1))  |
                                 (CCS_HIGH*(status&0x01)) |
                                 (CCS_OVER*((status&0x08)>>3)) |
                                 (CCS_PAR*((status&0x80)>>7)) ; }
  BTWRITE(STATUS,0x00); /* Reset CCVALID status bit */
  RETURN0 ; }

/* ------------------------------------------------------------------------ */
/* Debug and report routines */

#define DUMPREG(REG)   \
  fprintf(stderr,"%s: %-12s (0x%02X) = 0x%02X\n", \
    i2c_ident(gatos.bt829.addr),#REG,REG,BTREAD(REG))

void bt829_dumpregs(void) {
  DUMPREG(STATUS) ;
  DUMPREG(IFORM) ;
  DUMPREG(TDEC) ;
  DUMPREG(CROP) ;
  DUMPREG(VDELAY_LO) ;
  DUMPREG(VACTIVE_LO) ;
  DUMPREG(HDELAY_LO) ;
  DUMPREG(HACTIVE_LO) ;
  DUMPREG(HSCALE_HI) ;
  DUMPREG(HSCALE_LO) ;
  DUMPREG(BRIGHT) ;
  DUMPREG(CONTROL) ;
  DUMPREG(CONTRAST_LO) ;
  DUMPREG(SAT_U_LO) ;
  DUMPREG(SAT_V_LO) ;
  DUMPREG(HUE) ;
  if (gatos.bt829.deviceid >= BT827) {
    DUMPREG(SCLOOP) ;
    DUMPREG(WC_UP) ; }
  DUMPREG(OFORM) ;
  DUMPREG(VSCALE_HI) ;
  DUMPREG(VSCALE_LO) ;
  DUMPREG(TEST) ;
  DUMPREG(VPOLE) ;
  DUMPREG(IDCODE) ;
  DUMPREG(ADELAY) ;
  DUMPREG(BDELAY) ;
  DUMPREG(ADC) ;
  if (gatos.bt829.deviceid >= BT827) {
    DUMPREG(VTC) ;
    DUMPREG(CC_STATUS) ;
    DUMPREG(CC_DATA) ;
    DUMPREG(WC_DN) ;
    DUMPREG(P_IO) ; } }

/* ------------------------------------------------------------------------ */
/* Private routines */

static int bt829_register(u8 addr, int use, int vpole) {
  char *ident, name[64] ;
  gatos.bt829.addr = addr ;
  BTWRITE(SRESET,0x00) ;
  BTWRITE(VPOLE,vpole<<7) ;
  /* Chip device id, chip revision and chip ident string */ 
  gatos.bt829.deviceid = BTREAD(IDCODE) ;
  gatos.bt829.revision = gatos.bt829.deviceid & 0x0F ;
  gatos.bt829.deviceid = (gatos.bt829.deviceid & 0xF0) >> 4 ;
  switch (gatos.bt829.deviceid) {
    case BT815: ident = "Bt815A" ; break ;
    case BT817: ident = "Bt817A" ; break ;
    case BT819: ident = "Bt819A" ; break ;
    case BT827: ident = "Bt827A/B" ; break ;
    case BT829: ident = "Bt829A/B" ; break ;
    default: RETURN(ENODEV) ; break ; }
  snprintf(name,sizeof(name),
    "%s Video Decoder, Revision %d%s",ident,gatos.bt829.revision,
    (use)?"":" DISABLED") ;
  if (!use) gatos.bt829.addr = 0xFF ;
  i2c_register(addr,ident,name) ; RETURN0 ; }

static int bt829_crop(void) {
  BTWRITE(CROP,(H(vdelay)<<6)+(H(vtotal)<<4)+(H(hdelay)<<2)+H(gatos.xcapt)) ;
  RETURN0 ; }

static int bt829_ctrl(void) {
  BTWRITE(CONTROL,((gatos.mux==3)?0xC0:0x00)+
          (ldec<<5)+(cbsense<<4)+(H(luma)<<2)+(H(sat_u)<<1)+H(sat_v)) ;
  RETURN0 ; }

static int bt829_iform(void) {
  BTWRITE(IFORM,(gatos.mux<<5)|(xtsel<<3)|gatos.format) ; RETURN0 ; }
